import * as s3 from 'aws-cdk-lib/aws-s3';
import * as assets from 'aws-cdk-lib/aws-s3-assets';
import { Construct } from 'constructs';
import { IModel } from './model';
/**
 * The configuration needed to reference model artifacts.
 */
export interface ModelDataConfig {
    /**
     * The S3 path where the model artifacts, which result from model training, are stored. This path
     * must point to a single gzip compressed tar archive (.tar.gz suffix).
     */
    readonly uri: string;
}
/**
 * Model data represents the source of model artifacts, which will ultimately be loaded from an S3
 * location.
 */
export declare abstract class ModelData {
    /**
     * Constructs model data which is already available within S3.
     * @param bucket The S3 bucket within which the model artifacts are stored
     * @param objectKey The S3 object key at which the model artifacts are stored
     */
    static fromBucket(bucket: s3.IBucket, objectKey: string): ModelData;
    /**
     * Constructs model data that will be uploaded to S3 as part of the CDK app deployment.
     * @param path The local path to a model artifact file as a gzipped tar file
     * @param options The options to further configure the selected asset
     */
    static fromAsset(path: string, options?: assets.AssetOptions): ModelData;
    /**
     * This method is invoked by the SageMaker Model construct when it needs to resolve the model
     * data to a URI.
     * @param scope The scope within which the model data is resolved
     * @param model The Model construct performing the URI resolution
     */
    abstract bind(scope: Construct, model: IModel): ModelDataConfig;
}
