#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Fri Oct  4 13:47:20 2019
@author: asmith
"""
import multiprocessing
import os
import sys
from collections import Counter
from multiprocessing import SimpleQueue
from typing import List, Tuple, Union

import click
import numpy as np
import pandas as pd
import tqdm
from capcruncher.tools.deduplicate import (ReadDeduplicationParserProcess,
                                           ReadDuplicateRemovalProcess)
from capcruncher.tools.io import FastqReaderProcess, FastqWriterProcess
from capcruncher.tools.statistics import DeduplicationStatistics
from capcruncher.utils import get_file_type, load_dict, save_dict
from xopen import xopen


def parse(input_files: Tuple, output: os.PathLike = "out.json", read_buffer: int = 1e5):
    """
    Parses fastq file(s) into easy to deduplicate format.

    This command parses one or more fastq files and generates a dictionary containing
    hashed read identifiers together with hashed concatenated sequences. The hash dictionary
    is output in json format and the identify subcommand can be used to determine which read identifiers
    have duplicate sequences.

    \f
    Args:
     input_files (Tuple): One or more fastq files to process
     output (os.PathLike, optional): Output for parsed read identifiers and sequences. Defaults to "out.json".
     read_buffer (int, optional): Number of reads to process before outputting to file. Defaults to 1e5.
    """

    # Set up multiprocessing variables
    inputq = (
        multiprocessing.Queue()
    )  # Reads are placed into this queue for deduplication

    reader = FastqReaderProcess(
        input_files=input_files,
        outq=inputq,
        read_buffer=read_buffer,
    )

    parser = ReadDeduplicationParserProcess(inq=inputq, output_path=output)

    reader.start()
    parser.start()

    reader.join()
    parser.join()


def identify(input_files: Tuple, output: os.PathLike = "duplicates.json"):
    """
    Identifies fragments with duplicated sequences.

    Merges the hashed dictionaries (in json format) generated by the "parse" subcommand and
    identifies read with exactly the same sequence (share an identical hash). Duplicated read
    identifiers (hashed) are output in json format. The "remove" subcommand uses this dictionary
    to remove duplicates from fastq files.


    \f
    Args:
     input_files (Tuple): Paths to json files containing dictionaries with hashed read ids as the keys
                          and hashed sequences as the values.
     output (os.PathLike, optional): Duplicate read ids identified. Defaults to "duplicates.json".
    """

    input_files = np.array(input_files)
    np.random.shuffle(input_files)
    sequences_dedup = set()
    reads_duplicated = set()

    for ii, fn in enumerate(tqdm.tqdm(input_files)):

        input_file_type = get_file_type(fn)
        fastq_hashed_dict = load_dict(fn, format=input_file_type)

        for name_hash, sequence_hash in fastq_hashed_dict.items():

            if not sequence_hash in sequences_dedup:
                sequences_dedup.add(sequence_hash)
            else:
                reads_duplicated.add(name_hash)

    del sequences_dedup
    output_format = get_file_type(output)
    save_dict(reads_duplicated, output, format=output_format)

    return reads_duplicated


def remove(
    input_files: Tuple,
    duplicated_ids: os.PathLike,
    read_buffer: int = 1e5,
    output_prefix: os.PathLike = "",
    gzip: bool = False,
    compression_level: int = 5,
    sample_name: str = "",
    stats_prefix: os.PathLike = "",
    hash_read_name: bool = True,
    n_cores: int = 1,
):
    """
    Removes fragments with duplicated sequences from fastq files.

    Parses input fastq files and removes any duplicates from the fastq file(s) that are
    present in the json file supplied. This json dictionary should be produced by the
    "identify" subcommand.

    Statistics for the number of duplicated and unique reads are also provided.

    \f
    Args:
     input_files (Tuple): Input fastq files (in the same order as used for the parse command).
     duplicated_ids (os.PathLike): Duplicated read ids from identify command (hashed and in json format).
     read_buffer (int, optional): Number of reads to process before writing to file. Defaults to 1e5.
     output_prefix (os.PathLike, optional): Deduplicated fastq output prefix. Defaults to "".
     gzip (bool, optional): Determines if output is gzip compressed using pigz. Defaults to False.
     compression_level (int, optional): Level of compression if required (1-9). Defaults to 5.
     sample_name (str, optional): Name of sample processed e.g. DOX-treated_1. Defaults to "".
     stats_prefix (os.PathLike, optional): Output prefix for statistics. Defaults to "".

    """

    id_format = get_file_type(duplicated_ids)
    duplicated_ids_dict = load_dict(duplicated_ids, id_format)

    readq = (
        multiprocessing.Queue()
    )  # Reads are placed into this queue for deduplication
    writeq = (
        multiprocessing.Queue()
    )  # Deduplicated reads are placed into the queue for writing

    output_files = [
        f"{output_prefix}_{ii+1}.fastq{'.gz' if gzip else ''}"
        for ii in range(len(input_files))
    ]

    deduplicator = list()
    for ii in range(n_cores):
        stats_pipe = multiprocessing.Pipe()
        removal_process = ReadDuplicateRemovalProcess(
            inq=readq,
            outq=writeq,
            duplicated_ids=duplicated_ids_dict,
            stats_tx=stats_pipe[0],
            hash_read_name=hash_read_name,
        )
        stats_recv = stats_pipe[1]

        deduplicator.append((stats_recv, removal_process))

    del duplicated_ids_dict  # Reduces memory usage before starting (likely by forking) a new process

    reader = FastqReaderProcess(
        input_files=input_files,
        outq=readq,
        read_buffer=read_buffer,
    )

    writer = FastqWriterProcess(
        inq=writeq,
        output=output_files,
        compression_level=compression_level,
    )

    reader.start()
    for (conn, dedup) in deduplicator:
        dedup.start()

    writer.start()
    reader.join()


    for _ in range(n_cores):
        readq.put_nowait(None)

    stats = list()
    for (conn, dedup) in deduplicator:
        stats.append(conn.recv())
        dedup.join()

    writeq.put(None)
    writer.join()

    # Generate statistics
    df_stats = pd.DataFrame(stats)
    df_stats = df_stats.sum()
    df_stats = df_stats.to_frame("stat").rename_axis(index="stat_type").reset_index()
    df_stats["stage"] = "deduplication"
    df_stats["sample"] = sample_name
    df_stats["read_type"] = "pe"
    df_stats["read_number"] = 0
    df_stats.to_csv(f"{stats_prefix}.deduplication.csv", index=False)

    return df_stats
