"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StreamViewType = exports.ProjectionType = exports.BillingMode = exports.AttributeType = exports.Table = exports.TableEncryption = exports.Operation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const dynamodb_canned_metrics_generated_1 = require("./dynamodb-canned-metrics.generated");
const dynamodb_generated_1 = require("./dynamodb.generated");
const perms = require("./perms");
const replica_provider_1 = require("./replica-provider");
const scalable_table_attribute_1 = require("./scalable-table-attribute");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
const HASH_KEY_TYPE = 'HASH';
const RANGE_KEY_TYPE = 'RANGE';
// https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
const MAX_LOCAL_SECONDARY_INDEX_COUNT = 5;
/**
 * Supported DynamoDB table operations.
 *
 * @stability stable
 */
var Operation;
(function (Operation) {
    Operation["GET_ITEM"] = "GetItem";
    Operation["BATCH_GET_ITEM"] = "BatchGetItem";
    Operation["SCAN"] = "Scan";
    Operation["QUERY"] = "Query";
    Operation["GET_RECORDS"] = "GetRecords";
    Operation["PUT_ITEM"] = "PutItem";
    Operation["DELETE_ITEM"] = "DeleteItem";
    Operation["UPDATE_ITEM"] = "UpdateItem";
    Operation["BATCH_WRITE_ITEM"] = "BatchWriteItem";
})(Operation = exports.Operation || (exports.Operation = {}));
/**
 * What kind of server-side encryption to apply to this table.
 *
 * @stability stable
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["DEFAULT"] = "AWS_OWNED";
    TableEncryption["CUSTOMER_MANAGED"] = "CUSTOMER_MANAGED";
    TableEncryption["AWS_MANAGED"] = "AWS_MANAGED";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
class TableBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * @stability stable
         */
        this.regionalArns = new Array();
    }
    /**
     * Adds an IAM policy statement associated with this table to an IAM principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined).
     * @param actions The set of actions to allow (i.e. "dynamodb:PutItem", "dynamodb:GetItem", ...).
     * @stability stable
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                this.tableArn,
                core_1.Lazy.string({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE }),
                ...this.regionalArns,
                ...this.regionalArns.map(arn => core_1.Lazy.string({
                    produce: () => this.hasIndex ? `${arn}/index/*` : core_1.Aws.NO_VALUE,
                })),
            ],
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement associated with this table's stream to an IAM principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined).
     * @param actions The set of actions to allow (i.e. "dynamodb:DescribeStream", "dynamodb:GetRecords", ...).
     * @stability stable
     */
    grantStream(grantee, ...actions) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.tableStreamArn],
            scope: this,
        });
    }
    /**
     * Permits an IAM principal all data read operations from this table: BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantReadData(grantee) {
        return this.combinedGrant(grantee, { keyActions: perms.KEY_READ_ACTIONS, tableActions: perms.READ_DATA_ACTIONS });
    }
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined).
     * @stability stable
     */
    grantTableListStreams(grantee) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Permits an IAM principal all stream data read operations for this table's stream: DescribeStream, GetRecords, GetShardIterator, ListStreams.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantStreamRead(grantee) {
        this.grantTableListStreams(grantee);
        return this.combinedGrant(grantee, { keyActions: perms.KEY_READ_ACTIONS, streamActions: perms.READ_STREAM_DATA_ACTIONS });
    }
    /**
     * Permits an IAM principal all data write operations to this table: BatchWriteItem, PutItem, UpdateItem, DeleteItem.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantWriteData(grantee) {
        return this.combinedGrant(grantee, { keyActions: perms.KEY_WRITE_ACTIONS, tableActions: perms.WRITE_DATA_ACTIONS });
    }
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     *
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantReadWriteData(grantee) {
        const tableActions = perms.READ_DATA_ACTIONS.concat(perms.WRITE_DATA_ACTIONS);
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions });
    }
    /**
     * Permits all DynamoDB operations ("dynamodb:*") to an IAM principal.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to.
     * @stability stable
     */
    grantFullAccess(grantee) {
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions: ['dynamodb:*'] });
    }
    /**
     * Return the given named metric for this Table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/DynamoDB',
            metricName,
            dimensions: {
                TableName: this.tableName,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the consumed read capacity units this table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricConsumedReadCapacityUnits(props) {
        return this.cannedMetric(dynamodb_canned_metrics_generated_1.DynamoDBMetrics.consumedReadCapacityUnitsSum, props);
    }
    /**
     * Metric for the consumed write capacity units this table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricConsumedWriteCapacityUnits(props) {
        return this.cannedMetric(dynamodb_canned_metrics_generated_1.DynamoDBMetrics.consumedWriteCapacityUnitsSum, props);
    }
    /**
     * (deprecated) Metric for the system errors this table.
     *
     * @deprecated use `metricSystemErrorsForOperations`.
     */
    metricSystemErrors(props) {
        var _b, _c;
        if (!((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation)) {
            // 'Operation' must be passed because its an operational metric.
            throw new Error("'Operation' dimension must be passed for the 'SystemErrors' metric.");
        }
        const dimensions = {
            TableName: this.tableName,
            ...(_c = props === null || props === void 0 ? void 0 : props.dimensions) !== null && _c !== void 0 ? _c : {},
        };
        return this.metric('SystemErrors', { statistic: 'sum', ...props, dimensions });
    }
    /**
     * Metric for the user errors.
     *
     * Note that this metric reports user errors across all
     * the tables in the account and region the table resides in.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricUserErrors(props) {
        if (props === null || props === void 0 ? void 0 : props.dimensions) {
            throw new Error("'dimensions' is not supported for the 'UserErrors' metric");
        }
        // overriding 'dimensions' here because this metric is an account metric.
        // see 'UserErrors' in https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/metrics-dimensions.html
        return this.metric('UserErrors', { statistic: 'sum', ...props, dimensions: {} });
    }
    /**
     * Metric for the conditional check failed requests this table.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricConditionalCheckFailedRequests(props) {
        return this.metric('ConditionalCheckFailedRequests', { statistic: 'sum', ...props });
    }
    /**
     * How many requests are throttled on this table.
     *
     * Default: sum over 5 minutes
     *
     * @stability stable
     */
    metricThrottledRequests(props) {
        return this.cannedMetric(dynamodb_canned_metrics_generated_1.DynamoDBMetrics.throttledRequestsSum, props);
    }
    /**
     * Metric for the successful request latency this table.
     *
     * By default, the metric will be calculated as an average over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricSuccessfulRequestLatency(props) {
        var _b;
        if (!((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation)) {
            throw new Error("'Operation' dimension must be passed for the 'SuccessfulRequestLatency' metric.");
        }
        const dimensions = {
            TableName: this.tableName,
            Operation: props.dimensions.Operation,
        };
        return new cloudwatch.Metric({
            ...dynamodb_canned_metrics_generated_1.DynamoDBMetrics.successfulRequestLatencyAverage(dimensions),
            ...props,
            dimensions,
        }).attachTo(this);
    }
    /**
     * Metric for the system errors this table.
     *
     * This will sum errors across all possible operations.
     * Note that by default, each individual metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     *
     * @stability stable
     */
    metricSystemErrorsForOperations(props) {
        var _b, _c;
        if ((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation) {
            throw new Error("The Operation dimension is not supported. Use the 'operations' property.");
        }
        const operations = (_c = props === null || props === void 0 ? void 0 : props.operations) !== null && _c !== void 0 ? _c : Object.values(Operation);
        const values = this.createMetricsForOperations('SystemErrors', operations, { statistic: 'sum', ...props });
        const sum = new cloudwatch.MathExpression({
            expression: `${Object.keys(values).join(' + ')}`,
            usingMetrics: { ...values },
            color: props === null || props === void 0 ? void 0 : props.color,
            label: 'Sum of errors across all operations',
            period: props === null || props === void 0 ? void 0 : props.period,
        });
        return sum;
    }
    /**
     * Create a map of metrics that can be used in a math expression.
     *
     * Using the return value of this function as the `usingMetrics` property in `cloudwatch.MathExpression` allows you to
     * use the keys of this map as metric names inside you expression.
     *
     * @param metricName The metric name.
     * @param operations The list of operations to create metrics for.
     * @param props Properties for the individual metrics.
     * @param metricNameMapper Mapper function to allow controlling the individual metric name per operation.
     */
    createMetricsForOperations(metricName, operations, props, metricNameMapper) {
        var _b;
        const metrics = {};
        const mapper = metricNameMapper !== null && metricNameMapper !== void 0 ? metricNameMapper : (op => op.toLowerCase());
        if ((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation) {
            throw new Error('Invalid properties. Operation dimension is not supported when calculating operational metrics');
        }
        for (const operation of operations) {
            const metric = this.metric(metricName, {
                ...props,
                dimensions: {
                    TableName: this.tableName,
                    Operation: operation,
                    ...props === null || props === void 0 ? void 0 : props.dimensions,
                },
            });
            const operationMetricName = mapper(operation);
            const firstChar = operationMetricName.charAt(0);
            if (firstChar === firstChar.toUpperCase()) {
                // https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/using-metric-math.html#metric-math-syntax
                throw new Error(`Mapper generated an illegal operation metric name: ${operationMetricName}. Must start with a lowercase letter`);
            }
            metrics[operationMetricName] = metric;
        }
        return metrics;
    }
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param opts Options for keyActions, tableActions and streamActions
     */
    combinedGrant(grantee, opts) {
        if (opts.tableActions) {
            const resources = [this.tableArn,
                core_1.Lazy.string({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE }),
                ...this.regionalArns,
                ...this.regionalArns.map(arn => core_1.Lazy.string({
                    produce: () => this.hasIndex ? `${arn}/index/*` : core_1.Aws.NO_VALUE,
                }))];
            const ret = iam.Grant.addToPrincipal({
                grantee,
                actions: opts.tableActions,
                resourceArns: resources,
                scope: this,
            });
            if (this.encryptionKey && opts.keyActions) {
                this.encryptionKey.grant(grantee, ...opts.keyActions);
            }
            return ret;
        }
        if (opts.streamActions) {
            if (!this.tableStreamArn) {
                throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
            }
            const resources = [this.tableStreamArn];
            const ret = iam.Grant.addToPrincipal({
                grantee,
                actions: opts.streamActions,
                resourceArns: resources,
                scope: this,
            });
            return ret;
        }
        throw new Error(`Unexpected 'action', ${opts.tableActions || opts.streamActions}`);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ TableName: this.tableName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * Provides a DynamoDB table.
 *
 * @stability stable
 */
class Table extends TableBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.keySchema = new Array();
        this.attributeDefinitions = new Array();
        this.globalSecondaryIndexes = new Array();
        this.localSecondaryIndexes = new Array();
        this.secondaryIndexNames = new Set();
        this.nonKeyAttributes = new Set();
        this.tableScaling = {};
        this.indexScaling = new Map();
        this.globalReplicaCustomResources = new Array();
        const { sseSpecification, encryptionKey } = this.parseEncryption(props);
        let streamSpecification;
        if (props.replicationRegions) {
            if (props.stream && props.stream !== StreamViewType.NEW_AND_OLD_IMAGES) {
                throw new Error('`stream` must be set to `NEW_AND_OLD_IMAGES` when specifying `replicationRegions`');
            }
            streamSpecification = { streamViewType: StreamViewType.NEW_AND_OLD_IMAGES };
            this.billingMode = (_b = props.billingMode) !== null && _b !== void 0 ? _b : BillingMode.PAY_PER_REQUEST;
        }
        else {
            this.billingMode = (_c = props.billingMode) !== null && _c !== void 0 ? _c : BillingMode.PROVISIONED;
            if (props.stream) {
                streamSpecification = { streamViewType: props.stream };
            }
        }
        this.validateProvisioning(props);
        this.table = new dynamodb_generated_1.CfnTable(this, 'Resource', {
            tableName: this.physicalName,
            keySchema: this.keySchema,
            attributeDefinitions: this.attributeDefinitions,
            globalSecondaryIndexes: core_1.Lazy.any({ produce: () => this.globalSecondaryIndexes }, { omitEmptyArray: true }),
            localSecondaryIndexes: core_1.Lazy.any({ produce: () => this.localSecondaryIndexes }, { omitEmptyArray: true }),
            pointInTimeRecoverySpecification: props.pointInTimeRecovery ? { pointInTimeRecoveryEnabled: props.pointInTimeRecovery } : undefined,
            billingMode: this.billingMode === BillingMode.PAY_PER_REQUEST ? this.billingMode : undefined,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
            sseSpecification,
            streamSpecification,
            timeToLiveSpecification: props.timeToLiveAttribute ? { attributeName: props.timeToLiveAttribute, enabled: true } : undefined,
        });
        this.table.applyRemovalPolicy(props.removalPolicy);
        this.encryptionKey = encryptionKey;
        this.tableArn = this.getResourceArnAttribute(this.table.attrArn, {
            service: 'dynamodb',
            resource: 'table',
            resourceName: this.physicalName,
        });
        this.tableName = this.getResourceNameAttribute(this.table.ref);
        if (props.tableName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', this.tableName);
        }
        this.tableStreamArn = streamSpecification ? this.table.attrStreamArn : undefined;
        this.scalingRole = this.makeScalingRole();
        this.addKey(props.partitionKey, HASH_KEY_TYPE);
        this.tablePartitionKey = props.partitionKey;
        if (props.sortKey) {
            this.addKey(props.sortKey, RANGE_KEY_TYPE);
            this.tableSortKey = props.sortKey;
        }
        if (props.replicationRegions && props.replicationRegions.length > 0) {
            this.createReplicaTables(props.replicationRegions, props.replicationTimeout);
        }
    }
    /**
     * (deprecated) Permits an IAM Principal to list all DynamoDB Streams.
     *
     * @param grantee The principal (no-op if undefined).
     * @deprecated Use {@link #grantTableListStreams} for more granular permission
     */
    static grantListStreams(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Creates a Table construct that represents an external table via table name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableName The table's name.
     * @stability stable
     */
    static fromTableName(scope, id, tableName) {
        return Table.fromTableAttributes(scope, id, { tableName });
    }
    /**
     * Creates a Table construct that represents an external table via table arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableArn The table's ARN.
     * @stability stable
     */
    static fromTableArn(scope, id, tableArn) {
        return Table.fromTableAttributes(scope, id, { tableArn });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `TableAttributes` object.
     * @stability stable
     */
    static fromTableAttributes(scope, id, attrs) {
        class Import extends TableBase {
            constructor(_tableArn, tableName, tableStreamArn) {
                var _b, _c;
                super(scope, id);
                this.hasIndex = ((_b = attrs.globalIndexes) !== null && _b !== void 0 ? _b : []).length > 0 ||
                    ((_c = attrs.localIndexes) !== null && _c !== void 0 ? _c : []).length > 0;
                this.tableArn = _tableArn;
                this.tableName = tableName;
                this.tableStreamArn = tableStreamArn;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        let name;
        let arn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.tableName) {
            if (!attrs.tableArn) {
                throw new Error('One of tableName or tableArn is required!');
            }
            arn = attrs.tableArn;
            const maybeTableName = stack.parseArn(attrs.tableArn).resourceName;
            if (!maybeTableName) {
                throw new Error('ARN for DynamoDB table must be in the form: ...');
            }
            name = maybeTableName;
        }
        else {
            if (attrs.tableArn) {
                throw new Error('Only one of tableArn or tableName can be provided');
            }
            name = attrs.tableName;
            arn = stack.formatArn({
                service: 'dynamodb',
                resource: 'table',
                resourceName: attrs.tableName,
            });
        }
        return new Import(arn, name, attrs.tableStreamArn);
    }
    /**
     * Add a global secondary index of table.
     *
     * @param props the property of global secondary index.
     * @stability stable
     */
    addGlobalSecondaryIndex(props) {
        this.validateProvisioning(props);
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const gsiKeySchema = this.buildIndexKeySchema(props.partitionKey, props.sortKey);
        const gsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.add(props.indexName);
        this.globalSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: gsiKeySchema,
            projection: gsiProjection,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
        });
        this.indexScaling.set(props.indexName, {});
    }
    /**
     * Add a local secondary index of table.
     *
     * @param props the property of local secondary index.
     * @stability stable
     */
    addLocalSecondaryIndex(props) {
        // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
        if (this.localSecondaryIndexes.length >= MAX_LOCAL_SECONDARY_INDEX_COUNT) {
            throw new RangeError(`a maximum number of local secondary index per table is ${MAX_LOCAL_SECONDARY_INDEX_COUNT}`);
        }
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const lsiKeySchema = this.buildIndexKeySchema(this.tablePartitionKey, props.sortKey);
        const lsiProjection = this.buildIndexProjection(props);
        this.secondaryIndexNames.add(props.indexName);
        this.localSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: lsiKeySchema,
            projection: lsiProjection,
        });
    }
    /**
     * Enable read capacity scaling for this table.
     *
     * @returns An object to configure additional AutoScaling settings
     * @stability stable
     */
    autoScaleReadCapacity(props) {
        if (this.tableScaling.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'ReadScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for this table.
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     * @stability stable
     */
    autoScaleWriteCapacity(props) {
        if (this.tableScaling.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        this.tableScaling.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'WriteScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
        for (const globalReplicaCustomResource of this.globalReplicaCustomResources) {
            globalReplicaCustomResource.node.addDependency(this.tableScaling.scalableWriteAttribute);
        }
        return this.tableScaling.scalableWriteAttribute;
    }
    /**
     * Enable read capacity scaling for the given GSI.
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     * @stability stable
     */
    autoScaleGlobalSecondaryIndexReadCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this index');
        }
        return attributePair.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}ReadScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for the given GSI.
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     * @stability stable
     */
    autoScaleGlobalSecondaryIndexWriteCapacity(indexName, props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this index');
        }
        return attributePair.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}WriteScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Validate the table construct.
     *
     * @returns an array of validation error message
     * @stability stable
     */
    validate() {
        const errors = new Array();
        if (!this.tablePartitionKey) {
            errors.push('a partition key must be specified');
        }
        if (this.localSecondaryIndexes.length > 0 && !this.tableSortKey) {
            errors.push('a sort key of the table must be specified to add local secondary indexes');
        }
        if (this.globalReplicaCustomResources.length > 0 && this.billingMode === BillingMode.PROVISIONED) {
            const writeAutoScaleAttribute = this.tableScaling.scalableWriteAttribute;
            if (!writeAutoScaleAttribute) {
                errors.push('A global Table that uses PROVISIONED as the billing mode needs auto-scaled write capacity. ' +
                    'Use the autoScaleWriteCapacity() method to enable it.');
            }
            else if (!writeAutoScaleAttribute._scalingPolicyCreated) {
                errors.push('A global Table that uses PROVISIONED as the billing mode needs auto-scaled write capacity with a policy. ' +
                    'Call one of the scaleOn*() methods of the object returned from autoScaleWriteCapacity()');
            }
        }
        return errors;
    }
    /**
     * Validate read and write capacity are not specified for on-demand tables (billing mode PAY_PER_REQUEST).
     *
     * @param props read and write capacity properties
     */
    validateProvisioning(props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            if (props.readCapacity !== undefined || props.writeCapacity !== undefined) {
                throw new Error('you cannot provision read and write capacity for a table with PAY_PER_REQUEST billing mode');
            }
        }
    }
    /**
     * Validate index name to check if a duplicate name already exists.
     *
     * @param indexName a name of global or local secondary index
     */
    validateIndexName(indexName) {
        if (this.secondaryIndexNames.has(indexName)) {
            // a duplicate index name causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`a duplicate index name, ${indexName}, is not allowed`);
        }
        this.secondaryIndexNames.add(indexName);
    }
    /**
     * Validate non-key attributes by checking limits within secondary index, which may vary in future.
     *
     * @param nonKeyAttributes a list of non-key attribute names
     */
    validateNonKeyAttributes(nonKeyAttributes) {
        if (this.nonKeyAttributes.size + nonKeyAttributes.length > 100) {
            // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
            throw new RangeError('a maximum number of nonKeyAttributes across all of secondary indexes is 100');
        }
        // store all non-key attributes
        nonKeyAttributes.forEach(att => this.nonKeyAttributes.add(att));
    }
    buildIndexKeySchema(partitionKey, sortKey) {
        this.registerAttribute(partitionKey);
        const indexKeySchema = [
            { attributeName: partitionKey.name, keyType: HASH_KEY_TYPE },
        ];
        if (sortKey) {
            this.registerAttribute(sortKey);
            indexKeySchema.push({ attributeName: sortKey.name, keyType: RANGE_KEY_TYPE });
        }
        return indexKeySchema;
    }
    buildIndexProjection(props) {
        var _b, _c;
        if (props.projectionType === ProjectionType.INCLUDE && !props.nonKeyAttributes) {
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-dynamodb-projectionobject.html
            throw new Error(`non-key attributes should be specified when using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.projectionType !== ProjectionType.INCLUDE && props.nonKeyAttributes) {
            // this combination causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`non-key attributes should not be specified when not using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.nonKeyAttributes) {
            this.validateNonKeyAttributes(props.nonKeyAttributes);
        }
        return {
            projectionType: (_b = props.projectionType) !== null && _b !== void 0 ? _b : ProjectionType.ALL,
            nonKeyAttributes: (_c = props.nonKeyAttributes) !== null && _c !== void 0 ? _c : undefined,
        };
    }
    findKey(keyType) {
        return this.keySchema.find(prop => prop.keyType === keyType);
    }
    addKey(attribute, keyType) {
        const existingProp = this.findKey(keyType);
        if (existingProp) {
            throw new Error(`Unable to set ${attribute.name} as a ${keyType} key, because ${existingProp.attributeName} is a ${keyType} key`);
        }
        this.registerAttribute(attribute);
        this.keySchema.push({
            attributeName: attribute.name,
            keyType,
        });
        return this;
    }
    /**
     * Register the key attribute of table or secondary index to assemble attribute definitions of TableResourceProps.
     *
     * @param attribute the key attribute of table or secondary index
     */
    registerAttribute(attribute) {
        const { name, type } = attribute;
        const existingDef = this.attributeDefinitions.find(def => def.attributeName === name);
        if (existingDef && existingDef.attributeType !== type) {
            throw new Error(`Unable to specify ${name} as ${type} because it was already defined as ${existingDef.attributeType}`);
        }
        if (!existingDef) {
            this.attributeDefinitions.push({
                attributeName: name,
                attributeType: type,
            });
        }
    }
    /**
     * Return the role that will be used for AutoScaling
     */
    makeScalingRole() {
        // Use a Service Linked Role.
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/dynamodb.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_DynamoDBTable',
        }));
    }
    /**
     * Creates replica tables
     *
     * @param regions regions where to create tables
     */
    createReplicaTables(regions, timeout) {
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && regions.includes(stack.region)) {
            throw new Error('`replicationRegions` cannot include the region where this stack is deployed.');
        }
        const provider = replica_provider_1.ReplicaProvider.getOrCreate(this, { timeout });
        // Documentation at https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
        // is currently incorrect. AWS Support recommends `dynamodb:*` in both source and destination regions
        const onEventHandlerPolicy = new SourceTableAttachedPolicy(this, provider.onEventHandler.role);
        const isCompleteHandlerPolicy = new SourceTableAttachedPolicy(this, provider.isCompleteHandler.role);
        // Permissions in the source region
        this.grant(onEventHandlerPolicy, 'dynamodb:*');
        this.grant(isCompleteHandlerPolicy, 'dynamodb:DescribeTable');
        let previousRegion;
        let previousRegionCondition;
        for (const region of new Set(regions)) { // Remove duplicates
            // Use multiple custom resources because multiple create/delete
            // updates cannot be combined in a single API call.
            const currentRegion = new core_1.CustomResource(this, `Replica${region}`, {
                serviceToken: provider.provider.serviceToken,
                resourceType: 'Custom::DynamoDBReplica',
                properties: {
                    TableName: this.tableName,
                    Region: region,
                },
            });
            currentRegion.node.addDependency(onEventHandlerPolicy.policy, isCompleteHandlerPolicy.policy);
            this.globalReplicaCustomResources.push(currentRegion);
            // Deploy time check to prevent from creating a replica in the region
            // where this stack is deployed. Only needed for environment agnostic
            // stacks.
            let createReplica;
            if (core_1.Token.isUnresolved(stack.region)) {
                createReplica = new core_1.CfnCondition(this, `StackRegionNotEquals${region}`, {
                    expression: core_1.Fn.conditionNot(core_1.Fn.conditionEquals(region, core_1.Aws.REGION)),
                });
                const cfnCustomResource = currentRegion.node.defaultChild;
                cfnCustomResource.cfnOptions.condition = createReplica;
            }
            // Save regional arns for grantXxx() methods
            this.regionalArns.push(stack.formatArn({
                region,
                service: 'dynamodb',
                resource: 'table',
                resourceName: this.tableName,
            }));
            // We need to create/delete regions sequentially because we cannot
            // have multiple table updates at the same time. The `isCompleteHandler`
            // of the provider waits until the replica is in an ACTIVE state.
            if (previousRegion) {
                if (previousRegionCondition) {
                    // we can't simply use a Dependency,
                    // because the previousRegion is protected by the "different region" Condition,
                    // and you can't have Fn::If in DependsOn.
                    // Instead, rely on Ref adding a dependency implicitly!
                    const previousRegionCfnResource = previousRegion.node.defaultChild;
                    const currentRegionCfnResource = currentRegion.node.defaultChild;
                    currentRegionCfnResource.addMetadata('DynamoDbReplicationDependency', core_1.Fn.conditionIf(previousRegionCondition.logicalId, previousRegionCfnResource.ref, core_1.Aws.NO_VALUE));
                }
                else {
                    currentRegion.node.addDependency(previousRegion);
                }
            }
            previousRegion = currentRegion;
            previousRegionCondition = createReplica;
        }
        // Permissions in the destination regions (outside of the loop to
        // minimize statements in the policy)
        onEventHandlerPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['dynamodb:*'],
            resources: this.regionalArns,
        }));
    }
    /**
     * Whether this table has indexes.
     *
     * @stability stable
     */
    get hasIndex() {
        return this.globalSecondaryIndexes.length + this.localSecondaryIndexes.length > 0;
    }
    /**
     * Set up key properties and return the Table encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _b;
        let encryptionType = props.encryption;
        if (encryptionType != null && props.serverSideEncryption != null) {
            throw new Error('Only one of encryption and serverSideEncryption can be specified, but both were provided');
        }
        if (props.serverSideEncryption && props.encryptionKey) {
            throw new Error('encryptionKey cannot be specified when serverSideEncryption is specified. Use encryption instead');
        }
        if (encryptionType === undefined) {
            encryptionType = props.encryptionKey != null
                // If there is a configured encryptionKey, the encryption is implicitly CUSTOMER_MANAGED
                ? TableEncryption.CUSTOMER_MANAGED
                // Otherwise, if severSideEncryption is enabled, it's AWS_MANAGED; else undefined (do not set anything)
                : props.serverSideEncryption ? TableEncryption.AWS_MANAGED : undefined;
        }
        if (encryptionType !== TableEncryption.CUSTOMER_MANAGED && props.encryptionKey) {
            throw new Error('`encryptionKey cannot be specified unless encryption is set to TableEncryption.CUSTOMER_MANAGED (it was set to ${encryptionType})`');
        }
        if (encryptionType === TableEncryption.CUSTOMER_MANAGED && props.replicationRegions) {
            throw new Error('TableEncryption.CUSTOMER_MANAGED is not supported by DynamoDB Global Tables (where replicationRegions was set)');
        }
        switch (encryptionType) {
            case TableEncryption.CUSTOMER_MANAGED:
                const encryptionKey = (_b = props.encryptionKey) !== null && _b !== void 0 ? _b : new kms.Key(this, 'Key', {
                    description: `Customer-managed key auto-created for encrypting DynamoDB table at ${this.node.path}`,
                    enableKeyRotation: true,
                });
                return {
                    sseSpecification: { sseEnabled: true, kmsMasterKeyId: encryptionKey.keyArn, sseType: 'KMS' },
                    encryptionKey,
                };
            case TableEncryption.AWS_MANAGED:
                // Not specifying "sseType: 'KMS'" here because it would cause phony changes to existing stacks.
                return { sseSpecification: { sseEnabled: true } };
            case TableEncryption.DEFAULT:
                return { sseSpecification: { sseEnabled: false } };
            case undefined:
                // Not specifying "sseEnabled: false" here because it would cause phony changes to existing stacks.
                return { sseSpecification: undefined };
            default:
                throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
        }
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-dynamodb.Table", version: "1.96.0" };
/**
 * Data types for attributes within a table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes
 * @stability stable
 */
var AttributeType;
(function (AttributeType) {
    AttributeType["BINARY"] = "B";
    AttributeType["NUMBER"] = "N";
    AttributeType["STRING"] = "S";
})(AttributeType = exports.AttributeType || (exports.AttributeType = {}));
/**
 * DynamoDB's Read/Write capacity modes.
 *
 * @stability stable
 */
var BillingMode;
(function (BillingMode) {
    BillingMode["PAY_PER_REQUEST"] = "PAY_PER_REQUEST";
    BillingMode["PROVISIONED"] = "PROVISIONED";
})(BillingMode = exports.BillingMode || (exports.BillingMode = {}));
/**
 * The set of attributes that are projected into the index.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Projection.html
 * @stability stable
 */
var ProjectionType;
(function (ProjectionType) {
    ProjectionType["KEYS_ONLY"] = "KEYS_ONLY";
    ProjectionType["INCLUDE"] = "INCLUDE";
    ProjectionType["ALL"] = "ALL";
})(ProjectionType = exports.ProjectionType || (exports.ProjectionType = {}));
/**
 * When an item in the table is modified, StreamViewType determines what information is written to the stream for this table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_StreamSpecification.html
 * @stability stable
 */
var StreamViewType;
(function (StreamViewType) {
    StreamViewType["NEW_IMAGE"] = "NEW_IMAGE";
    StreamViewType["OLD_IMAGE"] = "OLD_IMAGE";
    StreamViewType["NEW_AND_OLD_IMAGES"] = "NEW_AND_OLD_IMAGES";
    StreamViewType["KEYS_ONLY"] = "KEYS_ONLY";
})(StreamViewType = exports.StreamViewType || (exports.StreamViewType = {}));
/**
 * An inline policy that is logically bound to the source table of a DynamoDB Global Tables
 * "cluster". This is here to ensure permissions are removed as part of (and not before) the
 * CleanUp phase of a stack update, when a replica is removed (or the entire "cluster" gets
 * replaced).
 *
 * If statements are added directly to the handler roles (as opposed to in a separate inline
 * policy resource), new permissions are in effect before clean up happens, and so replicas that
 * need to be dropped can no longer be due to lack of permissions.
 */
class SourceTableAttachedPolicy extends core_2.Construct {
    constructor(sourceTable, role) {
        super(sourceTable, `SourceTableAttachedManagedPolicy-${core_1.Names.nodeUniqueId(role.node)}`);
        const policy = new iam.ManagedPolicy(this, 'Resource', {
            // A CF update of the description property of a managed policy requires
            // a replacement. Use the table name in the description to force a managed
            // policy replacement when the table name changes. This way we preserve permissions
            // to delete old replicas in case of a table replacement.
            description: `DynamoDB replication managed policy for table ${sourceTable.tableName}`,
            roles: [role],
        });
        this.policy = policy;
        this.grantPrincipal = new SourceTableAttachedPrincipal(role, policy);
    }
}
/**
 * An `IPrincipal` entity that can be used as the target of `grant` calls, used by the
 * `SourceTableAttachedPolicy` class so it can act as an `IGrantable`.
 */
class SourceTableAttachedPrincipal extends iam.PrincipalBase {
    constructor(role, policy) {
        super();
        this.role = role;
        this.policy = policy;
    }
    get policyFragment() {
        return this.role.policyFragment;
    }
    addToPrincipalPolicy(statement) {
        this.policy.addStatements(statement);
        return {
            policyDependable: this.policy,
            statementAdded: true,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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