# How to Get Your API Key for Mirmer AI SDK

## Understanding Authentication

Your Mirmer AI application currently uses **Firebase Authentication**. The SDK's "API key" is actually a **Firebase ID token** from an authenticated user.

## Method 1: Get Token from Your Frontend (Easiest for Testing)

1. **Start your application:**
   ```bash
   ./start.sh
   ```

2. **Open your browser** and go to `http://localhost:5173`

3. **Sign in** with Google

4. **Open Browser DevTools:**
   - Press `F12` or right-click → "Inspect"
   - Go to the "Console" tab

5. **Get your Firebase token:**
   ```javascript
   // Paste this in the console:
   firebase.auth().currentUser.getIdToken().then(token => {
       console.log("Your API key:", token);
       // Copy this token
   });
   ```

6. **Copy the token** and use it in the SDK:
   ```python
   from mirmer import Client
   
   client = Client(
       api_key="your-copied-token-here",
       base_url="http://localhost:8001"
   )
   ```

## Method 2: Get Token Programmatically (For Production)

If you're building an application that uses the SDK, you can get the token programmatically:

```python
import firebase_admin
from firebase_admin import auth, credentials

# Initialize Firebase Admin (one time)
cred = credentials.Certificate("path/to/serviceAccountKey.json")
firebase_admin.initialize_app(cred)

# Create a custom token for a user
uid = "user-id-here"
custom_token = auth.create_custom_token(uid)

# User signs in with this token to get an ID token
# Then use that ID token with the SDK
```

## Method 3: For SDK Distribution (Recommended)

For a production SDK that others will use, you should implement a proper API key system:

### Backend Changes Needed:

1. **Create API Key Management:**
   ```python
   # backend/api_keys.py
   import secrets
   from datetime import datetime
   
   def generate_api_key():
       return f"mirmer_{secrets.token_urlsafe(32)}"
   
   def validate_api_key(api_key: str) -> bool:
       # Check against database
       # Return True if valid
       pass
   ```

2. **Update Authentication Middleware:**
   ```python
   # backend/main.py
   async def get_current_user(x_user_id: str = Header(...)):
       # Check if it's a Firebase token or API key
       if x_user_id.startswith("mirmer_"):
           # Validate API key
           user_id = validate_api_key(x_user_id)
       else:
           # Validate Firebase token
           user_id = validate_firebase_token(x_user_id)
       return user_id
   ```

3. **Add API Key Generation Endpoint:**
   ```python
   @app.post("/api/keys/generate")
   async def generate_api_key(current_user: str = Depends(get_current_user)):
       api_key = generate_api_key()
       # Store in database with user_id
       return {"api_key": api_key}
   ```

## Current Workaround for Testing

Since your backend doesn't have a dedicated API key system yet, here's how to test the SDK:

### Option A: Use a Test User ID

For local testing, you can temporarily modify your backend to accept a test user ID:

```python
# In backend/main.py, temporarily add:
@app.get("/api/test-token")
async def get_test_token():
    return {"token": "test-user-123"}  # Use this as your API key
```

Then in your SDK:
```python
client = Client(
    api_key="test-user-123",
    base_url="http://localhost:8001"
)
```

### Option B: Mock the Backend

For SDK development, create a mock server that returns test data:

```python
# sdk/tests/mock_server.py
from fastapi import FastAPI
from fastapi.responses import StreamingResponse

app = FastAPI()

@app.post("/api/conversations")
async def create_conversation():
    return {
        "id": "test-conv-123",
        "title": "Test Conversation",
        "created_at": "2024-01-01T00:00:00Z"
    }

# Add other endpoints...
```

## Recommendation

For a production SDK, I recommend:

1. **Add API Key System** to your backend
2. **Create API Key Management UI** in your frontend
3. **Document API Key Generation** in your SDK docs
4. **Keep Firebase Auth** for your web app
5. **Use API Keys** for SDK/programmatic access

This way:
- Web users use Firebase (Google Sign-In)
- SDK users use API keys (programmatic access)
- Both are secure and properly authenticated

## Need Help?

If you want to implement the API key system, I can help you:
1. Add API key generation to your backend
2. Create a management UI in your frontend
3. Update the SDK documentation

Let me know if you'd like me to implement this!
