# Mirmer AI CLI

Command-line interface for the Mirmer AI multi-LLM consultation system.

## Quick Start

```bash
# Install
pip install mirmer-ai

# Sign in
mirmer login

# Ask a question
mirmer query "What is the meaning of life?"
```

## Features

- 🔐 **Browser-based Google Sign-In** - Secure OAuth flow
- 💾 **Persistent Authentication** - Sign in once, use everywhere
- 🌊 **Real-time Streaming** - Watch the council process live
- 💬 **Conversation Management** - List, search, and continue conversations
- 📊 **Usage Tracking** - Monitor your API consumption
- 🎨 **Beautiful Output** - Colored, formatted terminal output

## Installation

```bash
pip install mirmer-ai
```

## Authentication

### Sign In

```bash
mirmer login
```

This will:
1. Open your browser
2. Prompt for Google Sign-In
3. Store your credentials locally
4. Return to the terminal

### Check Status

```bash
mirmer whoami
```

### Sign Out

```bash
mirmer logout
```

## Usage

### Asking Questions

**Basic query:**
```bash
mirmer query "What is artificial intelligence?"
```

**Continue a conversation:**
```bash
mirmer query "Tell me more about that" -c conv-abc123
```

**Disable streaming:**
```bash
mirmer query "Quick question" --no-stream
```

### Managing Conversations

**List all conversations:**
```bash
mirmer conversations
```

**Search conversations:**
```bash
mirmer search "machine learning"
```

### Usage Statistics

**Check your usage:**
```bash
mirmer usage
```

Output:
```
📊 Usage Statistics:
  Tier: pro
  Used: 45/100 queries
  Reset: 2024-12-03T00:00:00Z
  [█████████████░░░░░░░░░░░░░░░░░] 45.0%
```

## Examples

### Interactive Session

```bash
# Sign in
$ mirmer login
🔐 Authenticating with Mirmer AI...
Opening browser for Google Sign-In...
✅ Authentication successful!

# Ask a question
$ mirmer query "Explain quantum computing"
🚀 Starting council process...

Stage 1: Collecting responses...
✓ Stage 1: 3 responses received
Stage 2: Peer ranking...
✓ Stage 2: 3 rankings received
Stage 3: Chairman synthesis...
✓ Stage 3: Complete

============================================================
FINAL ANSWER:
============================================================
Quantum computing is a revolutionary approach to computation...
[detailed answer]

# Check usage
$ mirmer usage
📊 Usage Statistics:
  Tier: free
  Used: 1/10 queries
  Reset: 2024-12-03T00:00:00Z
  [███░░░░░░░░░░░░░░░░░░░░░░░░░░░] 10.0%
```

### Scripting

```bash
#!/bin/bash
# analyze.sh - Analyze multiple topics

topics=("AI" "blockchain" "quantum computing")

for topic in "${topics[@]}"; do
    echo "Analyzing: $topic"
    mirmer query "Explain $topic in simple terms" --no-stream > "${topic}.txt"
done
```

### Python Integration

```python
import subprocess
import json

# Use CLI from Python
result = subprocess.run(
    ["mirmer", "query", "What is AI?", "--no-stream"],
    capture_output=True,
    text=True
)

print(result.stdout)
```

## Configuration

### Custom API URL

For testing or self-hosted instances:

```bash
mirmer --base-url http://localhost:8001 query "test"
```

### Environment Variables

```bash
# Manual token (skips browser auth)
export MIRMER_API_KEY="your-firebase-token"

# Use CLI
mirmer query "What is AI?"
```

## Command Reference

### Global Options

```
--base-url TEXT  Base URL of Mirmer AI API (default: https://mirmer.ai)
--help           Show help message
```

### Commands

#### `login`
Sign in with Google via browser.

```bash
mirmer login
```

#### `logout`
Sign out and clear stored credentials.

```bash
mirmer logout
```

#### `whoami`
Show current authentication status.

```bash
mirmer whoami
```

#### `query`
Send a query to the Mirmer AI council.

```bash
mirmer query MESSAGE [OPTIONS]
```

**Options:**
- `-c, --conversation-id TEXT` - Continue existing conversation
- `--stream / --no-stream` - Enable/disable streaming (default: enabled)

**Examples:**
```bash
mirmer query "What is AI?"
mirmer query "Tell me more" -c conv-123
mirmer query "Quick question" --no-stream
```

#### `conversations`
List all your conversations.

```bash
mirmer conversations
```

#### `search`
Search conversations by content.

```bash
mirmer search QUERY
```

**Example:**
```bash
mirmer search "machine learning"
```

#### `usage`
Show API usage statistics.

```bash
mirmer usage
```

## Credentials Storage

Credentials are stored in `~/.mirmer/credentials.json` with restricted permissions (chmod 600).

**Location:**
- Linux/Mac: `~/.mirmer/credentials.json`
- Windows: `%USERPROFILE%\.mirmer\credentials.json`

**Format:**
```json
{
  "token": "firebase-id-token",
  "email": "user@example.com"
}
```

## Troubleshooting

### Authentication Issues

**Problem:** "Authentication failed or timed out"

**Solutions:**
- Check your internet connection
- Make sure the backend is accessible
- Try `mirmer logout` then `mirmer login` again
- Check if port 8765 is available

### Token Expired

**Problem:** "Invalid API key or unauthorized access"

**Solution:**
```bash
mirmer logout
mirmer login
```

Firebase tokens expire after 1 hour. Re-authenticate when needed.

### Port Already in Use

**Problem:** "Port 8765 already in use"

**Solutions:**
- Wait a few seconds and try again
- Close any application using port 8765
- Or use manual token: `export MIRMER_API_KEY="your-token"`

### Connection Errors

**Problem:** "Failed to connect to API"

**Solutions:**
- Check if the backend is running
- Verify the `--base-url` is correct
- Check your internet connection

## Development

### Local Testing

```bash
# Clone the repo
git clone https://github.com/mirmer-ai/python-sdk
cd sdk

# Install in development mode
pip install -e .

# Test with local backend
mirmer --base-url http://localhost:8001 query "test"
```

### Building from Source

```bash
# Install build tools
pip install build

# Build package
python -m build

# Install locally
pip install dist/mirmer_ai-*.whl
```

## Support

- **Documentation:** https://docs.mirmer.ai
- **Issues:** https://github.com/mirmer-ai/python-sdk/issues
- **Email:** support@mirmer.ai

## License

MIT License - See LICENSE file for details
