from .base import BaseGraph
from .utils import human_readable_number
from .utils import calculate_ticks
from .utils import match_ticks
from .utils import estimate_text_dimensions
import math
import numbers


def stacked_bar_range(data, series_types, secondary, maximum, minimum):
    non_secondary_bars_to_use = [
        not secondary[index] and series_types[index][0] == "bar"
        for index in range(len(secondary))
    ]

    non_secondary_non_bars_to_use = [
        not secondary[index] and series_types[index][0] != "bar"
        for index in range(len(secondary))
    ]

    if any(non_secondary_non_bars_to_use):
        non_secondary_non_bar_values = [
            value
            for values, non_bar_non_secondary in zip(
                data, non_secondary_non_bars_to_use
            )
            if non_bar_non_secondary
            for value in values
            if value is not None  # Exclude None values
        ]
        max_non_secondary_non_bar = max(non_secondary_non_bar_values)
        min_non_secondary_non_bar = min(non_secondary_non_bar_values)
        if maximum is not None:
            maximum = max(max_non_secondary_non_bar, maximum)
        else:
            maximum = max_non_secondary_non_bar
        if minimum is not None:
            minimum = min(min_non_secondary_non_bar, minimum)
        else:
            minimum = min_non_secondary_non_bar

    stacked_positive_data = [
        sum(
            max(value or 0, 0)
            for i, value in enumerate(column)
            if non_secondary_bars_to_use[i] and value is not None
        )
        for column in zip(*data)
    ]

    stacked_negative_data = [
        sum(
            min(value or 0, 0)
            for i, value in enumerate(column)
            if non_secondary_bars_to_use[i] and value is not None
        )
        for column in zip(*data)
    ]

    if stacked_positive_data and stacked_negative_data:
        # Only proceed if there are valid (non-None) data points
        if maximum is None:
            maximum = max(stacked_positive_data)
        if minimum is None:
            minimum = min(stacked_negative_data)
        return (
            min(min(filter(lambda x: x is not None, stacked_negative_data)), minimum),
            max(max(filter(lambda x: x is not None, stacked_positive_data)), maximum),
        )
    else:
        return (None, None)


def non_secondary_range(data, secondary, maximum, minimum):
    # Filter out None values and then compute the range for non-secondary values
    non_secondary_values = [
        value
        for values, is_secondary in zip(data, secondary)
        if not is_secondary
        for value in values
        if value is not None  # Exclude None values
    ]

    if non_secondary_values:
        # Proceed only if there are valid (non-None) data points
        if maximum is None:
            maximum = max(non_secondary_values)
        if minimum is None:
            minimum = min(non_secondary_values)
        return (
            min(non_secondary_values + [minimum]),
            max(non_secondary_values + [maximum]),
        )
    else:
        return (None, None)


class CategoricalGraph(BaseGraph):
    """
    The graphs generated by this class should have a categorical x-axis (e.g.
    user accounts). They can have a primary and secondary y-axis with different
    scales. Each series can display as bars (stacked or side by side), a line,
    or dots.
    """

    def __init__(
        self,
        width=300,
        height=200,
        bar_width=30,
        padding=20,
        x_padding=None,
        y_padding=None,
        y_top_padding=None,
        y_bottom_padding=None,
        x_left_padding=None,
        x_right_padding=None,
        colors=None,
        stacked=False,
        num_y_ticks=5,
        x_axis_label=None,
        primary_y_axis_label=None,
        secondary_y_axis_label=None,
        show_legend=True,
        rotate_x_labels=True,
        background_color=None,
        dark_mode=None,
        title=None,
        title_font_size=None,
        element_spacing=None,
        watermark=None,
        font_width_estimate_multiplier=1,
        scale_max=None,
        scale_min=None,
        secondary_scale_max=None,
        secondary_scale_min=None,
        primary_tick_prefix="",
        primary_tick_suffix="",
        secondary_tick_prefix="",
        secondary_tick_suffix="",
        legend_position="right",
        line_curvature=0,
    ):
        super().__init__(
            width=width,
            height=height,
            padding=padding,
            x_padding=x_padding,
            y_padding=y_padding,
            y_top_padding=y_top_padding,
            y_bottom_padding=y_bottom_padding,
            x_left_padding=x_left_padding,
            x_right_padding=x_right_padding,
            colors=colors,
            num_y_ticks=num_y_ticks,
            x_axis_label=x_axis_label,
            primary_y_axis_label=primary_y_axis_label,
            secondary_y_axis_label=secondary_y_axis_label,
            show_legend=show_legend,
            rotate_x_labels=rotate_x_labels,
            background_color=background_color,
            dark_mode=dark_mode,
            title=title,
            title_font_size=title_font_size,
            element_spacing=element_spacing,
            watermark=watermark,
            font_width_estimate_multiplier=font_width_estimate_multiplier,
        )
        self.stacked = stacked
        self.bar_width = bar_width
        self.scale_max = scale_max
        self.scale_min = scale_min
        self.secondary_scale_max = secondary_scale_max
        self.secondary_scale_min = secondary_scale_min
        self.primary_tick_prefix = primary_tick_prefix
        self.primary_tick_suffix = primary_tick_suffix
        self.secondary_tick_prefix = secondary_tick_prefix
        self.secondary_tick_suffix = secondary_tick_suffix
        self.legend_position = legend_position
        assert (
            line_curvature >= 0 and line_curvature <= 0.5
        ), f"Invalid line_curvature value: {line_curvature}. Must be between 0 and 0.5."
        self.line_curvature = line_curvature
        self.x_labels = []
        self.series_types = []
        self.secondary = []
        self.horizontal_lines = []
        self.vertical_lines = []
        self.stroke_width = []

    def add_series(
        self,
        series,
        legend_label=None,
        series_type="bar",
        print_values=False,
        secondary=False,
        stroke_width=1,
    ):
        # Deal with NaN values besides None (e.g. np.nan)
        cleaned_series = [
            (
                value
                if isinstance(value, numbers.Number) and not math.isnan(value)
                else None
            )
            for value in series
        ]
        self.data.append(cleaned_series)
        self.legend_labels.append(legend_label or None)
        self.series_types.append((series_type, print_values))
        self.secondary.append(secondary)
        self.stroke_width.append(stroke_width)

    def add_horizontal_line(
        self,
        y,
        color=None,
        label=None,
        label_x_position="right",
        label_y_position="top",
        stroke_width="1",
    ):
        if color is None:
            color = self.text_color
        assert label_x_position in ["left", "right", "center"], (
            f"Invalid label position: {label_x_position}. "
            + "Must be 'left', 'right', or 'center'."
        )
        assert label_y_position in ["top", "bottom"], (
            f"Invalid label position: {label_y_position}. "
            + "Must be 'top' or 'bottom'."
        )
        self.horizontal_lines.append(
            (y, color, stroke_width, label, label_x_position, label_y_position)
        )

    def add_vertical_line(
        self,
        x,
        color=None,
        label=None,
        label_x_position="right",
        label_y_position="top",
        rotate_label=False,
        stroke_width="1",
    ):
        if color is None:
            color = self.text_color
        assert label_x_position in ["left", "right"], (
            f"Invalid label position: {label_x_position}. "
            + "Must be 'left' or 'right'."
        )
        assert label_y_position in ["top", "bottom", "center"], (
            f"Invalid label position: {label_y_position}. "
            + "Must be 'top', 'bottom', or 'center'."
        )
        self.vertical_lines.append(
            (
                x,
                color,
                stroke_width,
                label,
                label_x_position,
                label_y_position,
                rotate_label,
            )
        )

    def _draw_bar(self, x, y, width, height, fill):
        assert x >= 0, f"Bars cannot start below 0: {x}"
        if height == 0:
            return ""
        if height < 0:
            y += height
            height *= -1
        return (
            f'<rect x="{x}" y="{y}" width="{width}" height="{height}" fill="{fill}" />'
        )

    def _draw_dot(self, x, y, fill, radius=5, stroke=None, stroke_width=1):
        if stroke is None:
            stroke_parameter = ""
        else:
            stroke_parameter = f'stroke="{stroke}" stroke-width="{stroke_width}"'
        self.most_extreme_dimensions["left"] = min(
            self.most_extreme_dimensions["left"], x - radius
        )
        self.most_extreme_dimensions["right"] = max(
            self.most_extreme_dimensions["right"], x + radius
        )
        self.most_extreme_dimensions["top"] = min(
            self.most_extreme_dimensions["top"], y - radius
        )
        self.most_extreme_dimensions["bottom"] = max(
            self.most_extreme_dimensions["bottom"], y + radius
        )
        return f'<circle cx="{x}" cy="{y}" r="{radius}" fill="{fill}" {stroke_parameter} />'

    def _draw_line(self, x1, y1, x2, y2, stroke="black", stroke_width="1"):
        self.most_extreme_dimensions["left"] = min(
            self.most_extreme_dimensions["left"], x1, x2
        )
        self.most_extreme_dimensions["right"] = max(
            self.most_extreme_dimensions["right"], x1, x2
        )
        self.most_extreme_dimensions["top"] = min(
            self.most_extreme_dimensions["top"], y1, y2
        )
        self.most_extreme_dimensions["bottom"] = max(
            self.most_extreme_dimensions["bottom"], y1, y2
        )
        return f'<line x1="{x1}" y1="{y1}" x2="{x2}" y2="{y2}" stroke="{stroke}" stroke-width="{stroke_width}" />'

    def _make_line_path(self, points, curvature=0):
        path_data = ""
        current_path = []

        def append_current_path():
            nonlocal path_data, current_path
            if not current_path:
                return

            path_data += f" M {current_path[0][0]} {current_path[0][1]}"
            if len(current_path) == 1:
                return  # Only one point, nothing more to draw

            if curvature == 0:
                # Draw straight lines between points
                for i in range(1, len(current_path)):
                    p1 = current_path[i]
                    path_data += f" L {p1[0]} {p1[1]}"
            else:
                for i in range(1, len(current_path)):
                    p0 = current_path[i - 1]
                    p1 = current_path[i]

                    if i < len(current_path) - 1:
                        p2 = current_path[i + 1]
                        is_valley = p0[1] <= p1[1] >= p2[1]
                        is_peak = p0[1] >= p1[1] <= p2[1]

                        if is_valley or is_peak:
                            # Handle peaks and valleys with a flatter approach
                            ctrl1_x = p0[0] + (p1[0] - p0[0]) * curvature
                            ctrl1_y = p1[1]  # Keep Y same to prevent overshoot
                            ctrl2_x = p1[0] - (p2[0] - p1[0]) * curvature
                            ctrl2_y = p1[1]  # Keep Y same to prevent overshoot
                        else:
                            # Adapt curvature based on the slope and overall direction
                            delta_x1 = p1[0] - p0[0]
                            delta_y1 = p1[1] - p0[1]
                            delta_x2 = p2[0] - p1[0]
                            delta_y2 = p2[1] - p1[1]
                            ctrl1_x = p0[0] + delta_x1 * curvature
                            ctrl1_y = p0[1] + delta_y1 * curvature
                            ctrl2_x = p1[0] - delta_x2 * curvature
                            ctrl2_y = p1[1] - delta_y2 * curvature
                    else:
                        # Last point, use the slope from the previous points
                        delta_x = p1[0] - p0[0]
                        delta_y = p1[1] - p0[1]
                        ctrl1_x = p0[0] + delta_x * curvature
                        ctrl1_y = p0[1] + delta_y * curvature
                        ctrl2_x = p1[0] + delta_x * curvature
                        ctrl2_y = p1[1] + delta_y * curvature

                    if i == 1:
                        path_data += f" C {ctrl1_x} {ctrl1_y} {ctrl2_x} {ctrl2_y} {p1[0]} {p1[1]}"
                    else:
                        path_data += f" S {ctrl2_x} {ctrl2_y} {p1[0]} {p1[1]}"

            current_path = []

        for point in points:
            if point is None:
                append_current_path()
            else:
                current_path.append(point)

        append_current_path()
        return path_data

    def _draw_line_path(self, points, stroke="black", stroke_width=1, curvature=0):
        path_data = self._make_line_path(points, curvature)
        return f'<path d="{path_data}" stroke="{stroke}" stroke-width="{stroke_width}" fill="none"/>'

    def _make_bar_path(self, bars):
        path_data = ""
        for x, y, width, height in bars:
            path_data += f"M {x} {y} h {width} v {height} h -{width} z "
        return path_data

    def _draw_bar_path(self, bars, fill):
        path_data = self._make_bar_path(bars)
        return f'<path d="{path_data}" fill="{fill}" />'

    def _make_dot_path(self, dots, radius=5):
        path_data = ""
        for x, y in dots:
            path_data += (
                f"M {x-radius} {y} "  # Move to the left edge of the circle
                f"a {radius},{radius} 0 1,0 {2*radius},0 "  # Draw the first arc
                f"a {radius},{radius} 0 1,0 -{2*radius},0 "  # Draw the second arc to complete the circle
            )
        return path_data

    def _draw_dot_path(self, dots, fill, radius=5):
        path_data = self._make_dot_path(dots, radius)
        return f'<path d="{path_data}" fill="{fill}" />'

    def render(self):
        self._reset_graph()
        self.line_paths = {}
        self.bar_paths = {}
        self.dot_paths = {}
        graph_width = self.width
        has_secondary = any(self.secondary)
        max_value_secondary = None
        min_value_secondary = None
        if self.stacked:
            bar_series_indices = [
                i for i, series in enumerate(self.series_types) if series[0] == "bar"
            ]

            assert all(self.secondary[i] for i in bar_series_indices) or not any(
                self.secondary[i] for i in bar_series_indices
            ), "All stacked bar series must be either primary or secondary."

            min_value_primary, max_value_primary = stacked_bar_range(
                self.data,
                self.series_types,
                self.secondary,
                self.scale_max,
                self.scale_min,
            )
            if has_secondary:
                min_value_secondary, max_value_secondary = stacked_bar_range(
                    self.data,
                    self.series_types,
                    [not sec for sec in self.secondary],
                    self.secondary_scale_max,
                    self.secondary_scale_min,
                )
        else:
            min_value_primary, max_value_primary = non_secondary_range(
                self.data, self.secondary, self.scale_max, self.scale_min
            )
            if has_secondary:
                min_value_secondary, max_value_secondary = non_secondary_range(
                    self.data,
                    [not sec for sec in self.secondary],
                    self.secondary_scale_max,
                    self.secondary_scale_min,
                )

        primary_ticks = calculate_ticks(
            min_value_primary,
            max_value_primary,
            include_zero=True,
            target_tick_count=self.num_y_ticks,
        )

        if has_secondary:
            secondary_ticks = calculate_ticks(
                min_value_secondary,
                max_value_secondary,
                include_zero=True,
                target_tick_count=self.num_y_ticks,
            )

            primary_ticks, secondary_ticks = match_ticks(primary_ticks, secondary_ticks)

            adjusted_max_value_secondary = secondary_ticks[-1]
            adjusted_min_value_secondary = secondary_ticks[0]

        adjusted_max_value_primary = primary_ticks[-1]
        adjusted_min_value_primary = primary_ticks[0]

        num_bars = 0
        if not self.stacked:
            for type, _ in self.series_types:
                if type == "bar":
                    num_bars += 1
        num_bars = max(num_bars, 1)
        max_bar_width = graph_width / (num_bars * (len(self.data[0]) + 0.5))

        bar_width = min(max_bar_width, self.bar_width)

        scale_primary = (self.height) / (
            adjusted_max_value_primary - adjusted_min_value_primary
        )
        if has_secondary:
            scale_secondary = (self.height) / (
                adjusted_max_value_secondary - adjusted_min_value_secondary
            )
        else:
            scale_secondary = None

        # Draw series
        bar_spacing = (self.width) / (len(self.data[0]) + 0.5)
        bar_series_across = (
            1
            if self.stacked
            else len(
                [
                    series_type
                    for series_type, _ in self.series_types
                    if series_type == "bar"
                ]
            )
        )
        total_bars_width = bar_series_across * bar_width

        num_categories = len(self.data[0])
        num_series = len(self.data)
        positive_bar_heights = [0] * num_categories
        negative_bar_heights = [0] * num_categories

        for sub_index in range(num_categories):
            bar_count = 0
            for index in range(num_series):
                value = self.data[index][sub_index]

                secondary_value = self.secondary[index]

                series_type, print_values = self.series_types[index]

                if value is None:
                    if series_type == "line":
                        if index not in self.line_paths:
                            self.line_paths[index] = []
                        self.line_paths[index].append(None)

                    continue

                if series_type == "dot" or series_type == "line" or self.stacked:
                    x = (0.5 + sub_index) * bar_spacing + (bar_spacing - bar_width) / 2
                else:
                    # Calculate the starting x-position of the bars in each category
                    start_x = (0.5 + sub_index) * bar_spacing + (
                        bar_spacing - total_bars_width
                    ) / 2
                    x = (
                        start_x + bar_count * bar_width - bar_width / 2
                    )  # Adjusting by half of the bar width
                    bar_count += 1
                scale = scale_secondary if secondary_value else scale_primary
                min_value = (
                    adjusted_min_value_secondary
                    if secondary_value
                    else adjusted_min_value_primary
                )
                y = self.height - (value - min_value) * scale

                if series_type == "bar" and self.stacked:
                    bar_height = value * scale
                    x -= bar_width / 2
                    if value >= 0:
                        y -= positive_bar_heights[sub_index]
                        positive_bar_heights[sub_index] += bar_height
                    else:
                        y -= negative_bar_heights[sub_index]
                        negative_bar_heights[sub_index] += bar_height
                    if index not in self.bar_paths:
                        self.bar_paths[index] = []
                    self.bar_paths[index].append((x, y, bar_width, bar_height))

                    # This is how we would draw bars individually
                    # self.svg_elements.append(
                    #     self._draw_bar(x, y, bar_width, bar_height, self.colors[index])
                    # )

                elif series_type == "bar":
                    bar_height = value * scale
                    if index not in self.bar_paths:
                        self.bar_paths[index] = []
                    self.bar_paths[index].append((x, y, bar_width, bar_height))

                    # self.svg_elements.append(
                    #     self._draw_bar(
                    #         x,
                    #         y,
                    #         bar_width,
                    #         value * scale,
                    #         self.colors[index],
                    #     )
                    # )

                elif series_type == "dot":
                    center_x = (
                        (sub_index + 0.5) * bar_spacing
                        + (bar_spacing - total_bars_width) / 2
                        + bar_width * (bar_series_across - 1) / 2
                    )
                    if index not in self.dot_paths:
                        self.dot_paths[index] = []
                    self.dot_paths[index].append((center_x, y))

                    # self.svg_elements.append(
                    #     self._draw_dot(
                    #         center_x,
                    #         y,
                    #         radius=5,
                    #         fill=self.colors[index],
                    #     )
                    # )

                elif series_type == "line":
                    if index not in self.line_paths:
                        self.line_paths[index] = []
                    self.line_paths[index].append((x, y))

                if print_values:
                    if series_type == "dot":
                        value_x = center_x
                    elif series_type == "line":
                        value_x = x
                    else:
                        value_x = x + bar_width / 2

                    value_y = y - 5 if series_type == "bar" else y - 10
                    self.svg_elements.append(
                        self._generate_text(
                            value, value_x, value_y, fill=self.text_color
                        )
                    )

        # Draw bars
        for index, bars in self.bar_paths.items():
            self.svg_elements.append(self._draw_bar_path(bars, self.colors[index]))

        # Draw paths
        for index, points in self.line_paths.items():
            self.svg_elements.append(
                self._draw_line_path(
                    points,
                    stroke=self.colors[index],
                    curvature=self.line_curvature,
                    stroke_width=self.stroke_width[index],
                )
            )

        # Draw dots
        for index, dots in self.dot_paths.items():
            self.svg_elements.append(self._draw_dot_path(dots, fill=self.colors[index]))

        # Draw horizontal lines
        for (
            y,
            color,
            stroke_width,
            label,
            label_x_position,
            label_y_position,
        ) in self.horizontal_lines:
            y_svg = self.height - (y - adjusted_min_value_primary) * scale_primary
            self.svg_elements.append(
                f'<line x1="0" y1="{y_svg}" x2="{self.width}" y2="{y_svg}" stroke="{color}" stroke-width="{stroke_width}" />'
            )
            if label:
                padding = 5  # Padding from the edge
                if label_x_position == "left":
                    x = padding
                    anchor = "start"
                elif label_x_position == "right":
                    x = self.width - padding
                    anchor = "end"
                else:  # Center position
                    x = self.width / 2
                    anchor = "middle"
                # Determine whether to place the label above or below the line
                dominant_baseline = (
                    "text-after-edge"
                    if label_y_position == "top"
                    else "text-before-edge"
                )
                text_y = (
                    y_svg - padding if label_y_position == "top" else y_svg + padding
                )

                self.svg_elements.append(
                    self._generate_text(
                        label,
                        x,
                        text_y,
                        fill=self.text_color,
                        anchor=anchor,
                        dominant_baseline=dominant_baseline,
                    )
                )

        # Corrected Draw vertical lines with proper alignment and rotation of labels
        for (
            x_index,
            color,
            stroke_width,
            label,
            label_x_position,
            label_y_position,
            rotate_label,
        ) in self.vertical_lines:
            x_svg = (
                (x_index + 0.5) * bar_spacing
                + (bar_spacing - total_bars_width) / 2
                + bar_width * (bar_series_across - 1) / 2
            )
            self.svg_elements.append(
                f'<line x1="{x_svg}" y1="0" x2="{x_svg}" y2="{self.height}" stroke="{color}" stroke-width="{stroke_width}" />'
            )
            if label:
                # Standard padding from the line
                padding = 5

                # Calculate the x position with respect to the line and padding
                x = x_svg + padding if label_x_position == "right" else x_svg - padding
                anchor = "start" if label_x_position == "right" else "end"

                # Calculate the y position based on the vertical label position
                if label_y_position == "top":
                    y = padding
                    vertical_anchor = "start" if label_x_position == "right" else "end"
                elif label_y_position == "bottom":
                    y = self.height - padding
                    vertical_anchor = "end" if label_x_position == "right" else "start"
                else:  # Center position
                    y = self.height / 2
                    vertical_anchor = "middle"

                if rotate_label:
                    # Set the rotation angle based on the side of the line
                    rotation = 90 if label_x_position == "right" else -90

                    # Apply rotation and adjustments
                    self.svg_elements.append(
                        self._generate_text(
                            label,
                            x,
                            y,
                            fill=self.text_color,
                            anchor=vertical_anchor,
                            dominant_baseline="text-after-edge",
                            rotation=rotation,
                        )
                    )
                else:
                    # For non-rotated labels, adjust the y position to ensure the label is always inside the graph area
                    y = y - padding if label_y_position == "bottom" else y + padding
                    # Non-rotated labels should have the baseline set to central if they are in the center position
                    dominant_baseline = (
                        "central"
                        if label_y_position == "center"
                        else "hanging" if label_y_position == "top" else "baseline"
                    )

                    self.svg_elements.append(
                        self._generate_text(
                            label,
                            x,
                            y,
                            fill=self.text_color,
                            anchor=anchor,
                            dominant_baseline=dominant_baseline,
                        )
                    )

        # Draw axis
        self.svg_elements.append(
            f'<line x1="0" y1="0" x2="0" y2="{self.height}" stroke="{self.text_color}" stroke-width="1" />'
        )
        zero_line_y = self.height + adjusted_min_value_primary * scale_primary
        self.svg_elements.append(
            f'<line x1="0" y1="{zero_line_y}" '
            + f'x2="{self.width}" y2="{zero_line_y}" '
            + f'stroke="{self.text_color}" stroke-width="1" />'
        )

        # Draw secondary y-axis if needed
        if has_secondary:
            self.svg_elements.append(
                f'<line x1="{self.width}" y1="0" x2="{self.width}" y2="{self.height}" stroke="{self.text_color}" stroke-width="1" />'
            )
            secondary_zero_line_y = (
                self.height + adjusted_min_value_secondary * scale_secondary
            )
            assert (
                abs(secondary_zero_line_y - zero_line_y) < 1e-9
            ), f"Secondary y-axis not aligned with primary y-axis: {secondary_zero_line_y} != {zero_line_y}"

        # Draw x tick labels
        for index, label in enumerate(self.x_labels):
            x = (
                (index + 0.5) * bar_spacing
                + (bar_spacing - total_bars_width) / 2
                + bar_width * (bar_series_across - 1) / 2
            )
            y = self.height + 5
            if label is not None and self.rotate_x_labels:
                self.svg_elements.append(
                    self._generate_text(
                        label, x, y, anchor="end", fill=self.text_color, rotation=-90
                    )
                )
            elif label is not None and not self.rotate_x_labels:
                self.svg_elements.append(
                    self._generate_text(label, x, y + 10, fill=self.text_color)
                )

        # Draw primary y-axis ticks and values
        for tick_value in primary_ticks:
            tick_y = (
                self.height - (tick_value - adjusted_min_value_primary) * scale_primary
            )
            tick_label = (
                self.primary_tick_prefix
                + human_readable_number(tick_value)
                + self.primary_tick_suffix
            )

            self.svg_elements.append(
                self._generate_text(
                    tick_label,
                    -5,
                    tick_y + 3,
                    fill=self.text_color,
                    anchor="end",
                    dominant_baseline="text-bottom",
                )
            )
            self.svg_elements.append(
                f'<line x1="0" y1="{tick_y}" x2="-3" y2="{tick_y}" stroke="{self.text_color}" stroke-width="1" />'
            )

        # Draw secondary y-axis ticks and values if needed
        if has_secondary:
            for tick_value in secondary_ticks:
                tick_y = (
                    self.height
                    - (tick_value - adjusted_min_value_secondary) * scale_secondary
                )
                tick_label = (
                    self.secondary_tick_prefix
                    + human_readable_number(tick_value)
                    + self.secondary_tick_suffix
                )

                self.svg_elements.append(
                    self._generate_text(
                        tick_label,
                        self.width + 5,
                        tick_y + 3,
                        fill=self.text_color,
                        anchor="start",
                        dominant_baseline="text-bottom",
                    )
                )
                self.svg_elements.append(
                    f'<line x1="{self.width}" y1="{tick_y}" x2="{self.width + 3}" y2="{tick_y}" stroke="{self.text_color}" stroke-width="1" />'
                )

        # Draw axis labels
        if self.x_axis_label:
            x_label_x = (self.width) / 2
            x_label_y = (
                max(self.height, self.most_extreme_dimensions["bottom"])
                + 1.5 * self.element_spacing
            )
            self.svg_elements.append(
                self._generate_text(
                    self.x_axis_label,
                    x_label_x,
                    x_label_y,
                    font_size=12,
                    fill=self.text_color,
                )
            )

        if self.primary_y_axis_label:
            y_label_x = (
                min(0, self.most_extreme_dimensions["left"]) - self.element_spacing
            )
            y_label_y = (self.height) / 2
            self.svg_elements.append(
                self._generate_text(
                    self.primary_y_axis_label,
                    y_label_x,
                    y_label_y,
                    font_size=12,
                    fill=self.text_color,
                    rotation=-90,
                )
            )

        if any(self.secondary) and self.secondary_y_axis_label:
            sec_y_label_x = (
                max(self.width, self.most_extreme_dimensions["right"])
                + self.element_spacing
            )
            sec_y_label_y = self.height / 2
            self.svg_elements.append(
                self._generate_text(
                    self.secondary_y_axis_label,
                    sec_y_label_x,
                    sec_y_label_y,
                    font_size=12,
                    fill=self.text_color,
                    rotation=-90,
                )
            )

        # Draw legend
        if self.show_legend:
            legend_rect_size = 10
            if self.legend_position == "right":
                legend_x = (
                    max(self.width, self.most_extreme_dimensions["right"])
                    + self.element_spacing
                )
                legend_y = 0

                for index, label in enumerate(self.legend_labels):
                    if label is not None:
                        series_type, _ = self.series_types[index]
                        if series_type == "dot":
                            self.svg_elements.append(
                                self._draw_dot(
                                    legend_x + legend_rect_size / 2,
                                    legend_y + legend_rect_size / 2,
                                    radius=5,
                                    fill=self.colors[index],
                                )
                            )
                        elif series_type == "line":
                            self.svg_elements.append(
                                self._draw_line(
                                    legend_x,
                                    legend_y + legend_rect_size / 2,
                                    legend_x + legend_rect_size,
                                    legend_y + legend_rect_size / 2,
                                    stroke=self.colors[index],
                                    stroke_width=self.stroke_width[index],
                                )
                            )
                        else:  # series_type == "bar"
                            self.svg_elements.append(
                                f'<rect x="{legend_x}" y="{legend_y}" width="{legend_rect_size}" '
                                + f'height="{legend_rect_size}" fill="{self.colors[index]}" />'
                            )
                        self.svg_elements.append(
                            self._generate_text(
                                label,
                                legend_x + legend_rect_size + self.element_spacing / 2,
                                legend_y + (2 / 3) * legend_rect_size,
                                fill=self.text_color,
                                anchor="start",
                            )
                        )
                        legend_y += self.element_spacing + legend_rect_size
                        if legend_y + legend_rect_size > self.height:
                            legend_y = 0
                            legend_x = (
                                max(self.width, self.most_extreme_dimensions["right"])
                                + (2 * self.element_spacing) / 3
                            )

            elif self.legend_position == "left":
                max_legend_label_width = max(
                    [
                        estimate_text_dimensions(
                            label, 10, self.font_width_estimate_multiplier
                        )
                        for label in self.legend_labels
                    ]
                )[0]

                legend_x = (
                    min(0, self.most_extreme_dimensions["left"])
                    - self.element_spacing
                    - max_legend_label_width
                    - legend_rect_size
                )
                legend_y = 0

                for index, label in enumerate(self.legend_labels):
                    if label is not None:
                        series_type, _ = self.series_types[index]
                        if series_type == "dot":
                            self.svg_elements.append(
                                self._draw_dot(
                                    legend_x + legend_rect_size / 2,
                                    legend_y + legend_rect_size / 2,
                                    radius=5,
                                    fill=self.colors[index],
                                )
                            )
                        elif series_type == "line":
                            self.svg_elements.append(
                                self._draw_line(
                                    legend_x,
                                    legend_y + legend_rect_size / 2,
                                    legend_x + legend_rect_size,
                                    legend_y + legend_rect_size / 2,
                                    stroke=self.colors[index],
                                    stroke_width=self.stroke_width[index],
                                )
                            )
                        else:  # series_type == "bar"
                            self.most_extreme_dimensions["left"] = min(
                                self.most_extreme_dimensions["left"],
                                legend_x,
                            )
                            self.svg_elements.append(
                                f'<rect x="{legend_x}" y="{legend_y}" width="{legend_rect_size}" '
                                + f'height="{legend_rect_size}" fill="{self.colors[index]}" />'
                            )
                        self.svg_elements.append(
                            self._generate_text(
                                label,
                                legend_x + legend_rect_size + self.element_spacing / 2,
                                legend_y + (2 / 3) * legend_rect_size,
                                fill=self.text_color,
                                anchor="start",
                            )
                        )
                        legend_y += self.element_spacing + legend_rect_size
                        if legend_y + legend_rect_size > self.height:
                            legend_y = 0
                            legend_x = (
                                min(0, self.most_extreme_dimensions["left"])
                                - (2 * self.element_spacing) / 3
                                - max_legend_label_width
                                - legend_rect_size
                            )

            elif self.legend_position == "top":
                legend_x = 0
                legend_y = (
                    min(0, self.most_extreme_dimensions["top"])
                    - self.element_spacing
                    - legend_rect_size
                )

                for index, label in enumerate(self.legend_labels):
                    if label is not None:
                        series_type, _ = self.series_types[index]
                        if series_type == "dot":
                            self.svg_elements.append(
                                self._draw_dot(
                                    legend_x + legend_rect_size / 2,
                                    legend_y + legend_rect_size / 2,
                                    radius=5,
                                    fill=self.colors[index],
                                )
                            )
                        elif series_type == "line":
                            self.svg_elements.append(
                                self._draw_line(
                                    legend_x,
                                    legend_y + legend_rect_size / 2,
                                    legend_x + legend_rect_size,
                                    legend_y + legend_rect_size / 2,
                                    stroke=self.colors[index],
                                    stroke_width=self.stroke_width[index],
                                )
                            )
                        else:  # series_type == "bar"
                            self.most_extreme_dimensions["top"] = min(
                                self.most_extreme_dimensions["top"],
                                legend_y,
                            )
                            self.svg_elements.append(
                                f'<rect x="{legend_x}" y="{legend_y}" width="{legend_rect_size}" '
                                + f'height="{legend_rect_size}" fill="{self.colors[index]}" />'
                            )
                        self.svg_elements.append(
                            self._generate_text(
                                label,
                                legend_x + legend_rect_size + self.element_spacing / 2,
                                legend_y + (2 / 3) * legend_rect_size,
                                fill=self.text_color,
                                anchor="start",
                            )
                        )
                        legend_x += (
                            (2 * self.element_spacing) / 3
                            + legend_rect_size
                            + estimate_text_dimensions(
                                label, 10, self.font_width_estimate_multiplier
                            )[0]
                        )

                        next_index_with_label = index + 1
                        next_label_width = 0
                        while next_index_with_label < len(self.legend_labels):
                            next_label_width = estimate_text_dimensions(
                                self.legend_labels[next_index_with_label],
                                10,
                                self.font_width_estimate_multiplier,
                            )[0]
                            if self.legend_labels[next_index_with_label] is not None:
                                break
                            next_index_with_label += 1

                        if (
                            legend_x
                            + legend_rect_size
                            + self.element_spacing / 2
                            + next_label_width
                            > self.width
                        ):
                            legend_y = (
                                min(0, self.most_extreme_dimensions["top"])
                                - self.element_spacing
                                - legend_rect_size
                            )
                            legend_x = 0

            elif self.legend_position == "bottom":
                legend_x = 0
                legend_y = (
                    max(self.height, self.most_extreme_dimensions["bottom"])
                    + self.element_spacing
                )

                for index, label in enumerate(self.legend_labels):
                    if label is not None:
                        series_type, _ = self.series_types[index]
                        if series_type == "dot":
                            self.svg_elements.append(
                                self._draw_dot(
                                    legend_x + legend_rect_size / 2,
                                    legend_y + legend_rect_size / 2,
                                    radius=5,
                                    fill=self.colors[index],
                                )
                            )
                        elif series_type == "line":
                            self.svg_elements.append(
                                self._draw_line(
                                    legend_x,
                                    legend_y + legend_rect_size / 2,
                                    legend_x + legend_rect_size,
                                    legend_y + legend_rect_size / 2,
                                    stroke=self.colors[index],
                                    stroke_width=self.stroke_width[index],
                                )
                            )
                        else:  # series_type == "bar"
                            self.most_extreme_dimensions["bottom"] = max(
                                self.most_extreme_dimensions["bottom"],
                                legend_y + legend_rect_size,
                            )
                            self.svg_elements.append(
                                f'<rect x="{legend_x}" y="{legend_y}" width="{legend_rect_size}" '
                                + f'height="{legend_rect_size}" fill="{self.colors[index]}" />'
                            )
                        self.svg_elements.append(
                            self._generate_text(
                                label,
                                legend_x + legend_rect_size + self.element_spacing / 2,
                                legend_y + (2 / 3) * legend_rect_size,
                                fill=self.text_color,
                                anchor="start",
                            )
                        )
                        legend_x += (
                            self.element_spacing
                            + legend_rect_size
                            + estimate_text_dimensions(
                                label, 10, self.font_width_estimate_multiplier
                            )[0]
                        )

                        next_index_with_label = index + 1
                        next_label_width = 0
                        while next_index_with_label < len(self.legend_labels):
                            next_label_width = estimate_text_dimensions(
                                self.legend_labels[next_index_with_label],
                                10,
                                self.font_width_estimate_multiplier,
                            )[0]
                            if self.legend_labels[next_index_with_label] is not None:
                                break
                            next_index_with_label += 1

                        if (
                            legend_x
                            + legend_rect_size
                            + self.element_spacing / 2
                            + next_label_width
                            > self.width
                        ):
                            legend_y = (
                                max(self.height, self.most_extreme_dimensions["bottom"])
                                + self.element_spacing
                            )
                            legend_x = 0

            else:
                raise ValueError(
                    f"Invalid legend position: {self.legend_position}. "
                    + "Must be 'right', 'left', 'top', or 'bottom'."
                )

        return self._generate_svg()
