#!/usr/bin/env python3
"""
Simple structure validation test that doesn't require dependencies.
"""

from pathlib import Path
import re

def test_files_exist():
    """Test that all required files exist."""
    print("Testing file structure...")
    
    root = Path(__file__).parent
    src = root / "src" / "dailystories_generator"
    
    required_files = [
        src / "image_evaluation.py",
        src / "image_prompt_improver.py",
        src / "image_statistics_tracker.py",
        src / "optimization_utils.py",
        root / "prompt_templates" / "cover_image_system_instruction.txt",
        root / "prompt_templates" / "page_image_system_instruction.txt",
        root / "optimize_prompts.py",  # Now has integrated support
    ]
    
    all_exist = True
    for file_path in required_files:
        if file_path.exists():
            print(f"✓ {file_path.relative_to(root)}")
        else:
            print(f"✗ Missing: {file_path.relative_to(root)}")
            all_exist = False
    
    return all_exist


def test_prompt_placeholders():
    """Test prompt templates have required placeholders."""
    print("\nTesting system instruction files...")
    
    root = Path(__file__).parent
    cover_template = (root / "prompt_templates" / "cover_image_system_instruction.txt").read_text()
    page_template = (root / "prompt_templates" / "page_image_system_instruction.txt").read_text()
    
    # System instructions don't need placeholders anymore
    cover_ok = "CRITICAL REQUIREMENTS" in cover_template
    page_ok = "CRITICAL REQUIREMENTS" in page_template
    
    if cover_ok:
        print("✓ Cover system instruction has required content")
    else:
        print("✗ Cover system instruction missing required content")
    
    if page_ok:
        print("✓ Page system instruction has required content")
    else:
        print("✗ Page system instruction missing required content")
    
    return cover_ok and page_ok


def test_code_structure():
    """Test that key code patterns exist."""
    print("\nTesting code structure...")
    
    root = Path(__file__).parent
    src = root / "src" / "dailystories_generator"
    
    # Check image_evaluation.py has required classes
    eval_code = (src / "image_evaluation.py").read_text()
    has_cover_eval = "class CoverImageEvaluation" in eval_code
    has_page_eval = "class PageImageEvaluation" in eval_code
    has_cover_func = "async def evaluate_cover_image" in eval_code
    has_page_func = "async def evaluate_page_image" in eval_code
    
    print(f"{'✓' if has_cover_eval else '✗'} CoverImageEvaluation class")
    print(f"{'✓' if has_page_eval else '✗'} PageImageEvaluation class")
    print(f"{'✓' if has_cover_func else '✗'} evaluate_cover_image function")
    print(f"{'✓' if has_page_func else '✗'} evaluate_page_image function")
    
    # Check image_prompt_improver.py has required functions
    improver_code = (src / "image_prompt_improver.py").read_text()
    has_cover_improve = "async def suggest_cover_prompt_improvements" in improver_code
    has_page_improve = "async def suggest_page_prompt_improvements" in improver_code
    has_validate = "def validate_placeholders" in improver_code
    
    print(f"{'✓' if has_cover_improve else '✗'} suggest_cover_prompt_improvements function")
    print(f"{'✓' if has_page_improve else '✗'} suggest_page_prompt_improvements function")
    print(f"{'✓' if has_validate else '✗'} validate_placeholders function")
    
    # Check image_statistics_tracker.py has required class
    tracker_code = (src / "image_statistics_tracker.py").read_text()
    has_tracker = "class ImageStatisticsTracker" in tracker_code
    has_log_method = "def log_iteration" in tracker_code
    
    print(f"{'✓' if has_tracker else '✗'} ImageStatisticsTracker class")
    print(f"{'✓' if has_log_method else '✗'} log_iteration method")
    
    # Check optimization_utils.py
    utils_code = (src / "optimization_utils.py").read_text()
    has_colors = "class Colors:" in utils_code
    has_load_ref = "def load_reference_image" in utils_code
    
    print(f"{'✓' if has_colors else '✗'} optimization_utils.py Colors class")
    print(f"{'✓' if has_load_ref else '✗'} load_reference_image function")
    
    # Check optimize script (now integrated)
    optimize_code = (root / "optimize_prompts.py").read_text()
    has_main = "def main():" in optimize_code
    has_optimize_func = "async def optimize_prompts" in optimize_code
    has_images_mode = '"images"' in optimize_code
    
    print(f"{'✓' if has_main else '✗'} optimize_prompts.py main function")
    print(f"{'✓' if has_optimize_func else '✗'} optimize_prompts function")
    print(f"{'✓' if has_images_mode else '✗'} Images mode supported")
    
    # Check __init__.py exports
    init_code = (src / "__init__.py").read_text()
    has_cover_eval_export = "evaluate_cover_image" in init_code
    has_page_eval_export = "evaluate_page_image" in init_code
    has_improver_exports = "suggest_cover_prompt_improvements" in init_code
    has_tracker_export = "ImageStatisticsTracker" in init_code
    
    print(f"{'✓' if has_cover_eval_export else '✗'} evaluate_cover_image exported")
    print(f"{'✓' if has_page_eval_export else '✗'} evaluate_page_image exported")
    print(f"{'✓' if has_improver_exports else '✗'} prompt improvement functions exported")
    print(f"{'✓' if has_tracker_export else '✗'} ImageStatisticsTracker exported")
    
    # Check GenerationRequest does NOT have old fields (they were removed)
    models_code = (src / "models.py").read_text()
    no_cover_field = "cover_image_reference_prompt" not in models_code
    no_page_field = "page_image_reference_prompt" not in models_code
    
    print(f"{'✓' if no_cover_field else '✗'} GenerationRequest removed cover_image_reference_prompt")
    print(f"{'✓' if no_page_field else '✗'} GenerationRequest removed page_image_reference_prompt")
    
    # Check GeminiClient has new signature
    client_code = (src / "gemini_client.py").read_text()
    has_new_sig = "system_instruction_file" in client_code
    no_old_methods = "generate_image_prompt" not in client_code and "generate_cover_prompt" not in client_code
    
    print(f"{'✓' if has_new_sig else '✗'} GeminiClient.generate_image has system_instruction_file")
    print(f"{'✓' if no_old_methods else '✗'} GeminiClient removed generate_image_prompt/generate_cover_prompt")
    
    all_good = all([
        has_cover_eval, has_page_eval, has_cover_func, has_page_func,
        has_cover_improve, has_page_improve, has_validate,
        has_tracker, has_log_method,
        has_colors, has_load_ref,
        has_main, has_optimize_func, has_images_mode,
        has_cover_eval_export, has_page_eval_export, has_improver_exports, has_tracker_export,
        no_cover_field, no_page_field, has_new_sig, no_old_methods,
    ])
    
    return all_good


def main():
    """Run all validation tests."""
    print("="*80)
    print("IMAGE OPTIMIZATION SYSTEM - STRUCTURE VALIDATION")
    print("="*80 + "\n")
    
    files_ok = test_files_exist()
    placeholders_ok = test_prompt_placeholders()
    structure_ok = test_code_structure()
    
    print("\n" + "="*80)
    if files_ok and placeholders_ok and structure_ok:
        print("✓ ALL VALIDATION CHECKS PASSED")
        print("="*80)
        print("\nThe image optimization system structure is complete!")
        print("\nNext steps:")
        print("1. Install dependencies if needed")
        print("2. Set GOOGLE_API_KEY environment variable")
        print("3. Run optimization:")
        print("   python optimize_prompts.py --mode images \\")
        print("     --reference-image story/original.png \\")
        print("     --iterations 5 \\")
        print("     --child-name Ludwig \\")
        print("     --child-age 6")
        print()
        return 0
    else:
        print("✗ VALIDATION FAILED")
        print("="*80)
        print("\nSome components are missing or incomplete.")
        return 1


if __name__ == "__main__":
    import sys
    sys.exit(main())

