#!/usr/bin/env python3
"""
Test script for dailystories-generator library.

Usage:
    # Set your API key
    export GOOGLE_API_KEY="your-gemini-api-key-here"
    
    # Run the test with defaults
    python test.py
    
    # Customize the story
    python test.py --title "My Adventure" --summary "A child finds a magic stone" --num-pages 8 --language English
    
    # Or run without images (faster)
    python test.py --no-images
    
    # Or generate only the outline (for iteration)
    python test.py --outline-only
"""

import argparse
import asyncio
import os
import sys
from pathlib import Path

# Add package to path for local testing
package_path = Path(__file__).parent / "src"
sys.path.insert(0, str(package_path))

from dailystories_generator import (
    StoryGenerator,
    GenerationRequest,
    ReferenceImage,
    Update,
    UpdateType,
)


class Colors:
    """ANSI color codes for pretty output"""
    HEADER = '\033[95m'
    OKBLUE = '\033[94m'
    OKCYAN = '\033[96m'
    OKGREEN = '\033[92m'
    WARNING = '\033[93m'
    FAIL = '\033[91m'
    ENDC = '\033[0m'
    BOLD = '\033[1m'


async def test_story_generation(args):
    """Test the story generator with a simple story."""
    
    # Get API key from environment
    api_key = os.environ.get("GOOGLE_API_KEY")
    if not api_key:
        print(f"{Colors.FAIL}❌ Error: GOOGLE_API_KEY environment variable not set{Colors.ENDC}")
        print(f"{Colors.WARNING}Set it with: export GOOGLE_API_KEY='your-key-here'{Colors.ENDC}")
        return False
    
    print(f"\n{Colors.HEADER}{'='*60}{Colors.ENDC}")
    print(f"{Colors.HEADER}🚀 Testing DailyStories Generator{Colors.ENDC}")
    print(f"{Colors.HEADER}{'='*60}{Colors.ENDC}\n")
    
    # Initialize generator
    print(f"{Colors.OKBLUE}📦 Initializing StoryGenerator...{Colors.ENDC}")
    generator = StoryGenerator(gemini_api_key=api_key)
    print(f"{Colors.OKGREEN}✓ Generator initialized{Colors.ENDC}\n")
    
    # Load reference image if it exists
    reference_images = []
    reference_image_path = Path(__file__).parent / "story" / "original.png"
    if reference_image_path.exists():
        print(f"{Colors.OKBLUE}📷 Loading reference image: {reference_image_path.name}{Colors.ENDC}")
        import base64
        with open(reference_image_path, "rb") as f:
            image_bytes = f.read()
            base64_data = base64.b64encode(image_bytes).decode('utf-8')
            reference_images.append(ReferenceImage(
                label=f"This is {args.child_name}, the child in the story",
                image_data=f"data:image/png;base64,{base64_data}"
            ))
        print(f"{Colors.OKGREEN}✓ Reference image loaded{Colors.ENDC}\n")
    
    # Create test request
    print(f"{Colors.OKBLUE}📝 Creating generation request...{Colors.ENDC}")
    request = GenerationRequest(
        title=args.title,
        summary=args.summary,
        num_pages=args.num_pages,
        child_name=args.child_name,
        child_age=args.child_age,
        language=args.language,
        illustration_style=args.illustration_style,
        generate_images=args.generate_images,
        reference_images=reference_images,
        outline_only=args.outline_only,
    )
    
    print(f"{Colors.OKGREEN}✓ Request created:{Colors.ENDC}")
    print(f"  • Title: {request.title}")
    print(f"  • Pages: {request.num_pages}")
    print(f"  • Images: {'Yes' if request.generate_images else 'No'}")
    print(f"  • Language: {request.language}")
    print(f"  • Mode: {'Outline only' if request.outline_only else 'Full story'}\n")
    
    # Track progress
    progress = {
        "outline": None,
        "pages": [],
        "cover_image_size": 0,
        "total_tokens": 0,
    }
    
    # Define callback
    async def on_update(update: Update) -> None:
        """Handle updates from the generator."""
        
        if update.type == UpdateType.GENERATING_OUTLINE:
            print(f"{Colors.OKCYAN}⏳ Generating story outline...{Colors.ENDC}")
        
        elif update.type == UpdateType.OUTLINE_COMPLETE:
            progress["outline"] = update.data.get("outline", "")
            tokens = update.data.get("tokens", 0)
            print(f"{Colors.OKGREEN}✓ Outline complete ({tokens} tokens){Colors.ENDC}")
            print(f"{Colors.WARNING}📋 Outline:{Colors.ENDC}")
            for line in progress["outline"].split('\n'):
                if line.strip():
                    print(f"  {line}")
            print()
        
        elif update.type == UpdateType.GENERATING_PAGE:
            page_num = update.data.get("page_number")
            print(f"{Colors.OKCYAN}⏳ Generating page {page_num}...{Colors.ENDC}")
        
        elif update.type == UpdateType.PAGE_COMPLETE:
            page = update.artifacts.get("current_page")
            if page:
                progress["pages"].append(page)
                text_preview = page.text_content[:100] + "..." if len(page.text_content) > 100 else page.text_content
                image_info = f", {len(page.image_data)} bytes" if page.image_data else ", no image"
                print(f"{Colors.OKGREEN}✓ Page {page.page_number} complete ({page.text_tokens} tokens{image_info}){Colors.ENDC}")
                print(f"  {Colors.WARNING}Text:{Colors.ENDC} {text_preview}")
                print()
        
        elif update.type == UpdateType.GENERATING_COVER:
            print(f"{Colors.OKCYAN}⏳ Generating cover image...{Colors.ENDC}")
        
        elif update.type == UpdateType.COVER_COMPLETE:
            cover_data = update.artifacts.get("cover_image_data")
            if cover_data:
                progress["cover_image_size"] = len(cover_data)
                print(f"{Colors.OKGREEN}✓ Cover complete ({len(cover_data)} bytes){Colors.ENDC}\n")
        
        elif update.type == UpdateType.COMPLETE:
            progress["total_tokens"] = update.data.get("total_tokens", 0)
            is_outline_only = update.data.get("outline_only", False)
            print(f"\n{Colors.OKGREEN}{'='*60}{Colors.ENDC}")
            if is_outline_only:
                print(f"{Colors.OKGREEN}🎉 Outline generation complete!{Colors.ENDC}")
            else:
                print(f"{Colors.OKGREEN}🎉 Story generation complete!{Colors.ENDC}")
            print(f"{Colors.OKGREEN}{'='*60}{Colors.ENDC}")
        
        elif update.type == UpdateType.FAILED:
            error = update.data.get("error", "Unknown error")
            print(f"\n{Colors.FAIL}{'='*60}{Colors.ENDC}")
            print(f"{Colors.FAIL}❌ Generation failed: {error}{Colors.ENDC}")
            print(f"{Colors.FAIL}{'='*60}{Colors.ENDC}")
    
    # Generate the story
    try:
        print(f"{Colors.BOLD}Starting generation...{Colors.ENDC}\n")
        story = await generator.generate(request, on_update=on_update)
        
        # Print summary
        print(f"\n{Colors.HEADER}📊 Generation Summary{Colors.ENDC}")
        print(f"{Colors.HEADER}{'='*60}{Colors.ENDC}")
        print(f"  • Outline length: {len(story.outline)} chars")
        
        if args.outline_only:
            print(f"  • Mode: Outline only")
            print(f"  • Total tokens: {story.total_text_tokens}")
        else:
            print(f"  • Pages generated: {len(story.pages)}")
            print(f"  • Total text tokens: {story.total_text_tokens}")
            print(f"  • Total image tokens: {story.total_image_tokens}")
            print(f"  • Total tokens: {story.total_text_tokens + story.total_image_tokens}")
            
            if story.cover_image_data:
                print(f"  • Cover image size: {len(story.cover_image_data)} bytes")
            else:
                print(f"  • Cover image: None")
        
        # Write story to HTML file
        output_file = Path(__file__).parent / "story.html"
        print(f"\n{Colors.OKBLUE}📝 Writing story to {output_file.name}...{Colors.ENDC}")
        
        import base64
        
        with open(output_file, "w", encoding="utf-8") as f:
            # Write HTML header
            f.write(f"""<!DOCTYPE html>
<html lang="{request.language.lower()}">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{request.title}</title>
    <style>
        body {{
            font-family: 'Georgia', serif;
            max-width: 900px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f5f5f5;
            color: #333;
        }}
        .container {{
            background-color: white;
            padding: 40px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }}
        h1 {{
            color: #2c3e50;
            text-align: center;
            font-size: 2.5em;
            margin-bottom: 10px;
        }}
        .metadata {{
            text-align: center;
            color: #7f8c8d;
            margin-bottom: 30px;
            font-size: 0.9em;
        }}
        .cover-image {{
            text-align: center;
            margin: 30px 0;
        }}
        .cover-image img {{
            max-width: 100%;
            height: auto;
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }}
        .outline {{
            background-color: #ecf0f1;
            padding: 20px;
            border-radius: 5px;
            margin: 20px 0;
        }}
        .outline h2 {{
            color: #34495e;
            margin-top: 0;
        }}
        .page {{
            margin: 40px 0;
            page-break-inside: avoid;
        }}
        .page h3 {{
            color: #3498db;
            border-bottom: 2px solid #3498db;
            padding-bottom: 10px;
        }}
        .page-image {{
            text-align: center;
            margin: 20px 0;
        }}
        .page-image img {{
            max-width: 100%;
            height: auto;
            border-radius: 8px;
            box-shadow: 0 3px 5px rgba(0,0,0,0.1);
        }}
        .page-text {{
            line-height: 1.8;
            font-size: 1.1em;
            text-align: justify;
        }}
        .footer {{
            text-align: center;
            color: #95a5a6;
            margin-top: 40px;
            padding-top: 20px;
            border-top: 1px solid #ddd;
            font-size: 0.9em;
        }}
        @media print {{
            body {{
                background-color: white;
            }}
            .container {{
                box-shadow: none;
            }}
        }}
    </style>
</head>
<body>
    <div class="container">
        <h1>{request.title}</h1>
        <div class="metadata">
            <p><strong>Child:</strong> {request.child_name}, age {request.child_age} | 
               <strong>Language:</strong> {request.language} | 
               <strong>Style:</strong> {request.illustration_style}</p>
""")
            
            if args.outline_only:
                f.write(f"            <p><strong>Mode:</strong> Outline only | <strong>Tokens:</strong> {story.total_text_tokens}</p>\n")
            else:
                f.write(f"            <p><strong>Pages:</strong> {len(story.pages)} | <strong>Total Tokens:</strong> {story.total_text_tokens + story.total_image_tokens}</p>\n")
            
            f.write("        </div>\n\n")
            
            # Add cover image if present
            if not args.outline_only and story.cover_image_data:
                cover_base64 = base64.b64encode(story.cover_image_data).decode('utf-8')
                f.write(f"""        <div class="cover-image">
            <img src="data:image/png;base64,{cover_base64}" alt="Cover Image">
        </div>

""")
            
            # Write outline
            f.write("""        <div class="outline">
            <h2>📋 Story Outline</h2>
            <pre style="white-space: pre-wrap; font-family: inherit;">""")
            f.write(story.outline.replace('<', '&lt;').replace('>', '&gt;'))
            f.write("""</pre>
        </div>

""")
            
            # Write story pages (only if not outline_only mode)
            if not args.outline_only:
                for page in story.pages:
                    f.write(f"""        <div class="page">
            <h3>Page {page.page_number}</h3>
""")
                    
                    # Add page image if present
                    if page.image_data:
                        page_base64 = base64.b64encode(page.image_data).decode('utf-8')
                        f.write(f"""            <div class="page-image">
                <img src="data:image/png;base64,{page_base64}" alt="Page {page.page_number} Illustration">
            </div>
""")
                    
                    # Add page text
                    f.write(f"""            <div class="page-text">
                <p>{page.text_content.replace('<', '&lt;').replace('>', '&gt;').replace(chr(10), '</p><p>')}</p>
            </div>
        </div>

""")
                
                # Add footer
                f.write(f"""        <div class="footer">
            <p>Generated with {story.total_text_tokens:,} text tokens and {story.total_image_tokens:,} image tokens</p>
        </div>
""")
            else:
                # Add footer for outline-only mode
                f.write(f"""        <div class="footer">
            <p>Outline generated with {story.total_text_tokens:,} tokens</p>
        </div>
""")
            
            # Close HTML
            f.write("""    </div>
</body>
</html>
""")
        
        print(f"{Colors.OKGREEN}✓ Story saved to {output_file.name}{Colors.ENDC}")
        print(f"\n{Colors.OKGREEN}✅ Test completed successfully!{Colors.ENDC}\n")
        return True
        
    except Exception as e:
        print(f"\n{Colors.FAIL}❌ Test failed with error:{Colors.ENDC}")
        print(f"{Colors.FAIL}{type(e).__name__}: {e}{Colors.ENDC}\n")
        import traceback
        traceback.print_exc()
        return False


def main():
    """Run the test."""
    parser = argparse.ArgumentParser(
        description="Test the DailyStories Generator",
        formatter_class=argparse.RawDescriptionHelpFormatter,
    )
    
    parser.add_argument("--title", default="Ludwig i legobyen", help="Story title")
    parser.add_argument("--summary", default="Ludwig er en legomann i legobyen. Han leker med vennen Theodor, og de finner en rar drage.", help="Story summary/theme")
    parser.add_argument("--num-pages", type=int, default=5, help="Number of pages (default: 5)")
    parser.add_argument("--child-name", default="Ludwig", help="Child's name")
    parser.add_argument("--child-age", type=int, default=6, help="Child's age")
    parser.add_argument("--language", default="Norwegian", help="Story language (default: Norwegian)")
    parser.add_argument("--illustration-style", default="colorful cartoon", help="Illustration style")
    parser.add_argument("--no-images", action="store_true", help="Skip image generation")
    parser.add_argument("--outline-only", action="store_true", help="Generate only outline")
    
    args = parser.parse_args()
    
    # Set generate_images based on no-images flag
    args.generate_images = not args.no_images
    
    print(f"{Colors.OKCYAN}DailyStories Generator - Test Script{Colors.ENDC}")
    
    # Run the async test
    success = asyncio.run(test_story_generation(args))
    
    # Exit with appropriate code
    sys.exit(0 if success else 1)


if __name__ == "__main__":
    main()

