"""Common utilities for prompt optimization."""

import base64
from pathlib import Path

from dailystories_generator.models import ReferenceImage


class Colors:
    """ANSI color codes for pretty output"""
    HEADER = '\033[95m'
    OKBLUE = '\033[94m'
    OKCYAN = '\033[96m'
    OKGREEN = '\033[92m'
    WARNING = '\033[93m'
    FAIL = '\033[91m'
    ENDC = '\033[0m'
    BOLD = '\033[1m'


def load_reference_image(image_path: Path) -> tuple[ReferenceImage, bytes]:
    """
    Load a reference image from disk and convert to base64.
    
    Args:
        image_path: Path to the reference image file
    
    Returns:
        Tuple of (ReferenceImage object, raw image bytes)
    """
    # Read image file
    image_bytes = image_path.read_bytes()
    
    # Encode to base64
    encoded = base64.b64encode(image_bytes).decode('utf-8')
    
    # Determine mime type from extension
    ext = image_path.suffix.lower()
    mime_type = {
        '.png': 'image/png',
        '.jpg': 'image/jpeg',
        '.jpeg': 'image/jpeg',
        '.gif': 'image/gif',
        '.webp': 'image/webp',
    }.get(ext, 'image/png')
    
    # Create data URI
    data_uri = f"data:{mime_type};base64,{encoded}"
    
    ref_image = ReferenceImage(
        image_data=data_uri,
        label="main character reference photo"
    )
    
    return ref_image, image_bytes

