"""Evaluation system for image generation quality assessment."""

from pydantic import BaseModel, Field
from typing import Dict
import json
import re

from dailystories_generator.gemini_client import GeminiClient


class ImageCategoryScore(BaseModel):
    """Score for a single image evaluation category."""
    
    score: int = Field(..., ge=1, le=5, description="Score from 1-5")
    explanation: str = Field(..., description="Explanation for the score")
    improvement_suggestion: str = Field(default="", description="Suggestion for improvement (only if score <= 3)")


class CoverImageEvaluation(BaseModel):
    """Complete evaluation result for cover image."""
    
    child_resemblance: ImageCategoryScore
    story_capture: ImageCategoryScore
    
    def get_scores_dict(self) -> Dict[str, int]:
        """Get a dictionary of category names to scores."""
        return {
            "child_resemblance": self.child_resemblance.score,
            "story_capture": self.story_capture.score,
        }
    
    def get_average_score(self) -> float:
        """Calculate the average score across all categories."""
        scores = self.get_scores_dict().values()
        return sum(scores) / len(scores)
    
    def all_scores_above_threshold(self, threshold: int = 4) -> bool:
        """Check if all category scores are above the threshold."""
        return all(score >= threshold for score in self.get_scores_dict().values())
    
    def get_improvement_suggestions(self) -> dict[str, str]:
        """Get improvement suggestions for categories scoring 3 or lower."""
        suggestions = {}
        for category, score in self.get_scores_dict().items():
            if score <= 3:
                category_score = getattr(self, category)
                if category_score.improvement_suggestion:
                    suggestions[category] = category_score.improvement_suggestion
        return suggestions


class PageImageEvaluation(BaseModel):
    """Complete evaluation result for page image."""
    
    child_resemblance: ImageCategoryScore
    page_content_accuracy: ImageCategoryScore
    style_consistency: ImageCategoryScore
    
    def get_scores_dict(self) -> Dict[str, int]:
        """Get a dictionary of category names to scores."""
        return {
            "child_resemblance": self.child_resemblance.score,
            "page_content_accuracy": self.page_content_accuracy.score,
            "style_consistency": self.style_consistency.score,
        }
    
    def get_average_score(self) -> float:
        """Calculate the average score across all categories."""
        scores = self.get_scores_dict().values()
        return sum(scores) / len(scores)
    
    def all_scores_above_threshold(self, threshold: int = 4) -> bool:
        """Check if all category scores are above the threshold."""
        return all(score >= threshold for score in self.get_scores_dict().values())
    
    def get_improvement_suggestions(self) -> dict[str, str]:
        """Get improvement suggestions for categories scoring 3 or lower."""
        suggestions = {}
        for category, score in self.get_scores_dict().items():
            if score <= 3:
                category_score = getattr(self, category)
                if category_score.improvement_suggestion:
                    suggestions[category] = category_score.improvement_suggestion
        return suggestions


async def evaluate_cover_image(
    reference_image_data: bytes,
    cover_image_data: bytes,
    story_outline: str,
    gemini_client: GeminiClient,
) -> CoverImageEvaluation:
    """
    Evaluate a cover image for quality across multiple dimensions.
    
    Args:
        reference_image_data: Raw bytes of the reference photo (original child photo)
        cover_image_data: Raw bytes of the generated cover image
        story_outline: The complete story outline text
        gemini_client: Gemini client for API calls
    
    Returns:
        CoverImageEvaluation with scores and explanations for all categories
    """
    evaluation_prompt = f"""You are a world-class expert evaluator of children's story book illustrations, specifically focused on character resemblance and story representation.

Please evaluate the GENERATED COVER IMAGE based on the REFERENCE IMAGE and STORY OUTLINE provided.

STORY OUTLINE:
{story_outline}

EVALUATION CATEGORIES:

1. **child_resemblance** (1-5): How well the main character in the generated cover matches the physical details from the reference image. Consider:
   - Hair color, style, and length
   - Facial features and structure
   - Clothing details, colors, and patterns
   - Accessories (glasses, hats, jewelry, etc.)
   - Body proportions and posture
   Score 5 = nearly identical to reference photo; Score 1 = completely different child

2. **story_capture** (1-5): How well the cover image captures and represents the overall story theme and mood. Consider:
   - Does it convey the main theme/adventure of the story?
   - Does the setting/background match the story?
   - Does it capture the emotional tone?
   - Would it attract readers and set proper expectations?
   Score 5 = perfectly captures story essence; Score 1 = completely unrelated to story

For EACH category, provide:
1. A score from 1 to 5 (where 1 = very poor, 2 = poor, 3 = acceptable, 4 = good, 5 = excellent)
2. A detailed explanation (2-3 sentences) justifying the score
3. If the score is 3 or lower, provide a specific improvement suggestion (1-2 sentences)

Respond in the following JSON format:
{{
  "child_resemblance": {{"score": X, "explanation": "...", "improvement_suggestion": "..."}},
  "story_capture": {{"score": X, "explanation": "...", "improvement_suggestion": "..."}}
}}

Note: For categories with score 4 or 5, you can omit "improvement_suggestion" or set it to empty string.

Be honest and constructive. Identify both strengths and areas for improvement."""

    # Prepare content with images
    content_parts = [
        {"mime_type": "image/png", "data": reference_image_data},
        "This is the REFERENCE IMAGE - the original photo of the child that should be the main character.",
        {"mime_type": "image/png", "data": cover_image_data},
        "This is the GENERATED COVER IMAGE to evaluate.",
        evaluation_prompt,
    ]
    
    # Generate evaluation using Gemini with vision
    gemini_model = gemini_client.client if hasattr(gemini_client, 'client') else None
    if gemini_model is None:
        import google.generativeai as genai
        gemini_model = genai.GenerativeModel("gemini-2.5-flash")  # type: ignore[attr-defined]
    else:
        import google.generativeai as genai
        gemini_model = genai.GenerativeModel("gemini-2.5-flash")  # type: ignore[attr-defined]
    
    response = await gemini_model.generate_content_async(contents=content_parts)
    response_text = response.text
    
    # Parse the JSON response
    json_match = re.search(r'```(?:json)?\s*(\{.*?\})\s*```', response_text, re.DOTALL)
    if json_match:
        json_str = json_match.group(1)
    else:
        json_str = response_text.strip()
    
    evaluation_data = json.loads(json_str)
    
    return CoverImageEvaluation(**evaluation_data)


async def evaluate_page_image(
    reference_image_data: bytes,
    page_image_data: bytes,
    page_text: str,
    previous_images: list[bytes],
    gemini_client: GeminiClient,
) -> PageImageEvaluation:
    """
    Evaluate a page image for quality across multiple dimensions.
    
    Args:
        reference_image_data: Raw bytes of the reference photo (original child photo)
        page_image_data: Raw bytes of the generated page image
        page_text: The text content of this page
        previous_images: List of raw bytes of previously generated images in the story
        gemini_client: Gemini client for API calls
    
    Returns:
        PageImageEvaluation with scores and explanations for all categories
    """
    # Build evaluation prompt
    evaluation_prompt = f"""You are a world-class expert evaluator of children's story book illustrations, focused on character resemblance, content accuracy, and visual consistency.

Please evaluate the GENERATED PAGE IMAGE based on the REFERENCE IMAGE, PAGE TEXT, and PREVIOUS IMAGES provided.

PAGE TEXT:
{page_text}

EVALUATION CATEGORIES:

1. **child_resemblance** (1-5): How well the main character matches the physical details from the REFERENCE IMAGE. Consider:
   - Hair color, style, and length
   - Facial features and structure
   - Clothing details, colors, and patterns
   - Accessories (glasses, hats, jewelry, etc.)
   - Body proportions and posture
   Score 5 = nearly identical to reference photo; Score 1 = completely different child

2. **page_content_accuracy** (1-5): How well the illustration depicts the events and details described in the page text. Consider:
   - Are key actions/events from the text shown?
   - Are mentioned objects/characters present?
   - Does the setting match the description?
   - Does it enhance understanding of the text?
   Score 5 = perfectly illustrates the page content; Score 1 = completely unrelated to text

3. **style_consistency** (1-5): How consistent the image is with previous story images. Consider:
   - Overall artistic style (colors, rendering, mood)
   - Character appearance across pages (does the child look the same as in previous images?)
   - Background/setting style consistency
   - Visual cohesion with the story so far
   Score 5 = perfectly consistent with previous images; Score 1 = completely different style/character
   Note: For the first page with no previous images, focus on internal consistency and artistic quality.

For EACH category, provide:
1. A score from 1 to 5 (where 1 = very poor, 2 = poor, 3 = acceptable, 4 = good, 5 = excellent)
2. A detailed explanation (2-3 sentences) justifying the score
3. If the score is 3 or lower, provide a specific improvement suggestion (1-2 sentences)

Respond in the following JSON format:
{{
  "child_resemblance": {{"score": X, "explanation": "...", "improvement_suggestion": "..."}},
  "page_content_accuracy": {{"score": X, "explanation": "...", "improvement_suggestion": "..."}},
  "style_consistency": {{"score": X, "explanation": "...", "improvement_suggestion": "..."}}
}}

Note: For categories with score 4 or 5, you can omit "improvement_suggestion" or set it to empty string.

Be honest and constructive. Identify both strengths and areas for improvement."""

    # Prepare content with images
    content_parts = [
        {"mime_type": "image/png", "data": reference_image_data},
        "This is the REFERENCE IMAGE - the original photo of the child that should be the main character.",
    ]
    
    # Add previous images if available
    if previous_images:
        for i, prev_img in enumerate(previous_images):
            content_parts.append({"mime_type": "image/png", "data": prev_img})
            content_parts.append(f"This is PREVIOUS IMAGE {i+1} from earlier in the story.")
    
    # Add the page image to evaluate
    content_parts.append({"mime_type": "image/png", "data": page_image_data})
    content_parts.append("This is the GENERATED PAGE IMAGE to evaluate.")
    
    # Add the evaluation prompt
    content_parts.append(evaluation_prompt)
    
    # Generate evaluation using Gemini with vision
    import google.generativeai as genai
    gemini_model = genai.GenerativeModel("gemini-2.5-flash")  # type: ignore[attr-defined]
    
    response = await gemini_model.generate_content_async(contents=content_parts)
    response_text = response.text
    
    # Parse the JSON response
    json_match = re.search(r'```(?:json)?\s*(\{.*?\})\s*```', response_text, re.DOTALL)
    if json_match:
        json_str = json_match.group(1)
    else:
        json_str = response_text.strip()
    
    evaluation_data = json.loads(json_str)
    
    return PageImageEvaluation(**evaluation_data)

