"""Gemini API client for text and image generation."""

import base64
from pathlib import Path
from typing import Optional, Any
import google.generativeai as genai
import google.ai.generativelanguage as glm

from dailystories_generator.models import ReferenceImage


# Gemini finish reason mapping
FINISH_REASON_NAMES = {
    0: "FINISH_REASON_UNSPECIFIED",
    1: "STOP",
    2: "MAX_TOKENS",
    3: "SAFETY",
    4: "RECITATION",
    5: "OTHER",
    6: "LANGUAGE",
    7: "BLOCKLIST",
    8: "PROHIBITED_CONTENT",
    9: "SPII",
    10: "MALFORMED_FUNCTION_CALL",
    11: "IMAGE_SAFETY",
}


class GeminiClient:
    """Client for interacting with Google's Gemini API."""
    
    def __init__(self, api_key: str):
        """
        Initialize the Gemini client.
        
        Args:
            api_key: Google API key for Gemini
        """
        self.api_key = api_key
        genai.configure(api_key=api_key)  # type: ignore[attr-defined]
    
    async def generate_text(
        self,
        prompt: str,
        system_prompt: str,
        model: str = "gemini-2.5-flash",
        max_tokens: int = 1000,
    ) -> tuple[str, int]:
        """
        Generate text using Gemini.
        
        Args:
            prompt: The user prompt
            system_prompt: The system instruction
            model: The Gemini model to use
            max_tokens: Maximum tokens (kept for compatibility, not directly used by Gemini)
        
        Returns:
            Tuple of (generated_text, total_tokens)
        """
        gemini_model = genai.GenerativeModel(model)  # type: ignore[attr-defined]
        full_prompt = f"{system_prompt}\n\n{prompt}"
        
        # Count prompt tokens
        prompt_tokens_response = await gemini_model.count_tokens_async(full_prompt)
        prompt_tokens = prompt_tokens_response.total_tokens
        
        # Generate content
        response = await gemini_model.generate_content_async(full_prompt)
        
        try:
            # Extract text and count output tokens
            response_text = response.text
            response_tokens_response = await gemini_model.count_tokens_async(response_text)
            response_tokens = response_tokens_response.total_tokens
        except Exception as e:
            print(f"Error counting tokens: {e}")
            response_tokens = 0
        
        return response_text, prompt_tokens + response_tokens
    
    async def generate_image(
        self,
        content_parts: list[str | dict[str, Any]],
        system_instruction: str,
        model: str = "gemini-2.5-flash-image",
    ) -> tuple[bytes | None, int]:
        """
        Generate an image using Gemini with system instruction.
        
        Args:
            content_parts: List of content parts (images, text) to pass to the model
            system_instruction: System instruction text
            model: The Gemini image model to use
        
        Returns:
            Tuple of (image_bytes, tokens_used)
            Returns (None, 0) if generation fails
        """
        # Create model with system instruction
        gemini_model = genai.GenerativeModel(  # type: ignore[attr-defined]
            model,
            system_instruction=system_instruction,
        )
        
        try:
            # Generate image
            response = await gemini_model.generate_content_async(contents=content_parts)
            
            # Check if response was blocked
            if hasattr(response, "prompt_feedback") and response.prompt_feedback:
                feedback = response.prompt_feedback
                block_reason = getattr(feedback, "block_reason", None)
                if block_reason:
                    print(f"ERROR: Gemini blocked the request. Block reason: {block_reason}")
                    if hasattr(feedback, "safety_ratings"):
                        print(f"Safety ratings: {feedback.safety_ratings}")
                    return None, 0
            
            # Extract image data
            raw_image_data = None
            if hasattr(response, "parts"):
                for part in response.parts:
                    if hasattr(part, "inline_data") and part.inline_data:
                        raw_image_data = part.inline_data.data
                        break
            
            if not raw_image_data:
                # Log detailed debugging information
                print(f"ERROR: No image data found in Gemini response.")
                
                # Log prompt feedback
                if hasattr(response, "prompt_feedback"):
                    feedback = response.prompt_feedback
                    print(f"  Prompt Feedback: {feedback}")
                    if hasattr(feedback, "block_reason"):
                        print(f"  Block Reason: {feedback.block_reason}")
                    if hasattr(feedback, "safety_ratings"):
                        print(f"  Safety Ratings: {feedback.safety_ratings}")
                else:
                    print("  No prompt_feedback available")
                
                # Log candidates with decoded finish reasons
                if hasattr(response, "candidates") and response.candidates:
                    print(f"  Number of candidates: {len(response.candidates)}")
                    for i, candidate in enumerate(response.candidates):
                        print(f"  Candidate {i}:")
                        if hasattr(candidate, "finish_reason"):
                            finish_reason_code = candidate.finish_reason
                            finish_reason_name = FINISH_REASON_NAMES.get(
                                finish_reason_code, 
                                f"UNKNOWN({finish_reason_code})"
                            )
                            print(f"    Finish Reason: {finish_reason_code} ({finish_reason_name})")
                            
                            # Provide context for specific finish reasons
                            if finish_reason_code == 3:  # SAFETY
                                print(f"    ⚠️  Generation blocked due to safety filters")
                            elif finish_reason_code == 11:  # IMAGE_SAFETY
                                print(f"    ⚠️  Image generation blocked due to image safety filters")
                            elif finish_reason_code == 8:  # PROHIBITED_CONTENT
                                print(f"    ⚠️  Content blocked due to policy violations")
                            elif finish_reason_code not in FINISH_REASON_NAMES:
                                print(f"    ⚠️  Unknown finish reason - possibly a rate limit, quota issue, or internal error")
                        
                        if hasattr(candidate, "safety_ratings"):
                            print(f"    Safety Ratings: {candidate.safety_ratings}")
                        if hasattr(candidate, "content"):
                            content = candidate.content
                            if hasattr(content, "parts"):
                                print(f"    Parts count: {len(list(content.parts))}")
                                for j, part in enumerate(content.parts):
                                    if hasattr(part, "text"):
                                        print(f"      Part {j} (text): {part.text[:100]}...")
                                    elif hasattr(part, "inline_data"):
                                        print(f"      Part {j}: inline_data present")
                                    else:
                                        print(f"      Part {j}: {type(part)}")
                            else:
                                print(f"    Content has no parts")
                else:
                    print("  No candidates in response")
                
                # Check usage metadata for quota/rate limit issues
                if hasattr(response, "usage_metadata"):
                    print(f"  Usage Metadata: {response.usage_metadata}")
                
                return None, 0
            
            # For simplicity, returning 1 token for now
            # This can be refined if Gemini provides token usage for images
            return raw_image_data, 1
            
        except Exception as e:
            print(f"ERROR generating image with Gemini: {e}")
            # Log exception details
            import traceback
            print(f"Exception traceback:")
            traceback.print_exc()
            return None, 0


# Import Any for type hints moved to top
