"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrowApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const lambda = require("aws-cdk-lib/aws-lambda");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const logs = require("aws-cdk-lib/aws-logs");
/**
 * For copying shared code to all paths
 */
const fse = require("fs-extra");
const DEFAULT_LAMBDA_CODE = `
exports.handler = async function (event, context, callback) {
  try {
    const data = {
      statusCode: 201,
    };
    return data;
  } catch (uncaughtError) {
    console.error(uncaughtError);
    throw uncaughtError;
  }
}
`;
/**
 * @experimental
 */
class CrowApi extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Pulling out props
        const { sourceDirectory = 'src', sharedDirectory = 'shared', useAuthorizerLambda = false, authorizerDirectory = 'authorizer', authorizerLambdaConfiguration = {}, tokenAuthorizerConfiguration = {}, createApiKey = false, logRetention = logs.RetentionDays.ONE_WEEK, apiGatewayConfiguration = {}, apiGatewayName = 'crow-api', lambdaConfigurations = {}, } = props;
        // Initializing constants
        const LAMBDA_RUNTIME = lambda.Runtime.NODEJS_14_X;
        const SPECIAL_DIRECTORIES = [
            sharedDirectory,
            authorizerDirectory,
        ];
        // Helpers functions for constructor
        // Prepares default Lambda props and overrides them with user input
        function bundleLambdaProps(codePath, userConfiguration, sharedLayer) {
            let layers;
            if (sharedLayer) {
                const { layers: userLayers = [], } = userConfiguration;
                layers = [sharedLayer, ...userLayers];
            }
            const defaultProps = {
                runtime: LAMBDA_RUNTIME,
                code: lambda.Code.fromAsset(codePath),
                handler: 'index.handler',
                logRetention,
            };
            const lambdaProps = {
                ...defaultProps,
                ...userConfiguration,
                layers,
            };
            return lambdaProps;
        }
        // Returns child directories given the path of a parent
        function getDirectoryChildren(parentDirectory) {
            try {
                const directories = fse.readdirSync(parentDirectory, { withFileTypes: true })
                    .filter((dirent) => dirent.isDirectory())
                    .map((dirent) => dirent.name);
                return directories;
            }
            catch {
                /**
                 * The only time I have run into this was when the src/ directory
                 * was empty.
                 * If it is empty, let CDK tree validation tell user that the
                 * REST API does not have any methods.
                 */
            }
            return [];
        }
        // A default Lambda function is needed for the API Gateway
        const defaultLambda = new lambda.Function(this, 'default-crow-lambda', {
            runtime: lambda.Runtime.NODEJS_12_X,
            code: new lambda.InlineCode(DEFAULT_LAMBDA_CODE),
            handler: 'index.handler',
            logRetention,
        });
        // API Gateway log group
        const gatewayLogGroup = new logs.LogGroup(this, 'api-access-logs', {
            retention: logs.RetentionDays.ONE_WEEK,
        });
        // The API Gateway itself
        const gateway = new apigateway.LambdaRestApi(this, apiGatewayName, {
            handler: defaultLambda,
            proxy: false,
            deploy: true,
            deployOptions: {
                loggingLevel: apigateway.MethodLoggingLevel.ERROR,
                accessLogDestination: new apigateway.LogGroupLogDestination(gatewayLogGroup),
            },
            apiKeySourceType: createApiKey ? apigateway.ApiKeySourceType.HEADER : undefined,
            ...apiGatewayConfiguration,
        });
        // Create API key if desired
        if (createApiKey) {
            const apiKey = gateway.addApiKey('api-key');
            const usagePlan = new apigateway.UsagePlan(this, 'usage-plan', {
                throttle: {
                    burstLimit: 5000,
                    rateLimit: 10000,
                },
                apiStages: [
                    {
                        api: gateway,
                        stage: gateway.deploymentStage,
                    },
                ],
            });
            usagePlan.addApiKey(apiKey);
        }
        // Create Lambda layer out of shared directory if it exists
        const sourceSharedDirectory = `${sourceDirectory}/${sharedDirectory}`;
        let sharedLayer;
        if (fse.existsSync(sourceSharedDirectory)) {
            sharedLayer = new lambda.LayerVersion(this, 'shared-layer', {
                code: lambda.Code.fromAsset(sourceSharedDirectory),
                compatibleRuntimes: [LAMBDA_RUNTIME],
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            this.lambdaLayer = sharedLayer;
        }
        // Create Lambda authorizer to be used in subsequent Methods
        let tokenAuthorizer;
        if (useAuthorizerLambda) {
            const fullAuthorizerDirectory = `${sourceDirectory}/${authorizerDirectory}`;
            const authorizerLambdaProps = bundleLambdaProps(fullAuthorizerDirectory, authorizerLambdaConfiguration, sharedLayer);
            const authorizerLambda = new lambda.Function(this, 'authorizer-lambda', authorizerLambdaProps);
            this.authorizerLambda = authorizerLambda;
            const bundledTokenAuthConfig = {
                handler: authorizerLambda,
                resultsCacheTtl: cdk.Duration.seconds(3600),
                ...tokenAuthorizerConfiguration,
            };
            tokenAuthorizer = new apigateway.TokenAuthorizer(this, 'token-authorizer', bundledTokenAuthConfig);
        }
        // Time to start walking the directories
        const root = sourceDirectory;
        const verbs = ['get', 'post', 'put', 'delete'];
        const graph = {};
        const lambdasByPath = {};
        // Initialize with root
        graph['/'] = {
            resource: gateway.root,
            path: root,
            paths: [],
            verbs: [],
        };
        // First element in tuple is directory path, second is API path
        const nodes = [[root, '/']];
        // BFS that creates API Gateway structure using addMethod
        while (nodes.length) {
            // The `|| ['type', 'script']` piece is needed or TS throws a fit
            const [directoryPath, apiPath] = nodes.shift() || ['type', 'script'];
            const children = getDirectoryChildren(directoryPath);
            // For debugging purposes
            // console.log(`${apiPath}'s children are: ${children}`);
            // Don't have to worry about previously visited nodes
            // since this is a file structure
            // ...unless there are symlinks? Haven't run into that
            children.forEach((child) => {
                const newDirectoryPath = `${directoryPath}/${child}`;
                // If we're on the root path, don't separate with a slash (/)
                //   because it ends up looking like //child-path
                const newApiPath = apiPath === '/' ? `/${child}` : `${apiPath}/${child}`;
                if (verbs.includes(child)) {
                    // If directory is a verb, we don't traverse it anymore
                    //   and need to create an API Gateway method and Lambda
                    const userConfiguration = lambdaConfigurations[newApiPath] || {};
                    const lambdaProps = bundleLambdaProps(newDirectoryPath, userConfiguration, sharedLayer);
                    const { useAuthorizerLambda: authorizerLambdaConfigured } = lambdaProps;
                    const newLambda = new lambda.Function(this, newDirectoryPath, lambdaProps);
                    let methodConfiguration;
                    if (authorizerLambdaConfigured && useAuthorizerLambda) {
                        methodConfiguration = {
                            authorizationType: apigateway.AuthorizationType.CUSTOM,
                            authorizer: tokenAuthorizer,
                        };
                    }
                    graph[apiPath].resource.addMethod(child.toUpperCase(), new apigateway.LambdaIntegration(newLambda), methodConfiguration);
                    graph[apiPath].verbs.push(child);
                    lambdasByPath[newApiPath] = newLambda;
                }
                else if (SPECIAL_DIRECTORIES.includes(child)) {
                    // The special directories should not result in an API path
                    // This means the API also cannot have a resource with the
                    //   same name
                }
                else {
                    // If directory is not a verb, create new API Gateway resource
                    //   for use by verb directory later
                    const newResource = graph[apiPath].resource
                        .resourceForPath(child);
                    nodes.push([newDirectoryPath, newApiPath]);
                    // Add child to parent's paths
                    graph[apiPath].paths.push(child);
                    // Initialize graph node to include child
                    graph[newApiPath] = {
                        resource: newResource,
                        path: newDirectoryPath,
                        paths: [],
                        verbs: [],
                    };
                }
            });
        }
        // For debugging purposes
        // console.log(graph);
        // Expose API Gateway
        this.gateway = gateway;
        this.lambdaFunctions = lambdasByPath;
    }
}
exports.CrowApi = CrowApi;
_a = JSII_RTTI_SYMBOL_1;
CrowApi[_a] = { fqn: "crow-api.CrowApi", version: "2.0.0" };
//# sourceMappingURL=data:application/json;base64,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