# -*- coding: utf-8 -*-
# pylint: disable=W0614,W0611,W0622
# flake8: noqa
# isort:skip_file
import logging
import sys

logger = logging.getLogger('adsorption_file_parser')
logger.setLevel(logging.DEBUG)

# create console handler
ch = logging.StreamHandler(stream=sys.stdout)
ch.setLevel(logging.INFO)

# add the handlers to the logger
logger.addHandler(ch)


class ParsingError(Exception):
    """Raised when parsing fails."""


_SUPPORTED_FORMATS = {
    'smsdvs': ('xlsx', ),
    'bel': ('csv', 'xl', 'dat'),
    'mic': ('xl', ),
    '3p': ('xl', ),
    'qnt': ('txt-raw', ),
}


def read(path, manufacturer, fmt, **options):
    """
    Parse aa file generated by commercial apparatus.

    Parameters
    ----------
    path: str
        the location of the file.
    manufacturer : {'mic', 'bel', '3p'}
        Manufacturer of the apparatus.
    fmt : {'xl', 'txt', ...}
        The format of the import for the isotherm.

    Returns
    -------
    meta: dict
        the metadata
    data: dict
        all available data
    """

    if manufacturer not in _SUPPORTED_FORMATS:
        raise ParsingError(
            f'Currently available manufacturers are {list(_SUPPORTED_FORMATS.keys())})'
        )

    if fmt not in _SUPPORTED_FORMATS[manufacturer]:
        raise ParsingError(f'Currently available formats are {_SUPPORTED_FORMATS[manufacturer]}')

    if manufacturer == 'smsdvs' and fmt == 'xlsx':
        from .smsdvs_excel import parse
    elif manufacturer == 'mic' and fmt == 'xl':
        from .mic_excel import parse
    elif manufacturer == 'bel':
        if fmt == 'xl':
            from .bel_excel import parse
        elif fmt == 'csv':
            from .bel_csv import parse
        elif fmt == 'dat':
            from .bel_dat import parse
    elif manufacturer == '3p' and fmt == 'xl':
        from .trp_excel import parse
    elif manufacturer == 'qnt' and fmt == 'txt-raw':
        from .qnt_txt import parse
    else:
        raise ParsingError('Something went wrong.')

    return parse(path, **options)
