"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToIot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("@aws-cdk/aws-apigateway");
const cdk = require("@aws-cdk/core");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayIot class.
 */
class ApiGatewayToIot extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayIot class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // IoT Core topic nesting. A topic in a publish or subscribe request can have no more than 7 forward slashes (/).
        // This excludes the first 3 slashes in the mandatory segments for Basic Ingest
        // Refer IoT Limits - https://docs.aws.amazon.com/general/latest/gr/iot-core.html#limits_iot
        this.topicNestingLevel = 7;
        // Assignment to local member variables to make these available to all member methods of the class.
        // (Split the string just in case user supplies fully qualified endpoint eg. ab123cdefghij4l-ats.iot.ap-south-1.amazonaws.com)
        this.iotEndpoint = props.iotEndpoint.trim().split('.')[0];
        // Mandatory params check
        if (!this.iotEndpoint || this.iotEndpoint.length < 0) {
            throw new Error('specify a valid iotEndpoint');
        }
        // Add additional params to the apiGatewayProps
        let extraApiGwProps = {
            binaryMediaTypes: ['application/octet-stream'],
            defaultMethodOptions: {
                apiKeyRequired: props.apiGatewayCreateApiKey
            }
        };
        // If apiGatewayProps are specified override the extra Api Gateway properties
        if (props.apiGatewayProps) {
            extraApiGwProps = defaults.overrideProps(props.apiGatewayProps, extraApiGwProps);
        }
        // Check whether an API Gateway execution role is specified?
        if (props.apiGatewayExecutionRole) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            // JSON that will be used for policy document
            const policyJSON = {
                Version: "2012-10-17",
                Statement: [
                    {
                        Action: [
                            "iot:UpdateThingShadow"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
                        Effect: "Allow"
                    },
                    {
                        Action: [
                            "iot:Publish"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
                        Effect: "Allow"
                    }
                ]
            };
            // Create a policy document
            const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
            // Props for IAM Role
            const iamRoleProps = {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                path: '/',
                inlinePolicies: { awsapigatewayiotpolicy: policyDocument }
            };
            // Create a policy that overrides the default policy that gets created with the construct
            this.apiGatewayRole = new iam.Role(this, 'apigateway-iot-role', iamRoleProps);
        }
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole,
            this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, extraApiGwProps, props.logGroupProps);
        // Validate the Query Params
        const requestValidatorProps = {
            restApi: this.apiGateway,
            validateRequestBody: false,
            validateRequestParameters: true
        };
        this.requestValidator = new api.RequestValidator(this, `aws-apigateway-iot-req-val`, requestValidatorProps);
        // Create a resource for messages '/message'
        const msgResource = this.apiGateway.root.addResource('message');
        // Create resources from '/message/{topic-level-1}' through '/message/{topic-level-1}/..../{topic-level-7}'
        let topicPath = 'topics';
        let parentNode = msgResource;
        let integParams = {};
        let methodParams = {};
        for (let pathLevel = 1; pathLevel <= this.topicNestingLevel; pathLevel++) {
            const topicName = `topic-level-${pathLevel}`;
            const topicResource = parentNode.addResource(`{${topicName}}`);
            const integReqParam = JSON.parse(`{"integration.request.path.${topicName}": "method.request.path.${topicName}"}`);
            const methodReqParam = JSON.parse(`{"method.request.path.${topicName}": true}`);
            topicPath = `${topicPath}/{${topicName}}`;
            integParams = Object.assign(integParams, integReqParam);
            methodParams = Object.assign(methodParams, methodReqParam);
            this.addResourceMethod(topicResource, props, topicPath, integParams, methodParams);
            parentNode = topicResource;
        }
        // Create a resource for shadow updates '/shadow'
        const shadowResource = this.apiGateway.root.addResource('shadow');
        // Create resource '/shadow/{thingName}'
        const defaultShadowResource = shadowResource.addResource('{thingName}');
        const shadowReqParams = { 'integration.request.path.thingName': 'method.request.path.thingName' };
        const methodShadowReqParams = { 'method.request.path.thingName': true };
        this.addResourceMethod(defaultShadowResource, props, 'things/{thingName}/shadow', shadowReqParams, methodShadowReqParams);
        // Create resource '/shadow/{thingName}/{shadowName}'
        const namedShadowResource = defaultShadowResource.addResource('{shadowName}');
        const namedShadowReqParams = Object.assign({
            'integration.request.path.shadowName': 'method.request.path.shadowName'
        }, shadowReqParams);
        const methodNamedShadowReqParams = Object.assign({
            'method.request.path.shadowName': true
        }, methodShadowReqParams);
        // For some reason path mapping to 'things/{thingName}/shadow/name/{shadowName}' results in 403 error, hence this shortcut
        this.addResourceMethod(namedShadowResource, props, 'topics/$aws/things/{thingName}/shadow/name/{shadowName}/update', namedShadowReqParams, methodNamedShadowReqParams);
    }
    /**
     * Adds a method to specified resource
     * @param resource API Gateway resource to which this method is added
     * @param resourcePath path of resource from root
     * @param integReqParams request paramters for the Integration method
     * @param methodReqParams request parameters at Method level
     */
    addResourceMethod(resource, props, resourcePath, integReqParams, methodReqParams) {
        const integResp = [
            {
                statusCode: "200",
                selectionPattern: "2\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "500",
                selectionPattern: "5\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "403",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            }
        ];
        // Method responses for the resource
        const methodResp = [
            {
                statusCode: "200"
            },
            {
                statusCode: "500"
            },
            {
                statusCode: "403"
            }
        ];
        // Override the default Integration Request Props
        const integrationReqProps = {
            subdomain: this.iotEndpoint,
            options: {
                requestParameters: integReqParams,
                integrationResponses: integResp,
                passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
            }
        };
        // Override the default Method Options
        const resourceMethodOptions = {
            requestParameters: methodReqParams,
            methodResponses: methodResp,
        };
        const resourceMethodParams = {
            service: 'iotdata',
            path: resourcePath,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: resource,
            requestTemplate: "$input.json('$')",
            requestValidator: this.requestValidator,
            awsIntegrationProps: integrationReqProps,
            methodOptions: resourceMethodOptions
        };
        const apiMethod = defaults.addProxyMethodToApiResource(resourceMethodParams);
        // cfn Nag doesn't like having a HTTP Method with Authorization Set to None, supress the warning
        if (props.apiGatewayCreateApiKey === true) {
            const cfnMethod = apiMethod.node.findChild('Resource');
            cfnMethod.cfnOptions.metadata = {
                cfn_nag: {
                    rules_to_suppress: [{
                            id: 'W59',
                            reason: 'When ApiKey is being created, we also set apikeyRequired to true, so techincally apiGateway still looks for apiKey even though user specified AuthorizationType to NONE'
                        }]
                }
            };
        }
    }
}
exports.ApiGatewayToIot = ApiGatewayToIot;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToIot[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-iot.ApiGatewayToIot", version: "1.85.0" };
//# sourceMappingURL=data:application/json;base64,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