"""组件构建"""
import os
import yaml
import shutil
import re
import json
from multiprocessing import Pool
import traceback
from tempfile import NamedTemporaryFile
from argparse import ArgumentParser
from jsonschema import validate, ValidationError
from git import Repo
from git.exc import InvalidGitRepositoryError
from mako.lookup import TemplateLookup
from lbkit.misc import Color, load_yml_with_json_schema_validate, get_json_schema_file, load_json_schema
from lbkit import errors
from lbkit.codegen.codegen import CodeGen
from lbkit.tools import Tools
from lbkit.build_conan_parallel import BuildConanParallel

tools = Tools("comp_build")
log = tools.log
cwd = os.getcwd()
lb_cwd = os.path.split(os.path.realpath(__file__))[0]


class DeployComponent():
    def __init__(self, package_ref, package_id, rootfs_dir):
        self.package_ref = package_ref
        self.package_id = package_id
        self.rootfs_dir = rootfs_dir

    def run(self):
        if self.package_ref.startswith("deploy"):
            return
        cmd = f"conan cache path {self.package_ref}:{self.package_id}"
        package_folder = tools.run(cmd).stdout.strip()
        log.info(f">>>> deploy {self.package_ref}")
        cmd = f"cp -rT {package_folder}/ {self.rootfs_dir}"
        cnt = 10
        while cnt > 0:
            try:
                cnt -= 1
                tools.exec(cmd)
                return
            except Exception as e:
                if cnt == 0:
                    log.warn("Copy failed, msg: " + str(e))
                    raise e
                log.info("Copy {self.package_ref} failed, try again")


class BuildComponent():
    def __init__(self, args_parser: ArgumentParser, args=None):
        self.deploy_success = True
        self.options = args_parser.parse_args(args)
        self.options.build_type = self.options.build_type.capitalize()
        if self.options.channel is None or self.options.channel.strip() == "":
            raise errors.ArgException("请正确指定-c, --channel指定conan包的channel通道")
        self.channel = self.options.channel
        self.build_type = self.options.build_type
        self.profile = self.options.profile
        self.profile_build = self.options.profile_build
        self.verbose = False if self.options.summary else True
        self.from_source = self.options.from_source
        # 当前组件及其依赖将被部署到rootfs目录
        self.rootfs_dir = os.path.join(cwd, ".temp", "rootfs")
        shutil.rmtree(self.rootfs_dir, ignore_errors=True)
        os.makedirs(self.rootfs_dir, exist_ok=True)

        self.pkg = None
        self.base_cmd = ""
        self.gen_conaninfo()
        self.base_cmd += f" --user {self.user} --channel {self.channel}"
        self.base_cmd += f" -pr {self.profile} -s build_type={self.build_type} -r " + self.options.remote
        self.base_cmd += f" -pr:b {self.profile_build}"
        if self.options.cov:
            self.base_cmd += f" -o {self.name}/*:gcov=True"
        if self.options.test:
            self.base_cmd += f" -o {self.name}/*:test=True"
        for pkg_option in self.options.pkg_options:
            self.base_cmd += " -o " + pkg_option

    def get_package_version(self):
        """
        从CMakeLists.txt读取版本号，格式需要满足正则表达式：project\((.*)VERSION ([0-9][1-9]*.[0-9][1-9]*.[0-9][1-9]*)\)
        示例: project(gcom LANGUAGES C VERSION 0.1.0)
        """
        try:
            with open("CMakeLists.txt", "r") as fp:
                content = fp.read()
            version = re.search("project\((.*)VERSION ([0-9][1-9]*.[0-9][1-9]*.[0-9][1-9]*)\)", content).group(2)
            return version.strip()
        except Exception as e:
            print(str(e))
            return None

    @property
    def _is_conanfile_tracked(self):
        """检查conanfile.py是否被git跟踪"""
        try:
            repo = Repo(".")
            for entry in repo.commit().tree.traverse():
                if entry.path == "conanfile.py":
                    return True
        except InvalidGitRepositoryError as e:
            log.error("Invalid git repository, conanfile.py will be generated by lbkit")
        return False

    def gen_conaninfo(self):
        package_yml = os.path.join(cwd, "metadata/package.yml")
        if not os.path.isfile(package_yml):
            raise FileNotFoundError("metadata/package.yml文件不存在")
        # 验证失败时抛异常，此处不用处理，由外层处理
        pkg = load_yml_with_json_schema_validate(package_yml, "/usr/share/litebmc/schema/cdf.v1.json")
        log.info(f"validate {package_yml} successfully")

        self.user = pkg.get("user")
        if self.user is None:
            raise errors.PackageConfigException("metadata/package.yml未正确配置user字段")
        # 构建命令未指定channel时从package.yml中读取
        pkg["channel"] = self.channel
        pkg["version"] = self.get_package_version()
        self.pkg = pkg
        # 从package.yml加载基础信息
        self.name = pkg.get("name")
        self.version = pkg.get("version")

        self.package = self.name + "/" + self.version + \
            "@" + self.user + "/" + self.channel
        # 准备部署依赖
        requires = pkg.get("requires")
        deps = []
        if requires is not None:
            if self.options.test:
                for rt in requires.get("test", []):
                    deps.append(rt)
            for rt in requires.get("compile", []):
                deps.append(rt)

        for dep in deps:
            option = dep.get("option", {})
            conan = dep.get("conan").split("/")[0]
            for k, v in option.items():
                self.base_cmd += f" -o {conan}/*:{k}={v}"

        # 生成conan构建脚本
        conanfile = os.path.join(cwd, "conanbase.py")
        # 当git未跟踪conanfile.py时生成新的conanfile.py
        if not self._is_conanfile_tracked:
            conanfile = os.path.join(cwd, "conanfile.py")

        # 使用litebmc.conanfile.mako模板生成基础litebmc公共conanfile
        lookup = TemplateLookup(directories=os.path.join(lb_cwd, "template"))
        template = lookup.get_template("conanbase.mako")
        conandata = template.render(lookup=lookup, pkg=pkg, conanfile_tracked=self._is_conanfile_tracked)
        # 写入文件
        fp = open(conanfile, "w")
        fp.write(conandata)
        fp.close()

    def upload(self):
        log.success(f"start upload {self.package}")
        cmd = "conan upload {}# -r {}".format(
            self.package, self.options.remote)
        if self.options.upload_recipe:
            cmd += " --only-recipe"
        tools.exec(cmd, verbose=True)

    def _copy_failed(self, result):
        print(result)
        self.deploy_success = False

    def deploy(self, graphfile):
        with open(graphfile, "r") as fp:
            graph = json.load(fp)
        nodes = graph.get("graph", {}).get("nodes", {})
        pool = Pool()
        for id, info in nodes.items():
            ref = info.get("ref")
            id = info.get("package_id")
            context = info.get("context")
            if context != "host":
                continue
            dep = DeployComponent(ref, id, self.rootfs_dir)
            pool.apply_async(dep.run, error_callback=self._copy_failed)
        pool.close()
        pool.join()

    def build(self):
        log.info(os.getcwd())

        export_cmd = f"conan export . --user={self.user} --channel={self.channel}"
        tools.run(export_cmd, capture_output=False)

        lockfile = os.path.join(cwd, ".temp", "conan.lock")
        graphfile = os.path.join(cwd, ".temp", "graph.info")
        lock_cmd = f"conan lock create . {self.base_cmd} --lockfile-out={lockfile}"
        tools.run(lock_cmd, capture_output=False)
        graph_cmd = f"conan graph info . {self.base_cmd} -f json --lockfile={lockfile}"
        tools.pipe([graph_cmd], out_file=graphfile)
        log.success(f"start build dependency packages of {self.package}")
        bcp = BuildConanParallel(graphfile, lockfile, self.base_cmd, self.from_source)
        bcp.build()

        cmd = f"conan create . {self.base_cmd}"
        log.info(f"start build {self.package}: {cmd}")
        tools.run(cmd, capture_output=False)

        log.success(f"start deploy {self.package} and is's dependency packages")
        self.deploy(graphfile)

        if not self.deploy_success:
            raise Exception("Deploy component failed")

        # 设置ROOTFS_DIR环境变量，为DT测试提供相对路径
        os.environ["ROOTFS_DIR"] = self.rootfs_dir
        os.chdir(cwd)

    def _validate_odf_object(self, name, obj):
        properties = obj.get("properties")
        # ODF支持properties置空，所以为None时免验证
        if properties is None:
            return True
        intf = obj.get("interface")
        intf_schema = f"usr/share/litebmc/schema/{intf}.json"
        real_schema = os.path.join(self.rootfs_dir, intf_schema)
        real_schema = os.path.relpath(real_schema, os.getcwd())
        log.info(f"Start validate object {name} with schema {real_schema}")
        if not os.path.exists(real_schema):
            log.error(f"The scheme file {real_schema} of interface not exist, validate object {name} failed")
            return False
        try:
            schema = load_json_schema(real_schema)
            validate(properties, schema)
        except FileNotFoundError as exc:
            log.error(f"validate object {name} failed, schema {real_schema} not exist, message: {str(exc)}\n")
        except ValidationError as exc:
            log.error(f"validate object {name} failed, schema {real_schema}, message: {exc.message}\n")
            if os.environ.get("LOG"):
                print(traceback.format_exc())
            return False
        return True

    def _validate_odf_file(self, file):
        ok = True
        with open(file, "r") as fp:
            odf = yaml.load(fp, yaml.FullLoader)
            objects = odf.get("objects", {})
            for name, obj in objects.items():
                obj_ok = self._validate_odf_object(name, obj)
                if not obj_ok:
                    ok = False
        return ok

    def _validate_odf_files(self):
        log.success("Start validate ODF files")
        ok = True
        for root, _, files in os.walk(self.rootfs_dir):
            for file in files:
                file = os.path.join(root, file)
                file = os.path.relpath(file, cwd)
                if not file.endswith(".yaml"):
                    log.debug(f"file {file} not endswith .yaml, skip validate")
                    continue

                schema_file = get_json_schema_file(file, None)
                if schema_file is None:
                    log.debug(f"the file {file} don't has validate 'yaml-language-server:', maybe not a valid odf file, skip it")
                    continue
                basename = os.path.basename(schema_file)
                if not re.match("^odf\\.v[0-9]+\\.json$", basename):
                    log.debug(f"the schema of file not match '^odf\\.v[0-9]+\\.json$', maybe not a valid odf file, skip it")
                    continue
                log.info(f"start validate {file} with schema {schema_file}")
                # 验证全局odf验证
                load_yml_with_json_schema_validate(file, schema_file)
                odf_ok = self._validate_odf_file(file)
                if not odf_ok:
                    ok = False
        if not ok:
            raise errors.OdfValidateException("Validate odf files with error, build failed")


    def run(self):
        cmd = f"conan remove {self.package} -c"
        tools.exec(cmd)
        gen = CodeGen(["-c", "./metadata/package.yml"])
        gen.run()
        # 部署依赖
        self.build()
        # start validate all odf(Object Description file) files
        self._validate_odf_files()
        if self.options.upload_recipe or self.options.upload_package:
            self.upload()
        log.success(f"build {self.package} successfully")

    @property
    def package_id(self):
        cmd = f"conan graph info . {self.base_cmd} --filter=package_id"
        res = tools.run(cmd)
        match = re.search(r"package_id: ([a-f0-9]{40})", res.stdout)
        if match is None:
            raise errors.LiteBmcException(f"Get package if of {self.package} failed")
        return match.group(1)

    @staticmethod
    def package_folder(self):
        cmd = f"conan cache path {self.package}#latest:{self.package_id}"
        res = tools.run(cmd)
        return res.stdout.strip()

    @property
    def build_folder(self):
        cmd = f"conan cache path {self.package}#latest:{self.package_id} --folder=build"
        res = tools.run(cmd)
        return res.stdout.strip()

    def test(self):
        try:
            self.run()
        except Exception as e:
            log.error(
                f"build {self.package} {Color.RED}failed{Color.RESET_ALL}")
            log.info(e)
            os._exit(-2)
