from turtle import forward
import torch
import torch.nn as nn
from torch.nn import LayerNorm

from torch.distributed import get_world_size
from torch.distributed.nn import all_reduce

from transformers.models.bloom.modeling_bloom import dropout_add, BloomGelu
from transformers.models.bloom.configuration_bloom import BloomConfig

from typing import Tuple, Optional

class ParallelAttention(nn.Module):
    def __init__(self, embed_dim, num_heads, hidden_dropout, bias=True,
                 device=None, dtype=None) -> None:
        factory_kwargs = {'device': device, 'dtype': dtype}
        super(ParallelAttention, self).__init__()
        self.embed_dim = embed_dim

        assert num_heads % get_world_size() == 0
        self.num_heads = num_heads // get_world_size()

        self.head_dim = embed_dim // num_heads
        assert self.head_dim * self.num_heads * get_world_size() == self.embed_dim, f"embed_dim must be divisible by num_heads, got embed_dim={self.embed_dim} and num_heads={self.num_heads} with world_size={get_world_size()} and head_dim={self.head_dim}"

        self.query_key_value = nn.Linear(embed_dim, 3 * embed_dim // get_world_size(), **factory_kwargs)

        self.dense = nn.Linear(embed_dim // get_world_size(), embed_dim, bias=bias, **factory_kwargs)

        self.hidden_dropout = hidden_dropout

    def __setstate__(self, state):
        # Support loading old MultiheadAttention checkpoints generated by v1.1.0

        # TODO: Тут пилим state по process_id
        super(ParallelAttention, self).__setstate__(state)

    def _split_heads(self, fused_qkv: torch.Tensor) -> Tuple[torch.Tensor, torch.Tensor, torch.Tensor]:
        """
        Split the last dimension into (num_heads, head_dim) without making any copies, results share same memory
        storage as `fused_qkv`

        Args:
            fused_qkv (`torch.tensor`, *required*): [batch_size, seq_length, num_heads * 3 * head_dim]

        Returns:
            query: [batch_size, seq_length, num_heads, head_dim] key: [batch_size, seq_length, num_heads, head_dim]
            value: [batch_size, seq_length, num_heads, head_dim]
        """
        batch_size, seq_length, three_times_hidden_size = fused_qkv.shape
        fused_qkv = fused_qkv.view(batch_size, seq_length, self.num_heads, 3, self.head_dim)
        return fused_qkv[..., 0, :], fused_qkv[..., 1, :], fused_qkv[..., 2, :]

    def _merge_heads(self, x: torch.Tensor) -> torch.Tensor:
        """
        Merge heads together over the last dimenstion

        Args:
            x: (`torch.tensor`, *required*): [batch_size * num_heads, seq_length, head_dim]

        Returns:
            torch.tensor: [batch_size, seq_length, num_heads * head_dim]
        """
        # What we want to achieve is:
        # batch_size * num_heads, seq_length, head_dim -> batch_size, seq_length, num_heads * head_dim
        batch_size_and_num_heads, seq_length, _ = x.shape
        batch_size = batch_size_and_num_heads // self.num_heads

        # First view to decompose the batch size
        # batch_size * num_heads, seq_length, head_dim -> batch_size, num_heads, seq_length, head_dim
        x = x.view(batch_size, self.num_heads, seq_length, self.head_dim)

        # batch_size, num_heads, seq_length, head_dim -> batch_size, seq_length, num_heads, head_dim
        x = x.permute(0, 2, 1, 3)

        # batch_size, seq_length, num_heads, head_dim -> batch_size, seq_length, num_heads * head_dim
        return x.reshape(batch_size, seq_length, self.num_heads * self.head_dim)

    def forward(self, hidden_states, residual):
        fused_qkv = self.query_key_value(hidden_states)
        # print("fused", fused_qkv.shape)
        (query_layer, key_layer, value_layer) = self._split_heads(fused_qkv)

        batch_size, q_length, _, _ = query_layer.shape

        query_layer = query_layer.transpose(1, 2).reshape(batch_size * self.num_heads, q_length, self.head_dim)
        # print("query", query_layer.shape)
        key_layer = key_layer.permute(0, 2, 3, 1).reshape(batch_size * self.num_heads, self.head_dim, q_length)
        # print("key", query_layer.shape)
        value_layer = value_layer.transpose(1, 2).reshape(batch_size * self.num_heads, q_length, self.head_dim)
        # print("value", query_layer.shape)

        _, _, kv_length = key_layer.shape

        matmul_result = query_layer @ key_layer

        attention_scores = matmul_result.view(batch_size, self.num_heads, q_length, kv_length)

        input_dtype = attention_scores.dtype

        attention_probs = torch.softmax(attention_scores, dim=-1, dtype=torch.float32).to(input_dtype)
        attention_probs_reshaped = attention_probs.view(batch_size * self.num_heads, q_length, kv_length)

        context_layer = torch.bmm(attention_probs_reshaped, value_layer)

        context_layer = self._merge_heads(context_layer)

        output_tensor = self.dense(context_layer)
        output_tensor = dropout_add(output_tensor, residual, self.hidden_dropout, self.training)
        output_tensor = all_reduce(output_tensor)

        return output_tensor


class ParallelMLP(nn.Module):
    def __init__(self, config: BloomConfig):
        super().__init__()
        hidden_size = config.hidden_size

        assert (4 * hidden_size) % get_world_size() == 0

        self.dense_h_to_4h = nn.Linear(hidden_size, (4 * hidden_size) // get_world_size())
        self.gelu_impl = BloomGelu()
        self.dense_4h_to_h = nn.Linear((4 * hidden_size) // get_world_size(), hidden_size)
        self.hidden_dropout = config.hidden_dropout

    def forward(self, hidden_states: torch.Tensor, residual: torch.Tensor) -> torch.Tensor:
        hidden_states = self.gelu_impl(self.dense_h_to_4h(hidden_states))
        intermediate_output = self.dense_4h_to_h(hidden_states)
        intermediate_output = all_reduce(intermediate_output)

        output = dropout_add(intermediate_output, residual, self.hidden_dropout, self.training)

        return output


class ParallelBlock(nn.Module):
    def __init__(self, config: BloomConfig):
        super().__init__()
        hidden_size = config.hidden_size

        self.input_layernorm = LayerNorm(hidden_size, eps=config.layer_norm_epsilon)
        self.num_heads = config.n_head
        self.self_attention = ParallelAttention(config.hidden_size, config.n_head, config.hidden_dropout)
        self.post_attention_layernorm = LayerNorm(hidden_size, eps=config.layer_norm_epsilon)

        self.mlp = ParallelMLP(config)

        self.apply_residual_connection_post_layernorm = config.apply_residual_connection_post_layernorm

    def forward(
        self,
        hidden_states: torch.Tensor,
    ):
        # hidden_states: [batch_size, seq_length, hidden_size]

        # Layer norm at the beginning of the transformer layer.
        layernorm_output = self.input_layernorm(hidden_states)

        # Layer norm post the self attention.
        if self.apply_residual_connection_post_layernorm:
            residual = layernorm_output
        else:
            residual = hidden_states

        # Self attention.
        attention_output = self.self_attention(
            layernorm_output,
            residual,
        )

        layernorm_output = self.post_attention_layernorm(attention_output)

        # Get residual
        if self.apply_residual_connection_post_layernorm:
            residual = layernorm_output
        else:
            residual = attention_output

        # MLP.
        output = self.mlp(layernorm_output, residual)

        return output
