/*
 * Decompiled with CFR 0.152.
 */
package org.bouncycastle.jce.provider.test;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigInteger;
import java.security.AlgorithmParameterGenerator;
import java.security.AlgorithmParameters;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.security.Signature;
import java.security.SignatureException;
import java.security.interfaces.DSAParams;
import java.security.interfaces.DSAPrivateKey;
import java.security.interfaces.DSAPublicKey;
import java.security.spec.DSAParameterSpec;
import java.security.spec.DSAPrivateKeySpec;
import java.security.spec.DSAPublicKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.eac.EACObjectIdentifiers;
import org.bouncycastle.asn1.nist.NISTNamedCurves;
import org.bouncycastle.asn1.nist.NISTObjectIdentifiers;
import org.bouncycastle.asn1.teletrust.TeleTrusTObjectIdentifiers;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.asn1.x9.X9ObjectIdentifiers;
import org.bouncycastle.crypto.params.DSAParameters;
import org.bouncycastle.crypto.params.DSAPublicKeyParameters;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.signers.DSASigner;
import org.bouncycastle.jce.interfaces.PKCS12BagAttributeCarrier;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECNamedCurveGenParameterSpec;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.jce.spec.ECPrivateKeySpec;
import org.bouncycastle.jce.spec.ECPublicKeySpec;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.BigIntegers;
import org.bouncycastle.util.Strings;
import org.bouncycastle.util.encoders.Hex;
import org.bouncycastle.util.test.FixedSecureRandom;
import org.bouncycastle.util.test.SimpleTest;
import org.bouncycastle.util.test.TestRandomBigInteger;
import org.bouncycastle.util.test.TestRandomData;

public class DSATest
extends SimpleTest {
    byte[] k1 = Hex.decode("d5014e4b60ef2ba8b6211b4062ba3224e0427dd3");
    byte[] k2 = Hex.decode("345e8d05c075c3a508df729a1685690e68fcfb8c8117847e89063bca1f85d968fd281540b6e13bd1af989a1fbf17e06462bf511f9d0b140fb48ac1b1baa5bded");
    SecureRandom random = new FixedSecureRandom(new byte[][]{this.k1, this.k2});
    static final DSAPrivateKeySpec PRIVATE_KEY = new DSAPrivateKeySpec(new BigInteger("15382583218386677486843706921635237927801862255437148328980464126979"), new BigInteger("18111848663142005571178770624881214696591339256823507023544605891411707081617152319519180201250440615163700426054396403795303435564101919053459832890139496933938670005799610981765220283775567361483662648340339405220348871308593627647076689407931875483406244310337925809427432681864623551598136302441690546585427193224254314088256212718983105131138772434658820375111735710449331518776858786793875865418124429269409118756812841019074631004956409706877081612616347900606555802111224022921017725537417047242635829949739109274666495826205002104010355456981211025738812433088757102520562459649777989718122219159982614304359"), new BigInteger("19689526866605154788513693571065914024068069442724893395618704484701"), new BigInteger("2859278237642201956931085611015389087970918161297522023542900348087718063098423976428252369340967506010054236052095950169272612831491902295835660747775572934757474194739347115870723217560530672532404847508798651915566434553729839971841903983916294692452760249019857108409189016993380919900231322610083060784269299257074905043636029708121288037909739559605347853174853410208334242027740275688698461842637641566056165699733710043802697192696426360843173620679214131951400148855611740858610821913573088059404459364892373027492936037789337011875710759208498486908611261954026964574111219599568903257472567764789616958430"));
    static final DSAPublicKeySpec PUBLIC_KEY = new DSAPublicKeySpec(new BigInteger("3846308446317351758462473207111709291533523711306097971550086650257733363793010331167387218552238580749873869644606313965369322235288232349768695167652078383049323372009684761500716177377559133181601169463467065599372409821150709457431511200322947508290005178002097442907264027681030630279992466889399803263077740944083143145889944752236964609401160683369911999691536496253347241224687497038281983541563359385775312520539189474547346202842754393945875580322395107808219776288693340128414248732205723681487826216650723066229432216070313248464681099019648414795585656947634409725447389416166053148132419345627682740529"), PRIVATE_KEY.getP(), PRIVATE_KEY.getQ(), PRIVATE_KEY.getG());
    static final String[] MODIFIED_SIGNATURES = new String[]{"303e02811c1e41b479ad576905b960fe14eadb91b0ccf34843dab916173bb8c9cd021d00ade65988d237d30f9ef41dd424a4e1c8f16967cf3365813fe8786236", "303f0282001c1e41b479ad576905b960fe14eadb91b0ccf34843dab916173bb8c9cd021d00ade65988d237d30f9ef41dd424a4e1c8f16967cf3365813fe8786236", "303e021d001e41b479ad576905b960fe14eadb91b0ccf34843dab916173bb8c9cd021d00ade65988d237d30f9ef41dd424a4e1c8f16967cf3365813fe8786236", "303e021c1e41b479ad576905b960fe14eadb91b0ccf34843dab916173bb8c9cd02811d00ade65988d237d30f9ef41dd424a4e1c8f16967cf3365813fe8786236", "303f021c1e41b479ad576905b960fe14eadb91b0ccf34843dab916173bb8c9cd0282001d00ade65988d237d30f9ef41dd424a4e1c8f16967cf3365813fe8786236", "303e021c1e41b479ad576905b960fe14eadb91b0ccf34843dab916173bb8c9cd021e0000ade65988d237d30f9ef41dd424a4e1c8f16967cf3365813fe8786236", "30813d021c1e41b479ad576905b960fe14eadb91b0ccf34843dab916173bb8c9cd021d00ade65988d237d30f9ef41dd424a4e1c8f16967cf3365813fe8786236", "3082003d021c1e41b479ad576905b960fe14eadb91b0ccf34843dab916173bb8c9cd021d00ade65988d237d30f9ef41dd424a4e1c8f16967cf3365813fe8786236", "303d021c1e41b479ad576905b960fe14eadb91b0ccf34843dab916173bb8c9cd021d00ade65988d237d30f9ef41dd424a4e1c8f16967cf3365813fe87862360000", "3040021c57b10411b54ab248af03d8f2456676ebc6d3db5f1081492ac87e9ca8021d00942b117051d7d9d107fc42cac9c5a36a1fd7f0f8916ccca86cec4ed3040100", "303e021c57b10411b54ab248af03d8f2456676ebc6d3db5f1081492ac87e9ca802811d00942b117051d7d9d107fc42cac9c5a36a1fd7f0f8916ccca86cec4ed3"};

    private void testModified() throws Exception {
        KeyFactory kFact = KeyFactory.getInstance("DSA", "BC");
        PublicKey pubKey = kFact.generatePublic(PUBLIC_KEY);
        Signature sig = Signature.getInstance("DSA", "BC");
        int i = 0;
        while (i != MODIFIED_SIGNATURES.length) {
            boolean failed;
            sig.initVerify(pubKey);
            sig.update(Strings.toByteArray("Hello"));
            try {
                failed = !sig.verify(Hex.decode(MODIFIED_SIGNATURES[i]));
            }
            catch (SignatureException e) {
                failed = true;
            }
            this.isTrue("sig verified when shouldn't", failed);
            ++i;
        }
    }

    private void testCompat() throws Exception {
        if (Security.getProvider("SUN") == null) {
            return;
        }
        Signature s = Signature.getInstance("DSA", "SUN");
        KeyPairGenerator g = KeyPairGenerator.getInstance("DSA", "SUN");
        byte[] byArray = new byte[10];
        byArray[0] = 1;
        byArray[1] = 2;
        byArray[2] = 3;
        byArray[3] = 4;
        byArray[4] = 5;
        byArray[5] = 6;
        byArray[6] = 7;
        byArray[7] = 8;
        byArray[8] = 9;
        byte[] data = byArray;
        g.initialize(512, new SecureRandom());
        KeyPair p = g.generateKeyPair();
        PrivateKey sKey = p.getPrivate();
        PublicKey vKey = p.getPublic();
        s.initSign(sKey);
        s.update(data);
        byte[] sigBytes = s.sign();
        s = Signature.getInstance("DSA", "BC");
        s.initVerify(vKey);
        s.update(data);
        if (!s.verify(sigBytes)) {
            this.fail("SUN -> BC verification failed");
        }
        s.initSign(sKey);
        s.update(data);
        sigBytes = s.sign();
        s = Signature.getInstance("DSA", "SUN");
        s.initVerify(vKey);
        s.update(data);
        if (!s.verify(sigBytes)) {
            this.fail("BC -> SUN verification failed");
        }
        KeyFactory f = KeyFactory.getInstance("DSA", "BC");
        X509EncodedKeySpec x509s = new X509EncodedKeySpec(vKey.getEncoded());
        DSAPublicKey k1 = (DSAPublicKey)f.generatePublic(x509s);
        this.checkPublic(k1, vKey);
        PKCS8EncodedKeySpec pkcs8 = new PKCS8EncodedKeySpec(sKey.getEncoded());
        DSAPrivateKey k2 = (DSAPrivateKey)f.generatePrivate(pkcs8);
        this.checkPrivateKey(k2, sKey);
        f = KeyFactory.getInstance("DSA", "SUN");
        x509s = new X509EncodedKeySpec(k1.getEncoded());
        vKey = (DSAPublicKey)f.generatePublic(x509s);
        this.checkPublic(k1, vKey);
        pkcs8 = new PKCS8EncodedKeySpec(k2.getEncoded());
        sKey = f.generatePrivate(pkcs8);
        this.checkPrivateKey(k2, sKey);
    }

    private void testNullParameters() throws Exception {
        KeyFactory f = KeyFactory.getInstance("DSA", "BC");
        X509EncodedKeySpec x509s = new X509EncodedKeySpec(new SubjectPublicKeyInfo(new AlgorithmIdentifier(X9ObjectIdentifiers.id_dsa), new ASN1Integer(10001L)).getEncoded());
        DSAPublicKey key1 = (DSAPublicKey)f.generatePublic(x509s);
        DSAPublicKey key2 = (DSAPublicKey)f.generatePublic(x509s);
        this.isTrue("parameters not absent", key1.getParams() == null && key2.getParams() == null);
        this.isTrue("hashCode mismatch", key1.hashCode() == key2.hashCode());
        this.isTrue("not equal", key1.equals(key2));
        this.isTrue("encoding mismatch", Arrays.areEqual(x509s.getEncoded(), key1.getEncoded()));
    }

    private void testValidate() throws Exception {
        DSAParameterSpec dsaParams = new DSAParameterSpec(new BigInteger("F56C2A7D366E3EBDEAA1891FD2A0D099436438A673FED4D75F594959CFFEBCA7BE0FC72E4FE67D91D801CBA0693AC4ED9E411B41D19E2FD1699C4390AD27D94C69C0B143F1DC88932CFE2310C886412047BD9B1C7A67F8A25909132627F51A0C866877E672E555342BDF9355347DBD43B47156B2C20BAD9D2B071BC2FDCF9757F75C168C5D9FC43131BE162A0756D1BDEC2CA0EB0E3B018A8B38D3EF2487782AEB9FBF99D8B30499C55E4F61E5C7DCEE2A2BB55BD7F75FCDF00E48F2E8356BDB59D86114028F67B8E07B127744778AFF1CF1399A4D679D92FDE7D941C5C85C5D7BFF91BA69F9489D531D1EBFA727CFDA651390F8021719FA9F7216CEB177BD75", 16), new BigInteger("C24ED361870B61E0D367F008F99F8A1F75525889C89DB1B673C45AF5867CB467", 16), new BigInteger("8DC6CC814CAE4A1C05A3E186A6FE27EABA8CDB133FDCE14A963A92E809790CBA096EAA26140550C129FA2B98C16E84236AA33BF919CD6F587E048C52666576DB6E925C6CBE9B9EC5C16020F9A44C9F1C8F7A8E611C1F6EC2513EA6AA0B8D0F72FED73CA37DF240DB57BBB27431D618697B9E771B0B301D5DF05955425061A30DC6D33BB6D2A32BD0A75A0A71D2184F506372ABF84A56AEEEA8EB693BF29A640345FA1298A16E85421B2208D00068A5A42915F82CF0B858C8FA39D43D704B6927E0B2F916304E86FB6A1B487F07D8139E428BB096C6D67A76EC0B8D4EF274B8A2CF556D279AD267CCEF5AF477AFED029F485B5597739F5D0240F67C2D948A6279", 16));
        KeyFactory f = KeyFactory.getInstance("DSA", "BC");
        try {
            f.generatePublic(new DSAPublicKeySpec(BigInteger.valueOf(1L), dsaParams.getP(), dsaParams.getG(), dsaParams.getQ()));
            this.fail("no exception");
        }
        catch (Exception e) {
            this.isTrue("mismatch", "invalid KeySpec: y value does not appear to be in correct group".equals(e.getMessage()));
        }
    }

    private void testNONEwithDSA() throws Exception {
        byte[] dummySha1 = Hex.decode("01020304050607080910111213141516");
        KeyPairGenerator kpGen = KeyPairGenerator.getInstance("DSA", "BC");
        kpGen.initialize(512);
        KeyPair kp = kpGen.generateKeyPair();
        Signature sig = Signature.getInstance("NONEwithDSA", "BC");
        sig.initSign(kp.getPrivate());
        sig.update(dummySha1);
        byte[] sigBytes = sig.sign();
        sig.initVerify(kp.getPublic());
        sig.update(dummySha1);
        sig.verify(sigBytes);
        sig.update(dummySha1);
        if (!sig.verify(sigBytes)) {
            this.fail("NONEwithDSA failed to reset");
        }
        DSAPublicKey key = (DSAPublicKey)kp.getPublic();
        DSAParameters params = new DSAParameters(key.getParams().getP(), key.getParams().getQ(), key.getParams().getG());
        DSAPublicKeyParameters keyParams = new DSAPublicKeyParameters(key.getY(), params);
        DSASigner signer = new DSASigner();
        ASN1Sequence derSig = ASN1Sequence.getInstance(ASN1Primitive.fromByteArray(sigBytes));
        signer.init(false, keyParams);
        if (!signer.verifySignature(dummySha1, ASN1Integer.getInstance(derSig.getObjectAt(0)).getValue(), ASN1Integer.getInstance(derSig.getObjectAt(1)).getValue())) {
            this.fail("NONEwithDSA not really NONE!");
        }
    }

    private void checkPublic(DSAPublicKey k1, PublicKey vKey) {
        if (!k1.getY().equals(((DSAPublicKey)vKey).getY())) {
            this.fail("public number not decoded properly");
        }
        if (!k1.getParams().getG().equals(((DSAPublicKey)vKey).getParams().getG())) {
            this.fail("public generator not decoded properly");
        }
        if (!k1.getParams().getP().equals(((DSAPublicKey)vKey).getParams().getP())) {
            this.fail("public p value not decoded properly");
        }
        if (!k1.getParams().getQ().equals(((DSAPublicKey)vKey).getParams().getQ())) {
            this.fail("public q value not decoded properly");
        }
    }

    private void checkPrivateKey(DSAPrivateKey k2, PrivateKey sKey) {
        if (!k2.getX().equals(((DSAPrivateKey)sKey).getX())) {
            this.fail("private number not decoded properly");
        }
        if (!k2.getParams().getG().equals(((DSAPrivateKey)sKey).getParams().getG())) {
            this.fail("private generator not decoded properly");
        }
        if (!k2.getParams().getP().equals(((DSAPrivateKey)sKey).getParams().getP())) {
            this.fail("private p value not decoded properly");
        }
        if (!k2.getParams().getQ().equals(((DSAPrivateKey)sKey).getParams().getQ())) {
            this.fail("private q value not decoded properly");
        }
    }

    private Object serializeDeserialize(Object o) throws Exception {
        ByteArrayOutputStream bOut = new ByteArrayOutputStream();
        ObjectOutputStream oOut = new ObjectOutputStream(bOut);
        oOut.writeObject(o);
        oOut.close();
        ObjectInputStream oIn = new ObjectInputStream(new ByteArrayInputStream(bOut.toByteArray()));
        return oIn.readObject();
    }

    private void testECDSA239bitPrime() throws Exception {
        BigInteger[] sig;
        BigInteger r = new BigInteger("308636143175167811492622547300668018854959378758531778147462058306432176");
        BigInteger s = new BigInteger("323813553209797357708078776831250505931891051755007842781978505179448783");
        byte[] kData = BigIntegers.asUnsignedByteArray(new BigInteger("700000017569056646655505781757157107570501575775705779575555657156756655"));
        TestRandomBigInteger k = new TestRandomBigInteger(kData);
        ECCurve.Fp curve = new ECCurve.Fp(new BigInteger("883423532389192164791648750360308885314476597252960362792450860609699839"), new BigInteger("7fffffffffffffffffffffff7fffffffffff8000000000007ffffffffffc", 16), new BigInteger("6b016c3bdcf18941d0d654921475ca71a9db2fb27d1d37796185c2942c0a", 16));
        ECParameterSpec spec = new ECParameterSpec(curve, curve.decodePoint(Hex.decode("020ffa963cdca8816ccc33b8642bedf905c3d358573d3f27fbbd3b3cb9aaaf")), new BigInteger("883423532389192164791648750360308884807550341691627752275345424702807307"));
        ECPrivateKeySpec priKey = new ECPrivateKeySpec(new BigInteger("876300101507107567501066130761671078357010671067781776716671676178726717"), spec);
        ECPublicKeySpec pubKey = new ECPublicKeySpec(curve.decodePoint(Hex.decode("025b6dc53bc61a2548ffb0f671472de6c9521a9d2d2534e65abfcbd5fe0c70")), spec);
        Signature sgr = Signature.getInstance("ECDSA", "BC");
        KeyFactory f = KeyFactory.getInstance("ECDSA", "BC");
        PrivateKey sKey = f.generatePrivate(priKey);
        PublicKey vKey = f.generatePublic(pubKey);
        sgr.initSign(sKey, k);
        byte[] message = new byte[]{97, 98, 99};
        sgr.update(message);
        byte[] sigBytes = sgr.sign();
        sgr.initVerify(vKey);
        sgr.update(message);
        if (!sgr.verify(sigBytes)) {
            this.fail("239 Bit EC verification failed");
        }
        if (!r.equals((sig = this.derDecode(sigBytes))[0])) {
            this.fail("r component wrong." + Strings.lineSeparator() + " expecting: " + r + Strings.lineSeparator() + " got      : " + sig[0]);
        }
        if (!s.equals(sig[1])) {
            this.fail("s component wrong." + Strings.lineSeparator() + " expecting: " + s + Strings.lineSeparator() + " got      : " + sig[1]);
        }
    }

    private void testNONEwithECDSA239bitPrime() throws Exception {
        ECCurve.Fp curve = new ECCurve.Fp(new BigInteger("883423532389192164791648750360308885314476597252960362792450860609699839"), new BigInteger("7fffffffffffffffffffffff7fffffffffff8000000000007ffffffffffc", 16), new BigInteger("6b016c3bdcf18941d0d654921475ca71a9db2fb27d1d37796185c2942c0a", 16));
        ECParameterSpec spec = new ECParameterSpec(curve, curve.decodePoint(Hex.decode("020ffa963cdca8816ccc33b8642bedf905c3d358573d3f27fbbd3b3cb9aaaf")), new BigInteger("883423532389192164791648750360308884807550341691627752275345424702807307"));
        ECPrivateKeySpec priKey = new ECPrivateKeySpec(new BigInteger("876300101507107567501066130761671078357010671067781776716671676178726717"), spec);
        ECPublicKeySpec pubKey = new ECPublicKeySpec(curve.decodePoint(Hex.decode("025b6dc53bc61a2548ffb0f671472de6c9521a9d2d2534e65abfcbd5fe0c70")), spec);
        Signature sgr = Signature.getInstance("NONEwithECDSA", "BC");
        KeyFactory f = KeyFactory.getInstance("ECDSA", "BC");
        PrivateKey sKey = f.generatePrivate(priKey);
        PublicKey vKey = f.generatePublic(pubKey);
        byte[] message = "abc".getBytes();
        byte[] sig = Hex.decode("3040021e2cb7f36803ebb9c427c58d8265f11fc5084747133078fc279de874fbecb0021e64cb19604be06c57e761b3de5518f71de0f6e0cd2df677cec8a6ffcb690d");
        this.checkMessage(sgr, sKey, vKey, message, sig);
        message = "abcdefghijklmnopqrstuvwxyz".getBytes();
        sig = Hex.decode("3040021e2cb7f36803ebb9c427c58d8265f11fc5084747133078fc279de874fbecb0021e43fd65b3363d76aabef8630572257dbb67c82818ad9fad31256539b1b02c");
        this.checkMessage(sgr, sKey, vKey, message, sig);
        message = "a very very long message gauranteed to cause an overflow".getBytes();
        sig = Hex.decode("3040021e2cb7f36803ebb9c427c58d8265f11fc5084747133078fc279de874fbecb0021e7d5be84b22937a1691859a3c6fe45ed30b108574431d01b34025825ec17a");
        this.checkMessage(sgr, sKey, vKey, message, sig);
    }

    private void testECDSAP256sha3(ASN1ObjectIdentifier sigOid, int size, BigInteger s) throws Exception {
        X9ECParameters p = NISTNamedCurves.getByName("P-256");
        KeyFactory ecKeyFact = KeyFactory.getInstance("EC", "BC");
        ECDomainParameters params = new ECDomainParameters(p.getCurve(), p.getG(), p.getN(), p.getH());
        ECCurve curve = p.getCurve();
        ECParameterSpec spec = new ECParameterSpec(curve, p.getG(), p.getN());
        ECPrivateKeySpec priKey = new ECPrivateKeySpec(new BigInteger("20186677036482506117540275567393538695075300175221296989956723148347484984008"), spec);
        ECPublicKeySpec pubKey = new ECPublicKeySpec(params.getCurve().decodePoint(Hex.decode("03596375E6CE57E0F20294FC46BDFCFD19A39F8161B58695B3EC5B3D16427C274D")), spec);
        this.doEcDsaTest("SHA3-" + size + "withECDSA", s, ecKeyFact, pubKey, priKey);
        this.doEcDsaTest(sigOid.getId(), s, ecKeyFact, pubKey, priKey);
    }

    private void doEcDsaTest(String sigName, BigInteger s, KeyFactory ecKeyFact, ECPublicKeySpec pubKey, ECPrivateKeySpec priKey) throws NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException, InvalidKeySpecException, SignatureException {
        BigInteger sigS;
        TestRandomBigInteger k = new TestRandomBigInteger(BigIntegers.asUnsignedByteArray(new BigInteger("72546832179840998877302529996971396893172522460793442785601695562409154906335")));
        byte[] M = Hex.decode("1BD4ED430B0F384B4E8D458EFF1A8A553286D7AC21CB2F6806172EF5F94A06AD");
        Signature dsa = Signature.getInstance(sigName, "BC");
        dsa.initSign(ecKeyFact.generatePrivate(priKey), k);
        dsa.update(M, 0, M.length);
        byte[] encSig = dsa.sign();
        ASN1Sequence sig = ASN1Sequence.getInstance(encSig);
        BigInteger r = new BigInteger("97354732615802252173078420023658453040116611318111190383344590814578738210384");
        BigInteger sigR = ASN1Integer.getInstance(sig.getObjectAt(0)).getValue();
        if (!r.equals(sigR)) {
            this.fail("r component wrong." + Strings.lineSeparator() + " expecting: " + r.toString(16) + Strings.lineSeparator() + " got      : " + sigR.toString(16));
        }
        if (!s.equals(sigS = ASN1Integer.getInstance(sig.getObjectAt(1)).getValue())) {
            this.fail("s component wrong." + Strings.lineSeparator() + " expecting: " + s.toString(16) + Strings.lineSeparator() + " got      : " + sigS.toString(16));
        }
        dsa.initVerify(ecKeyFact.generatePublic(pubKey));
        dsa.update(M, 0, M.length);
        if (!dsa.verify(encSig)) {
            this.fail("signature fails");
        }
    }

    private void testDSAsha3(ASN1ObjectIdentifier sigOid, int size, BigInteger s) throws Exception {
        DSAParameterSpec dsaParams = new DSAParameterSpec(new BigInteger("F56C2A7D366E3EBDEAA1891FD2A0D099436438A673FED4D75F594959CFFEBCA7BE0FC72E4FE67D91D801CBA0693AC4ED9E411B41D19E2FD1699C4390AD27D94C69C0B143F1DC88932CFE2310C886412047BD9B1C7A67F8A25909132627F51A0C866877E672E555342BDF9355347DBD43B47156B2C20BAD9D2B071BC2FDCF9757F75C168C5D9FC43131BE162A0756D1BDEC2CA0EB0E3B018A8B38D3EF2487782AEB9FBF99D8B30499C55E4F61E5C7DCEE2A2BB55BD7F75FCDF00E48F2E8356BDB59D86114028F67B8E07B127744778AFF1CF1399A4D679D92FDE7D941C5C85C5D7BFF91BA69F9489D531D1EBFA727CFDA651390F8021719FA9F7216CEB177BD75", 16), new BigInteger("C24ED361870B61E0D367F008F99F8A1F75525889C89DB1B673C45AF5867CB467", 16), new BigInteger("8DC6CC814CAE4A1C05A3E186A6FE27EABA8CDB133FDCE14A963A92E809790CBA096EAA26140550C129FA2B98C16E84236AA33BF919CD6F587E048C52666576DB6E925C6CBE9B9EC5C16020F9A44C9F1C8F7A8E611C1F6EC2513EA6AA0B8D0F72FED73CA37DF240DB57BBB27431D618697B9E771B0B301D5DF05955425061A30DC6D33BB6D2A32BD0A75A0A71D2184F506372ABF84A56AEEEA8EB693BF29A640345FA1298A16E85421B2208D00068A5A42915F82CF0B858C8FA39D43D704B6927E0B2F916304E86FB6A1B487F07D8139E428BB096C6D67A76EC0B8D4EF274B8A2CF556D279AD267CCEF5AF477AFED029F485B5597739F5D0240F67C2D948A6279", 16));
        BigInteger x = new BigInteger("0CAF2EF547EC49C4F3A6FE6DF4223A174D01F2C115D49A6F73437C29A2A8458C", 16);
        BigInteger y = new BigInteger("2828003D7C747199143C370FDD07A2861524514ACC57F63F80C38C2087C6B795B62DE1C224BF8D1D1424E60CE3F5AE3F76C754A2464AF292286D873A7A30B7EACBBC75AAFDE7191D9157598CDB0B60E0C5AA3F6EBE425500C611957DBF5ED35490714A42811FDCDEB19AF2AB30BEADFF2907931CEE7F3B55532CFFAEB371F84F01347630EB227A419B1F3F558BC8A509D64A765D8987D493B007C4412C297CAF41566E26FAEE475137EC781A0DC088A26C8804A98C23140E7C936281864B99571EE95C416AA38CEEBB41FDBFF1EB1D1DC97B63CE1355257627C8B0FD840DDB20ED35BE92F08C49AEA5613957D7E5C7A6D5A5834B4CB069E0831753ECF65BA02B", 16);
        DSAPrivateKeySpec priKey = new DSAPrivateKeySpec(x, dsaParams.getP(), dsaParams.getQ(), dsaParams.getG());
        DSAPublicKeySpec pubKey = new DSAPublicKeySpec(y, dsaParams.getP(), dsaParams.getQ(), dsaParams.getG());
        KeyFactory dsaKeyFact = KeyFactory.getInstance("DSA", "BC");
        this.doDsaTest("SHA3-" + size + "withDSA", s, dsaKeyFact, pubKey, priKey);
        this.doDsaTest(sigOid.getId(), s, dsaKeyFact, pubKey, priKey);
    }

    private void doDsaTest(String sigName, BigInteger s, KeyFactory ecKeyFact, DSAPublicKeySpec pubKey, DSAPrivateKeySpec priKey) throws NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException, InvalidKeySpecException, SignatureException {
        BigInteger sigS;
        FixedSecureRandom k = new FixedSecureRandom(new FixedSecureRandom.Source[]{new FixedSecureRandom.BigInteger(BigIntegers.asUnsignedByteArray(new BigInteger("72546832179840998877302529996971396893172522460793442785601695562409154906335"))), new FixedSecureRandom.Data(Hex.decode("01020304"))});
        byte[] M = Hex.decode("1BD4ED430B0F384B4E8D458EFF1A8A553286D7AC21CB2F6806172EF5F94A06AD");
        Signature dsa = Signature.getInstance(sigName, "BC");
        dsa.initSign(ecKeyFact.generatePrivate(priKey), k);
        dsa.update(M, 0, M.length);
        byte[] encSig = dsa.sign();
        ASN1Sequence sig = ASN1Sequence.getInstance(encSig);
        BigInteger r = new BigInteger("4864074fe30e6601268ee663440e4d9b703f62673419864e91e9edb0338ce510", 16);
        BigInteger sigR = ASN1Integer.getInstance(sig.getObjectAt(0)).getValue();
        if (!r.equals(sigR)) {
            this.fail("r component wrong." + Strings.lineSeparator() + " expecting: " + r.toString(16) + Strings.lineSeparator() + " got      : " + sigR.toString(16));
        }
        if (!s.equals(sigS = ASN1Integer.getInstance(sig.getObjectAt(1)).getValue())) {
            this.fail("s component wrong." + Strings.lineSeparator() + " expecting: " + s.toString(16) + Strings.lineSeparator() + " got      : " + sigS.toString(16));
        }
        dsa.initVerify(ecKeyFact.generatePublic(pubKey));
        dsa.update(M, 0, M.length);
        if (!dsa.verify(encSig)) {
            this.fail("signature fails");
        }
    }

    private void checkMessage(Signature sgr, PrivateKey sKey, PublicKey vKey, byte[] message, byte[] sig) throws InvalidKeyException, SignatureException {
        byte[] kData = BigIntegers.asUnsignedByteArray(new BigInteger("700000017569056646655505781757157107570501575775705779575555657156756655"));
        TestRandomBigInteger k = new TestRandomBigInteger(kData);
        sgr.initSign(sKey, k);
        sgr.update(message);
        byte[] sigBytes = sgr.sign();
        if (!Arrays.areEqual(sigBytes, sig)) {
            this.fail(String.valueOf(new String(message)) + " signature incorrect");
        }
        sgr.initVerify(vKey);
        sgr.update(message);
        if (!sgr.verify(sigBytes)) {
            this.fail(String.valueOf(new String(message)) + " verification failed");
        }
    }

    private void testECDSA239bitBinary() throws Exception {
        BigInteger[] sig;
        BigInteger r = new BigInteger("21596333210419611985018340039034612628818151486841789642455876922391552");
        BigInteger s = new BigInteger("197030374000731686738334997654997227052849804072198819102649413465737174");
        byte[] kData = BigIntegers.asUnsignedByteArray(new BigInteger("171278725565216523967285789236956265265265235675811949404040041670216363"));
        TestRandomBigInteger k = new TestRandomBigInteger(kData);
        ECCurve.F2m curve = new ECCurve.F2m(239, 36, new BigInteger("32010857077C5431123A46B808906756F543423E8D27877578125778AC76", 16), new BigInteger("790408F2EEDAF392B012EDEFB3392F30F4327C0CA3F31FC383C422AA8C16", 16));
        ECParameterSpec params = new ECParameterSpec(curve, curve.decodePoint(Hex.decode("0457927098FA932E7C0A96D3FD5B706EF7E5F5C156E16B7E7C86038552E91D61D8EE5077C33FECF6F1A16B268DE469C3C7744EA9A971649FC7A9616305")), new BigInteger("220855883097298041197912187592864814557886993776713230936715041207411783"), BigInteger.valueOf(4L));
        ECPrivateKeySpec priKeySpec = new ECPrivateKeySpec(new BigInteger("145642755521911534651321230007534120304391871461646461466464667494947990"), params);
        ECPublicKeySpec pubKeySpec = new ECPublicKeySpec(curve.decodePoint(Hex.decode("045894609CCECF9A92533F630DE713A958E96C97CCB8F5ABB5A688A238DEED6DC2D9D0C94EBFB7D526BA6A61764175B99CB6011E2047F9F067293F57F5")), params);
        Signature sgr = Signature.getInstance("ECDSA", "BC");
        KeyFactory f = KeyFactory.getInstance("ECDSA", "BC");
        PrivateKey sKey = f.generatePrivate(priKeySpec);
        PublicKey vKey = f.generatePublic(pubKeySpec);
        byte[] message = new byte[]{97, 98, 99};
        sgr.initSign(sKey, k);
        sgr.update(message);
        byte[] sigBytes = sgr.sign();
        sgr.initVerify(vKey);
        sgr.update(message);
        if (!sgr.verify(sigBytes)) {
            this.fail("239 Bit EC verification failed");
        }
        if (!r.equals((sig = this.derDecode(sigBytes))[0])) {
            this.fail("r component wrong." + Strings.lineSeparator() + " expecting: " + r + Strings.lineSeparator() + " got      : " + sig[0]);
        }
        if (!s.equals(sig[1])) {
            this.fail("s component wrong." + Strings.lineSeparator() + " expecting: " + s + Strings.lineSeparator() + " got      : " + sig[1]);
        }
    }

    private void testECDSA239bitBinary(String algorithm, ASN1ObjectIdentifier oid) throws Exception {
        byte[] kData = BigIntegers.asUnsignedByteArray(new BigInteger("171278725565216523967285789236956265265265235675811949404040041670216363"));
        TestRandomBigInteger k = new TestRandomBigInteger(kData);
        ECCurve.F2m curve = new ECCurve.F2m(239, 36, new BigInteger("32010857077C5431123A46B808906756F543423E8D27877578125778AC76", 16), new BigInteger("790408F2EEDAF392B012EDEFB3392F30F4327C0CA3F31FC383C422AA8C16", 16));
        ECParameterSpec params = new ECParameterSpec(curve, curve.decodePoint(Hex.decode("0457927098FA932E7C0A96D3FD5B706EF7E5F5C156E16B7E7C86038552E91D61D8EE5077C33FECF6F1A16B268DE469C3C7744EA9A971649FC7A9616305")), new BigInteger("220855883097298041197912187592864814557886993776713230936715041207411783"), BigInteger.valueOf(4L));
        ECPrivateKeySpec priKeySpec = new ECPrivateKeySpec(new BigInteger("145642755521911534651321230007534120304391871461646461466464667494947990"), params);
        ECPublicKeySpec pubKeySpec = new ECPublicKeySpec(curve.decodePoint(Hex.decode("045894609CCECF9A92533F630DE713A958E96C97CCB8F5ABB5A688A238DEED6DC2D9D0C94EBFB7D526BA6A61764175B99CB6011E2047F9F067293F57F5")), params);
        Signature sgr = Signature.getInstance(algorithm, "BC");
        KeyFactory f = KeyFactory.getInstance("ECDSA", "BC");
        PrivateKey sKey = f.generatePrivate(priKeySpec);
        PublicKey vKey = f.generatePublic(pubKeySpec);
        byte[] message = new byte[]{97, 98, 99};
        sgr.initSign(sKey, k);
        sgr.update(message);
        byte[] sigBytes = sgr.sign();
        sgr = Signature.getInstance(oid.getId(), "BC");
        sgr.initVerify(vKey);
        sgr.update(message);
        if (!sgr.verify(sigBytes)) {
            this.fail("239 Bit EC RIPEMD160 verification failed");
        }
    }

    private void testGeneration() throws Exception {
        Signature s = Signature.getInstance("DSA", "BC");
        KeyPairGenerator g = KeyPairGenerator.getInstance("DSA", "BC");
        byte[] byArray = new byte[10];
        byArray[0] = 1;
        byArray[1] = 2;
        byArray[2] = 3;
        byArray[3] = 4;
        byArray[4] = 5;
        byArray[5] = 6;
        byArray[6] = 7;
        byArray[7] = 8;
        byArray[8] = 9;
        byte[] data = byArray;
        try {
            g.initialize(513, new SecureRandom());
            this.fail("illegal parameter 513 check failed.");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            g.initialize(510, new SecureRandom());
            this.fail("illegal parameter 510 check failed.");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            g.initialize(1025, new SecureRandom());
            this.fail("illegal parameter 1025 check failed.");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        g.initialize(512, new SecureRandom());
        KeyPair p = g.generateKeyPair();
        PrivateKey sKey = p.getPrivate();
        PublicKey vKey = p.getPublic();
        s.initSign(sKey);
        s.update(data);
        byte[] sigBytes = s.sign();
        s = Signature.getInstance("DSA", "BC");
        s.initVerify(vKey);
        s.update(data);
        if (!s.verify(sigBytes)) {
            this.fail("DSA verification failed");
        }
        DSAPublicKey k1 = (DSAPublicKey)this.serializeDeserialize(vKey);
        this.checkPublic(k1, vKey);
        this.checkEquals(k1, vKey);
        DSAPrivateKey k2 = (DSAPrivateKey)this.serializeDeserialize(sKey);
        this.checkPrivateKey(k2, sKey);
        this.checkEquals(k2, sKey);
        if (!(k2 instanceof PKCS12BagAttributeCarrier)) {
            this.fail("private key not implementing PKCS12 attribute carrier");
        }
        s = Signature.getInstance("ECDSA", "BC");
        g = KeyPairGenerator.getInstance("ECDSA", "BC");
        ECCurve curve = new ECCurve.Fp(new BigInteger("883423532389192164791648750360308885314476597252960362792450860609699839"), new BigInteger("7fffffffffffffffffffffff7fffffffffff8000000000007ffffffffffc", 16), new BigInteger("6b016c3bdcf18941d0d654921475ca71a9db2fb27d1d37796185c2942c0a", 16));
        ECParameterSpec ecSpec = new ECParameterSpec(curve, curve.decodePoint(Hex.decode("020ffa963cdca8816ccc33b8642bedf905c3d358573d3f27fbbd3b3cb9aaaf")), new BigInteger("883423532389192164791648750360308884807550341691627752275345424702807307"));
        g.initialize(ecSpec, new SecureRandom());
        p = g.generateKeyPair();
        sKey = p.getPrivate();
        vKey = p.getPublic();
        s.initSign(sKey);
        s.update(data);
        sigBytes = s.sign();
        s = Signature.getInstance("ECDSA", "BC");
        s.initVerify(vKey);
        s.update(data);
        if (!s.verify(sigBytes)) {
            this.fail("ECDSA verification failed");
        }
        PublicKey eck1 = (PublicKey)this.serializeDeserialize(vKey);
        this.checkEquals(eck1, vKey);
        PrivateKey eck2 = (PrivateKey)this.serializeDeserialize(sKey);
        this.checkEquals(eck2, sKey);
        g.initialize(new ECNamedCurveGenParameterSpec("P-256"), new SecureRandom());
        p = g.generateKeyPair();
        sKey = p.getPrivate();
        vKey = p.getPublic();
        s.initSign(sKey);
        s.update(data);
        sigBytes = s.sign();
        s = Signature.getInstance("ECDSA", "BC");
        s.initVerify(vKey);
        s.update(data);
        if (!s.verify(sigBytes)) {
            this.fail("ECDSA verification failed");
        }
        eck1 = (PublicKey)this.serializeDeserialize(vKey);
        this.checkEquals(eck1, vKey);
        eck2 = (PrivateKey)this.serializeDeserialize(sKey);
        this.checkEquals(eck2, sKey);
        s = Signature.getInstance("ECDSA", "BC");
        g = KeyPairGenerator.getInstance("ECDSA", "BC");
        curve = new ECCurve.F2m(239, 36, new BigInteger("32010857077C5431123A46B808906756F543423E8D27877578125778AC76", 16), new BigInteger("790408F2EEDAF392B012EDEFB3392F30F4327C0CA3F31FC383C422AA8C16", 16));
        ecSpec = new ECParameterSpec(curve, curve.decodePoint(Hex.decode("0457927098FA932E7C0A96D3FD5B706EF7E5F5C156E16B7E7C86038552E91D61D8EE5077C33FECF6F1A16B268DE469C3C7744EA9A971649FC7A9616305")), new BigInteger("220855883097298041197912187592864814557886993776713230936715041207411783"), BigInteger.valueOf(4L));
        g.initialize(ecSpec, new SecureRandom());
        p = g.generateKeyPair();
        sKey = p.getPrivate();
        vKey = p.getPublic();
        s.initSign(sKey);
        s.update(data);
        sigBytes = s.sign();
        s = Signature.getInstance("ECDSA", "BC");
        s.initVerify(vKey);
        s.update(data);
        if (!s.verify(sigBytes)) {
            this.fail("ECDSA verification failed");
        }
        eck1 = (PublicKey)this.serializeDeserialize(vKey);
        this.checkEquals(eck1, vKey);
        eck2 = (PrivateKey)this.serializeDeserialize(sKey);
        this.checkEquals(eck2, sKey);
        if (!(eck2 instanceof PKCS12BagAttributeCarrier)) {
            this.fail("private key not implementing PKCS12 attribute carrier");
        }
    }

    private void checkEquals(Object o1, Object o2) {
        if (!o1.equals(o2)) {
            this.fail("comparison test failed");
        }
        if (o1.hashCode() != o2.hashCode()) {
            this.fail("hashCode test failed");
        }
    }

    private void testParameters() throws Exception {
        AlgorithmParameterGenerator a = AlgorithmParameterGenerator.getInstance("DSA", "BC");
        a.init(512, this.random);
        AlgorithmParameters params = a.generateParameters();
        byte[] encodeParams = params.getEncoded();
        AlgorithmParameters a2 = AlgorithmParameters.getInstance("DSA", "BC");
        a2.init(encodeParams);
        byte[] encodeParams_2 = a2.getEncoded();
        if (!this.areEqual(encodeParams, encodeParams_2)) {
            this.fail("encode/decode parameters failed");
        }
        DSAParameterSpec dsaP = params.getParameterSpec(DSAParameterSpec.class);
        KeyPairGenerator g = KeyPairGenerator.getInstance("DSA", "BC");
        g.initialize(dsaP, new SecureRandom());
        KeyPair p = g.generateKeyPair();
        PrivateKey sKey = p.getPrivate();
        PublicKey vKey = p.getPublic();
        Signature s = Signature.getInstance("DSA", "BC");
        byte[] byArray = new byte[10];
        byArray[0] = 1;
        byArray[1] = 2;
        byArray[2] = 3;
        byArray[3] = 4;
        byArray[4] = 5;
        byArray[5] = 6;
        byArray[6] = 7;
        byArray[7] = 8;
        byArray[8] = 9;
        byte[] data = byArray;
        s.initSign(sKey);
        s.update(data);
        byte[] sigBytes = s.sign();
        s = Signature.getInstance("DSA", "BC");
        s.initVerify(vKey);
        s.update(data);
        if (!s.verify(sigBytes)) {
            this.fail("DSA verification failed");
        }
    }

    private void testDSA2Parameters() throws Exception {
        byte[] seed = Hex.decode("4783081972865EA95D43318AB2EAF9C61A2FC7BBF1B772A09017BDF5A58F4FF0");
        AlgorithmParameterGenerator a = AlgorithmParameterGenerator.getInstance("DSA", "BC");
        a.init(2048, (SecureRandom)new DSATestSecureRandom(seed));
        AlgorithmParameters params = a.generateParameters();
        DSAParameterSpec dsaP = params.getParameterSpec(DSAParameterSpec.class);
        if (!dsaP.getQ().equals(new BigInteger("C24ED361870B61E0D367F008F99F8A1F75525889C89DB1B673C45AF5867CB467", 16))) {
            this.fail("Q incorrect");
        }
        if (!dsaP.getP().equals(new BigInteger("F56C2A7D366E3EBDEAA1891FD2A0D099436438A673FED4D75F594959CFFEBCA7BE0FC72E4FE67D91D801CBA0693AC4ED9E411B41D19E2FD1699C4390AD27D94C69C0B143F1DC88932CFE2310C886412047BD9B1C7A67F8A25909132627F51A0C866877E672E555342BDF9355347DBD43B47156B2C20BAD9D2B071BC2FDCF9757F75C168C5D9FC43131BE162A0756D1BDEC2CA0EB0E3B018A8B38D3EF2487782AEB9FBF99D8B30499C55E4F61E5C7DCEE2A2BB55BD7F75FCDF00E48F2E8356BDB59D86114028F67B8E07B127744778AFF1CF1399A4D679D92FDE7D941C5C85C5D7BFF91BA69F9489D531D1EBFA727CFDA651390F8021719FA9F7216CEB177BD75", 16))) {
            this.fail("P incorrect");
        }
        if (!dsaP.getG().equals(new BigInteger("8DC6CC814CAE4A1C05A3E186A6FE27EABA8CDB133FDCE14A963A92E809790CBA096EAA26140550C129FA2B98C16E84236AA33BF919CD6F587E048C52666576DB6E925C6CBE9B9EC5C16020F9A44C9F1C8F7A8E611C1F6EC2513EA6AA0B8D0F72FED73CA37DF240DB57BBB27431D618697B9E771B0B301D5DF05955425061A30DC6D33BB6D2A32BD0A75A0A71D2184F506372ABF84A56AEEEA8EB693BF29A640345FA1298A16E85421B2208D00068A5A42915F82CF0B858C8FA39D43D704B6927E0B2F916304E86FB6A1B487F07D8139E428BB096C6D67A76EC0B8D4EF274B8A2CF556D279AD267CCEF5AF477AFED029F485B5597739F5D0240F67C2D948A6279", 16))) {
            this.fail("G incorrect");
        }
        KeyPairGenerator g = KeyPairGenerator.getInstance("DSA", "BC");
        g.initialize(dsaP, (SecureRandom)new TestRandomBigInteger(Hex.decode("0CAF2EF547EC49C4F3A6FE6DF4223A174D01F2C115D49A6F73437C29A2A8458C")));
        KeyPair p = g.generateKeyPair();
        DSAPrivateKey sKey = (DSAPrivateKey)p.getPrivate();
        DSAPublicKey vKey = (DSAPublicKey)p.getPublic();
        if (!vKey.getY().equals(new BigInteger("2828003D7C747199143C370FDD07A2861524514ACC57F63F80C38C2087C6B795B62DE1C224BF8D1D1424E60CE3F5AE3F76C754A2464AF292286D873A7A30B7EACBBC75AAFDE7191D9157598CDB0B60E0C5AA3F6EBE425500C611957DBF5ED35490714A42811FDCDEB19AF2AB30BEADFF2907931CEE7F3B55532CFFAEB371F84F01347630EB227A419B1F3F558BC8A509D64A765D8987D493B007C4412C297CAF41566E26FAEE475137EC781A0DC088A26C8804A98C23140E7C936281864B99571EE95C416AA38CEEBB41FDBFF1EB1D1DC97B63CE1355257627C8B0FD840DDB20ED35BE92F08C49AEA5613957D7E5C7A6D5A5834B4CB069E0831753ECF65BA02B", 16))) {
            this.fail("Y value incorrect");
        }
        if (!sKey.getX().equals(new BigInteger("0CAF2EF547EC49C4F3A6FE6DF4223A174D01F2C115D49A6F73437C29A2A8458C", 16))) {
            this.fail("X value incorrect");
        }
        byte[] encodeParams = params.getEncoded();
        AlgorithmParameters a2 = AlgorithmParameters.getInstance("DSA", "BC");
        a2.init(encodeParams);
        byte[] encodeParams_2 = a2.getEncoded();
        if (!this.areEqual(encodeParams, encodeParams_2)) {
            this.fail("encode/decode parameters failed");
        }
        Signature s = Signature.getInstance("DSA", "BC");
        byte[] byArray = new byte[10];
        byArray[0] = 1;
        byArray[1] = 2;
        byArray[2] = 3;
        byArray[3] = 4;
        byArray[4] = 5;
        byArray[5] = 6;
        byArray[6] = 7;
        byArray[7] = 8;
        byArray[8] = 9;
        byte[] data = byArray;
        s.initSign(sKey);
        s.update(data);
        byte[] sigBytes = s.sign();
        s = Signature.getInstance("DSA", "BC");
        s.initVerify(vKey);
        s.update(data);
        if (!s.verify(sigBytes)) {
            this.fail("DSA verification failed");
        }
    }

    private void testKeyGeneration(int keysize) throws Exception {
        KeyPairGenerator generator = KeyPairGenerator.getInstance("DSA", "BC");
        generator.initialize(keysize);
        KeyPair keyPair = generator.generateKeyPair();
        DSAPrivateKey priv = (DSAPrivateKey)keyPair.getPrivate();
        DSAParams params = priv.getParams();
        this.isTrue("keysize mismatch", keysize == params.getP().bitLength());
        int qsize = params.getQ().bitLength();
        switch (keysize) {
            case 1024: {
                this.isTrue("Invalid qsize for 1024 bit key:" + qsize, qsize >= 160);
                break;
            }
            case 2048: {
                this.isTrue("Invalid qsize for 2048 bit key:" + qsize, qsize >= 224);
                break;
            }
            case 3072: {
                this.isTrue("Invalid qsize for 3072 bit key:" + qsize, qsize >= 256);
                break;
            }
            default: {
                this.fail("Invalid key size:" + keysize);
            }
        }
        this.isTrue("privkey error", priv.getX().bitLength() >= qsize - 32);
    }

    private void testKeyGenerationAll() throws Exception {
        this.testKeyGeneration(1024);
        this.testKeyGeneration(2048);
        this.testKeyGeneration(3072);
    }

    @Override
    public void performTest() throws Exception {
        this.testCompat();
        this.testNONEwithDSA();
        this.testDSAsha3(NISTObjectIdentifiers.id_dsa_with_sha3_224, 224, new BigInteger("613202af2a7f77e02b11b5c3a5311cf6b412192bc0032aac3ec127faebfc6bd0", 16));
        this.testDSAsha3(NISTObjectIdentifiers.id_dsa_with_sha3_256, 256, new BigInteger("2450755c5e15a691b121bc833b97864e34a61ee025ecec89289c949c1858091e", 16));
        this.testDSAsha3(NISTObjectIdentifiers.id_dsa_with_sha3_384, 384, new BigInteger("7aad97c0b71bb1e1a6483b6948a03bbe952e4780b0cee699a11731f90d84ddd1", 16));
        this.testDSAsha3(NISTObjectIdentifiers.id_dsa_with_sha3_512, 512, new BigInteger("725ad64d923c668e64e7c3898b5efde484cab49ce7f98c2885d2a13a9e355ad4", 16));
        this.testECDSA239bitPrime();
        this.testNONEwithECDSA239bitPrime();
        this.testECDSA239bitBinary();
        this.testECDSA239bitBinary("RIPEMD160withECDSA", TeleTrusTObjectIdentifiers.ecSignWithRipemd160);
        this.testECDSA239bitBinary("SHA1withECDSA", TeleTrusTObjectIdentifiers.ecSignWithSha1);
        this.testECDSA239bitBinary("SHA224withECDSA", X9ObjectIdentifiers.ecdsa_with_SHA224);
        this.testECDSA239bitBinary("SHA256withECDSA", X9ObjectIdentifiers.ecdsa_with_SHA256);
        this.testECDSA239bitBinary("SHA384withECDSA", X9ObjectIdentifiers.ecdsa_with_SHA384);
        this.testECDSA239bitBinary("SHA512withECDSA", X9ObjectIdentifiers.ecdsa_with_SHA512);
        this.testECDSA239bitBinary("SHA1withCVC-ECDSA", EACObjectIdentifiers.id_TA_ECDSA_SHA_1);
        this.testECDSA239bitBinary("SHA224withCVC-ECDSA", EACObjectIdentifiers.id_TA_ECDSA_SHA_224);
        this.testECDSA239bitBinary("SHA256withCVC-ECDSA", EACObjectIdentifiers.id_TA_ECDSA_SHA_256);
        this.testECDSA239bitBinary("SHA384withCVC-ECDSA", EACObjectIdentifiers.id_TA_ECDSA_SHA_384);
        this.testECDSA239bitBinary("SHA512withCVC-ECDSA", EACObjectIdentifiers.id_TA_ECDSA_SHA_512);
        this.testECDSAP256sha3(NISTObjectIdentifiers.id_ecdsa_with_sha3_224, 224, new BigInteger("84d7d8e68e405064109cd9fc3e3026d74d278aada14ce6b7a9dd0380c154dc94", 16));
        this.testECDSAP256sha3(NISTObjectIdentifiers.id_ecdsa_with_sha3_256, 256, new BigInteger("99a43bdab4af989aaf2899079375642f2bae2dce05bcd8b72ec8c4a8d9a143f", 16));
        this.testECDSAP256sha3(NISTObjectIdentifiers.id_ecdsa_with_sha3_384, 384, new BigInteger("aa27726509c37aaf601de6f7e01e11c19add99530c9848381c23365dc505b11a", 16));
        this.testECDSAP256sha3(NISTObjectIdentifiers.id_ecdsa_with_sha3_512, 512, new BigInteger("f8306b57a1f5068bf12e53aabaae39e2658db39bc56747eaefb479995130ad16", 16));
        this.testGeneration();
        this.testParameters();
        this.testDSA2Parameters();
        this.testNullParameters();
        this.testValidate();
        this.testModified();
        this.testKeyGenerationAll();
    }

    protected BigInteger[] derDecode(byte[] encoding) throws IOException {
        ByteArrayInputStream bIn = new ByteArrayInputStream(encoding);
        ASN1InputStream aIn = new ASN1InputStream(bIn);
        ASN1Sequence s = (ASN1Sequence)aIn.readObject();
        BigInteger[] sig = new BigInteger[]{((ASN1Integer)s.getObjectAt(0)).getValue(), ((ASN1Integer)s.getObjectAt(1)).getValue()};
        return sig;
    }

    @Override
    public String getName() {
        return "DSA/ECDSA";
    }

    public static void main(String[] args) {
        Security.addProvider(new BouncyCastleProvider());
        DSATest.runTest(new DSATest());
    }

    private class DSATestSecureRandom
    extends TestRandomData {
        private boolean first;

        public DSATestSecureRandom(byte[] value) {
            super(value);
            this.first = true;
        }

        @Override
        public void nextBytes(byte[] bytes) {
            if (this.first) {
                super.nextBytes(bytes);
                this.first = false;
            } else {
                bytes[bytes.length - 1] = 2;
            }
        }
    }
}

