"""
Contributors: Alexander Jüstel, Arthur Endlein Correia, Florian Wellmann, Marius Pischke

GemGIS is a Python-based, open-source spatial data processing library.
It is capable of preprocessing spatial data such as vector data
raster data, data obtained from online services and many more data formats.
GemGIS wraps and extends the functionality of packages known to the geo-community
such as GeoPandas, Rasterio, OWSLib, Shapely, PyVista, Pandas, and NumPy.

GemGIS is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

GemGIS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License (LICENSE.md) for more details.

"""

import pytest
import rasterio
import numpy as np
import pandas as pd
import geopandas as gpd
from shapely.geometry import Point, LineString, MultiLineString, Polygon, MultiPolygon, GeometryCollection
from typing import Collection
import pyvista as pv
import gemgis as gg

gg.download_gemgis_data.download_tutorial_data(filename='test_vector.zip',
                                               dirpath='../docs/getting_started/tutorial/data/test_vector/')

# Definition of GeoDataFrames
###########################################################

lines1 = LineString([[0.256327195431048, 264.86214748436396],
                     [10.59346813871597, 276.73370778641777],
                     [17.134940141888464, 289.089821570188],
                     [19.150128045807676, 293.313485355882],
                     [27.79511673965105, 310.571692592952]])

lines2 = LineString([[0.1881868620686138, 495.787213546976],
                     [8.840672956663411, 504.1418419288791],
                     [41.09257661030145, 546.423052386348],
                     [71.72834900044251, 604.1435562935771],
                     [87.60971085508348, 626.5358946968061]])

lines3 = LineString([[970.6766251230017, 833.0526164998309],
                     [959.3724321757514, 800.0232029873156],
                     [941.2919969974961, 754.801239252839],
                     [925.3512303815758, 711.3618892206762],
                     [908.0193245862978, 675.9805395602916]])

gdf_interfaces1_lines = gpd.GeoDataFrame(geometry=[lines1, lines2, lines3], crs='EPSG:4326')
gdf_interfaces1_lines['formation'] = ['Sand1', 'Ton', 'Ton']
gdf_interfaces1_lines['id'] = None

points1 = Point(19.150128045807676, 293.313485355882)

points2 = Point(61.93436666575576, 381.4593263680641)

points3 = Point(109.3578600758187, 480.9455679783049)

points4 = Point(157.812298994796, 615.9994296460927)

points5 = Point(191.3180280345144, 719.0939805375339)

gdf_interfaces1_points = gpd.GeoDataFrame(geometry=[points1, points2, points3, points4, points5], crs='EPSG:4326')
gdf_interfaces1_points['formation'] = 'Ton'
gdf_interfaces1_points['id'] = None

lines1 = LineString([[0.7408806771479846, 475.4410147469845],
                     [35.62873136073459, 429.2469161566801],
                     [77.30033078835194, 340.0890755208477],
                     [104.7583614189525, 269.3442671902416],
                     [127.0478215779106, 207.6444571850097]])

lines2 = LineString([[645.9649974657704, 0.524960824189406],
                     [685.1409268045179, 61.8662186045969],
                     [724.8323288977228, 114.4444395592319],
                     [753.6988031473263, 158.7750964425515],
                     [781.5343318880155, 194.858189254556]])

lines3 = LineString([[490.2922256196942, 0.5249608241894066],
                     [505.7564082534104, 40.73183567185146],
                     [519.1586998692977, 79.39229225614187],
                     [529.4681549584418, 95.88742039877246],
                     [575.8607028595903, 158.2596236880943]])

gdf_topo1 = gpd.GeoDataFrame(geometry=[lines1, lines2, lines3], crs='EPSG:4326')
gdf_topo1['Z'] = [400, 300, 400]
gdf_topo1['id'] = None

points1 = Point(281.5257600645256, 902.0868083698422)

points2 = Point(-144.4205416504286, 120.8290807190565)

points3 = Point(1127.312693889784, -140.9983501274578)

points4 = Point(925.866703136033, 618.5767934183793)

points5 = Point(331.0111444924173, 255.6839742805063)

gdf_randompoints1 = gpd.GeoDataFrame(geometry=[points1, points2, points3, points4, points5], crs='EPSG:4326')
gdf_randompoints1['id'] = None

points1 = Point(62.51291694589889, 382.8322599248675)

points2 = Point(141.4890451853447, 555.1552982533094)

points3 = Point(191.943436551061, 720.9493534434355)

points4 = Point(287.5407959609224, 893.3021490959683)

points5 = Point(481.9075000924391, 893.3021490959683)

points6 = Point(594.4999939926145, 720.9419248062279)

points7 = Point(684.5145067836722, 555.153221276689)

points8 = Point(807.4407530936584, 213.5143073406334)

points9 = Point(878.3919189490275, 213.5161371712952)

points10 = Point(927.4888326152723, 382.8272837360255)

gdf_points_strike = gpd.GeoDataFrame(
    geometry=[points1, points2, points3, points4, points5, points6, points7, points8, points9, points10],
    crs='EPSG:4326')
gdf_points_strike['id'] = [2, 3, 4, 5, 5, 4, 3, 1, 1, 2]
gdf_points_strike['formation'] = 'Ton'
gdf_points_strike['Z'] = [400, 500, 600, 700, 700, 600, 500, 300, 300, 400]

points1 = Point(96.47104121438838, 451.5636209742439)

points2 = Point(172.7610088740548, 661.8765047927839)

points3 = Point(383.0738926925949, 957.7578658512201)

points4 = Point(592.3558310022205, 722.7022898187342)

points5 = Point(766.5856220087561, 348.4690700828027)

points6 = Point(843.906535177337, 167.0226605138662)

points7 = Point(941.8463585242062, 428.8828197781268)

points8 = Point(22.14220797837953, 299.5527565162123)

gdf_orientations1 = gpd.GeoDataFrame(
    geometry=[points1, points2, points3, points4, points5, points6, points7, points8, ], crs='EPSG:4326')
gdf_orientations1['id'] = None
gdf_orientations1['formation'] = 'Ton'
gdf_orientations1['dip'] = 30.5
gdf_orientations1['azimuth'] = 180

lines1 = LineString([[0.7408806771479846, 475.4410147469845],
                     [35.62873136073459, 429.2469161566801],
                     [77.30033078835194, 340.0890755208477],
                     [104.7583614189525, 269.3442671902416],
                     [127.0478215779106, 207.6444571850097],
                     [147.7221034644805, 141.4221480170908],
                     [173.2419201682151, 76.81501712156006],
                     [202.9612003801592, 0.901638319311445]])

lines2 = LineString([[645.9649974657704, 0.5249608241894066],
                     [685.1409268045179, 61.8662186045969],
                     [724.8323288977228, 114.4444395592319],
                     [753.6988031473263, 158.7750964425515],
                     [781.5343318880155, 194.858189254556],
                     [802.6687148207609, 214.9616266783869],
                     [832.0506618248215, 222.693717995245],
                     [858.8552450565962, 218.5699359595874],
                     [884.6288827794565, 204.1366988347856],
                     [904.2168474488303, 161.3524602148376],
                     [916.5881935558033, 123.7229491394615],
                     [925.8667031360329, 68.56736441254054],
                     [922.7738666092898, 31.45332609162176],
                     [919.6810300825465, 1.555906333103849]])

lines3 = LineString([[490.2922256196942, 0.5249608241894066],
                     [505.7564082534104, 40.73183567185146],
                     [519.1586998692977, 79.39229225614187],
                     [529.4681549584418, 95.88742039877246],
                     [575.8607028595903, 158.2596236880943],
                     [626.8925055508537, 223.7246635041595],
                     [675.8624172242883, 287.1278123023957],
                     [718.1311830897791, 342.798869783774],
                     [749.5750211116687, 385.5831084037221],
                     [771.7403495533285, 423.2126194790981],
                     [791.8437869771595, 451.0481482197872],
                     [810.9162788920762, 463.9349670812173],
                     [827.9268797891639, 469.6051673802466],
                     [849.0612627219093, 462.904021572303],
                     [877.4122642170557, 436.0994383405283],
                     [898.5466471498012, 412.3876916354968],
                     [923.2893393637471, 382.4902718769788],
                     [936.1761582251772, 320.1180685876569],
                     [944.9391950509496, 239.7043188923328],
                     [958.3414866668369, 169.6000242861528],
                     [971.7437782827242, 25.78312579259239]])

lines4 = LineString([[911.4334660112313, 1068.58450805952],
                     [908.8561022389451, 1026.831214948486],
                     [909.3715749934024, 989.20170387311],
                     [920.7119755914609, 947.9638835165335],
                     [934.1142672073483, 901.571335615385],
                     [957.8260139123797, 867.0346610667522],
                     [971.2283055282671, 853.1168966964076]])

lines5 = LineString([[228.4320663554334, 1068.58450805952],
                     [239.7724669534919, 1017.037232613799],
                     [251.6283403060076, 970.6446847126506],
                     [264.5151591674378, 930.4378098649886],
                     [281.5257600645256, 902.0868083698422],
                     [314.0005435953295, 878.3750616648108],
                     [347.5062726350479, 864.4572972944661],
                     [382.0429471836807, 864.4572972944661],
                     [436.1675864016873, 869.096552084581],
                     [464.0031151423764, 877.3441161558962],
                     [486.1684435840363, 894.3547170529841],
                     [516.5813360970114, 941.2627377085898],
                     [547.5097013644438, 984.046976328538],
                     [570.7059753150181, 1018.068178122714],
                     [602.6652860913648, 1066.522617041691]])

lines6 = LineString([[114.5125876203909, 1068.069035305062],
                     [151.6266259413098, 930.4378098649886],
                     [182.5549912087421, 806.7243487952592],
                     [188.2251915077713, 748.475927541595],
                     [197.503701088001, 712.3928347295906],
                     [227.401120846519, 664.4538685650705],
                     [260.9068498862374, 628.8862485075233],
                     [300.082779224985, 600.5352470123769],
                     [335.134926528075, 587.6484281509468],
                     [378.4346379024803, 579.9163368340887],
                     [421.2188765224284, 586.1020098875752],
                     [455.7555510710612, 611.8756476104354],
                     [499.0552624454665, 642.2885401234106],
                     [539.7776100475858, 677.3406874265006],
                     [573.2833390873041, 704.1452706582753],
                     [595.4486675289639, 722.1868170642775],
                     [620.706832497367, 758.7853826307392],
                     [656.7899253093715, 843.8383871161781],
                     [679.9861992599458, 892.8082987896125],
                     [698.5432184204051, 944.8710469897903],
                     [716.5847648264073, 1002.603995488997],
                     [734.1108384779524, 1068.069035305062]])

lines7 = LineString([[28.42863762603758, 1067.553562550605],
                     [71.72834900044288, 866.0037155578379],
                     [100.5948232500463, 729.9189083811357],
                     [109.3578600758188, 678.3716329354152],
                     [133.0696067808503, 591.2567374321472],
                     [156.2658807314245, 524.7607521071677],
                     [181.0085729453704, 447.9553116930441],
                     [216.0607202484604, 385.067635649265],
                     [251.1128675515504, 329.912050922344],
                     [290.288796890298, 285.5813940390242],
                     [331.0111444924173, 255.6839742805063],
                     [373.2799103579081, 244.8590464369049],
                     [432.0438043660296, 246.9209374547337],
                     [486.6839163384935, 283.0040302667381],
                     [525.8598456772411, 320.6335413421142],
                     [562.9738839981599, 355.170215890747],
                     [608.8509591448512, 398.4699272651523],
                     [638.7483789033691, 435.0684928316139],
                     [668.1303259074301, 497.9561688753931],
                     [690.8111271035472, 555.173644620143],
                     [730.502529196752, 640.2266491055819],
                     [755.760694165155, 695.8977065869601],
                     [781.0188591335582, 767.0329467020546],
                     [799.5758782940175, 807.2398215497167],
                     [818.6483702089341, 827.8587317280048],
                     [834.6280255971076, 830.9515682547481],
                     [848.545789967452, 824.7658952012616],
                     [862.9790270922538, 809.8171853220026],
                     [876.8967914625985, 774.2495652644554],
                     [892.3609740963145, 719.6094532919917],
                     [903.701374694373, 664.9693413195278],
                     [925.866703136033, 618.5767934183793],
                     [946.4856133143212, 587.648428150947],
                     [970.7128327738099, 568.5759362360303]])

gdf_topo1 = gpd.GeoDataFrame(geometry=[lines1, lines2, lines3, lines4, lines5, lines6, lines7], crs='EPSG:4326')
gdf_topo1['id'] = None
gdf_topo1['Z'] = [400, 300, 400, 600, 700, 600, 500]

lines1 = LineString([[151.5909534995675, 8.547404541679612],
                     [227.0320471834991, 59.21679582193212],
                     [273.1974925721735, 96.37434942745062],
                     [355.394505093472, 134.6578895058636],
                     [430.8355987774034, 160.555578382437],
                     [497.268800678179, 156.0516324908591],
                     [571.583907889216, 160.555578382437],
                     [629.0092180068355, 165.0595242740151],
                     [702.1983387449779, 172.9414295842765],
                     [702.1983387449779, 172.9414295842765],
                     [767.5055541728589, 184.2012943132216],
                     [822.6788913446894, 211.2249696626895],
                     [822.6788913446894, 211.2249696626895],
                     [892.4900526641484, 250.6344962139971],
                     [954.4193086733458, 279.9101445092541],
                     [1016.348564682543, 290.0440227653045],
                     [1074.899861273057, 275.4061986176761],
                     [1132.325171390677, 248.382523268208],
                     [1178.490616779351, 221.3588479187401],
                     [1213.396197439081, 198.8391184608501],
                     [1253.931710463283, 162.8075513282261],
                     [1282.081372285645, 136.9098624516525],
                     [1298.971169379063, 108.7602006292901],
                     [1318.112939418269, 82.86251175271661],
                     [1327.120831201425, 38.94903930983116],
                     [1327.120831201425, 23.18522868930813]])

lines2 = LineString([[2174.988645290984, 10.79937748746863],
                     [2161.47680761625, 34.44509341825312],
                     [2141.209051104149, 103.1302682648176],
                     [2101.799524552841, 174.0674160571711],
                     [2070.271903311795, 235.9966720663686],
                     [2036.49230912496, 300.1779010213551],
                     [2002.712714938125, 367.7370893950251],
                     [1965.555161332607, 445.4301560247455],
                     [1914.885770052354, 507.359412033943],
                     [1851.830527570262, 591.8083975010305],
                     [1795.531203925537, 675.1313964952235],
                     [1754.995690901335, 733.6826930857375],
                     [1731.34997497055, 774.2182061099395],
                     [1720.090110241605, 818.131678552825],
                     [1732.475961443445, 890.1948128180729],
                     [1743.73582617239, 949.8720958814814],
                     [1748.239772063968, 969.0138659206879],
                     [1766.25555563028, 975.7697847580549],
                     [1783.145352723697, 970.1398523935824],
                     [1807.917055127376, 956.6280147188484],
                     [1843.94862226, 923.9744070049079],
                     [1872.098284082363, 899.2027046012289],
                     [1904.751891796303, 867.675083360183],
                     [1930.649580672877, 850.7852862667654],
                     [1979.06699900734, 831.643516227559],
                     [2011.720606721281, 819.2576650257195],
                     [2093.917619242579, 786.6040573117789],
                     [2136.70510521257, 783.2260978930955],
                     [2165.980753507827, 767.4622872725724],
                     [2217.776131260974, 747.1945307604715],
                     [2251.555725447809, 730.3047336670539],
                     [2251.555725447809, 730.3047336670539],
                     [2302.225116728062, 703.281058317586],
                     [2338.256683860685, 677.3833694410125],
                     [2377.666210411993, 635.721869943916],
                     [2421.579682854879, 596.3123433926085],
                     [2511.658600686439, 543.390979166567],
                     [2546.564181346168, 507.359412033943],
                     [2591.603640261948, 460.0679801723741],
                     [2605.115477936682, 430.792331877117],
                     [2645.650990960884, 385.7528729613371],
                     [2677.17861220193, 348.5953193558186],
                     [2677.17861220193, 348.5953193558186],
                     [2715.462152280343, 313.6897386960891],
                     [2757.12365177744, 273.154225671887],
                     [2861.840393756628, 202.2170778795335],
                     [2925.458629475167, 162.2445580917787],
                     [2973.876047809631, 137.4728556880997],
                     [2973.876047809631, 137.4728556880997]])

lines3 = LineString([[1870.40930437302, 4.04345865010157],
                     [1860.838419353417, 33.88210018180581],
                     [1848.452568151577, 69.35067407798255],
                     [1839.444676368421, 93.55938324521429],
                     [1822.554879275004, 132.9689097965218],
                     [1802.287122762903, 166.1855107469095],
                     [1780.33038654146, 207.2840170075588],
                     [1758.936643556465, 245.0045638495245],
                     [1731.912968206997, 284.4140904008319],
                     [1709.956231985554, 324.3866101885867],
                     [1687.436502527664, 350.8472923016074],
                     [1667.73173925201, 380.6859338333117],
                     [1652.530921867934, 394.7607647444929],
                     [1623.818266809125, 429.1033521677751],
                     [1593.416632040973, 465.6979125368464],
                     [1569.770916110189, 494.9735608321033],
                     [1546.688193415852, 517.4932902899933],
                     [1505.026693918755, 554.0878506590647],
                     [1478.003018569287, 569.8516612795877],
                     [1448.72737027403, 607.5722081215533],
                     [1430.711586707718, 653.1746602737807],
                     [1425.081654343245, 683.0133018054848],
                     [1418.325735505879, 723.5488148296869],
                     [1414.947776087195, 757.8914022529691],
                     [1412.132809904959, 783.2260978930954],
                     [1408.754850486275, 805.7458273509853],
                     [1401.435938412461, 855.8522253947906],
                     [1398.057978993777, 896.9507316554398],
                     [1394.680019575094, 929.0413461329329],
                     [1391.865053392858, 958.3169944281899],
                     [1389.050087210622, 998.8525074523918],
                     [1387.36110750128, 1036.573054294357],
                     [1383.983148082596, 1075.419587609218],
                     [1371.03430364431, 1153.675647475386],
                     [1354.144506550892, 1200.404086100507],
                     [1337.817702693922, 1243.191572070498],
                     [1322.616885309846, 1276.971166257333],
                     [1307.979061162218, 1316.943686045088],
                     [1291.652257305248, 1349.597293759028],
                     [1274.76246021183, 1394.636752674808],
                     [1270.8215075567, 1443.617164245719],
                     [1270.8215075567, 1467.825873412951],
                     [1260.124636064202, 1500.479481126891],
                     [1252.24273075394, 1532.570095604385],
                     [1248.864771335257, 1565.223703318325],
                     [1243.797832207231, 1605.759216342527],
                     [1253.931710463281, 1627.152959327522],
                     [1260.687629300648, 1640.101803765809],
                     [1270.258514320252, 1650.798675258307],
                     [1280.392392576302, 1654.739627913438],
                     [1298.971169379061, 1647.420715839623],
                     [1324.868858255635, 1627.152959327522],
                     [1353.581513314445, 1604.633229869632],
                     [1382.857161609702, 1577.609554520164],
                     [1411.006823432064, 1546.081933279118],
                     [1439.156485254427, 1524.688190294123],
                     [1471.24709973192, 1499.916487890444],
                     [1499.396761554282, 1481.337711087685],
                     [1529.798396322434, 1467.825873412951],
                     [1555.13309196256, 1450.936076319533],
                     [1577.65282142045, 1433.483285989669],
                     [1614.810375025969, 1413.215529477568],
                     [1626.070239754914, 1406.459610640201],
                     [1665.479766306221, 1377.746955581391],
                     [1696.44439431082, 1352.975253177712],
                     [1722.90507642384, 1341.15239521232],
                     [1741.4838532266, 1333.270489902058],
                     [1788.775285088169, 1308.498787498379],
                     [1815.798960437637, 1286.542051276936],
                     [1851.267534333814, 1262.333342109705],
                     [1870.40930437302, 1244.317558543393],
                     [1900.810939141172, 1217.293883193925],
                     [1916.011756525247, 1192.522180790246],
                     [1948.665364239188, 1153.675647475386],
                     [1959.362235731685, 1137.911836854863],
                     [1989.20087726339, 1104.132242668028],
                     [2012.283599957727, 1080.486526737243],
                     [2042.122241489431, 1055.151831097117],
                     [2065.767957420216, 1032.632101639227],
                     [2101.236531316392, 1015.74230454581],
                     [2134.45313226678, 1003.919446580417],
                     [2172.173679108746, 991.533595378578],
                     [2207.079259768475, 984.7776765412109],
                     [2249.866745738466, 981.3997171225275],
                     [2321.366886767267, 977.4587644673967],
                     [2416.512743726852, 973.517811812266],
                     [2451.418324386581, 971.8288321029243],
                     [2475.064040317366, 971.8288321029243],
                     [2511.09560744999, 963.9469267926627],
                     [2533.61533690788, 962.2579470833209],
                     [2570.772890513398, 941.99019057122],
                     [2594.98159968063, 917.218488167541],
                     [2615.812349429179, 886.2538601629421],
                     [2641.710038305752, 855.2892321583435],
                     [2661.977794817853, 835.0214756462425],
                     [2680.556571620612, 813.0647394247998],
                     [2728.410996718628, 792.2339896762514],
                     [2747.552766757835, 787.7300437846735],
                     [2792.592225673615, 782.1001114202011],
                     [2816.800934840846, 777.596165528623],
                     [2857.336447865048, 770.840246691256],
                     [2908.005839145301, 756.2024225436276],
                     [2958.675230425553, 742.1275916324463],
                     [2976.128020755418, 737.0606525044211]])

lines4 = LineString([[1998.771762282993, 1855.165220088658],
                     [2031.425369996933, 1825.889571793401],
                     [2069.708910075346, 1804.495828808405],
                     [2104.614490735075, 1795.487937025249],
                     [2136.142111976121, 1780.850112877621],
                     [2169.921706162956, 1753.826437528153],
                     [2203.701300349791, 1718.920856868423],
                     [2231.850962172154, 1677.259357371327],
                     [2283.646339925301, 1645.731736130281],
                     [2346.701582407392, 1613.07812841634],
                     [2412.008797835274, 1575.920574810822],
                     [2469.434107952893, 1553.400845352932],
                     [2521.22948570604, 1542.140980623987],
                     [2555.009079892875, 1548.896899461354],
                     [2603.426498227338, 1555.652818298721],
                     [2631.576160049701, 1574.794588337928],
                     [2648.465957143118, 1608.574182524763],
                     [2645.087997724435, 1642.353776711598],
                     [2630.450173576806, 1670.50343853396],
                     [2613.560376483389, 1715.54289744974],
                     [2585.410714661026, 1756.078410473942],
                     [2564.016971676031, 1804.495828808405],
                     [2546.001188109719, 1848.409301251291],
                     [2527.985404543407, 1900.204679004438],
                     [2507.717648031306, 1937.362232609956],
                     [2486.32390504631, 1965.511894432319],
                     [2452.544310859475, 1985.77965094442],
                     [2401.874919579223, 2004.921420983626],
                     [2352.331514771865, 2009.425366875204],
                     [2310.670015274769, 2018.43325865836],
                     [2248.740759265571, 2027.441150441516],
                     [2189.063476202163, 2038.701015170461],
                     [2135.016125503227, 2042.078974589145],
                     [2071.960883021135, 2047.708906953617],
                     [2029.173397051144, 2051.0868663723],
                     [1980.755978716681, 2045.456934007828],
                     [1944.724411584056, 2035.323055751777],
                     [1930.086587436428, 2018.43325865836],
                     [1930.086587436428, 1983.527677998631],
                     [1940.220465692478, 1947.496110866007],
                     [1964.992168096157, 1905.83461136891],
                     [1989.763870499836, 1868.677057763392]])

lines5 = LineString([[2977.25400722831, 2515.55628644128],
                     [2959.238223661998, 2525.127171460884],
                     [2928.836588893847, 2533.009076771145],
                     [2905.190872963062, 2545.957921209432],
                     [2883.797129978067, 2558.906765647719],
                     [2852.269508737021, 2576.359555977584],
                     [2832.00175222492, 2581.989488342056],
                     [2800.474130983874, 2600.568265144815],
                     [2781.895354181115, 2607.887177218629],
                     [2740.233854684018, 2619.710035184022],
                     [2688.438476930872, 2638.851805223228],
                     [2593.855613207734, 2671.505412937168],
                     [2535.867309853667, 2692.336162685716],
                     [2493.079823883676, 2704.159020651108],
                     [2437.906486711846, 2732.871675709918],
                     [2384.985122485804, 2762.710317241622],
                     [2332.063758259763, 2790.859979063985],
                     [2270.134502250566, 2812.25372204898],
                     [2213.272185369393, 2830.832498851739],
                     [2169.358712926508, 2839.277397398448],
                     [2108.555443390205, 2858.982160674102],
                     [2083.783740986526, 2869.679032166599],
                     [2064.078977710873, 2876.434951003966],
                     [2056.760065637058, 2875.871957767519],
                     [2049.441153563244, 2870.242025403047],
                     [2044.937207671666, 2860.108147146997],
                     [2044.374214435219, 2849.411275654499],
                     [2051.130133272586, 2842.655356817132],
                     [2054.508092691269, 2836.462431216212],
                     [2068.019930366003, 2813.379708521875],
                     [2092.228639533235, 2783.54106699017],
                     [2111.370409572442, 2757.643378113597],
                     [2128.260206665859, 2730.619702764129],
                     [2140.083064631251, 2702.470040941766],
                     [2148.52796317796, 2678.824325010982],
                     [2158.661841434011, 2646.170717297042],
                     [2169.921706162955, 2624.776974312046],
                     [2181.744564128348, 2596.627312489684],
                     [2196.382388275976, 2565.099691248638],
                     [2208.205246241369, 2544.26894150009],
                     [2232.976948645048, 2518.934245859964],
                     [2267.31953606833, 2495.288529929179],
                     [2288.713279053325, 2480.087712545104],
                     [2330.937771786869, 2465.449888397475],
                     [2352.894508008312, 2455.879003377872],
                     [2396.807980451197, 2441.804172466691],
                     [2427.209615219349, 2427.72934155551],
                     [2467.182135007103, 2418.158456535906],
                     [2507.717648031305, 2402.957639151831],
                     [2531.363363962089, 2395.075733841569],
                     [2574.15084993208, 2391.134781186438],
                     [2608.493437355362, 2379.311923221046],
                     [2644.525004487987, 2366.36307878276],
                     [2683.371537802847, 2346.658315507106],
                     [2712.647186098104, 2329.768518413689],
                     [2749.241746467175, 2299.366883645537],
                     [2785.273313599799, 2263.89830974936],
                     [2825.808826624001, 2214.91789817845],
                     [2844.38760342676, 2190.146195774771],
                     [2872.537265249123, 2146.232723331885],
                     [2897.871960889249, 2115.831088563733],
                     [2919.828697110692, 2075.858568775979],
                     [2938.970467149898, 2043.204961062038],
                     [2954.171284533974, 2017.870265421912],
                     [2978.379993701205, 1971.141826796791]])

lines6 = LineString([[496.1428142052848, 3720.924805674841],
                     [546.8122054855372, 3670.255414394588],
                     [595.2296238200006, 3617.334050168547],
                     [631.2611909526246, 3557.656767105138],
                     [648.150988046042, 3512.617308189358],
                     [654.906906883409, 3452.94002512595],
                     [656.0328933563035, 3380.876890860702],
                     [663.9147986665649, 3333.585458999133],
                     [697.6943928533999, 3295.30191892072],
                     [730.3480005673404, 3271.656202989936],
                     [744.9858247149688, 3250.26246000494],
                     [759.6236488625973, 3215.356879345211],
                     [773.1354865373313, 3171.443406902325],
                     [781.0173918475928, 3127.52993445944],
                     [784.3953512662762, 3080.238502597872],
                     [775.3874594831202, 3043.080948992353],
                     [756.2456894439138, 3018.309246588674],
                     [728.0960276215513, 3013.805300697096],
                     [695.4424199076109, 3020.561219534463],
                     [668.418744558143, 3029.569111317619],
                     [639.143096262886, 3041.954962519459],
                     [598.6075832386841, 3053.214827248403],
                     [532.1743813379087, 3066.726664923137],
                     [483.7569630034453, 3076.860543179188],
                     [435.3395446689819, 3073.482583760504],
                     [376.788248078468, 3074.608570233399],
                     [324.9928703253212, 3074.608570233399],
                     [274.3234790450688, 3075.734556706293],
                     [230.4100066021833, 3080.238502597872],
                     [175.2366694303529, 3097.128299691289],
                     [131.3231969874676, 3129.78190740523],
                     [91.91367043616012, 3174.821366321009],
                     [57.00808977643069, 3228.868717019945],
                     [35.61434679143524, 3269.404230044147],
                     [35.61434679143524, 3269.404230044147],
                     [18.72454969801778, 3306.561783649665],
                     [4.086725550389303, 3356.105188457023],
                     [1.834752604600311, 3396.640701481225],
                     [1.834752604600311, 3396.640701481225]])

lines7 = LineString([[1416.073762560089, 3715.294873310368],
                     [1467.869140313236, 3684.893238542216],
                     [1527.546423376645, 3666.877454975905],
                     [1597.357584696103, 3645.483711990909],
                     [1657.034867759512, 3619.586023114336],
                     [1657.034867759512, 3619.586023114336],
                     [1709.956231985553, 3597.066293656445],
                     [1775.263447413434, 3568.916631834083],
                     [1822.554879275003, 3540.766970011721],
                     [1877.728216446833, 3502.483429933308],
                     [1928.397607727086, 3467.577849273578],
                     [1985.822917844705, 3428.168322722271],
                     [2038.744282070747, 3397.766687954119],
                     [2096.169592188366, 3367.365053185968],
                     [2153.594902305985, 3336.963418417817],
                     [2221.154090679655, 3312.191716014137],
                     [2307.855049092531, 3289.671986556248],
                     [2388.926075140935, 3270.530216517041],
                     [2442.973425839871, 3260.396338260991],
                     [2504.902681849069, 3245.758514113362],
                     [2526.296424834064, 3242.380554694679]])

lines8 = LineString([[793.4032430494308, 3720.92480567484],
                     [805.7890942512703, 3672.507387340377],
                     [823.8048778175822, 3609.452144858285],
                     [848.5765802212611, 3547.522888849088],
                     [867.7183502604677, 3487.845605785679],
                     [901.4979444473025, 3420.28641741201],
                     [935.2775386341375, 3354.979201984128],
                     [972.4350922396559, 3309.939743068348],
                     [1017.474551155436, 3253.640419423624],
                     [1047.876185923587, 3196.215109306004],
                     [1080.529793637528, 3137.66381271549],
                     [1106.427482514101, 3102.758232055761],
                     [1106.427482514101, 3102.758232055761],
                     [1144.711022592514, 3044.206935465247],
                     [1176.23864383356, 2977.773733564472],
                     [1189.750481508294, 2929.356315230008],
                     [1186.372522089611, 2875.308964531072],
                     [1173.986670887771, 2830.269505615292],
                     [1143.58503611962, 2796.489911428458],
                     [1096.293604258051, 2777.348141389251],
                     [1052.380131815165, 2761.584330768728],
                     [982.5689704957067, 2761.584330768728],
                     [885.7341338267798, 2762.710317241623],
                     [885.7341338267798, 2762.710317241623],
                     [829.4348101820549, 2775.096168443462],
                     [770.883513591541, 2782.978073753723],
                     [708.9542575823438, 2793.111952009774],
                     [670.6707175039307, 2793.111952009774],
                     [606.4894885489445, 2793.111952009774],
                     [554.6941107957974, 2782.978073753723],
                     [508.5286654071231, 2757.08038487715],
                     [464.6151929642376, 2753.702425458466],
                     [400.4339640092512, 2754.828411931361],
                     [305.8511002861135, 2761.584330768728],
                     [243.9218442769161, 2771.718209024778],
                     [196.6304124153472, 2777.348141389251],
                     [135.8271428790443, 2779.60011433504],
                     [95.2916298548424, 2781.852087280829],
                     [71.64591392405796, 2777.348141389251],
                     [52.50414388485149, 2760.458344295833],
                     [45.74822504748451, 2739.064601310838],
                     [46.87421152037902, 2709.788953015581],
                     [55.88210330353499, 2676.009358828746],
                     [80.65380570721393, 2630.969899912966],
                     [131.3231969874663, 2567.914657430874],
                     [159.4728588098287, 2513.867306731939],
                     [195.5044259424526, 2447.434104831163],
                     [239.417898385338, 2367.489065255654],
                     [281.0793978824344, 2292.047971571723],
                     [318.2369514879529, 2213.228918469108],
                     [347.5125997832098, 2140.039797730966],
                     [347.5125997832098, 2140.039797730966],
                     [376.7882480784666, 2019.559245131255],
                     [381.2921939700447, 1929.480327299695],
                     [379.0402210242556, 1843.905355359713],
                     [367.7803562953108, 1774.094194040254],
                     [343.0086538916318, 1731.306708070263],
                     [295.7172220300628, 1702.031059775006],
                     [239.4178983853379, 1690.771195046061],
                     [201.134358306925, 1697.527113883428],
                     [156.0948993911452, 1724.550789232896],
                     [115.5593863669433, 1760.58235636552],
                     [72.77190039695233, 1783.10208582341],
                     [9.716657914860434, 1837.149436522346],
                     [9.716657914860434, 1837.149436522346]])

lines9 = LineString([[5.21271202328245, 1394.636752674808],
                     [33.36237384564492, 1376.620969108496],
                     [81.7797921801083, 1349.597293759028],
                     [142.5830617164112, 1301.179875424565],
                     [211.2682365629755, 1246.006538252735],
                     [266.441573734806, 1195.337146972482],
                     [266.441573734806, 1195.337146972482],
                     [328.3708297440034, 1139.037823327757],
                     [379.0402210242559, 1092.872377939083],
                     [444.3474364521367, 1048.958905496197],
                     [497.2688006781781, 1026.439176038307],
                     [549.064178431325, 1018.557270728046],
                     [609.8674479676279, 1016.305297782257],
                     [661.6628257207748, 1027.565162511202],
                     [688.6865010702427, 1034.321081348569],
                     [711.2062305281327, 1066.974689062509],
                     [725.8440546757611, 1112.014147978289],
                     [747.2377976607565, 1152.549661002491],
                     [773.1354865373301, 1216.730889957477],
                     [799.0331754139035, 1280.912118912464],
                     [799.0331754139035, 1280.912118912464],
                     [815.9229725073209, 1349.597293759028],
                     [812.5450130886375, 1402.518657985069],
                     [792.2772565765365, 1466.699886940056],
                     [777.639432428908, 1529.755129422148],
                     [764.1275947541742, 1608.574182524763],
                     [759.6236488625963, 1672.755411479749],
                     [743.8598382420732, 1745.944532217891],
                     [737.1039194047063, 1805.6218152813],
                     [730.3480005673393, 1834.897463576557],
                     [730.3480005673393, 1834.897463576557],
                     [707.8282711094494, 1861.921138926025],
                     [680.8045957599815, 1897.952706058649],
                     [680.8045957599815, 1897.952706058649],
                     [650.40296099183, 1932.858286718378],
                     [650.40296099183, 1932.858286718378],
                     [631.2611909526232, 1963.259921486529],
                     [614.3713938592058, 1997.039515673364],
                     [607.6154750218388, 2025.189177495727],
                     [592.9776508742103, 2067.976663465718],
                     [580.5917996723708, 2108.51217648992],
                     [574.9618673078984, 2159.181567770172],
                     [567.0799619976369, 2199.717080794374],
                     [556.9460837415864, 2254.890417966204],
                     [536.6783272294855, 2298.80389040909],
                     [508.5286654071231, 2347.221308743553],
                     [481.5049900576551, 2384.378862349072],
                     [455.6073011810817, 2427.166348319063],
                     [455.6073011810817, 2450.812064249847],
                     [461.2372335455541, 2493.599550219838],
                     [482.6309765305496, 2510.489347313256],
                     [516.4105707173845, 2517.245266150623],
                     [580.5917996723709, 2517.245266150623],
                     [644.7730286273573, 2517.245266150623],
                     [696.5684063805041, 2517.245266150623],
                     [747.2377976607565, 2507.111387894572],
                     [801.2851483596925, 2500.355469057205],
                     [844.0726343296834, 2493.599550219838],
                     [885.7341338267798, 2493.599550219838],
                     [931.8995792154542, 2496.977509638522],
                     [998.3327811162296, 2490.221590801155],
                     [1050.128158869376, 2485.717644909577],
                     [1096.293604258051, 2480.087712545104],
                     [1137.955103755147, 2481.213699017999],
                     [1168.356738523299, 2477.835739599315],
                     [1214.522183911973, 2457.567983087215],
                     [1251.679737517492, 2441.804172466691],
                     [1298.97116937906, 2422.662402427485],
                     [1333.87675003879, 2406.898591806962],
                     [1377.790222481675, 2395.638727078017],
                     [1416.073762560088, 2379.874916457494],
                     [1462.239207948763, 2364.111105836971],
                     [1501.64873450007, 2353.97722758092],
                     [1554.570098726111, 2356.229200526709],
                     [1590.601665858735, 2365.237092309865],
                     [1624.38126004557, 2395.638727078017],
                     [1640.145070666093, 2420.410429481696],
                     [1652.530921867933, 2464.323901924581],
                     [1648.026975976355, 2496.977509638521],
                     [1642.397043611882, 2558.906765647719],
                     [1639.019084193199, 2590.434386888765],
                     [1632.263165355832, 2638.851805223228],
                     [1625.507246518465, 2680.513304720325],
                     [1614.24738178952, 2706.410993596898],
                     [1596.231598223208, 2737.938614837944],
                     [1562.452004036373, 2799.867870847142],
                     [1532.050369268221, 2859.54515391055],
                     [1498.270775081387, 2907.962572245013],
                     [1475.751045623497, 2945.120125850532],
                     [1441.971451436661, 2993.537544184996],
                     [1400.309951939565, 3038.577003100775],
                     [1357.522465969574, 3094.8763267455],
                     [1357.522465969574, 3094.8763267455],
                     [1319.238925891161, 3162.43551511917],
                     [1319.238925891161, 3162.43551511917],
                     [1285.459331704326, 3207.47497403495],
                     [1251.679737517492, 3262.64831120678],
                     [1212.270210966184, 3314.443688959927],
                     [1178.490616779349, 3368.491039658863],
                     [1146.962995538303, 3422.538390357799],
                     [1110.931428405679, 3486.719619312785],
                     [1069.269928908583, 3540.766970011721],
                     [1034.364348248853, 3591.436361291974],
                     [994.9548216975459, 3640.979766099331],
                     [974.6870651854449, 3678.13731970485],
                     [951.0413492546605, 3715.294873310369]])

lines10 = LineString([[2768.383516506382, 2869.679032166601],
                      [2719.966098171918, 2893.324748097386],
                      [2659.162828635615, 2916.97046402817],
                      [2605.115477936679, 2927.104342284221],
                      [2522.918465415381, 2949.624071742111],
                      [2435.09152052961, 2962.00992294395],
                      [2383.296142776463, 2976.647747091579],
                      [2319.114913821477, 2989.033598293418],
                      [2251.555725447807, 3011.553327751308],
                      [2194.130415330188, 3027.317138371831],
                      [2117.563335173362, 3053.214827248405],
                      [2065.767957420215, 3082.490475543661],
                      [2022.980471450224, 3114.018096784707],
                      [1963.303188386815, 3152.30163686312],
                      [1913.759783579458, 3190.585176941533],
                      [1868.720324663678, 3216.482865818107],
                      [1815.798960437636, 3262.648311206781],
                      [1769.633515048962, 3302.057837758089],
                      [1718.964123768709, 3341.467364309396],
                      [1679.554597217402, 3375.246958496231],
                      [1639.0190841932, 3407.900566210171],
                      [1595.105611750315, 3424.790363303589],
                      [1547.814179888746, 3450.688052180162],
                      [1503.90070744586, 3466.451862800685],
                      [1439.719478490874, 3482.215673421208],
                      [1410.443830195617, 3473.207781638052],
                      [1393.5540331022, 3456.317984544635],
                      [1386.798114264833, 3438.302200978323],
                      [1395.806006047989, 3420.286417412011],
                      [1422.829681397457, 3380.876890860703],
                      [1447.601383801135, 3351.601242565446],
                      [1447.601383801135, 3351.601242565446],
                      [1487.010910352443, 3317.821648378611],
                      [1527.546423376645, 3276.160148881515],
                      [1573.711868765319, 3240.128581748891],
                      [1601.861530587682, 3208.600960507845],
                      [1627.759219464255, 3170.317420429432],
                      [1658.160854232407, 3134.285853296808],
                      [1676.176637798719, 3099.380272637079],
                      [1702.074326675292, 3068.978637868928],
                      [1706.57827256687, 3040.828976046565],
                      [1726.846029078971, 2996.91550360368],
                      [1739.23188028081, 2975.521760618684],
                      [1754.995690901334, 2940.616179958954],
                      [1769.633515048962, 2906.83658577212],
                      [1788.775285088168, 2870.805018639496],
                      [1804.539095708691, 2846.033316235817],
                      [1823.680865747898, 2815.631681467666],
                      [1836.066716949737, 2794.23793848267],
                      [1857.460459934733, 2767.214263133202],
                      [1877.728216446834, 2743.568547202418],
                      [1912.633797106564, 2700.781061232427],
                      [1927.271621254192, 2670.379426464276],
                      [1956.547269549449, 2625.339967548496],
                      [1980.192985480234, 2575.796562741138],
                      [1989.200877263389, 2536.38703618983],
                      [2000.460741992335, 2490.221590801156],
                      [2012.846593194174, 2448.560091304059],
                      [2034.240336179169, 2386.630835294862],
                      [2061.264011528637, 2355.103214053816],
                      [2078.153808622055, 2331.457498123032],
                      [2135.579118739674, 2298.803890409091],
                      [2185.122523547032, 2275.158174478307],
                      [2232.413955408601, 2266.150282695151],
                      [2232.413955408601, 2266.150282695151],
                      [2301.099130255166, 2251.512458547522],
                      [2346.138589170945, 2248.134499128839],
                      [2346.138589170945, 2248.134499128839],
                      [2410.319818125931, 2221.110823779371],
                      [2448.603358204345, 2199.717080794375],
                      [2495.894790065913, 2180.575310755169],
                      [2537.55628956301, 2161.433540715962],
                      [2562.327991966689, 2144.543743622545],
                      [2593.855613207735, 2115.268095327288],
                      [2620.879288557203, 2096.126325288082],
                      [2620.879288557203, 2096.126325288082],
                      [2652.406909798249, 2064.598704047036],
                      [2682.8085445664, 2017.307272185467],
                      [2707.580246970079, 1968.889853851004],
                      [2733.477935846652, 1917.094476097857],
                      [2759.375624723226, 1860.795152453132],
                      [2777.391408289538, 1793.235964079462],
                      [2801.037124220322, 1718.920856868425],
                      [2813.422975422161, 1667.125479115278],
                      [2826.934813096896, 1613.078128416342],
                      [2843.824610190313, 1545.518940042673],
                      [2856.210461392153, 1497.101521708209],
                      [2856.210461392153, 1497.101521708209],
                      [2874.226244958465, 1453.188049265324],
                      [2893.368014997671, 1394.636752674809],
                      [2904.627879726616, 1346.219334340346],
                      [2894.494001470566, 1302.305861897461],
                      [2874.226244958465, 1264.022321819048],
                      [2838.194677825841, 1225.738781740635],
                      [2838.194677825841, 1225.738781740635],
                      [2799.911137747428, 1218.982862903268],
                      [2730.099976427969, 1220.108849376162],
                      [2665.918747472983, 1231.368714105107],
                      [2608.493437355363, 1240.376605888263],
                      [2527.422411306959, 1253.888443562998],
                      [2465.493155297762, 1282.03810538536],
                      [2395.681993978303, 1318.069672517984],
                      [2330.374778550422, 1339.463415502979],
                      [2257.18565781228, 1374.368996162709],
                      [2211.020212423606, 1393.510766201915],
                      [2156.97286172467, 1407.022603876649],
                      [2096.169592188367, 1427.29036038875],
                      [2057.886052109954, 1457.691995156901],
                      [2001.586728465229, 1512.865332328732],
                      [1953.169310130766, 1579.298534229507],
                      [1953.169310130766, 1579.298534229507],
                      [1897.995972958935, 1620.960033726604],
                      [1842.822635787105, 1671.629425006856],
                      [1776.389433886329, 1714.416910976847],
                      [1707.704259039765, 1756.078410473943],
                      [1619.877314153995, 1816.881680010246],
                      [1547.814179888747, 1845.031341832608],
                      [1480.254991515077, 1878.810936019443],
                      [1426.207640816141, 1906.960597841806],
                      [1362.026411861155, 1938.488219082852],
                      [1305.72708821643, 1984.653664471526],
                      [1266.317561665122, 2018.433258658361],
                      [1215.64817038487, 2037.575028697567],
                      [1162.726806158829, 2046.582920480723],
                      [1112.057414878576, 2067.976663465718],
                      [1065.891969489902, 2090.496392923608],
                      [1025.3564564657, 2105.134217071237],
                      [966.805159875186, 2118.646054745971],
                      [917.2617550678281, 2120.89802769176],
                      [886.8601202996766, 2115.268095327287],
                      [871.0963096791536, 2109.638162962815],
                      [857.5844720044197, 2090.496392923608],
                      [875.6002555707316, 2034.197069278883],
                      [895.8680120828326, 1997.039515673365],
                      [930.7735927425621, 1927.228354353906],
                      [963.4272004565025, 1868.677057763392],
                      [989.3248893330759, 1810.125761172878],
                      [1008.466659372282, 1747.070518690786],
                      [1026.482442938594, 1680.637316790011],
                      [1043.372240032012, 1625.463979618181],
                      [1061.388023598324, 1571.416628919245],
                      [1061.388023598324, 1571.416628919245],
                      [1062.514010071218, 1522.999210584781],
                      [1062.514010071218, 1475.707778723213],
                      [1064.765983017007, 1430.668319807433],
                      [1064.765983017007, 1391.258793256125],
                      [1061.388023598324, 1356.353212596396],
                      [1072.647888327269, 1323.699604882455],
                      [1082.781766583319, 1285.416064804042],
                      [1094.041631312264, 1240.376605888262],
                      [1098.545577203842, 1202.09306580985],
                      [1103.04952309542, 1155.927620421175],
                      [1095.167617785159, 1109.762175032501],
                      [1090.663671893581, 1054.588837860671],
                      [1069.269928908585, 1005.045433053313],
                      [1051.254145342273, 925.1003934778033],
                      [1037.742307667539, 860.9191645228169],
                      [1010.718632318071, 779.8481384744131],
                      [989.3248893330758, 723.5488148296882],
                      [958.9232545649244, 664.9975182391744],
                      [958.9232545649244, 664.9975182391744],
                      [925.1436603780894, 614.328126958922],
                      [899.245971501516, 579.4225462991925],
                      [846.3246072754746, 546.768938585252],
                      [846.3246072754746, 546.768938585252],
                      [802.4111348325891, 511.8633579255227],
                      [773.1354865373322, 488.2176419947382],
                      [729.2220140944469, 475.8317907928987],
                      [729.2220140944469, 475.8317907928987],
                      [663.914798666566, 476.9577772657932],
                      [597.4815967657905, 492.7215878863162],
                      [535.5523407565931, 506.2334255610501],
                      [482.6309765305518, 529.8791414918346],
                      [433.0875717231939, 558.028803314197],
                      [381.292193970047, 600.816289284188],
                      [365.5283833495245, 626.7139781607614],
                      [321.614910906639, 672.8794235494358],
                      [276.5754519908592, 701.0290853717983],
                      [233.7879660208682, 728.0527607212662],
                      [172.9846964845653, 771.9662331641516],
                      [116.6853728398405, 809.12378676967],
                      [66.01598155958806, 838.3994350649269],
                      [6.338698496179688, 890.1948128180738]])

gdf_topo5 = gpd.GeoDataFrame(geometry=[lines1, lines2, lines3, lines4, lines5, lines6, lines7, lines8, lines9, lines10],
                             crs='EPSG:4326')
gdf_topo5['id'] = None
gdf_topo5['Z'] = [200, 100, 150, 250, 150, 350, 250, 300, 250, 200]

polygons1 = Polygon([[0.256327195431048, 264.862147484364],
                     [10.59346813871597, 276.7337077864178],
                     [17.13494014188846, 289.089821570188],
                     [19.15012804580768, 293.313485355882],
                     [27.79511673965105, 310.571692592952],
                     [34.41734765644295, 324.1391900810135],
                     [40.7165429187572, 338.5142767052691],
                     [49.27698776241503, 352.566327675047],
                     [55.33390628387104, 364.1148523226231],
                     [60.98703023722999, 376.3094482791545],
                     [61.93436666575576, 381.4593263680641],
                     [74.31225098443318, 404.8981037004269],
                     [89.49492674488292, 440.4320256929689],
                     [100.8011746516008, 465.6288067422258],
                     [109.3578600758187, 480.9455679783049],
                     [122.121527847126, 511.4998696780527],
                     [134.7199183717545, 543.1573638168628],
                     [146.0261662784724, 575.1378936101505],
                     [154.748128949369, 602.2728885862734],
                     [157.812298994796, 615.9994296460927],
                     [170.2538403642964, 655.5737715750863],
                     [179.944909998626, 686.9082300594188],
                     [191.3180280345144, 719.0939805375339],
                     [200.6191918851958, 750.2232183370392],
                     [210.9563328284808, 774.7739280773408],
                     [224.8237570742327, 798.4767847239435],
                     [240.6756130404249, 822.9062405945112],
                     [255.2122174919194, 846.1648077169024],
                     [264.903287126249, 861.9935547863074],
                     [264.903287126249, 861.9935547863074],
                     [272.7627232387529, 874.2512796291529],
                     [291.3922107934166, 899.788726360193],
                     [308.8361361352099, 915.9405090840756],
                     [323.2790531755591, 927.3449733382452],
                     [341.462737237453, 937.9069335885562],
                     [355.0302347255144, 947.2749675684081],
                     [366.0632917955071, 952.6031383066481],
                     [384.7495149374586, 957.2890728572154],
                     [394.4405845717882, 955.9969302393048],
                     [407.8165849065408, 950.0257745343622],
                     [423.1907578202995, 941.4603257878103],
                     [441.2807544710481, 929.1849709176595],
                     [456.2710238255182, 917.5509910035582],
                     [474.5534268822465, 900.757833323626],
                     [490.0591382971738, 879.4374801281008],
                     [503.949671439713, 861.3474834773522],
                     [515.0349178336396, 847.9621691518354],
                     [530.7616307613583, 821.9371336310784],
                     [545.6212708673304, 798.3555308542095],
                     [559.8348396643472, 775.7430350407737],
                     [575.3452301051327, 753.6306550861667],
                     [592.1384051121127, 724.0573303243491],
                     [604.0907243277858, 703.706084092257],
                     [617.9812574703249, 678.1862673885223],
                     [629.6105410315205, 658.804128119863],
                     [638.748378903369, 641.7730673689532],
                     [651.2539298815232, 612.9330651840361],
                     [660.6219638613752, 594.1969972243322],
                     [669.3439265322719, 575.4609292646282],
                     [674.8314717153735, 563.4212086914579],
                     [685.4957092561546, 542.1882568534299],
                     [698.7401710897384, 515.053261877307],
                     [710.6924903054116, 486.9491599377511],
                     [720.0605242852636, 462.3984501974494],
                     [727.4903443382495, 442.3702396198349],
                     [734.1108384779523, 426.3054560058411],
                     [745.9033766434759, 391.3306062123655],
                     [759.1478384770597, 360.3191833825108],
                     [767.5467654934788, 341.5831154228068],
                     [771.7403495533283, 327.3346871500575],
                     [779.1760490546743, 305.080086466832],
                     [785.3137264897498, 279.2372341086196],
                     [795.967569012817, 245.8899919458189],
                     [805.9880083763195, 216.8913527944325],
                     [812.4487214658725, 198.4783204892062],
                     [818.648370208934, 185.5796796743258],
                     [826.9853259173669, 173.9276107489045],
                     [841.3291714050511, 164.9607694960376],
                     [857.0276417837888, 173.6045750944268],
                     [867.1028091279114, 188.1570434466119],
                     [874.1485314711044, 203.323855306371],
                     [886.7469219957329, 240.1499199168235],
                     [893.8537063942413, 268.2540218563794],
                     [900.3144194837944, 285.6979471981728],
                     [903.701374694373, 295.8908491281679],
                     [909.6824534636465, 319.6166909183264],
                     [916.4662022076773, 337.706687569075],
                     [920.1965028370037, 355.1702158907466],
                     [929.3876283867835, 383.2547148504243],
                     [936.1713771308141, 409.0975672086366],
                     [945.539411110666, 440.7550613474467],
                     [950.0939225955215, 453.1100392376158],
                     [955.5535163994733, 474.3507694131226],
                     [962.6603007979818, 497.6093365355138],
                     [966.0735779836949, 510.3275149823656],
                     [972.0889039630482, 529.79176361285],
                     [971.8062477653798, -0.006899458905866851],
                     [0.2361374670297192, 0.1546183683329865],
                     [0.256327195431048, 264.862147484364]])

polygons2 = Polygon([[0.256327195431048, 264.862147484364],
                     [0.1881868620686138, 495.787213546976],
                     [8.840672956663411, 504.1418419288791],
                     [41.09257661030145, 546.423052386348],
                     [71.72834900044251, 604.1435562935771],
                     [87.60971085508348, 626.5358946968061],
                     [116.3598841035946, 686.6205264296495],
                     [140.8016980977082, 751.5687640683381],
                     [181.6130863080805, 848.1383536684762],
                     [223.2773388108611, 950.5412472888196],
                     [238.4673614961476, 988.0127920572999],
                     [253.1747585693791, 1022.707432912829],
                     [265.6023564722704, 1045.513138554322],
                     [278.5237826513765, 1068.771705676713],
                     [511.6747662706227, 1068.852464590333],
                     [526.3753184316984, 1045.388234108946],
                     [560.1099247138658, 990.6172670215257],
                     [608.8509591448512, 912.3962634589865],
                     [636.0233035161142, 859.787826958076],
                     [677.9243082421169, 782.4971293357705],
                     [700.6304344116447, 739.6185634923892],
                     [723.285910634351, 693.8358155691311],
                     [740.0407842579184, 658.5366142184982],
                     [757.3071124285267, 626.8243574896944],
                     [772.6673853601612, 603.6205529572973],
                     [790.2973687137878, 583.524646115289],
                     [810.139521279569, 572.9321657819203],
                     [831.5351890703644, 569.0914089904873],
                     [852.4571920161416, 577.4546649446074],
                     [869.6801729001976, 594.8650467133475],
                     [882.8225435370409, 621.7105496080459],
                     [898.5466471498011, 648.474213176897],
                     [908.0193245862978, 675.9805395602916],
                     [925.3512303815758, 711.3618892206762],
                     [941.2919969974961, 754.801239252839],
                     [959.3724321757514, 800.0232029873156],
                     [971.7381074320145, 832.9758676364308],
                     [972.0889039630482, 529.79176361285],
                     [966.0735779836949, 510.3275149823656],
                     [962.6603007979818, 497.6093365355138],
                     [955.5535163994733, 474.3507694131226],
                     [950.0939225955215, 453.1100392376158],
                     [945.539411110666, 440.7550613474467],
                     [936.1713771308141, 409.0975672086366],
                     [929.3876283867835, 383.2547148504243],
                     [920.1965028370037, 355.1702158907466],
                     [916.4662022076773, 337.706687569075],
                     [909.6824534636465, 319.6166909183264],
                     [903.701374694373, 295.8908491281679],
                     [893.8537063942413, 268.2540218563794],
                     [886.7469219957329, 240.1499199168235],
                     [874.1485314711044, 203.323855306371],
                     [867.1028091279114, 188.1570434466119],
                     [857.0276417837888, 173.6045750944268],
                     [841.3291714050511, 164.9607694960376],
                     [826.9853259173669, 173.9276107489045],
                     [818.648370208934, 185.5796796743258],
                     [812.4487214658725, 198.4783204892062],
                     [805.9880083763195, 216.8913527944325],
                     [795.967569012817, 245.8899919458189],
                     [785.3137264897498, 279.2372341086196],
                     [779.1760490546743, 305.080086466832],
                     [771.7403495533283, 327.3346871500575],
                     [767.5467654934788, 341.5831154228068],
                     [759.1478384770597, 360.3191833825108],
                     [745.9033766434759, 391.3306062123655],
                     [734.1108384779523, 426.3054560058411],
                     [727.4903443382495, 442.3702396198349],
                     [720.0605242852636, 462.3984501974494],
                     [710.6924903054116, 486.9491599377511],
                     [698.7401710897384, 515.053261877307],
                     [685.4957092561546, 542.1882568534299],
                     [674.8314717153735, 563.4212086914579],
                     [669.3439265322719, 575.4609292646282],
                     [660.6219638613752, 594.1969972243322],
                     [651.2539298815232, 612.9330651840361],
                     [638.748378903369, 641.7730673689532],
                     [629.6105410315205, 658.804128119863],
                     [617.9812574703249, 678.1862673885223],
                     [604.0907243277858, 703.706084092257],
                     [575.3452301051327, 753.6306550861667],
                     [559.8348396643472, 775.7430350407737],
                     [545.6212708673304, 798.3555308542095],
                     [530.7616307613583, 821.9371336310784],
                     [515.0349178336396, 847.9621691518354],
                     [503.949671439713, 861.3474834773522],
                     [490.0591382971738, 879.4374801281008],
                     [474.5534268822465, 900.757833323626],
                     [456.2710238255182, 917.5509910035582],
                     [441.2807544710481, 929.1849709176595],
                     [423.1907578202995, 941.4603257878103],
                     [407.8165849065408, 950.0257745343622],
                     [394.4405845717882, 955.9969302393048],
                     [384.7495149374586, 957.2890728572154],
                     [366.0632917955071, 952.6031383066481],
                     [355.0302347255144, 947.2749675684081],
                     [341.462737237453, 937.9069335885562],
                     [323.2790531755591, 927.3449733382452],
                     [308.8361361352099, 915.9405090840756],
                     [291.3922107934166, 899.788726360193],
                     [291.3922107934166, 899.788726360193],
                     [272.7627232387529, 874.2512796291529],
                     [264.903287126249, 861.9935547863074],
                     [255.2122174919194, 846.1648077169024],
                     [240.6756130404249, 822.9062405945112],
                     [224.8237570742327, 798.4767847239435],
                     [210.9563328284808, 774.7739280773408],
                     [200.6191918851958, 750.2232183370392],
                     [191.3180280345144, 719.0939805375339],
                     [179.944909998626, 686.9082300594188],
                     [170.2538403642964, 655.5737715750863],
                     [157.812298994796, 615.9994296460927],
                     [154.748128949369, 602.2728885862734],
                     [146.0261662784724, 575.1378936101505],
                     [134.7199183717545, 543.1573638168628],
                     [122.121527847126, 511.4998696780527],
                     [109.3578600758187, 480.9455679783049],
                     [100.8011746516008, 465.6288067422258],
                     [89.49492674488292, 440.4320256929689],
                     [74.31225098443318, 404.8981037004269],
                     [61.93436666575576, 381.4593263680641],
                     [60.98703023722999, 376.3094482791545],
                     [55.33390628387104, 364.1148523226231],
                     [49.27698776241503, 352.566327675047],
                     [40.7165429187572, 338.5142767052691],
                     [34.41734765644295, 324.1391900810135],
                     [27.79511673965105, 310.571692592952],
                     [19.15012804580768, 293.313485355882],
                     [17.13494014188846, 289.089821570188],
                     [10.59346813871597, 276.7337077864178],
                     [0.256327195431048, 264.862147484364]])

polygons3 = Polygon([[0.1881868620686138, 495.787213546976],
                     [0.2489663569543978, 1068.759507715801],
                     [278.5237826513765, 1068.771705676713],
                     [253.1747585693791, 1022.707432912829],
                     [238.4673614961476, 988.0127920572999],
                     [223.2773388108611, 950.5412472888196],
                     [181.6130863080805, 848.1383536684762],
                     [140.8016980977082, 751.5687640683381],
                     [116.3598841035946, 686.6205264296495],
                     [87.60971085508348, 626.5358946968061],
                     [71.72834900044251, 604.1435562935771],
                     [41.09257661030145, 546.423052386348],
                     [8.840672956663411, 504.1418419288791],
                     [0.1881868620686138, 495.787213546976]])

polygons4 = Polygon([[511.6747662706227, 1068.852464590333],
                     [971.6979382849257, 1068.79988717261],
                     [971.7381074320145, 832.9758676364308],
                     [959.3724321757514, 800.0232029873156],
                     [941.2919969974961, 754.801239252839],
                     [925.3512303815758, 711.3618892206762],
                     [908.0193245862978, 675.9805395602916],
                     [898.5466471498011, 648.474213176897],
                     [882.8225435370409, 621.7105496080459],
                     [869.6801729001976, 594.8650467133475],
                     [852.4571920161416, 577.4546649446074],
                     [831.5351890703644, 569.0914089904873],
                     [810.139521279569, 572.9321657819203],
                     [790.2973687137878, 583.524646115289],
                     [772.6673853601612, 603.6205529572973],
                     [757.3071124285267, 626.8243574896944],
                     [740.0407842579184, 658.5366142184982],
                     [723.285910634351, 693.8358155691311],
                     [700.6304344116447, 739.6185634923892],
                     [677.9243082421169, 782.4971293357705],
                     [636.0233035161142, 859.787826958076],
                     [608.8509591448512, 912.3962634589865],
                     [560.1099247138658, 990.6172670215257],
                     [526.3753184316984, 1045.388234108946],
                     [511.6747662706227, 1068.852464590333]])

gdf_geolmap1 = gpd.GeoDataFrame(geometry=[polygons1, polygons2, polygons3, polygons4], crs='EPSG:4326')
gdf_geolmap1['id'] = None
gdf_geolmap1['formation'] = ['Sand1', 'Ton', 'Sand2', 'Sand2']


# Testing extract_xy_linestrings
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_linestrings(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy_linestrings

    assert isinstance(gdf_interfaces1_lines, gpd.geodataframe.GeoDataFrame)
    gdf = extract_xy_linestrings(gdf=gdf_interfaces1_lines)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert gdf.crs == 'EPSG:4326'
    assert len(gdf) == 15
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'formation', 'geometry'}.issubset(gdf.columns)
    assert not {'id', 'index', 'points'}.issubset(gdf.columns)
    assert gdf.loc[0].X == 0.256327195431048
    assert gdf.loc[0].Y == 264.86214748436396


# Testing extract_xy_linestrings_index
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_linestrings_index(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy_linestrings

    gdf = extract_xy_linestrings(gdf=gdf_interfaces1_lines, drop_index=False)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert gdf.crs == 'EPSG:4326'
    assert len(gdf) == 15
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'formation', 'geometry', 'index'}.issubset(gdf.columns)
    assert not {'id', 'points'}.issubset(gdf.columns)
    assert gdf.loc[0].X == 0.256327195431048
    assert gdf.loc[0].Y == 264.86214748436396


# Testing extract_xy_linestrings_id
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_linestrings_id(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy_linestrings

    gdf = extract_xy_linestrings(gdf=gdf_interfaces1_lines, drop_id=False)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert gdf.crs == 'EPSG:4326'
    assert len(gdf) == 15
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'formation', 'geometry', 'id'}.issubset(gdf.columns)
    assert not {'index', 'points'}.issubset(gdf.columns)
    assert gdf.loc[0].X == 0.256327195431048
    assert gdf.loc[0].Y == 264.86214748436396


# Testing extract_xy_linestrings_points
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_linestrings_points(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy_linestrings

    gdf = extract_xy_linestrings(gdf=gdf_interfaces1_lines, drop_points=False)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert gdf.crs == 'EPSG:4326'
    assert len(gdf) == 15
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'formation', 'geometry', 'points'}.issubset(gdf.columns)
    assert not {'index', 'id'}.issubset(gdf.columns)
    assert gdf.loc[0].X == 0.256327195431048
    assert gdf.loc[0].Y == 264.86214748436396


# Testing extract_xy_linestrings_all
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_linestrings_all(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy_linestrings

    gdf = extract_xy_linestrings(gdf=gdf_interfaces1_lines, drop_points=False, drop_id=False, drop_index=False)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert gdf.crs == 'EPSG:4326'
    assert len(gdf) == 15
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'formation', 'geometry', 'points', 'index', 'id'}.issubset(gdf.columns)
    assert gdf.loc[0].X == 0.256327195431048
    assert gdf.loc[0].Y == 264.86214748436396


# Testing extract_xy_linestrings_crs
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_linestrings_crs(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy_linestrings

    gdf = extract_xy_linestrings(gdf=gdf_interfaces1_lines, target_crs='EPSG:4647')

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert gdf.crs == 'EPSG:4647'
    assert len(gdf) == 15
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'formation', 'geometry'}.issubset(gdf.columns)
    assert not {'id', 'index', 'points'}.issubset(gdf.columns)


# Testing extract_xy_points
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
def test_extract_xy_points(gdf_interfaces1_points):
    from gemgis.vector import extract_xy_points

    gdf = extract_xy_points(gdf=gdf_interfaces1_points)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert gdf.crs == 'EPSG:4326'
    assert len(gdf) == 5
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'formation', 'geometry'}.issubset(gdf.columns)
    assert not {'id', 'index', 'points'}.issubset(gdf.columns)
    assert gdf.loc[0].X == 19.150128045807676
    assert gdf.loc[0].Y == 293.313485355882


# Testing extract_xy_points_id
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
def test_extract_xy_points_id(gdf_interfaces1_points):
    from gemgis.vector import extract_xy_points

    gdf = extract_xy_points(gdf=gdf_interfaces1_points, drop_id=False)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert gdf.crs == 'EPSG:4326'
    assert len(gdf) == 5
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'formation', 'geometry', 'id'}.issubset(gdf.columns)
    assert 'points' not in gdf
    assert gdf.loc[0].X == 19.150128045807676
    assert gdf.loc[0].Y == 293.313485355882


# Testing extract_xy_points_id
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
def test_extract_xy_points_points(gdf_interfaces1_points):
    from gemgis.vector import extract_xy_points

    gdf = extract_xy_points(gdf=gdf_interfaces1_points, drop_id=False)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert gdf.crs == 'EPSG:4326'
    assert len(gdf) == 5
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'formation', 'geometry', 'id'}.issubset(gdf.columns)
    assert 'points' not in gdf
    assert gdf.loc[0].X == 19.150128045807676
    assert gdf.loc[0].Y == 293.313485355882


# Testing extract_xy_points_id
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
def test_extract_xy_points_id3(gdf_interfaces1_points):
    from gemgis.vector import extract_xy_points

    gdf = extract_xy_points(gdf=gdf_interfaces1_points, drop_id=False)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert gdf.crs == 'EPSG:4326'
    assert len(gdf) == 5
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'formation', 'geometry', 'id'}.issubset(gdf.columns)
    assert 'points' not in gdf
    assert gdf.loc[0].X == 19.150128045807676
    assert gdf.loc[0].Y == 293.313485355882


# Testing explode MultilineStrings
###########################################################
def test_explode_multilinestrings():
    from gemgis.vector import explode_multilinestrings

    from shapely.geometry import MultiLineString
    coords = [((0, 0), (1, 1)), ((-1, 0), (1, 0)), ((-5, 0), (1, 0))]
    lines = MultiLineString(coords)

    gdf_multilines = gpd.GeoDataFrame(geometry=[lines])

    gdf = explode_multilinestrings(gdf=gdf_multilines)
    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert len(gdf) == 3
    assert all(gdf.geom_type == 'LineString')
    assert 'geometry' in gdf
    assert not {'level_0', 'level_1'}.issubset(gdf.columns)


# Testing explode MultilineStrings_level0
###########################################################
def test_explode_multilinestrings_level0():
    from gemgis.vector import explode_multilinestrings

    from shapely.geometry import MultiLineString
    coords = [((0, 0), (1, 1)), ((-1, 0), (1, 0)), ((-5, 0), (1, 0))]
    lines = MultiLineString(coords)

    gdf_multilines = gpd.GeoDataFrame(geometry=[lines])

    gdf = explode_multilinestrings(gdf=gdf_multilines, reset_index=True, drop_level0=False)
    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert len(gdf) == 3
    assert all(gdf.geom_type == 'LineString')
    assert {'geometry', 'level_0'}.issubset(gdf.columns)
    assert not {'level_1'}.issubset(gdf.columns)


# Testing explode MultilineStrings_level1
###########################################################
def test_explode_multilinestrings_level1():
    from gemgis.vector import explode_multilinestrings

    from shapely.geometry import MultiLineString
    coords = [((0, 0), (1, 1)), ((-1, 0), (1, 0)), ((-5, 0), (1, 0))]
    lines = MultiLineString(coords)

    gdf_multilines = gpd.GeoDataFrame(geometry=[lines])

    gdf = explode_multilinestrings(gdf=gdf_multilines, reset_index=True, drop_level1=False)
    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert len(gdf) == 3
    assert all(gdf.geom_type == 'LineString')
    assert {'geometry', 'level_1'}.issubset(gdf.columns)
    assert not {'level_0'}.issubset(gdf.columns)


# Testing explode_polygons
###########################################################
@pytest.mark.parametrize("gdf",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/GeologicalMapAachen.shp')
                         ])
def test_explode_polygons(gdf):
    from gemgis.vector import explode_polygons

    gdf_collection = explode_polygons(gdf=gdf)

    assert all(gdf.geom_type == 'Polygon')
    assert isinstance(gdf_collection, gpd.GeoDataFrame)
    assert 'geometry' in gdf_collection
    assert np.unique(np.array([i for i in gdf_collection.geom_type])).tolist() == ['LineString', 'MultiLineString']


# Testing extract_xy
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
def test_extract_xy_points(gdf_interfaces1_points):
    from gemgis.vector import extract_xy
    gdf_new = extract_xy(gdf=gdf_interfaces1_points)
    # Assert type on input
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_points
    assert all(gdf_interfaces1_points.geom_type == 'Point')

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)
    assert 'id' not in gdf_new

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796, 191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
def test_extract_xy_points_drop_id(gdf_interfaces1_points):
    from gemgis.vector import extract_xy
    gdf_new = extract_xy(gdf=gdf_interfaces1_points, reset_index=True, drop_id=False)
    # Assert type on input
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_points
    assert all(gdf_interfaces1_points.geom_type == 'Point')

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'id'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796, 191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_lines(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy

    gdf_new = extract_xy(gdf=gdf_interfaces1_lines)
    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)
    assert not {'index', 'id', 'points'}.issubset(gdf_interfaces1_lines.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676, 27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_lines_drop_points(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy

    gdf_new = extract_xy(gdf=gdf_interfaces1_lines, drop_points=False)
    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'points'}.issubset(gdf_new.columns)
    assert not {'index', 'id'}.issubset(gdf_interfaces1_lines.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676, 27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_lines_drop_id(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy

    gdf_new = extract_xy(gdf=gdf_interfaces1_lines, drop_id=False)
    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'id'}.issubset(gdf_new.columns)
    assert not {'index', 'points'}.issubset(gdf_interfaces1_lines.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676, 27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_lines_drop_index(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy

    gdf_new = extract_xy(gdf=gdf_interfaces1_lines, drop_index=False)
    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)
    assert not {'id', 'points', 'index'}.issubset(gdf_interfaces1_lines.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676, 27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]


@pytest.mark.parametrize("gdf_topo1", [gdf_topo1])
def test_extract_xy_lines(gdf_topo1):
    from gemgis.vector import extract_xy
    gdf_new = extract_xy(gdf=gdf_topo1)
    # Assert type on input
    assert isinstance(gdf_topo1, gpd.GeoDataFrame)
    assert 'geometry' in gdf_topo1
    assert all(gdf_topo1.geom_type == 'LineString')

    # Assert CRS
    assert gdf_topo1.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_topo1.columns)
    assert 'Z' in gdf_topo1

    # Assert type of output
    assert isinstance(gdf_topo1, gpd.GeoDataFrame)
    assert gdf_topo1 is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert not {'index', 'id', 'points'}.issubset(gdf_topo1.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.7408806771479846, 35.62873136073459, 77.30033078835194,
                                            104.7583614189525, 127.0478215779106]
    assert gdf_new['Y'].head().tolist() == [475.4410147469845, 429.2469161566801, 340.0890755208477,
                                            269.3442671902416, 207.6444571850097]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_geojson_multiline(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy

    gdf_new = extract_xy(gdf=gdf_interfaces1_lines)
    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'MultiLineString')

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676,
                                            27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]


def test_extract_xy_multilinestring():
    from gemgis.vector import extract_xy

    coords = [((0, 0), (1, 1)), ((-1, 0), (1, 0)), ((-5, 0), (1, 0))]
    lines = MultiLineString(coords)
    gdf_multilinestring = gpd.GeoDataFrame(geometry=[lines], crs='EPSG:4326')

    gdf_new = extract_xy(gdf=gdf_multilinestring)
    # Assert type on input
    assert isinstance(gdf_multilinestring, gpd.GeoDataFrame)
    assert 'geometry' in gdf_multilinestring
    assert all(gdf_multilinestring.geom_type == 'MultiLineString')

    # Assert CRS
    assert gdf_multilinestring.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_multilinestring.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_multilinestring is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)
    assert not {'index', 'id', 'points', 'level_0', 'level_1'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.0, 1.0, -1.0, 1.0, -5.0]
    assert gdf_new['Y'].head().tolist() == [0.0, 1.0, 0.0, 0.0, 0.0]


def test_extract_xy_multilinestring_drop_points():
    from gemgis.vector import extract_xy

    coords = [((0, 0), (1, 1)), ((-1, 0), (1, 0)), ((-5, 0), (1, 0))]
    lines = MultiLineString(coords)
    gdf_multilinestring = gpd.GeoDataFrame(geometry=[lines], crs='EPSG:4326')

    gdf_new = extract_xy(gdf=gdf_multilinestring, drop_points=False)
    # Assert type on input
    assert isinstance(gdf_multilinestring, gpd.GeoDataFrame)
    assert 'geometry' in gdf_multilinestring
    assert all(gdf_multilinestring.geom_type == 'MultiLineString')

    # Assert CRS
    assert gdf_multilinestring.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_multilinestring.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_multilinestring is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'points'}.issubset(gdf_new.columns)
    assert not {'index', 'id', 'level_0', 'level_1'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.0, 1.0, -1.0, 1.0, -5.0]
    assert gdf_new['Y'].head().tolist() == [0.0, 1.0, 0.0, 0.0, 0.0]


def test_extract_xy_multilinestring_drop_leve0():
    from gemgis.vector import extract_xy

    coords = [((0, 0), (1, 1)), ((-1, 0), (1, 0)), ((-5, 0), (1, 0))]
    lines = MultiLineString(coords)
    gdf_multilinestring = gpd.GeoDataFrame(geometry=[lines], crs='EPSG:4326')

    gdf_new = extract_xy(gdf=gdf_multilinestring, drop_level0=False)
    # Assert type on input
    assert isinstance(gdf_multilinestring, gpd.GeoDataFrame)
    assert 'geometry' in gdf_multilinestring
    assert all(gdf_multilinestring.geom_type == 'MultiLineString')

    # Assert CRS
    assert gdf_multilinestring.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_multilinestring.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_multilinestring is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'level_0'}.issubset(gdf_new.columns)
    assert not {'points', 'id', 'level_1'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.0, 1.0, -1.0, 1.0, -5.0]
    assert gdf_new['Y'].head().tolist() == [0.0, 1.0, 0.0, 0.0, 0.0]


def test_extract_xy_multilinestring_drop_leve1():
    from gemgis.vector import extract_xy

    coords = [((0, 0), (1, 1)), ((-1, 0), (1, 0)), ((-5, 0), (1, 0))]
    lines = MultiLineString(coords)
    gdf_multilinestring = gpd.GeoDataFrame(geometry=[lines], crs='EPSG:4326')

    gdf_new = extract_xy(gdf=gdf_multilinestring, drop_level1=False)
    # Assert type on input
    assert isinstance(gdf_multilinestring, gpd.GeoDataFrame)
    assert 'geometry' in gdf_multilinestring
    assert all(gdf_multilinestring.geom_type == 'MultiLineString')

    # Assert CRS
    assert gdf_multilinestring.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_multilinestring.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_multilinestring is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'level_1'}.issubset(gdf_new.columns)
    assert not {'points', 'id', 'level_0'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.0, 1.0, -1.0, 1.0, -5.0]
    assert gdf_new['Y'].head().tolist() == [0.0, 1.0, 0.0, 0.0, 0.0]


def test_extract_xy_multilinestring_drop_index():
    from gemgis.vector import extract_xy

    coords = [((0, 0), (1, 1)), ((-1, 0), (1, 0)), ((-5, 0), (1, 0))]
    lines = MultiLineString(coords)
    gdf_multilinestring = gpd.GeoDataFrame(geometry=[lines], crs='EPSG:4326')

    # No index column will be created!
    gdf_new = extract_xy(gdf=gdf_multilinestring, reset_index=True, drop_index=False)
    # Assert type on input
    assert isinstance(gdf_multilinestring, gpd.GeoDataFrame)
    assert 'geometry' in gdf_multilinestring
    assert all(gdf_multilinestring.geom_type == 'MultiLineString')

    # Assert CRS
    assert gdf_multilinestring.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_multilinestring.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_multilinestring is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)
    assert not {'points', 'id', 'level_0', 'level_1'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.0, 1.0, -1.0, 1.0, -5.0]
    assert gdf_new['Y'].head().tolist() == [0.0, 1.0, 0.0, 0.0, 0.0]


# Testing set_dtype
###########################################################
@pytest.mark.parametrize("gdf_orientations1", [gdf_orientations1])
def test_set_dtype(gdf_orientations1):
    from gemgis.vector import set_dtype, extract_xy

    gdf_points = extract_xy(gdf_orientations1)
    gdf_points['polarity'] = 1
    gdf_types = set_dtype(gdf=gdf_points)

    assert isinstance(gdf_types, gpd.GeoDataFrame)
    assert 'geometry' in gdf_types
    assert all(gdf_types.geom_type == 'Point')

    assert gdf_types['formation'].dtype == 'O'
    assert gdf_types['polarity'].dtype == 'float64'
    assert gdf_types['azimuth'].dtype == 'float64'
    assert gdf_types['dip'].dtype == 'float64'
    assert gdf_types['X'].dtype == 'float64'
    assert gdf_types['Y'].dtype == 'float64'


# Testing extract_xy_polygons
###########################################################
@pytest.mark.parametrize("gdf",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/GeologicalMapAachen.shp')
                         ])
def test_extract_xy_polygons(gdf):
    from gemgis.vector import extract_xy

    gdf_new = extract_xy(gdf=gdf)
    # Assert type on input
    assert isinstance(gdf, gpd.GeoDataFrame)
    assert 'geometry' in gdf
    assert all(gdf.geom_type == 'Polygon')

    # Assert CRS
    assert gdf.crs == 'EPSG:4647'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4647'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', }.issubset(gdf_new.columns)
    assert not {'points', 'id', 'level_0', 'level_1'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [32299083.70919895, 32299164.005298954, 32299123.22539895,
                                            32299088.346098952, 32298996.61839895]
    assert gdf_new['Y'].head().tolist() == [5631034.98260157, 5630970.06570157, 5630909.550101571, 5630931.022001569,
                                            5630993.45760157]


# Testing extract_xy
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
def test_extract_xy_points(gdf_interfaces1_points):
    from gemgis.vector import extract_xy
    gdf_new = extract_xy(gdf=gdf_interfaces1_points)
    # Assert type on input
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_points
    assert all(gdf_interfaces1_points.geom_type == 'Point')

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796, 191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
def test_extract_xy_points_inplace(gdf_interfaces1_points):
    from gemgis.vector import extract_xy
    gdf_new = extract_xy(gdf=gdf_interfaces1_points)
    # Assert type on input
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_points
    assert all(gdf_interfaces1_points.geom_type == 'Point')

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796, 191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_lines(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy

    gdf_new = extract_xy(gdf=gdf_interfaces1_lines)
    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676, 27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_lines2(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy

    gdf_new = extract_xy(gdf=gdf_interfaces1_lines)
    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676, 27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_lines_inplace(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy
    gdf_new = extract_xy(gdf=gdf_interfaces1_lines)
    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676, 27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]


@pytest.mark.parametrize("gdf_topo1", [gdf_topo1])
def test_extract_xy_lines(gdf_topo1):
    from gemgis.vector import extract_xy
    gdf_new = extract_xy(gdf=gdf_topo1)
    # Assert type on input
    assert isinstance(gdf_topo1, gpd.GeoDataFrame)
    assert 'geometry' in gdf_topo1
    assert all(gdf_topo1.geom_type == 'LineString')

    # Assert CRS
    assert gdf_topo1.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_topo1.columns)
    assert 'Z' in gdf_topo1

    # Assert type of output
    assert isinstance(gdf_topo1, gpd.GeoDataFrame)
    assert gdf_topo1 is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.7408806771479846, 35.62873136073459, 77.30033078835194,
                                            104.7583614189525, 127.0478215779106]
    assert gdf_new['Y'].head().tolist() == [475.4410147469845, 429.2469161566801, 340.0890755208477,
                                            269.3442671902416, 207.6444571850097]


@pytest.mark.parametrize("gdf_topo1", [gdf_topo1])
def test_extract_xy_lines(gdf_topo1):
    from gemgis.vector import extract_xy
    gdf_new = extract_xy(gdf=gdf_topo1)
    # Assert type on input
    assert isinstance(gdf_topo1, gpd.GeoDataFrame)
    assert 'geometry' in gdf_topo1
    assert all(gdf_topo1.geom_type == 'LineString')

    # Assert CRS
    assert gdf_topo1.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_topo1.columns)
    assert 'Z' in gdf_topo1

    # Assert type of output
    assert isinstance(gdf_topo1, gpd.GeoDataFrame)
    assert gdf_topo1 is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.7408806771479846, 35.62873136073459, 77.30033078835194,
                                            104.7583614189525, 127.0478215779106]
    assert gdf_new['Y'].head().tolist() == [475.4410147469845, 429.2469161566801, 340.0890755208477,
                                            269.3442671902416, 207.6444571850097]


@pytest.mark.parametrize("gdf",
                         [
                             gpd.read_file(
                                 '../docs/getting_started/tutorial/data/test_vector/interfaces1_lines_geojson.geojson')
                         ])
def test_extract_xy_geojson_multiline(gdf):
    from gemgis.vector import extract_xy
    gdf_new = extract_xy(gdf=gdf)
    # Assert type on input
    assert isinstance(gdf, gpd.GeoDataFrame)
    assert 'geometry' in gdf
    assert all(gdf.geom_type == 'MultiLineString')

    # Assert CRS
    assert gdf.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676,
                                            27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]


# Testing extract_z_rasterio
###########################################################
@pytest.mark.parametrize("gdf_randompoints1", [gdf_randompoints1])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_rasterio(gdf_randompoints1, dem):
    from gemgis.vector import extract_xyz_rasterio

    gdf_z = extract_xyz_rasterio(gdf=gdf_randompoints1, dem=dem)

    # Assert type on input
    assert isinstance(gdf_z, gpd.GeoDataFrame)
    assert 'geometry' in gdf_z
    assert all(gdf_randompoints1.geom_type == 'Point')

    # Assert CRS
    assert gdf_randompoints1.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_randompoints1.columns)

    # Assert type of output
    assert gdf_randompoints1 is not gdf_z

    # Assert CRS
    assert gdf_z.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_z.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Y'}.issubset(gdf_z.columns)
    assert not {'points', 'id', 'level_0', 'level_1'}.issubset(gdf_z.columns)

    # Assert if values are correct
    assert gdf_z['X'].head().tolist() == [281.5257600645256, -144.4205416504286, 1127.312693889784, 925.866703136033,
                                          331.0111444924173]
    assert gdf_z['Y'].head().tolist() == [902.0868083698422, 120.8290807190565, -140.9983501274578, 618.5767934183793,
                                          255.6839742805063]
    assert gdf_z['Z'].head().tolist() == [700.2296752929688, -3.402823e+38, -3.402823e+38, 500.2345275878906,
                                          499.8694763183594]


@pytest.mark.parametrize("gdf_randompoints1", [gdf_randompoints1])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_rasterio_drop_id(gdf_randompoints1, dem):
    from gemgis.vector import extract_xyz_rasterio

    gdf_z = extract_xyz_rasterio(gdf=gdf_randompoints1, dem=dem, drop_id=False)

    # Assert type on input
    assert isinstance(gdf_z, gpd.GeoDataFrame)
    assert 'geometry' in gdf_z
    assert all(gdf_randompoints1.geom_type == 'Point')

    # Assert CRS
    assert gdf_randompoints1.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_randompoints1.columns)

    # Assert type of output
    assert gdf_randompoints1 is not gdf_z

    # Assert CRS
    assert gdf_z.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_z.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Y', 'id'}.issubset(gdf_z.columns)
    assert not {'points', 'level_0', 'level_1'}.issubset(gdf_z.columns)

    # Assert if values are correct
    assert gdf_z['X'].head().tolist() == [281.5257600645256, -144.4205416504286, 1127.312693889784, 925.866703136033,
                                          331.0111444924173]
    assert gdf_z['Y'].head().tolist() == [902.0868083698422, 120.8290807190565, -140.9983501274578, 618.5767934183793,
                                          255.6839742805063]
    assert gdf_z['Z'].head().tolist() == [700.2296752929688, -3.402823e+38, -3.402823e+38, 500.2345275878906,
                                          499.8694763183594]


# Testing clip_by_bbox
###########################################################
@pytest.mark.parametrize("gdf_randompoints1", [gdf_randompoints1])
def test_clip_by_bbox(gdf_randompoints1):
    from gemgis.vector import clip_by_bbox

    gdf_clipped = clip_by_bbox(gdf=gdf_randompoints1, bbox=[0, 972.0, 0, 1069.0])

    # Assert type on input
    assert isinstance(gdf_clipped, gpd.GeoDataFrame)
    assert 'geometry' in gdf_clipped
    assert all(gdf_randompoints1.geom_type == 'Point')
    assert len(gdf_randompoints1) == 5
    assert len(gdf_clipped) == 3

    # Assert CRS
    assert gdf_randompoints1.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_randompoints1.columns)

    # Assert type of output
    assert gdf_randompoints1 is not gdf_clipped

    # Assert CRS
    assert gdf_clipped.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_clipped.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Y', }.issubset(gdf_clipped.columns)
    assert not {'points', 'level_0', 'level_1', 'index', 'id'}.issubset(gdf_clipped.columns)


@pytest.mark.parametrize("gdf_randompoints1", [gdf_randompoints1])
def test_clip_by_bbox_drop_id(gdf_randompoints1):
    from gemgis.vector import clip_by_bbox

    gdf_clipped = clip_by_bbox(gdf=gdf_randompoints1, bbox=[0, 972.0, 0, 1069.0], drop_id=False)

    # Assert type on input
    assert isinstance(gdf_clipped, gpd.GeoDataFrame)
    assert 'geometry' in gdf_clipped
    assert all(gdf_randompoints1.geom_type == 'Point')
    assert len(gdf_randompoints1) == 5
    assert len(gdf_clipped) == 3

    # Assert CRS
    assert gdf_randompoints1.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_randompoints1.columns)

    # Assert type of output
    assert gdf_randompoints1 is not gdf_clipped

    # Assert CRS
    assert gdf_clipped.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_clipped.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'id'}.issubset(gdf_clipped.columns)
    assert not {'points', 'level_0', 'level_1', 'index'}.issubset(gdf_clipped.columns)


# Testing extract_xyz_array
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_array(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz_array

    gdf_array = extract_xyz_array(gdf=gdf_interfaces1_points, dem=dem.read(1), extent=[0, 972.0, 0, 1069.0])

    # Assert type on input
    assert isinstance(gdf_array, gpd.GeoDataFrame)
    assert 'geometry' in gdf_array
    assert all(gdf_interfaces1_points.geom_type == 'Point')
    assert len(gdf_interfaces1_points) == 5
    assert len(gdf_array) == 5

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert gdf_interfaces1_points is not gdf_array

    # Assert CRS
    assert gdf_array.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_array.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_array.columns)
    assert not {'points', 'level_0', 'level_1', 'index', 'id'}.issubset(gdf_array.columns)

    # Assert if values are correct
    assert gdf_array['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                              157.812298994796, 191.3180280345144]
    assert gdf_array['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                              719.0939805375339]
    assert gdf_array['Z'].head().tolist() == [366.612548828125, 402.09912109375, 460.61810302734375, 529.015625,
                                              597.6325073242188]


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_array_drop_id(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz_array

    gdf_array = extract_xyz_array(gdf=gdf_interfaces1_points, dem=dem.read(1), extent=[0, 972.0, 0, 1069.0],
                                  drop_id=False)

    # Assert type on input
    assert isinstance(gdf_array, gpd.GeoDataFrame)
    assert 'geometry' in gdf_array
    assert all(gdf_interfaces1_points.geom_type == 'Point')
    assert len(gdf_interfaces1_points) == 5
    assert len(gdf_array) == 5

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert gdf_interfaces1_points is not gdf_array

    # Assert CRS
    assert gdf_array.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_array.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z', 'id'}.issubset(gdf_array.columns)
    assert not {'points', 'level_0', 'level_1', 'index'}.issubset(gdf_array.columns)

    # Assert if values are correct
    assert gdf_array['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                              157.812298994796, 191.3180280345144]
    assert gdf_array['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                              719.0939805375339]
    assert gdf_array['Z'].head().tolist() == [366.612548828125, 402.09912109375, 460.61810302734375, 529.015625,
                                              597.6325073242188]


# Testing extract_z_array
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_raster(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz

    gdf_raster = extract_xyz(gdf=gdf_interfaces1_points, dem=dem)

    # Assert type on input
    assert isinstance(gdf_raster, gpd.GeoDataFrame)
    assert 'geometry' in gdf_raster
    assert all(gdf_interfaces1_points.geom_type == 'Point')
    assert len(gdf_interfaces1_points) == 5
    assert len(gdf_raster) == 5

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert gdf_interfaces1_points is not gdf_raster

    # Assert CRS
    assert gdf_raster.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_raster.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_raster.columns)
    assert not {'points', 'level_0', 'level_1', 'index', 'id'}.issubset(gdf_raster.columns)

    # Assert if values are correct
    assert gdf_raster['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                               157.812298994796, 191.3180280345144]
    assert gdf_raster['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049,
                                               615.9994296460927,
                                               719.0939805375339]
    assert gdf_raster['Z'].head().tolist() == [364.994873046875, 400.3435974121094, 459.54931640625, 525.6910400390625,
                                               597.6325073242188]


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_arrays(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz

    gdf_raster = extract_xyz(gdf=gdf_interfaces1_points, dem=dem.read(1), extent=[0, 972.0, 0, 1069.0])

    # Assert type on input
    assert isinstance(gdf_raster, gpd.GeoDataFrame)
    assert 'geometry' in gdf_raster
    assert all(gdf_interfaces1_points.geom_type == 'Point')
    assert len(gdf_interfaces1_points) == 41
    assert len(gdf_raster) == 41

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert gdf_interfaces1_points is not gdf_raster

    # Assert CRS
    assert gdf_raster.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_raster.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_raster.columns)
    assert not {'points', 'level_0', 'level_1', 'index', 'id'}.issubset(gdf_raster.columns)

    # Assert if values are correct
    assert gdf_raster['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                               157.812298994796, 191.3180280345144]
    assert gdf_raster['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049,
                                               615.9994296460927,
                                               719.0939805375339]
    assert gdf_raster['Z'].head().tolist() == [366.612548828125, 402.09912109375, 460.61810302734375, 529.015625,
                                               597.6325073242188]


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_raster_drop_id(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz

    gdf_raster = extract_xyz(gdf=gdf_interfaces1_points, dem=dem, drop_id=False)

    # Assert type on input
    assert isinstance(gdf_raster, gpd.GeoDataFrame)
    assert 'geometry' in gdf_raster
    assert all(gdf_interfaces1_points.geom_type == 'Point')
    assert len(gdf_interfaces1_points) == 5
    assert len(gdf_raster) == 5

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert gdf_interfaces1_points is not gdf_raster

    # Assert CRS
    assert gdf_raster.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_raster.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z', 'id'}.issubset(gdf_raster.columns)
    assert not {'points', 'level_0', 'level_1', 'index'}.issubset(gdf_raster.columns)

    # Assert if values are correct
    assert gdf_raster['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                               157.812298994796, 191.3180280345144]
    assert gdf_raster['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049,
                                               615.9994296460927,
                                               719.0939805375339]
    assert gdf_raster['Z'].head().tolist() == [364.994873046875, 400.3435974121094, 459.54931640625, 525.6910400390625,
                                               597.6325073242188]


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_arrays(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz

    gdf_raster = extract_xyz(gdf=gdf_interfaces1_points, dem=dem.read(1), extent=[0, 972.0, 0, 1069.0], drop_id=False)

    # Assert type on input
    assert isinstance(gdf_raster, gpd.GeoDataFrame)
    assert 'geometry' in gdf_raster
    assert all(gdf_interfaces1_points.geom_type == 'Point')
    assert len(gdf_interfaces1_points) == 5
    assert len(gdf_raster) == 5

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert gdf_interfaces1_points is not gdf_raster

    # Assert CRS
    assert gdf_raster.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_raster.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z', 'id'}.issubset(gdf_raster.columns)
    assert not {'points', 'level_0', 'level_1', 'index'}.issubset(gdf_raster.columns)

    # Assert if values are correct
    assert gdf_raster['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                               157.812298994796, 191.3180280345144]
    assert gdf_raster['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049,
                                               615.9994296460927,
                                               719.0939805375339]
    assert gdf_raster['Z'].head().tolist() == [366.612548828125, 402.09912109375, 460.61810302734375, 529.015625,
                                               597.6325073242188]


# Testing extract_coordinates
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_coordinates_lines_dem_false(gdf_interfaces1_lines, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_lines, dem=dem)

    assert dem.read(1).ndim == 2
    assert dem.read(1).shape == (275, 250)
    assert isinstance(dem, rasterio.io.DatasetReader)
    assert isinstance(gdf_interfaces1_lines, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_lines.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676,
                                            27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]
    assert gdf_new['Z'].head().tolist() == [353.9727783203125, 359.03631591796875, 364.28497314453125, 364.994873046875,
                                            372.81036376953125]
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'
    assert dem.crs == {'init': 'epsg:4326'}
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_coordinates_lines_dem_true(gdf_interfaces1_lines, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_lines, dem=dem)

    assert dem.read(1).ndim == 2
    assert dem.read(1).shape == (275, 250)
    assert isinstance(dem, rasterio.io.DatasetReader)
    assert isinstance(gdf_interfaces1_lines, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_lines.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676,
                                            27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]
    assert gdf_new['Z'].head().tolist() == [353.9727783203125, 359.03631591796875, 364.28497314453125, 364.994873046875,
                                            372.81036376953125]
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'
    assert dem.crs == {'init': 'epsg:4326'}
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_coordinates_points_dem_false(gdf_interfaces1_lines, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_lines, dem=dem)

    assert dem.read(1).ndim == 2
    assert dem.read(1).shape == (275, 250)
    assert isinstance(dem, rasterio.io.DatasetReader)
    assert isinstance(gdf_interfaces1_lines, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_lines.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796,
                                            191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]
    assert gdf_new['Z'].head().tolist() == [364.994873046875, 400.3435974121094, 459.54931640625, 525.6910400390625,
                                            597.6325073242188]
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'
    assert dem.crs == {'init': 'epsg:4326'}
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'Point')


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_coordinates_points_dem_true(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_points, dem=dem)

    assert dem.read(1).ndim == 2
    assert dem.read(1).shape == (275, 250)
    assert isinstance(dem, rasterio.io.DatasetReader)
    assert isinstance(gdf_interfaces1_points, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796,
                                            191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]
    assert gdf_new['Z'].head().tolist() == [364.994873046875, 400.3435974121094, 459.54931640625, 525.6910400390625,
                                            597.6325073242188]
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'
    assert dem.crs == {'init': 'epsg:4326'}
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_interfaces1_points
    assert all(gdf_interfaces1_points.geom_type == 'Point')


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_coordinates_points_dem_false(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_points, dem=dem)

    assert dem.read(1).ndim == 2
    assert dem.read(1).shape == (275, 250)
    assert isinstance(dem, rasterio.io.DatasetReader)
    assert isinstance(gdf_interfaces1_points, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796,
                                            191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]
    assert gdf_new['Z'].head().tolist() == [364.994873046875, 400.3435974121094, 459.54931640625, 525.6910400390625,
                                            597.6325073242188]
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'
    assert dem.crs == {'init': 'epsg:4326'}
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_interfaces1_points
    assert all(gdf_interfaces1_points.geom_type == 'Point')


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             np.load('../docs/getting_started/tutorial/data/test_vector/array_rbf.npy')
                         ])
def test_extract_coordinates_points_array_false(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_points, dem=dem, extent=[0, 972, 0, 1069])

    assert dem.ndim == 2
    assert dem.shape == (1069, 972)
    assert isinstance(dem, np.ndarray)
    assert isinstance(gdf_interfaces1_points, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796,
                                            191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]
    assert gdf_new['Z'].head().tolist() == [469.09802654928296, 473.44941380590296, 483.88114008172556,
                                            485.0516805807032,
                                            472.7250883449502]
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_interfaces1_points
    assert all(gdf_interfaces1_points.geom_type == 'Point')


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             np.load('../docs/getting_started/tutorial/data/test_vector/array_rbf.npy')
                         ])
def test_extract_coordinates_points_array_true(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_points, dem=dem, extent=[0, 972, 0, 1069])

    assert dem.ndim == 2
    assert dem.shape == (1069, 972)
    assert isinstance(dem, np.ndarray)
    assert isinstance(gdf_interfaces1_points, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796,
                                            191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]
    assert gdf_new['Z'].head().tolist() == [469.09802654928296, 473.44941380590296, 483.88114008172556,
                                            485.0516805807032,
                                            472.7250883449502]
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_interfaces1_points
    assert all(gdf_interfaces1_points.geom_type == 'Point')


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
@pytest.mark.parametrize("dem",
                         [
                             np.load('../docs/getting_started/tutorial/data/test_vector/array_rbf.npy')
                         ])
def test_extract_coordinates_lines_array_false(gdf_interfaces1_lines, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_lines, dem=dem, extent=[0, 972, 0, 1069])

    assert dem.ndim == 2
    assert dem.shape == (1069, 972)
    assert isinstance(dem, np.ndarray)
    assert isinstance(gdf_interfaces1_lines, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_lines.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676,
                                            27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]
    assert gdf_new['Z'].head().tolist() == [466.7501589231589, 468.49775671714633, 468.9434645548434,
                                            469.09802654928296,
                                            469.77232323980155]
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
@pytest.mark.parametrize("dem",
                         [
                             np.load('../docs/getting_started/tutorial/data/test_vector/array_rbf.npy')
                         ])
def test_extract_coordinates_lines_array_true(gdf_interfaces1_lines, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_lines, dem=dem, extent=[0, 972, 0, 1069])

    assert dem.ndim == 2
    assert dem.shape == (1069, 972)
    assert isinstance(dem, np.ndarray)
    assert isinstance(gdf_interfaces1_lines, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_lines.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676,
                                            27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]
    assert gdf_new['Z'].head().tolist() == [466.7501589231589, 468.49775671714633, 468.9434645548434,
                                            469.09802654928296,
                                            469.77232323980155]
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
@pytest.mark.parametrize("dem",
                         [
                             np.load('../docs/getting_started/tutorial/data/test_vector/array_rbf.npy')
                         ])
def test_extract_coordinates_error(gdf_interfaces1_lines, dem):
    from gemgis.vector import extract_xyz
    with pytest.raises(TypeError):
        extract_xyz([gdf_interfaces1_lines], dem, extent=[0, 972, 0, 1069])
    with pytest.raises(TypeError):
        extract_xyz(gdf_interfaces1_lines, [dem], extent=[0, 972, 0, 1069])
    with pytest.raises(TypeError):
        extract_xyz(gdf_interfaces1_lines, dem, extent=(0, 972, 0, 1069))
    with pytest.raises(ValueError):
        extract_xyz(gdf_interfaces1_lines, dem, extent=[0, 972, 0, 1069, 100])


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_coordinates_points_dem_false(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz
    from gemgis.vector import extract_xy
    gdf_xy = extract_xy(gdf=gdf_interfaces1_points)
    gdf_new = extract_xyz(gdf=gdf_xy, dem=dem)

    assert dem.read(1).ndim == 2
    assert dem.read(1).shape == (275, 250)

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)
    assert isinstance(dem, rasterio.io.DatasetReader)
    assert isinstance(gdf_interfaces1_points, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796,
                                            191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]
    assert gdf_new['Z'].head().tolist() == [364.994873046875, 400.3435974121094, 459.54931640625, 525.6910400390625,
                                            597.6325073242188]
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'
    assert dem.crs == {'init': 'epsg:4326'}
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_interfaces1_points
    assert all(gdf_interfaces1_points.geom_type == 'Point')


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_coordinates_points_dem_false(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz
    from gemgis.vector import extract_xy
    gdf_xy = extract_xy(gdf=gdf_interfaces1_points)
    gdf_new = extract_xyz(gdf=gdf_xy, dem=dem)

    assert dem.read(1).ndim == 2
    assert dem.read(1).shape == (275, 250)

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)
    assert isinstance(dem, rasterio.io.DatasetReader)
    assert isinstance(gdf_interfaces1_points, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796,
                                            191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]
    assert gdf_new['Z'].head().tolist() == [364.994873046875, 400.3435974121094, 459.54931640625, 525.6910400390625,
                                            597.6325073242188]
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'
    assert dem.crs == {'init': 'epsg:4326'}
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_interfaces1_points
    assert all(gdf_interfaces1_points.geom_type == 'Point')


@pytest.mark.parametrize("gdf_topo1", [gdf_topo1])
def test_extract_coordinates_countours(gdf_topo1):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_topo1, dem=None)

    assert isinstance(gdf_topo1, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf_new, gpd.geodataframe.GeoDataFrame)
    assert 'Z' in gdf_topo1.columns
    assert not {'X', 'Y'}.issubset(gdf_topo1.columns)
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)
    assert gdf_new['X'].head().tolist() == [0.7408806771479846, 35.62873136073459, 77.30033078835194,
                                            104.7583614189525,
                                            127.0478215779106]
    assert gdf_new['Y'].head().tolist() == [475.4410147469845, 429.2469161566801, 340.0890755208477,
                                            269.3442671902416,
                                            207.6444571850097]
    assert gdf_new['Z'].head().tolist() == [400, 400, 400, 400, 400]
    assert gdf_topo1 is not gdf_new

    # Assert CRS
    assert gdf_topo1.crs == 'EPSG:4326'
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    assert 'geometry' in gdf_topo1
    assert all(gdf_topo1.geom_type == 'LineString')


# Testing extract_z
###########################################################

@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_points(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_points, dem=dem)

    # Assert type on input
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_points
    assert isinstance(dem, rasterio.io.DatasetReader)
    assert all(gdf_new.geom_type == 'Point')

    assert dem.read(1).ndim == 2
    assert dem.read(1).shape == (275, 250)

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'
    assert dem.crs == {'init': 'epsg:4326'}

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796, 191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_points_inplace(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_points, dem=dem)

    # Assert type on input
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_points
    assert isinstance(dem, rasterio.io.DatasetReader)
    assert all(gdf_new.geom_type == 'Point')

    assert dem.read(1).ndim == 2
    assert dem.read(1).shape == (275, 250)

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'
    assert dem.crs == {'init': 'epsg:4326'}

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796, 191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]
    assert gdf_new['Z'].head().tolist() == [364.994873046875, 400.3435974121094, 459.54931640625, 525.6910400390625,
                                            597.6325073242188]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_lines_inplace(gdf_interfaces1_lines, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_lines, dem=dem)

    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert isinstance(dem, rasterio.io.DatasetReader)
    assert all(gdf_new.geom_type == 'LineString')

    assert dem.read(1).ndim == 2
    assert dem.read(1).shape == (275, 250)

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'
    assert dem.crs == {'init': 'epsg:4326'}

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'LineString')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676, 27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]
    assert gdf_new['Z'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
@pytest.mark.parametrize("dem",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/raster1.tif')
                         ])
def test_extract_z_lines_inplace(gdf_interfaces1_lines, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_lines, dem=dem)

    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert isinstance(dem, rasterio.io.DatasetReader)
    assert all(gdf_new.geom_type == 'Point')

    assert dem.read(1).ndim == 2
    assert dem.read(1).shape == (275, 250)

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'
    assert dem.crs == {'init': 'epsg:4326'}

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676, 27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]
    assert gdf_new['Z'].head().tolist() == [353.9727783203125, 359.03631591796875, 364.28497314453125, 364.994873046875,
                                            372.81036376953125]


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             np.load('../docs/getting_started/tutorial/data/test_vector/array_rbf.npy')
                         ])
def test_extract_z_points_array(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_points, dem=dem, extent=[0, 972, 0, 1069])

    # Assert type on input
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_points
    assert isinstance(dem, np.ndarray)
    assert all(gdf_new.geom_type == 'Point')

    assert dem.ndim == 2
    assert dem.shape == (1069, 972)

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796, 191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]
    assert gdf_new['Z'].head().tolist() == [387.2258761923539, 387.154888907343, 387.3960957643691, 387.5444087461885,
                                            388.6688927116212]


@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
@pytest.mark.parametrize("dem",
                         [
                             np.load('../docs/getting_started/tutorial/data/test_vector/array_rbf.npy')
                         ])
def test_extract_z_points_array(gdf_interfaces1_points, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_points, dem=dem, extent=[0, 972, 0, 1069])

    # Assert type on input
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_points
    assert isinstance(dem, np.ndarray)
    assert all(gdf_new.geom_type == 'Point')

    assert dem.ndim == 2
    assert dem.shape == (1069, 972)

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796, 191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049, 615.9994296460927,
                                            719.0939805375339]
    assert gdf_new['Z'].head().tolist() == [387.2258761923539, 387.154888907343, 387.3960957643691, 387.5444087461885,
                                            388.6688927116212]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
@pytest.mark.parametrize("dem",
                         [
                             np.load('../docs/getting_started/tutorial/data/test_vector/array_rbf.npy')
                         ])
def test_extract_z_points_array(gdf_interfaces1_lines, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_lines, dem=dem, extent=[0, 972, 0, 1069])

    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert isinstance(dem, np.ndarray)
    assert all(gdf_new.geom_type == 'Point')

    assert dem.ndim == 2
    assert dem.shape == (1069, 972)

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676,
                                            27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]
    assert gdf_new['Z'].head().tolist() == [466.7501589231589, 468.49775671714633, 468.9434645548434,
                                            469.09802654928296,
                                            469.77232323980155]


@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
@pytest.mark.parametrize("dem",
                         [
                             np.load('../docs/getting_started/tutorial/data/test_vector/array_rbf.npy')
                         ])
def test_extract_z_values_points_array(gdf_interfaces1_lines, dem):
    from gemgis.vector import extract_xyz
    gdf_new = extract_xyz(gdf=gdf_interfaces1_lines, dem=dem, extent=[0, 972, 0, 1069])

    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert isinstance(dem, np.ndarray)
    assert all(gdf_new.geom_type == 'Point')

    assert dem.ndim == 2
    assert dem.shape == (1069, 972)

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y', 'Z'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y', 'Z'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [0.256327195431048, 10.59346813871597, 17.134940141888464,
                                            19.150128045807676, 27.79511673965105]
    assert gdf_new['Y'].head().tolist() == [264.86214748436396, 276.73370778641777, 289.089821570188, 293.313485355882,
                                            310.571692592952]
    assert gdf_new['Z'].head().tolist() == [466.7501589231589, 468.49775671714633, 468.9434645548434,
                                            469.09802654928296,
                                            469.77232323980155]


# Testing clip_vector_data_by_extent
###########################################################
@pytest.mark.parametrize("gdf_randompoints1", [gdf_randompoints1])
def test_clip_vector_data_by_extent(gdf_randompoints1):
    from gemgis.vector import clip_by_bbox

    gdf = clip_by_bbox(gdf=gdf_randompoints1, bbox=[0, 1069, 0, 972])

    assert len(gdf_randompoints1) == 5
    assert len(gdf) == 3

    assert gdf_randompoints1.bounds.min()[0] == -144.4205416504286
    assert gdf_randompoints1.bounds.min()[1] == -140.9983501274578
    assert gdf_randompoints1.bounds.min()[2] == -144.4205416504286
    assert gdf_randompoints1.bounds.min()[3] == -140.9983501274578

    assert gdf_randompoints1.bounds.max()[0] == 1127.312693889784
    assert gdf_randompoints1.bounds.max()[1] == 902.0868083698422
    assert gdf_randompoints1.bounds.max()[2] == 1127.312693889784
    assert gdf_randompoints1.bounds.max()[3] == 902.0868083698422

    assert gdf.bounds.min()[0] == 281.5257600645256
    assert gdf.bounds.min()[1] == 255.6839742805063
    assert gdf.bounds.min()[2] == 281.5257600645256
    assert gdf.bounds.min()[3] == 255.6839742805063

    assert gdf.bounds.max()[0] == 925.866703136033
    assert gdf.bounds.max()[1] == 902.0868083698422
    assert gdf.bounds.max()[2] == 925.866703136033
    assert gdf.bounds.max()[3] == 902.0868083698422

    assert isinstance(gdf_randompoints1, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_randompoints1.columns)
    assert {'X', 'Y'}.issubset(gdf.columns)

    assert gdf is not gdf_randompoints1

    # Assert CRS
    assert gdf.crs == 'EPSG:4326'
    assert gdf_randompoints1.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_randompoints1.geom_type == 'Point')

    assert 'geometry' in gdf
    assert all(gdf.geom_type == 'Point')


@pytest.mark.parametrize("gdf_randompoints1", [gdf_randompoints1])
def test_clip_vector_data_by_extent_error(gdf_randompoints1):
    from gemgis.vector import clip_by_bbox

    with pytest.raises(TypeError):
        clip_by_bbox(gdf=[gdf_randompoints1], bbox=[0, 1069, 0, 972])
    with pytest.raises(TypeError):
        clip_by_bbox(gdf=gdf_randompoints1, bbox=(0, 1069, 0, 972))


# Testing extract_xy dropping columns
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_points", [gdf_interfaces1_points])
def test_extract_xy_drop_id(gdf_interfaces1_points):
    from gemgis.vector import extract_xy

    gdf_new = extract_xy(gdf=gdf_interfaces1_points)
    # Assert type on input
    assert isinstance(gdf_interfaces1_points, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_points
    assert all(gdf_interfaces1_points.geom_type == 'Point')

    # Assert CRS
    assert gdf_interfaces1_points.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_points.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_interfaces1_points is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)

    # Assert if values are correct
    assert gdf_new['X'].head().tolist() == [19.150128045807676, 61.93436666575576, 109.3578600758187,
                                            157.812298994796, 191.3180280345144]
    assert gdf_new['Y'].head().tolist() == [293.313485355882, 381.4593263680641, 480.9455679783049,
                                            615.9994296460927, 719.0939805375339]

    assert not {'id'}.issubset(gdf_new.columns)


# Testing extract_xy dropping columns 2
###########################################################
@pytest.mark.parametrize("gdf_interfaces1_lines", [gdf_interfaces1_lines])
def test_extract_xy_drop_index(gdf_interfaces1_lines):
    from gemgis.vector import extract_xy

    gdf_new = extract_xy(gdf=gdf_interfaces1_lines)
    # Assert type on input
    assert isinstance(gdf_interfaces1_lines, gpd.GeoDataFrame)
    assert 'geometry' in gdf_interfaces1_lines
    assert all(gdf_interfaces1_lines.geom_type == 'LineString')

    # Assert CRS
    assert gdf_interfaces1_lines.crs == 'EPSG:4326'

    # Assert if columns are already in input gdf
    assert not {'X', 'Y'}.issubset(gdf_interfaces1_lines.columns)

    # Assert type of output
    assert isinstance(gdf_new, gpd.GeoDataFrame)
    assert gdf_interfaces1_lines is not gdf_new

    # Assert CRS
    assert gdf_new.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_new.geom_type == 'Point')

    # Assert if columns are in gdf_new
    assert {'X', 'Y'}.issubset(gdf_new.columns)

    assert not {'index'}.issubset(gdf_new.columns)


# Testing extract_xy for MultiLineStrings and LineStrings
###########################################################
@pytest.mark.parametrize("gdf",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/GeologicalMapAachen.shp')
                         ])
def test_extract_xy_multilinestrings2(gdf):
    from gemgis.vector import explode_polygons
    from gemgis.vector import extract_xy

    gdf_linestrings = explode_polygons(gdf=gdf)

    gdf_linestrings_xy = extract_xy(gdf=gdf_linestrings)

    assert isinstance(gdf_linestrings_xy, gpd.geodataframe.GeoDataFrame)

    assert gdf_linestrings_xy.crs == 'EPSG:4647'

    assert not {'id'}.issubset(gdf_linestrings_xy.columns)
    assert not {'index'}.issubset(gdf_linestrings_xy.columns)
    assert {'X', 'Y', 'geometry'}.issubset(gdf_linestrings_xy.columns)


# Testing interpolate_raster
###########################################################

@pytest.mark.parametrize("gdf_topo1", [gdf_topo1])
def test_interpolate_raster_nearest(gdf_topo1):
    from gemgis.vector import interpolate_raster
    from gemgis.vector import extract_xy

    gdf_xyz = extract_xy(gdf=gdf_topo1)
    raster = interpolate_raster(gdf=gdf_xyz, method='nearest')

    assert isinstance(gdf_topo1, gpd.geodataframe.GeoDataFrame)
    assert {'X', 'Y', 'Z'}.issubset(gdf_xyz.columns)

    assert isinstance(raster, np.ndarray)


@pytest.mark.parametrize("gdf_topo1", [gdf_topo1])
def test_interpolate_raster_linear(gdf_topo1):
    from gemgis.vector import interpolate_raster
    from gemgis.vector import extract_xy

    gdf_xyz = extract_xy(gdf=gdf_topo1)
    raster = interpolate_raster(gdf=gdf_xyz, method='linear')

    assert isinstance(gdf_topo1, gpd.geodataframe.GeoDataFrame)
    assert {'X', 'Y', 'Z'}.issubset(gdf_xyz.columns)

    assert isinstance(raster, np.ndarray)


@pytest.mark.parametrize("gdf_topo1", [gdf_topo1])
def test_interpolate_raster_cubic(gdf_topo1):
    from gemgis.vector import interpolate_raster
    from gemgis.vector import extract_xy

    gdf_xyz = extract_xy(gdf=gdf_topo1)
    raster = interpolate_raster(gdf=gdf_xyz, method='cubic')

    assert isinstance(gdf_topo1, gpd.geodataframe.GeoDataFrame)
    assert {'X', 'Y', 'Z'}.issubset(gdf_xyz.columns)

    assert isinstance(raster, np.ndarray)


@pytest.mark.parametrize("gdf_topo1", [gdf_topo1])
def test_interpolate_raster_rbf(gdf_topo1):
    from gemgis.vector import interpolate_raster
    from gemgis.vector import extract_xy

    gdf_xyz = extract_xy(gdf=gdf_topo1)
    raster = interpolate_raster(gdf=gdf_xyz, method='rbf')

    assert isinstance(gdf_topo1, gpd.geodataframe.GeoDataFrame)
    assert {'X', 'Y', 'Z'}.issubset(gdf_xyz.columns)

    assert isinstance(raster, np.ndarray)


@pytest.mark.parametrize("gdf_topo1", [gdf_topo1])
def test_interpolate_raster_error(gdf_topo1):
    from gemgis.vector import interpolate_raster
    from gemgis.vector import extract_xy

    gdf_xyz = extract_xy(gdf_topo1)
    with pytest.raises(TypeError):
        interpolate_raster(gdf=[gdf_xyz], method='linear')
    with pytest.raises(TypeError):
        interpolate_raster(gdf=gdf_xyz, method=['linear'])


@pytest.mark.parametrize("gdf_topo1", [gdf_topo1])
def test_interpolate_raster_rbf_samples(gdf_topo1):
    from gemgis.vector import interpolate_raster
    from gemgis.vector import extract_xy

    gdf_xyz = extract_xy(gdf=gdf_topo1)
    raster = interpolate_raster(gdf=gdf_xyz, method='rbf')

    assert isinstance(gdf_topo1, gpd.geodataframe.GeoDataFrame)
    assert {'X', 'Y', 'Z'}.issubset(gdf_xyz.columns)

    assert isinstance(raster, np.ndarray)


@pytest.mark.parametrize("gdf_topo1", [gdf_topo1])
def test_interpolate_raster_rbf_samples_error(gdf_topo1):
    from gemgis.vector import interpolate_raster
    from gemgis.vector import extract_xy

    gdf_xyz = extract_xy(gdf=gdf_topo1)

    with pytest.raises(ValueError):
        interpolate_raster(gdf_xyz, method='rbf', n=500)


@pytest.mark.parametrize("gdf_topo5", [gdf_topo5])
def test_interpolate_raster_rbf_linalg_error(gdf_topo5):
    from gemgis.vector import interpolate_raster
    from gemgis.vector import extract_xy

    gdf_xyz = extract_xy(gdf=gdf_topo5)

    interpolate_raster(gdf=gdf_xyz, method='rbf', n=30)


@pytest.mark.parametrize("gdf_topo5", [gdf_topo5])
def test_interpolate_raster_rbf_linalg_no_error(gdf_topo5):
    from gemgis.vector import interpolate_raster
    from gemgis.vector import extract_xy

    np.random.seed(1)
    gdf_xyz = extract_xy(gdf=gdf_topo5)
    raster = interpolate_raster(gdf=gdf_xyz, method='rbf', n=30)

    assert isinstance(gdf_topo5, gpd.geodataframe.GeoDataFrame)
    assert {'X', 'Y', 'Z'}.issubset(gdf_xyz.columns)

    assert isinstance(raster, np.ndarray)


# Testing clip_vector_data_by_shape
###########################################################
@pytest.mark.parametrize("gdf_randompoints1", [gdf_randompoints1])
def test_clip_vector_data_by_shape(gdf_randompoints1):
    from gemgis.vector import clip_by_polygon

    polygon = Polygon([[-3.425391036716071e-06, 1068.999995417988],
                       [972.0000001040772, 1068.999998985101],
                       [972.0001356544709, 8.639147955555315e-05],
                       [4.294711243593707e-05, -6.580545516307601e-05],
                       [-3.425391036716071e-06, 1068.999995417988]])

    gdf = clip_by_polygon(gdf=gdf_randompoints1, polygon=polygon)

    assert len(gdf_randompoints1) == 5
    assert len(gdf) == 3

    assert gdf_randompoints1.bounds.min()[0] == -144.4205416504286
    assert gdf_randompoints1.bounds.min()[1] == -140.9983501274578
    assert gdf_randompoints1.bounds.min()[2] == -144.4205416504286
    assert gdf_randompoints1.bounds.min()[3] == -140.9983501274578

    assert gdf_randompoints1.bounds.max()[0] == 1127.312693889784
    assert gdf_randompoints1.bounds.max()[1] == 902.0868083698422
    assert gdf_randompoints1.bounds.max()[2] == 1127.312693889784
    assert gdf_randompoints1.bounds.max()[3] == 902.0868083698422

    assert gdf.bounds.min()[0] == 281.5257600645256
    assert gdf.bounds.min()[1] == 255.6839742805063
    assert gdf.bounds.min()[2] == 281.5257600645256
    assert gdf.bounds.min()[3] == 255.6839742805063

    assert gdf.bounds.max()[0] == 925.866703136033
    assert gdf.bounds.max()[1] == 902.0868083698422
    assert gdf.bounds.max()[2] == 925.866703136033
    assert gdf.bounds.max()[3] == 902.0868083698422

    assert isinstance(polygon, Polygon)
    assert isinstance(gdf_randompoints1, gpd.geodataframe.GeoDataFrame)
    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert not {'X', 'Y', 'Z'}.issubset(gdf_randompoints1.columns)
    assert not {'X', 'Y'}.issubset(gdf.columns)

    assert gdf is not gdf_randompoints1

    # Assert CRS
    assert gdf.crs == 'EPSG:4326'
    assert gdf_randompoints1.crs == 'EPSG:4326'

    # Assert Type of shape file
    assert all(gdf_randompoints1.geom_type == 'Point')

    assert 'geometry' in gdf
    assert all(gdf.geom_type == 'Point')


@pytest.mark.parametrize("gdf_randompoints1", [gdf_randompoints1])
def test_clip_vector_data_by_shape_error(gdf_randompoints1):
    from gemgis.vector import clip_by_polygon

    polygon = Polygon([[-3.425391036716071e-06, 1068.999995417988],
                       [972.0000001040772, 1068.999998985101],
                       [972.0001356544709, 8.639147955555315e-05],
                       [4.294711243593707e-05, -6.580545516307601e-05],
                       [-3.425391036716071e-06, 1068.999995417988]])

    with pytest.raises(TypeError):
        clip_by_polygon(gdf=[gdf_randompoints1], polygon=polygon)
    with pytest.raises(TypeError):
        clip_by_polygon(gdf=gdf_randompoints1, polygon=[polygon])


# Testing create_buffer
###########################################################
def test_create_buffer_point():
    from gemgis.vector import create_buffer

    point = Point(0.0, 0.0)

    polygon = create_buffer(geom_object=point, distance=5)

    assert isinstance(point, Point)
    assert isinstance(polygon, Polygon)
    assert polygon.area == 78.41371226364848


def test_create_buffer_linestring():
    from gemgis.vector import create_buffer

    line = LineString([(0, 0), (2, 2)])

    polygon = create_buffer(geom_object=line, distance=5)

    assert isinstance(line, LineString)
    assert isinstance(polygon, Polygon)
    assert polygon.area == 106.69798351111038


# Testing subtract_geom_objects
###########################################################
def test_subtract_geom_objects():
    from gemgis.vector import subtract_geom_objects

    polygon = Polygon([[0, 0], [2, 0], [2, 2], [0, 2], [0, 0]])

    line = LineString([(0, 0), (2, 2)])

    assert isinstance(polygon, Polygon)
    assert isinstance(line, LineString)

    result = subtract_geom_objects(geom_object1=polygon, geom_object2=line.buffer(0.2))

    assert isinstance(result, MultiPolygon)

    result = subtract_geom_objects(geom_object1=line.buffer(2), geom_object2=polygon)

    assert isinstance(result, Polygon)


# Testing remove_object_within_buffer
###########################################################
@pytest.mark.parametrize("faults",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/GK50_Tektonik.shp')
                         ])
@pytest.mark.parametrize("interfaces",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/GeologicalMapAachen.shp')

                         ])
def test_remove_object_within_buffer(faults, interfaces):
    from gemgis.vector import remove_object_within_buffer

    interfaces_linestrings = [interfaces.boundary[i] for i in range(len(interfaces))]
    interfaces_gdf = gpd.GeoDataFrame({'geometry': interfaces_linestrings}, crs=interfaces.crs)

    fault = faults.loc[782].geometry
    interface_points = interfaces_gdf.iloc[710].geometry

    result_out, result_in = remove_object_within_buffer(buffer_object=fault,
                                                        buffered_object=interface_points,
                                                        distance=500)

    assert isinstance(result_out, LineString)
    assert isinstance(result_in, LineString)


# Testing remove_objects_within_buffer
###########################################################
@pytest.mark.parametrize("faults",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/GK50_Tektonik.shp')
                         ])
@pytest.mark.parametrize("interfaces",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/GeologicalMapAachen.shp')

                         ])
def test_remove_objects_within_buffer(faults, interfaces):
    from gemgis.vector import remove_objects_within_buffer

    interfaces_linestrings = [interfaces.boundary[i] for i in range(len(interfaces))]
    interfaces_gdf = gpd.GeoDataFrame({'geometry': interfaces_linestrings}, crs=interfaces.crs)

    fault = faults.loc[782].geometry
    result_out, result_in = remove_objects_within_buffer(buffer_object=fault,
                                                         buffered_objects_gdf=interfaces_gdf,
                                                         distance=500,
                                                         return_gdfs=False,
                                                         remove_empty_geometries=False,
                                                         extract_coordinates=False)

    assert isinstance(result_out, list)
    assert isinstance(result_in, list)

    result_out, result_in = remove_objects_within_buffer(buffer_object=fault,
                                                         buffered_objects_gdf=interfaces_gdf,
                                                         distance=500,
                                                         return_gdfs=True,
                                                         remove_empty_geometries=False,
                                                         extract_coordinates=False)

    assert isinstance(result_out, gpd.geodataframe.GeoDataFrame)
    assert isinstance(result_in, gpd.geodataframe.GeoDataFrame)
    assert len(result_out) == 848
    assert len(result_in) == 848

    result_out, result_in = remove_objects_within_buffer(buffer_object=fault,
                                                         buffered_objects_gdf=interfaces_gdf,
                                                         distance=500,
                                                         return_gdfs=True,
                                                         remove_empty_geometries=True,
                                                         extract_coordinates=False)

    assert isinstance(result_out, gpd.geodataframe.GeoDataFrame)
    assert isinstance(result_in, gpd.geodataframe.GeoDataFrame)
    assert len(result_out) == 848
    assert len(result_in) == 0

    result_out, result_in = remove_objects_within_buffer(buffer_object=fault,
                                                         buffered_objects_gdf=interfaces_gdf,
                                                         distance=500,
                                                         return_gdfs=True,
                                                         remove_empty_geometries=True,
                                                         extract_coordinates=True)

    assert isinstance(result_out, gpd.geodataframe.GeoDataFrame)
    assert isinstance(result_in, gpd.geodataframe.GeoDataFrame)
    assert len(result_out) == 47621
    assert len(result_in) == 0
    assert all(result_out.geom_type == 'Point')


# Testing remove_objects_within_buffers
###########################################################
@pytest.mark.parametrize("faults",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/GK50_Tektonik.shp')
                         ])
@pytest.mark.parametrize("interfaces",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/GeologicalMapAachen.shp')

                         ])
def test_remove_interfaces_within_fault_buffers(faults, interfaces):
    from gemgis.vector import remove_interfaces_within_fault_buffers

    interfaces_linestrings = [interfaces.boundary[i] for i in range(len(interfaces))]
    interfaces_gdf = gpd.GeoDataFrame({'geometry': interfaces_linestrings}, crs=interfaces.crs)

    result_out, result_in = remove_interfaces_within_fault_buffers(fault_gdf=faults.loc[:10],
                                                                   interfaces_gdf=interfaces_gdf,
                                                                   distance=500,
                                                                   remove_empty_geometries=False,
                                                                   extract_coordinates=False)

    assert isinstance(result_out, gpd.geodataframe.GeoDataFrame)
    assert isinstance(result_in, gpd.geodataframe.GeoDataFrame)

    result_out, result_in = remove_interfaces_within_fault_buffers(fault_gdf=faults.loc[:10],
                                                                   interfaces_gdf=interfaces_gdf,
                                                                   distance=500,
                                                                   remove_empty_geometries=False,
                                                                   extract_coordinates=False)

    assert isinstance(result_out, gpd.geodataframe.GeoDataFrame)
    assert isinstance(result_in, gpd.geodataframe.GeoDataFrame)
    assert len(result_out) == 848
    assert len(result_in) == 848

    result_out, result_in = remove_interfaces_within_fault_buffers(fault_gdf=faults.loc[:10],
                                                                   interfaces_gdf=interfaces_gdf,
                                                                   distance=500,
                                                                   remove_empty_geometries=True,
                                                                   extract_coordinates=False)

    assert isinstance(result_out, gpd.geodataframe.GeoDataFrame)
    assert isinstance(result_in, gpd.geodataframe.GeoDataFrame)
    assert len(result_out) == 848
    assert len(result_in) == 0

    result_out, result_in = remove_interfaces_within_fault_buffers(fault_gdf=faults.loc[:10],
                                                                   interfaces_gdf=interfaces_gdf,
                                                                   distance=500,
                                                                   remove_empty_geometries=True,
                                                                   extract_coordinates=True)

    assert isinstance(result_out, gpd.geodataframe.GeoDataFrame)
    assert isinstance(result_in, gpd.geodataframe.GeoDataFrame)
    assert len(result_out) == 47621
    assert len(result_in) == 0
    assert all(result_out.geom_type == 'Point')


# Testing calculate angle
###########################################################

def test_calculate_angle():
    from gemgis.vector import calculate_angle

    linestring = LineString([(0, 0), (2, 3)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_angle(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 146.30993247402023

    linestring = LineString([(0, 0), (2, -3)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_angle(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 33.690067525979785

    linestring = LineString([(2, 3), (0, 0)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_angle(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 33.690067525979785

    linestring = LineString([(2, -3), (0, 0)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_angle(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 146.30993247402023


# Testing calculate_strike_direction_straight_linestring
###########################################################
def test_calculate_strike_direction_straight_linestring():
    from gemgis.vector import calculate_strike_direction_straight_linestring

    linestring = LineString([(0, 0), (2, 3)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_strike_direction_straight_linestring(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 33.69006752597977

    linestring = LineString([(0, 0), (2, -3)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_strike_direction_straight_linestring(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 146.30993247402023

    linestring = LineString([(2, 3), (0, 0)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_strike_direction_straight_linestring(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 213.69006752597977

    linestring = LineString([(2, -3), (0, 0)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_strike_direction_straight_linestring(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 326.30993247402023

    linestring = LineString([(0, 0), (10, 0)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_strike_direction_straight_linestring(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 90

    linestring = LineString([(10, 0), (0, 0)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_strike_direction_straight_linestring(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 270

    linestring = LineString([(0, 0), (0, -10)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_strike_direction_straight_linestring(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 180

    linestring = LineString([(0, 0), (0, 10)])

    assert isinstance(linestring, LineString)
    assert len(linestring.coords) == 2

    angle = calculate_strike_direction_straight_linestring(linestring=linestring)

    assert isinstance(angle, float)
    assert angle == 0


# Testing calculate_strike_direction_straight_linestring
###########################################################
def test_explode_linestring():
    from gemgis.vector import explode_linestring_to_elements

    linestring = LineString([(0, 0), (1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

    splitted_linestrings = explode_linestring_to_elements(linestring=linestring)

    assert isinstance(splitted_linestrings, list)
    assert all(isinstance(n, LineString) for n in splitted_linestrings)
    assert len(splitted_linestrings) == 5
    assert splitted_linestrings[0].wkt == 'LINESTRING (0 0, 1 1)'
    assert splitted_linestrings[1].wkt == 'LINESTRING (1 1, 2 2)'
    assert splitted_linestrings[2].wkt == 'LINESTRING (2 2, 3 3)'
    assert splitted_linestrings[3].wkt == 'LINESTRING (3 3, 4 4)'
    assert splitted_linestrings[4].wkt == 'LINESTRING (4 4, 5 5)'

    linestring = LineString([(0, 0), (1, 1)])

    splitted_linestrings = explode_linestring_to_elements(linestring=linestring)

    assert isinstance(splitted_linestrings, list)
    assert all(isinstance(n, LineString) for n in splitted_linestrings)
    assert len(splitted_linestrings) == 1
    assert splitted_linestrings[0].wkt == 'LINESTRING (0 0, 1 1)'


# Testing calculate_strike_direction_bent_linestring
###########################################################
def test_calculate_strike_direction_bent_linestring():
    from gemgis.vector import calculate_strike_direction_bent_linestring

    linestring = LineString([(0, 0), (1, 2), (2, 1), (3, 5), (4, 7), (6, 3)])

    angles = calculate_strike_direction_bent_linestring(linestring=linestring)

    assert isinstance(angles, list)
    assert all(isinstance(n, float) for n in angles)
    assert len(angles) == 5
    assert angles[0] == 26.565051177078004
    assert angles[1] == 135
    assert angles[2] == 14.036243467926482
    assert angles[3] == 26.565051177078004
    assert angles[4] == 153.434948822922

    linestring = LineString([(6, 3), (4, 7), (3, 5), (2, 1), (1, 2), (0, 0)])

    angles = calculate_strike_direction_bent_linestring(linestring=linestring)

    assert isinstance(angles, list)
    assert all(isinstance(n, float) for n in angles)
    assert len(angles) == 5
    assert angles[0] == 333.434948822922
    assert angles[1] == 206.565051177078
    assert angles[2] == 194.03624346792648
    assert angles[3] == 315
    assert angles[4] == 206.565051177078


# Testing calculate_dipping_angle_linestring
###########################################################
def test_calculate_dipping_angle_linestring():
    from gemgis.vector import calculate_dipping_angle_linestring

    linestring = LineString([(0, 0), (1, -1)])

    dip = calculate_dipping_angle_linestring(linestring=linestring)
    assert isinstance(dip, float)
    assert dip == 45

    linestring = LineString([(0, 0), (1, -2)])

    dip = calculate_dipping_angle_linestring(linestring=linestring)
    assert isinstance(dip, float)
    assert dip == 63.43494882292201

    linestring = LineString([(0, 0), (1, -0.5)])

    dip = calculate_dipping_angle_linestring(linestring=linestring)
    assert isinstance(dip, float)
    assert dip == 26.56505117707799

    linestring = LineString([(1, -1), (0, 0)])

    dip = calculate_dipping_angle_linestring(linestring=linestring)
    assert isinstance(dip, float)
    assert dip == 45

    linestring = LineString([(1, -2), (0, 0)])

    dip = calculate_dipping_angle_linestring(linestring=linestring)
    assert isinstance(dip, float)
    assert dip == 63.43494882292201

    linestring = LineString([(1, -0.5), (0, 0)])

    dip = calculate_dipping_angle_linestring(linestring=linestring)
    assert isinstance(dip, float)
    assert dip == 26.56505117707799


# Testing calculate_dipping_angles_linestrings
##########################################################
def test_calculate_dipping_angles_linestrings():
    from gemgis.vector import calculate_dipping_angles_linestrings

    linestring_list = [LineString([(0, 0), (1, 1)]),
                       LineString([(0, 0), (1, -2)]),
                       LineString([(0, 0), (1, -0.5)]),
                       LineString([(1, -1), (0, 0)]),
                       LineString([(1, -2), (0, 0)]),
                       LineString([(1, -0.5), (0, 0)])]

    angles = calculate_dipping_angles_linestrings(linestring_list=linestring_list)

    assert isinstance(angles, list)
    assert all(isinstance(n, float) for n in angles)
    assert len(angles) == 6
    assert angles == [45, 63.43494882292201, 26.56505117707799, 45, 63.43494882292201, 26.56505117707799]

    linestring_list = [LineString([(0, 0), (1, 1)]),
                       LineString([(0, 0), (1, -2)]),
                       LineString([(0, 0), (1, -0.5)]),
                       LineString([(1, -1), (0, 0)]),
                       LineString([(1, -2), (0, 0)]),
                       LineString([(1, -0.5), (0, 0)])]

    linestring_gdf = gpd.GeoDataFrame(geometry=linestring_list)

    angles = calculate_dipping_angles_linestrings(linestring_list=linestring_gdf)

    assert isinstance(angles, list)
    assert all(isinstance(n, float) for n in angles)
    assert len(angles) == 6
    assert angles == [45, 63.43494882292201, 26.56505117707799, 45, 63.43494882292201, 26.56505117707799]


# Testing calculate_coordinates_for_point_on_cross_section
##########################################################
def test_calculate_coordinates_for_point_on_cross_section():
    from gemgis.vector import calculate_coordinates_for_point_on_cross_section

    linestring = LineString([(0, 0), (10, 0)])

    point = Point(5, 10)

    coordinates = calculate_coordinates_for_point_on_cross_section(linestring=linestring,
                                                                   point=point)

    assert isinstance(coordinates, Point)
    assert coordinates.wkt == 'POINT (5 0)'

    linestring = LineString([(0, 0), (10, 10)])

    point = Point(5, 10)

    coordinates = calculate_coordinates_for_point_on_cross_section(linestring=linestring,
                                                                   point=point)

    assert isinstance(coordinates, Point)
    assert coordinates.wkt == 'POINT (3.535533905932737 3.535533905932737)'


# Testing calculate_coordinates_for_linestring_on_straight_cross_sections
##########################################################
def test_calculate_coordinates_for_linestring_on_straight_cross_sections():
    from gemgis.vector import calculate_coordinates_for_linestring_on_cross_sections

    linestring = LineString([(0, 0), (10, 10)])

    interfaces = LineString([(5, -5), (6, -10)])

    points = calculate_coordinates_for_linestring_on_cross_sections(linestring=linestring,
                                                                    interfaces=interfaces)

    assert isinstance(points, list)
    assert all(isinstance(n, Point) for n in points)
    assert len(points) == 2
    assert points[0].wkt == 'POINT (3.535533905932737 3.535533905932737)'
    assert points[1].wkt == 'POINT (4.242640687119285 4.242640687119285)'


# Testing calculate_coordinates_for_linestrings_on_straight_cross_sections
##########################################################
def test_calculate_coordinates_for_linestrings_on_straight_cross_sections():
    from gemgis.vector import calculate_coordinates_for_linestrings_on_cross_sections

    linestring = LineString([(0, 0), (10, 10)])

    interfaces_list = [LineString([(5, -5), (6, -10)]), LineString([(4, -4), (7, -11)])]

    points = calculate_coordinates_for_linestrings_on_cross_sections(
        linestring=linestring,
        linestring_interfaces_list=interfaces_list)

    assert isinstance(points, list)
    assert all(isinstance(n, Point) for n in points)
    assert len(points) == 4
    assert points[0].wkt == 'POINT (3.535533905932737 3.535533905932737)'
    assert points[1].wkt == 'POINT (4.242640687119285 4.242640687119285)'
    assert points[2].wkt == 'POINT (2.82842712474619 2.82842712474619)'
    assert points[3].wkt == 'POINT (4.949747468305833 4.949747468305833)'


# Testing extract_interfaces_coordinates_from_cross_section
##########################################################
def test_extract_interfaces_coordinates_from_cross_section():
    from gemgis.vector import extract_interfaces_coordinates_from_cross_section

    linestring = LineString([(0, 0), (10, 12)])

    interfaces_list = [LineString([(5, -5), (6, -10)]), LineString([(4, -4), (7, -11)])]

    interfaces_gdf = gpd.GeoDataFrame(geometry=interfaces_list)

    gdf = extract_interfaces_coordinates_from_cross_section(linestring=linestring,
                                                            interfaces_gdf=interfaces_gdf,
                                                            extract_coordinates=True)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert len(gdf) == 4
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'Z', 'geometry'}.issubset(gdf.columns)
    assert gdf.loc[0][['X', 'Y', 'Z']].values.tolist() == [3.2009219983223995, 3.84110639798688, -5]
    assert gdf.loc[1][['X', 'Y', 'Z']].values.tolist() == [3.8411063979868794, 4.609327677584255, -10]
    assert gdf.loc[2][['X', 'Y', 'Z']].values.tolist() == [2.5607375986579193, 3.0728851183895034, -4]
    assert gdf.loc[3][['X', 'Y', 'Z']].values.tolist() == [4.48129079765136, 5.377548957181631, -11]


# Testing extract_xyz_from_cross_sections
##########################################################
def test_extract_xyz_from_cross_sections():
    from gemgis.vector import extract_xyz_from_cross_sections

    names = ['Profile1', 'Profile2']
    formation = ['Formation1', 'Formation2']

    linestrings = [LineString([(0, 0), (10, 12)]), LineString([(0, 5), (10, 12)])]

    profile_gdf = gpd.GeoDataFrame(data=names,
                                   geometry=linestrings)

    profile_gdf.columns = ['name', 'geometry']

    interfaces_list = [LineString([(5, -5), (6, -10)]), LineString([(4, -4), (7, -11)])]

    interfaces_gdf = gpd.GeoDataFrame(data=pd.DataFrame([names, formation]).T,
                                      geometry=interfaces_list)

    interfaces_gdf.columns = ['name', 'formation', 'geometry']

    gdf = extract_xyz_from_cross_sections(profile_gdf=profile_gdf,
                                          interfaces_gdf=interfaces_gdf)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert len(gdf) == 4
    assert all(gdf.geom_type == 'Point')
    assert {'X', 'Y', 'Z', 'name', 'geometry'}.issubset(gdf.columns)
    assert gdf.loc[0][['X', 'Y', 'Z']].values.tolist() == [3.2009219983223995, 3.84110639798688, -5]
    assert gdf.loc[1][['X', 'Y', 'Z']].values.tolist() == [3.8411063979868794, 4.609327677584255, -10]
    assert gdf.loc[2][['X', 'Y', 'Z']].values.tolist() == [3.2769276820761624, 7.293849377453314, -4.0]
    assert gdf.loc[3][['X', 'Y', 'Z']].values.tolist() == [5.734623443633283, 9.014236410543297, -11.0]


# Testing calculating_midpoint_linestrings
##########################################################
def test_calculate_midpoint_linestring():
    from gemgis.vector import calculate_midpoint_linestring

    linestring = LineString([(0, 0), (10, 10)])

    midpoint = calculate_midpoint_linestring(linestring=linestring)

    assert isinstance(midpoint, Point)
    assert midpoint.wkt == 'POINT (5 5)'

    linestring = LineString([(0, 0), (10, 0)])

    midpoint = calculate_midpoint_linestring(linestring=linestring)

    assert isinstance(midpoint, Point)
    assert midpoint.wkt == 'POINT (5 0)'


# Testing calculating_midpoints_linestrings
##########################################################
def test_calculate_midpoints_linestrings():
    from gemgis.vector import calculate_midpoints_linestrings

    linestrings = [LineString([(0, 0), (10, 10)]), LineString([(0, 0), (10, 0)])]

    midpoints = calculate_midpoints_linestrings(linestring_gdf=linestrings)

    assert isinstance(midpoints, list)
    assert all(isinstance(n, Point) for n in midpoints)
    assert len(midpoints) == 2
    assert midpoints[0].wkt == 'POINT (5 5)'
    assert midpoints[1].wkt == 'POINT (5 0)'


# Testing calculate_orientation_from_cross_section
##########################################################
def test_calculate_orientation_from_cross_section():
    from gemgis.vector import calculate_orientation_from_cross_section
    from gemgis.vector import calculate_strike_direction_straight_linestring
    from gemgis.vector import calculate_midpoint_linestring
    from gemgis.vector import calculate_coordinates_for_point_on_cross_section

    linestring = LineString([(0, 0), (10, 0)])

    orientation_linestring = LineString([(2, 0), (4, -2)])

    orientation = calculate_orientation_from_cross_section(profile_linestring=linestring,
                                                           orientation_linestring=orientation_linestring)

    assert isinstance(orientation, list)
    assert len(orientation) == 5
    assert isinstance(orientation[0], Point)
    assert isinstance(orientation[1], float)
    assert isinstance(orientation[2], float)
    assert isinstance(orientation[3], float)
    assert isinstance(orientation[4], int)

    azimuth_profile = calculate_strike_direction_straight_linestring(linestring)
    assert azimuth_profile == 90

    midpoint = calculate_midpoint_linestring(orientation_linestring)
    assert midpoint.wkt == 'POINT (3 -1)'

    assert orientation[0].wkt == 'POINT (3 0)'
    assert orientation[1] == -1
    assert orientation[2] == 45
    assert orientation[3] == 90
    assert orientation[4] == 1

    linestring = LineString([(0, 0), (-10, 0)])

    orientation_linestring = LineString([(2, 0), (4, -2)])

    orientation = calculate_orientation_from_cross_section(profile_linestring=linestring,
                                                           orientation_linestring=orientation_linestring)

    assert isinstance(orientation, list)
    assert len(orientation) == 5
    assert isinstance(orientation[0], Point)
    assert isinstance(orientation[1], float)
    assert isinstance(orientation[2], float)
    assert isinstance(orientation[3], float)
    assert isinstance(orientation[4], int)

    azimuth_profile = calculate_strike_direction_straight_linestring(linestring)
    assert azimuth_profile == 270

    midpoint = calculate_midpoint_linestring(orientation_linestring)
    assert midpoint.wkt == 'POINT (3 -1)'

    coordinates = calculate_coordinates_for_point_on_cross_section(linestring, midpoint)
    assert coordinates.wkt == 'POINT (-3 0)'

    assert orientation[0].wkt == 'POINT (-3 0)'
    assert orientation[1] == -1
    assert orientation[2] == 45
    assert orientation[3] == 270
    assert orientation[4] == 1

    linestring = LineString([(0, 0), (0, -10)])

    orientation_linestring = LineString([(2, 0), (4, -2)])

    orientation = calculate_orientation_from_cross_section(profile_linestring=linestring,
                                                           orientation_linestring=orientation_linestring)

    assert isinstance(orientation, list)
    assert len(orientation) == 5
    assert isinstance(orientation[0], Point)
    assert isinstance(orientation[1], float)
    assert isinstance(orientation[2], float)
    assert isinstance(orientation[3], float)
    assert isinstance(orientation[4], int)

    azimuth_profile = calculate_strike_direction_straight_linestring(linestring)
    assert azimuth_profile == 180

    midpoint = calculate_midpoint_linestring(orientation_linestring)
    assert midpoint.wkt == 'POINT (3 -1)'

    coordinates = calculate_coordinates_for_point_on_cross_section(linestring, midpoint)
    assert coordinates.wkt == 'POINT (0 -3)'

    assert orientation[0].wkt == 'POINT (0 -3)'
    assert orientation[1] == -1
    assert orientation[2] == 45
    assert orientation[3] == 180
    assert orientation[4] == 1

    linestring = LineString([(0, 0), (0, 10)])

    orientation_linestring = LineString([(2, 0), (4, -2)])

    orientation = calculate_orientation_from_cross_section(profile_linestring=linestring,
                                                           orientation_linestring=orientation_linestring)

    assert isinstance(orientation, list)
    assert len(orientation) == 5
    assert isinstance(orientation[0], Point)
    assert isinstance(orientation[1], float)
    assert isinstance(orientation[2], float)
    assert isinstance(orientation[3], float)
    assert isinstance(orientation[4], int)

    azimuth_profile = calculate_strike_direction_straight_linestring(linestring)
    assert azimuth_profile == 0

    midpoint = calculate_midpoint_linestring(orientation_linestring)
    assert midpoint.wkt == 'POINT (3 -1)'

    coordinates = calculate_coordinates_for_point_on_cross_section(linestring, midpoint)
    assert coordinates.wkt == 'POINT (0 3)'

    assert orientation[0].wkt == 'POINT (0 3)'
    assert orientation[1] == -1
    assert orientation[2] == 45
    assert orientation[3] == 0
    assert orientation[4] == 1

    linestring = LineString([(0, 0), (10, 5)])

    orientation_linestring = LineString([(2, 0), (4, -3)])

    orientation = calculate_orientation_from_cross_section(profile_linestring=linestring,
                                                           orientation_linestring=orientation_linestring)

    assert isinstance(orientation, list)
    assert len(orientation) == 5
    assert isinstance(orientation[0], Point)
    assert isinstance(orientation[1], float)
    assert isinstance(orientation[2], float)
    assert isinstance(orientation[3], float)
    assert isinstance(orientation[4], int)

    azimuth_profile = calculate_strike_direction_straight_linestring(linestring)
    assert azimuth_profile == 63.43494882292201

    midpoint = calculate_midpoint_linestring(orientation_linestring)
    assert midpoint.wkt == 'POINT (3 -1.5)'

    assert orientation[0].wkt == 'POINT (2.683281572999747 1.341640786499874)'
    assert orientation[1] == -1.5
    assert orientation[2] == 56.309932474020215
    assert orientation[3] == 63.43494882292201
    assert orientation[4] == 1


# Testing calculate_orientation_from_cross_section
##########################################################
def test_calculate_orientation_from_bent_cross_section():
    from gemgis.vector import calculate_orientation_from_bent_cross_section
    from gemgis.vector import calculate_midpoint_linestring
    from gemgis.vector import calculate_coordinates_for_linestring_on_cross_sections

    linestring = LineString([(0, 0), (5, 0), (10, 0)])

    orientation_linestring = LineString([(2, 0), (4, -2)])

    orientation = calculate_orientation_from_bent_cross_section(profile_linestring=linestring,
                                                                orientation_linestring=orientation_linestring)

    assert isinstance(orientation, list)
    assert len(orientation) == 5
    assert isinstance(orientation[0], Point)
    assert isinstance(orientation[1], float)
    assert isinstance(orientation[2], float)
    assert isinstance(orientation[3], float)
    assert isinstance(orientation[4], int)

    midpoint = calculate_midpoint_linestring(orientation_linestring)
    assert midpoint.wkt == 'POINT (3 -1)'

    points = calculate_coordinates_for_linestring_on_cross_sections(linestring, orientation_linestring)
    assert points[0].wkt == 'POINT (2 0)'
    assert points[1].wkt == 'POINT (4 0)'

    assert orientation[0].wkt == 'POINT (3 0)'
    assert orientation[1] == -1
    assert orientation[2] == 45
    assert orientation[3] == 90
    assert orientation[4] == 1

    linestring = LineString([(0, 0), (5, 0), (10, 5)])

    orientation_linestring = LineString([(6, 0), (8, -2)])

    orientation = calculate_orientation_from_bent_cross_section(profile_linestring=linestring,
                                                                orientation_linestring=orientation_linestring)

    assert isinstance(orientation, list)
    assert len(orientation) == 5
    assert isinstance(orientation[0], Point)
    assert isinstance(orientation[1], float)
    assert isinstance(orientation[2], float)
    assert isinstance(orientation[3], float)
    assert isinstance(orientation[4], int)

    midpoint = calculate_midpoint_linestring(linestring=orientation_linestring)
    assert midpoint.wkt == 'POINT (7 -1)'

    points = calculate_coordinates_for_linestring_on_cross_sections(linestring=linestring,
                                                                    interfaces=orientation_linestring)

    assert points[0].wkt == 'POINT (5.707106781186548 0.7071067811865475)'
    assert points[1].wkt == 'POINT (7.121320343559642 2.121320343559642)'

    assert orientation[0].wkt == 'POINT (6.414213562373095 1.414213562373095)'
    assert orientation[1] == -1
    assert orientation[2] == 45
    assert orientation[3] == 45
    assert orientation[4] == 1


# Testing calculate_orientations_from_cross_section
##########################################################
def test_calculate_orientations_from_cross_section():
    from gemgis.vector import calculate_orientations_from_cross_section

    linestring = LineString([(0, 0), (5, 0), (10, 5)])

    orientation_linestrings = [LineString([(2, 0), (4, -2)]), LineString([(6, 0), (9, -3)])]

    orientation = calculate_orientations_from_cross_section(profile_linestring=linestring,
                                                            orientation_linestrings=orientation_linestrings)

    assert isinstance(orientation, gpd.geodataframe.GeoDataFrame)
    assert all(orientation.geom_type == 'Point')
    assert {'X', 'Y', 'Z', 'dip', 'azimuth', 'polarity', 'geometry'}.issubset(orientation.columns)


# Testing calculate_orientations_from_cross_sections
##########################################################
def test_extract_orientations_from_cross_sections():
    from gemgis.vector import extract_orientations_from_cross_sections
    names = ['Profile1', 'Profile2']
    formation = ['Formation1', 'Formation2']

    linestrings = [LineString([(0, 0), (5, 0), (10, 5)]), LineString([(0, 0), (5, 0), (10, 5)])]

    profile_gdf = gpd.GeoDataFrame(data=names,
                                   geometry=linestrings)

    profile_gdf.columns = ['name', 'geometry']

    orientation_linestrings = [LineString([(2, 0), (4, -2)]), LineString([(6, 0), (9, -3)])]

    orientations_gdf = gpd.GeoDataFrame(data=pd.DataFrame([names, formation]).T,
                                        geometry=orientation_linestrings)

    orientations_gdf.columns = ['name', 'formation', 'geometry']

    orientation = extract_orientations_from_cross_sections(profile_gdf=profile_gdf,
                                                           orientations_gdf=orientations_gdf,
                                                           profile_name_column='name')

    assert isinstance(orientation, gpd.geodataframe.GeoDataFrame)
    assert all(orientation.geom_type == 'Point')
    assert {'X', 'Y', 'Z', 'dip', 'azimuth', 'polarity', 'geometry'}.issubset(orientation.columns)


# Testing intersection_polygon_polygon
##########################################################
def test_intersection_polygon_polygon():
    from gemgis.vector import intersection_polygon_polygon

    polygon1 = Polygon([(0, 0), (10, 0), (10, 10), (0, 10)])

    polygon2 = Polygon([(10, 0), (20, 0), (20, 10), (10, 10)])

    intersection = intersection_polygon_polygon(polygon1=polygon1,
                                                polygon2=polygon2)

    assert isinstance(intersection, LineString)
    assert intersection.wkt == 'LINESTRING (10 0, 10 10)'

    polygon1 = Polygon([(0, 0), (10, 0), (10, 10), (0, 10)])

    polygon2 = Polygon([(5, 0), (15, 0), (15, 10), (5, 10)])

    intersection = intersection_polygon_polygon(polygon1=polygon1,
                                                polygon2=polygon2)

    assert isinstance(intersection, Polygon)
    assert intersection.wkt == 'POLYGON ((10 0, 5 0, 5 10, 10 10, 10 0))'


# Testing intersection_polygon_polygons
##########################################################
def test_intersections_polygon_polygons():
    from gemgis.vector import intersections_polygon_polygons

    polygon1 = Polygon([(0, 0), (10, 0), (10, 10), (0, 10)])

    polygons2 = [Polygon([(10, 0), (20, 0), (20, 10), (10, 10)]),
                 Polygon([(5, 0), (15, 0), (15, 10), (5, 10)])]

    intersections = intersections_polygon_polygons(polygon1=polygon1,
                                                   polygons2=polygons2)

    assert isinstance(intersections, list)
    assert len(intersections) == 2
    assert isinstance(intersections[0], LineString)
    assert intersections[0].wkt == 'LINESTRING (10 0, 10 10)'

    assert isinstance(intersections[1], Polygon)
    assert intersections[1].wkt == 'POLYGON ((10 0, 5 0, 5 10, 10 10, 10 0))'


# Testing intersection_polygon_polygons
##########################################################
def test_intersections_polygons_polygons():
    from gemgis.vector import intersections_polygons_polygons

    polygons = [Polygon([(0, 0), (10, 0), (10, 10), (0, 10)]),
                Polygon([(10, 0), (20, 0), (20, 10), (10, 10)]),
                Polygon([(5, 0), (15, 0), (15, 10), (5, 10)])]

    intersections = intersections_polygons_polygons(polygons1=polygons,
                                                    polygons2=polygons)

    assert isinstance(intersections, list)
    assert len(intersections) == 9

    assert isinstance(intersections[0], Polygon)
    assert isinstance(intersections[1], LineString)
    assert isinstance(intersections[2], Polygon)
    assert isinstance(intersections[3], LineString)
    assert isinstance(intersections[4], Polygon)
    assert isinstance(intersections[5], Polygon)
    assert isinstance(intersections[6], Polygon)
    assert isinstance(intersections[7], Polygon)
    assert isinstance(intersections[8], Polygon)

    assert intersections[0].wkt == 'POLYGON ((10 0, 0 0, 0 10, 10 10, 10 0))'
    assert intersections[1].wkt == 'LINESTRING (10 0, 10 10)'
    assert intersections[2].wkt == 'POLYGON ((10 0, 5 0, 5 10, 10 10, 10 0))'
    assert intersections[3].wkt == 'LINESTRING (10 10, 10 0)'
    assert intersections[4].wkt == 'POLYGON ((20 0, 10 0, 10 10, 20 10, 20 0))'
    assert intersections[5].wkt == 'POLYGON ((15 0, 10 0, 10 10, 15 10, 15 0))'
    assert intersections[6].wkt == 'POLYGON ((10 0, 5 0, 5 10, 10 10, 10 0))'
    assert intersections[7].wkt == 'POLYGON ((15 0, 10 0, 10 10, 15 10, 15 0))'
    assert intersections[8].wkt == 'POLYGON ((15 0, 5 0, 5 10, 15 10, 15 0))'


# Testing explode_linestrings
##########################################################
def test_explode_linestring_points():
    from gemgis.vector import explode_linestring
    from shapely.geometry import LineString

    linestring = LineString([(0, 0), (5, 5), (10, 0), (15, 5)])

    point_list = explode_linestring(linestring=linestring)

    assert isinstance(point_list, list)
    assert all(isinstance(n, Point) for n in point_list)
    assert len(point_list) == 4
    assert point_list[0].wkt == 'POINT (0 0)'
    assert point_list[1].wkt == 'POINT (5 5)'
    assert point_list[2].wkt == 'POINT (10 0)'
    assert point_list[3].wkt == 'POINT (15 5)'


# Testing explode_polygons
##########################################################
def test_explode_polygon():
    from gemgis.vector import explode_polygon
    from shapely.geometry import Polygon

    polygon = Polygon([(0, 0), (10, 0), (10, 10), (0, 10)])

    point_list = explode_polygon(polygon=polygon)

    assert isinstance(point_list, list)
    assert all(isinstance(n, Point) for n in point_list)
    assert len(point_list) == 5
    assert point_list[0].wkt == 'POINT (0 0)'
    assert point_list[1].wkt == 'POINT (10 0)'
    assert point_list[2].wkt == 'POINT (10 10)'
    assert point_list[3].wkt == 'POINT (0 10)'
    assert point_list[4].wkt == 'POINT (0 0)'


# Testing explode_multilinestring
##########################################################
def test_explode_multilinestring():
    from gemgis.vector import explode_multilinestring

    multilinestring = MultiLineString([((0, 0), (5, 5)), ((10, 0), (15, 5))])

    multilinestring_list = explode_multilinestring(multilinestring=multilinestring)

    assert isinstance(multilinestring_list, list)
    assert all(isinstance(n, LineString) for n in multilinestring_list)
    assert len(multilinestring_list) == 2

    assert multilinestring_list[0].wkt == 'LINESTRING (0 0, 5 5)'
    assert multilinestring_list[1].wkt == 'LINESTRING (10 0, 15 5)'


# Testing sort by stratigraphy
##########################################################
@pytest.mark.parametrize("gdf_geolmap1", [gdf_geolmap1])
def test_sort_by_stratigraphy(gdf_geolmap1):
    from gemgis.vector import sort_by_stratigraphy

    stratigraphy = ['Sand2', 'Sand1', 'Ton']

    sorted_gdf = sort_by_stratigraphy(gdf=gdf_geolmap1,
                                      stratigraphy=stratigraphy)

    assert isinstance(sorted_gdf, gpd.geodataframe.GeoDataFrame)
    assert len(sorted_gdf) == 4
    assert sorted_gdf['formation'].tolist() == ['Sand2', 'Sand2', 'Sand1', 'Ton']


# Testing extract xy from polygon intersections
##########################################################
@pytest.mark.parametrize("gdf_geolmap1", [gdf_geolmap1])
def test_extract_xy_from_polygon_intersections(gdf_geolmap1):
    from gemgis.vector import extract_xy_from_polygon_intersections

    intersections = extract_xy_from_polygon_intersections(gdf=gdf_geolmap1,
                                                          extract_coordinates=False)

    assert isinstance(intersections, gpd.geodataframe.GeoDataFrame)
    assert len(intersections) == 2

    assert all(intersections.geom_type == 'MultiLineString')
    assert intersections['formation'].tolist() == ['Sand1', 'Ton']


# Testing create_unified_buffer
###########################################################
@pytest.mark.parametrize("faults",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/GK50_Tektonik.shp')
                         ])
def test_create_unified_buffer(faults):
    from gemgis.vector import create_unified_buffer

    polygon = create_unified_buffer(geom_object=faults,
                                    distance=250)

    assert isinstance(polygon, MultiPolygon)


# Testing explode_geometry_collections
###########################################################
def test_explode_geometry_collections():
    from gemgis.vector import explode_geometry_collections
    from shapely.geometry import LineString

    line1 = LineString([(0, 0), (1, 1), (1, 2), (2, 2)])
    line2 = LineString([(0, 0), (1, 1), (2, 1), (2, 2)])
    collection = line1.intersection(line2)

    gdf = gpd.GeoDataFrame(geometry=[collection, line1, line2])

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert any(gdf.geom_type == 'GeometryCollection')

    gdf_exploded = explode_geometry_collections(gdf=gdf,
                                                reset_index=True,
                                                drop_level0=True,
                                                drop_level1=True,
                                                remove_points=True)

    assert isinstance(gdf_exploded, gpd.geodataframe.GeoDataFrame)
    assert not any(gdf_exploded.geom_type == 'GeometryCollection')
    assert all(gdf_exploded.geom_type == 'LineString')

    gdf_exploded = explode_geometry_collections(gdf=gdf,
                                                reset_index=True,
                                                drop_level0=True,
                                                drop_level1=True,
                                                remove_points=False)

    assert isinstance(gdf_exploded, gpd.geodataframe.GeoDataFrame)
    assert not any(gdf_exploded.geom_type == 'GeometryCollection')
    assert gdf_exploded.geom_type.isin(('Point', 'LineString')).all()


# Testing extract_xy with Geometry collections
###########################################################
def test_extract_xy_geometry_collection():
    from shapely.geometry import LineString
    from gemgis.vector import extract_xy

    line1 = LineString([(0, 0), (1, 1), (1, 2), (2, 2)])
    line2 = LineString([(0, 0), (1, 1), (2, 1), (2, 2)])
    collection = line1.intersection(line2)

    gdf = gpd.GeoDataFrame(geometry=[collection, line1, line2])

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert any(gdf.geom_type == 'GeometryCollection')

    gdf_xy = extract_xy(gdf=gdf)

    assert isinstance(gdf_xy, gpd.geodataframe.GeoDataFrame)
    assert not any(gdf_xy.geom_type == 'GeometryCollection')
    assert all(gdf_xy.geom_type == 'Point')


# Testing create_bbox
###########################################################
def test_create_bbox():
    from gemgis.vector import create_bbox

    bbox = create_bbox(extent=[0, 100, 0, 100])

    assert isinstance(bbox, Polygon)


def test_create_bbox_error():
    from gemgis.vector import create_bbox

    with pytest.raises(TypeError):
        create_bbox(1, 10, 1, 10)

    with pytest.raises(TypeError):
        create_bbox(extent=[1, 10, 1, '10'])


# Testing create_linestring
###########################################################
@pytest.mark.parametrize("gdf_points_strike", [gdf_points_strike])
def test_create_linestring(gdf_points_strike):
    from gemgis.vector import create_linestring_from_points

    linestring = create_linestring_from_points(gdf=gdf_points_strike,
                                               formation='Ton',
                                               altitude=400)
    assert len(linestring.coords) == 2
    assert isinstance(linestring, LineString)


# Testing create_linestring_gdf
###########################################################
@pytest.mark.parametrize("gdf_points_strike", [gdf_points_strike])
def test_create_linestring_gdf(gdf_points_strike):
    from gemgis.vector import create_linestring_gdf

    linestring_gdf = create_linestring_gdf(gdf=gdf_points_strike)

    assert isinstance(linestring_gdf, gpd.geodataframe.GeoDataFrame)
    assert all(linestring_gdf.geom_type == 'LineString')
    assert linestring_gdf.crs == 'EPSG:4326'
    assert len(linestring_gdf) == 5


# Testing polygons_to_linestrings
###########################################################
@pytest.mark.parametrize("gdf",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/GeologicalMapAachen.shp')
                         ])
def test_polygons_to_linestrings(gdf):
    from gemgis.vector import explode_polygons

    gdf_linestrings = explode_polygons(gdf)

    no_geom_types = np.unique(np.array([gdf_linestrings.geom_type[i] for i in range(len(gdf_linestrings))]))

    assert len(no_geom_types) == 2
    assert no_geom_types[0] == 'LineString'
    assert no_geom_types[1] == 'MultiLineString'
    assert isinstance(gdf_linestrings, gpd.geodataframe.GeoDataFrame)
    assert gdf_linestrings.crs == 'EPSG:4647'
    assert len(gdf_linestrings) == 848


# Testing calculate_azimuth
###########################################################
@pytest.mark.parametrize("gdf",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/bottom_cret_orient.shp')
                         ])
def test_calculate_azimuth(gdf):
    from gemgis.vector import calculate_azimuth

    azimuth = calculate_azimuth(gdf=gdf)

    assert isinstance(azimuth, list)


# Testing extract_orientations_from_map
###########################################################
@pytest.mark.parametrize("gdf",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/bottom_cret_orient.shp')
                         ])
def test_extract_orientations_from_map(gdf):
    from gemgis.vector import extract_orientations_from_map

    orientations = extract_orientations_from_map(gdf=gdf)

    assert isinstance(orientations, gpd.geodataframe.GeoDataFrame)

    assert {'dip', 'azimuth', 'polarity', 'X', 'Y'}.issubset(orientations.columns)


# Testing calculate_orientations_from_strike_lines
###########################################################
@pytest.mark.parametrize("gdf_points_strike", [gdf_points_strike])
def test_calculate_orientations_from_strike_lines(gdf_points_strike):
    from gemgis.vector import calculate_orientations_from_strike_lines, create_linestring_gdf

    linestring_gdf = create_linestring_gdf(gdf=gdf_points_strike)

    orientations = calculate_orientations_from_strike_lines(gdf=linestring_gdf)

    assert isinstance(orientations, gpd.geodataframe.GeoDataFrame)
    assert len(orientations) == 4


# Testing calculate_distance_linestrings
###########################################################
def test_calculate_distance_linestrings():
    from gemgis.vector import calculate_distance_linestrings

    ls1 = LineString([(0, 0), (10, 0)])
    ls2 = LineString([(0, 5), (10, 5)])

    distance = calculate_distance_linestrings(ls1=ls1,
                                              ls2=ls2)

    assert isinstance(distance, float)
    assert distance == 5


# Testing explode_geometry_collection
###########################################################
def test_explode_geometry_collection():
    from gemgis.vector import explode_geometry_collection

    a = LineString([(0, 0), (1, 1), (1, 2), (2, 2)])
    b = LineString([(0, 0), (1, 1), (2, 1), (2, 2)])
    collection = a.intersection(b)

    assert isinstance(collection, GeometryCollection)

    collection_exploded = explode_geometry_collection(collection=collection)

    assert isinstance(collection_exploded, list)
    assert len(collection_exploded) == 2
    assert isinstance(collection_exploded[0], Point)
    assert isinstance(collection_exploded[1], LineString)


# Testing extract_xy_linestring
###########################################################
def test_extract_xy_linestring():
    from gemgis.vector import extract_xy_linestring

    a = LineString([(0, 0), (1, 1), (1, 2), (2, 2)])
    b = LineString([(0, 0), (1, 1), (2, 1), (2, 2)])

    gdf = gpd.GeoDataFrame(geometry=[a, b])

    gdf_exploded = extract_xy_linestring(gdf=gdf)

    assert isinstance(gdf_exploded, gpd.geodataframe.GeoDataFrame)
    assert {'X', 'Y'}.issubset(gdf_exploded.columns)
    assert isinstance(gdf_exploded.loc[0]['X'], list)
    assert isinstance(gdf_exploded.loc[1]['Y'], list)
    assert isinstance(gdf_exploded.loc[0]['Y'], list)
    assert isinstance(gdf_exploded.loc[1]['X'], list)


# Testing extract_xyz_points
###########################################################
def test_extract_xyz_points():
    from gemgis.vector import extract_xyz_points

    point = Point(1, 2, 3)

    gdf = gpd.GeoDataFrame(geometry=[point, point])

    gdf = extract_xyz_points(gdf=gdf)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert {'X', 'Y', 'Z'}.issubset(gdf.columns)


# Testing extract_xyz_points
###########################################################
def test_extract_xyz_linestrings():
    from gemgis.vector import extract_xyz_linestrings

    linestring1 = LineString([[1, 2, 3], [4, 5, 6]])
    linestring2 = LineString([[1, 2, 3], [4, 5, 7]])

    gdf = gpd.GeoDataFrame(geometry=[linestring1, linestring2])

    gdf = extract_xyz_linestrings(gdf=gdf)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert {'X', 'Y', 'Z'}.issubset(gdf.columns)


# Testing load_gpx
###########################################################
def test_load_gpx():
    from gemgis.vector import load_gpx

    gpx = load_gpx(path='../docs/getting_started/tutorial/data/test_vector/Run.gpx',
                   layer='tracks')

    assert isinstance(gpx, Collection)
    assert gpx.crs == {'init': 'epsg:4326'}
    assert gpx.bounds == (8.460906, 52.694879, 8.501507, 52.732331)


# Testing load_gpx
###########################################################
def test_load_gpx_as_dict():
    from gemgis.vector import load_gpx_as_dict

    gpx = load_gpx_as_dict(path='../docs/getting_started/tutorial/data/test_vector/Run.gpx',
                           layer='tracks')

    assert isinstance(gpx, dict)


# Testing load_gpx
###########################################################
def test_load_gpx_as_geometry():
    from gemgis.vector import load_gpx_as_geometry
    import shapely

    gpx = load_gpx_as_geometry(path='../docs/getting_started/tutorial/data/test_vector/Run.gpx',
                               layer='tracks')

    assert isinstance(gpx, shapely.geometry.base.BaseGeometry)
    assert isinstance(gpx, MultiLineString)


# Testing create_linestring_from_xyz_points
###########################################################
def test_create_linestring_from_xyz_points():
    from gemgis.vector import create_linestring_from_xyz_points

    points = np.array([[3.23, 5.69, 2.03], [3.24, 5.68, 2.02], [3.25, 5.67, 1.97], [3.26, 5.66, 1.95]])
    linestring = create_linestring_from_xyz_points(points=points)

    assert isinstance(linestring, LineString)
    assert linestring.has_z


# Testing create_linestring_from_xyz_points
###########################################################
@pytest.mark.parametrize("roads",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/Major_Roads.shp')
                         ])
@pytest.mark.parametrize("raster",
                         [
                             rasterio.open('../docs/getting_started/tutorial/data/test_vector/DEM50.tif')
                         ])
def test_create_linestrings_from_xyz_points(roads, raster):
    from gemgis.vector import create_linestrings_from_xyz_points, extract_xyz

    autobahns = roads[roads['STRKL'] == 'A']

    autobahns_xyz = extract_xyz(gdf=autobahns,
                                dem=raster,
                                reset_index=False)

    assert {'X', 'Y', 'Z'}.issubset(autobahns_xyz.columns)

    autobahns_linestring_z = create_linestrings_from_xyz_points(gdf=autobahns_xyz,
                                                                groupby='ABS')

    assert isinstance(autobahns_linestring_z, gpd.geodataframe.GeoDataFrame)


# Testing create_linestring_contours
###########################################################
@pytest.mark.parametrize("contours",
                         [
                             pv.read('../docs/getting_started/tutorial/data/test_vector/contours.vtk')
                         ])
def test_create_linestrings_from_contours(contours):
    from gemgis.vector import create_linestrings_from_contours

    gdf = create_linestrings_from_contours(contours=contours)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert all(gdf.geom_type == 'LineString')
    assert all(gdf.has_z)


# Testing create_polygons_from_faces
###########################################################
@pytest.mark.parametrize("mesh",
                         [
                             pv.read('../docs/getting_started/tutorial/data/test_vector/mesh.vtk')
                         ])
def test_create_polygons_from_faces(mesh):
    from gemgis.vector import create_polygons_from_faces

    polygons = create_polygons_from_faces(mesh=mesh, crs='EPSG:25832')

    assert isinstance(polygons, gpd.geodataframe.GeoDataFrame)


# Testing unify_polygons
###########################################################
@pytest.mark.parametrize("mesh",
                         [
                             pv.read('../docs/getting_started/tutorial/data/test_vector/mesh.vtk')
                         ])
def test_unify_polygons(mesh):
    from gemgis.vector import create_polygons_from_faces, unify_polygons

    polygons = create_polygons_from_faces(mesh=mesh, crs='EPSG:25832')

    assert isinstance(polygons, gpd.geodataframe.GeoDataFrame)

    polygons = polygons[polygons.is_valid]

    polygons_merged = unify_polygons(polygons=polygons)

    assert isinstance(polygons_merged, gpd.geodataframe.GeoDataFrame)


# Testing unify_linestrings
###########################################################
@pytest.mark.parametrize("contours",
                         [
                             pv.read('../docs/getting_started/tutorial/data/test_vector/contours.vtk')
                         ])
def test_unify_linestrings(contours):
    from gemgis.vector import create_linestrings_from_contours, unify_linestrings

    gdf = create_linestrings_from_contours(contours=contours)

    assert isinstance(gdf, gpd.geodataframe.GeoDataFrame)
    assert all(gdf.geom_type == 'LineString')
    assert all(gdf.has_z)

    gdf_merged = unify_linestrings(linestrings=gdf,
                                   crs='EPSG:4647')

    assert isinstance(gdf_merged, gpd.geodataframe.GeoDataFrame)
    assert all(gdf_merged.geom_type == 'LineString')
    assert all(gdf_merged.has_z)


# Testing unify_linestrings
###########################################################
@pytest.mark.parametrize("gdf",
                         [
                             gpd.read_file('../docs/getting_started/tutorial/data/test_vector/interfaces12.shp')
                         ])
def test_calculate_orientation_for_three_point_problem(gdf):
    from gemgis.vector import calculate_orientation_for_three_point_problem

    orientation = calculate_orientation_for_three_point_problem(gdf=gdf)

    assert isinstance(orientation, gpd.geodataframe.GeoDataFrame)

    assert {'X', 'Y', 'Z', 'dip', 'azimuth', 'polarity', 'formation'}.issubset(orientation.columns)
