# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['flatten_dict', 'flatten_dict.tests']

package_data = \
{'': ['*']}

install_requires = \
['six>=1.12,<2.0']

extras_require = \
{':python_version < "3.4"': ['pathlib2>=2.3,<3.0'],
 ':python_version < "3.8"': ['importlib-metadata']}

setup_kwargs = {
    'name': 'flatten-dict',
    'version': '0.4.2',
    'description': 'A flexible utility for flattening and unflattening dict-like objects in Python.',
    'long_description': 'flatten-dict\n============\n.. image:: https://github.com/ianlini/flatten-dict/actions/workflows/main.yml/badge.svg\n   :target: https://github.com/ianlini/flatten-dict/actions\n.. image:: https://img.shields.io/pypi/v/flatten-dict.svg\n   :target: https://pypi.org/project/flatten-dict/\n.. image:: https://img.shields.io/pypi/l/flatten-dict.svg\n   :target: https://github.com/ianlini/flatten-dict/blob/master/LICENSE\n.. image:: https://img.shields.io/github/stars/ianlini/flatten-dict.svg?style=social\n   :target: https://github.com/ianlini/flatten-dict\n\nA flexible utility for flattening and unflattening dict-like objects in Python.\n\n\nIntroduction\n------------\nThis package provides a function ``flatten()`` for flattening dict-like objects in Python 2.7 and 3.5~3.8.\nIt also provides some key joining methods (reducer), and you can choose the reducer you want or even implement your own reducer.\nYou can also invert the resulting flat dict using ``unflatten()``.\n\nInstallation\n------------\n\n.. code-block:: bash\n\n   pip install flatten-dict\n\nDocumentation\n-------------\n\nFlatten\n```````\n\n.. code-block:: python\n\n   def flatten(d, reducer=\'tuple\', inverse=False, enumerate_types=(), keep_empty_types=()):\n       """Flatten `Mapping` object.\n\n       Parameters\n       ----------\n       d : dict-like object\n           The dict that will be flattened.\n       reducer : {\'tuple\', \'path\', \'underscore\', \'dot\', Callable}\n           The key joining method. If a `Callable` is given, the `Callable` will be\n           used to reduce.\n           \'tuple\': The resulting key will be tuple of the original keys.\n           \'path\': Use `os.path.join` to join keys.\n           \'underscore\': Use underscores to join keys.\n           \'dot\': Use dots to join keys.\n       inverse : bool\n           Whether you want invert the resulting key and value.\n       max_flatten_depth : int\n           Maximum depth to merge.\n       enumerate_types : Sequence[type]\n           Flatten these types using `enumerate`.\n           For example, if we set `enumerate_types` to ``(list,)``,\n           `list` indices become keys: ``{\'a\': [\'b\', \'c\']}`` -> ``{(\'a\', 0): \'b\', (\'a\', 1): \'c\'}``.\n       keep_empty_types : Sequence[type]\n           By default, ``flatten({1: 2, 3: {}})`` will give you ``{(1,): 2}``, that is, the key ``3``\n           will disappear.\n           This is also applied for the types in `enumerate_types`, that is,\n           ``flatten({1: 2, 3: []}, enumerate_types=(list,))`` will give you ``{(1,): 2}``.\n           If you want to keep those empty values, you can specify the types in `keep_empty_types`:\n\n           >>> flatten({1: 2, 3: {}}, keep_empty_types=(dict,))\n           {(1,): 2, (3,): {}}\n\n       Returns\n       -------\n       flat_dict : dict\n       """\n\nExamples\n::::::::\n\n>>> from flatten_dict import flatten\n>>> from pprint import pprint\n>>> normal_dict = {\n...     \'a\': \'0\',\n...     \'b\': {\n...         \'a\': \'1.0\',\n...         \'b\': \'1.1\',\n...     },\n...     \'c\': {\n...         \'a\': \'2.0\',\n...         \'b\': {\n...             \'a\': \'2.1.0\',\n...             \'b\': \'2.1.1\',\n...         },\n...     },\n... }\n>>> pprint(flatten(normal_dict))\n{(\'a\',): \'0\',\n (\'b\', \'a\'): \'1.0\',\n (\'b\', \'b\'): \'1.1\',\n (\'c\', \'a\'): \'2.0\',\n (\'c\', \'b\', \'a\'): \'2.1.0\',\n (\'c\', \'b\', \'b\'): \'2.1.1\'}\n>>> pprint(flatten(normal_dict, reducer=\'path\'))\n{\'a\': \'0\',\n \'b/a\': \'1.0\',\n \'b/b\': \'1.1\',\n \'c/a\': \'2.0\',\n \'c/b/a\': \'2.1.0\',\n \'c/b/b\': \'2.1.1\'}\n>>> pprint(flatten(normal_dict, reducer=\'path\', inverse=True))\n{\'0\': \'a\',\n \'1.0\': \'b/a\',\n \'1.1\': \'b/b\',\n \'2.0\': \'c/a\',\n \'2.1.0\': \'c/b/a\',\n \'2.1.1\': \'c/b/b\'}\n>>> pprint(flatten(normal_dict, reducer=\'path\', max_flatten_depth=2))\n{\'a\': \'0\',\n \'b/a\': \'1.0\',\n \'b/b\': \'1.1\',\n \'c/a\': \'2.0\',\n \'c/b\': {\'a\': \'2.1.0\', \'b\': \'2.1.1\'}}\n\nThe `reducer` parameter supports ``\'tuple\'``, ``\'path\'``, ``\'underscore\'``, ``\'dot\'`` and `Callable`. We can customize the reducer using a function:\n\n>>> def underscore_reducer(k1, k2):\n...     if k1 is None:\n...         return k2\n...     else:\n...         return k1 + "_" + k2\n...\n>>> pprint(flatten(normal_dict, reducer=underscore_reducer))\n{\'a\': \'0\',\n \'b_a\': \'1.0\',\n \'b_b\': \'1.1\',\n \'c_a\': \'2.0\',\n \'c_b_a\': \'2.1.0\',\n \'c_b_b\': \'2.1.1\'}\n\nThere is also a factory function `make_reducer()` to help you create customized reducer. The function currently only supports customized delimiter:\n\n>>> from flatten_dict.reducers import make_reducer\n>>> pprint(flatten(normal_dict, reducer=make_reducer(delimiter=\'_\')))\n{\'a\': \'0\',\n \'b_a\': \'1.0\',\n \'b_b\': \'1.1\',\n \'c_a\': \'2.0\',\n \'c_b_a\': \'2.1.0\',\n \'c_b_b\': \'2.1.1\'}\n\nIf we have some iterable (e.g., `list`) in the `dict`, we will normally get this:\n\n>>> flatten({\'a\': [1, 2, 3], \'b\': \'c\'})\n{(\'a\',): [1, 2, 3], (\'b\',): \'c\'}\n\nIf we want to use its indices as keys, then we can use the parameter `enumerate_types`:\n\n>>> flatten({\'a\': [1, 2, 3], \'b\': \'c\'}, enumerate_types=(list,))\n{(\'a\', 0): 1, (\'a\', 1): 2, (\'a\', 2): 3, (\'b\',): \'c\'}\n\nWe can even flatten a `list` directly:\n\n>>> flatten([1, 2, 3], enumerate_types=(list,))\n{(0,): 1, (1,): 2, (2,): 3}\n\nIf there is an empty dict in the values, by default, it will disappear after flattened:\n\n>>> flatten({1: 2, 3: {}})\n{(1,): 2}\n\nWe can keep the empty dict in the result using ``keep_empty_types=(dict,)``:\n\n>>> flatten({1: 2, 3: {}}, keep_empty_types=(dict,))\n{(1,): 2, (3,): {}}\n\nUnflatten\n`````````\n\n.. code-block:: python\n\n   def unflatten(d, splitter=\'tuple\', inverse=False):\n       """Unflatten dict-like object.\n\n       Parameters\n       ----------\n       d : dict-like object\n           The dict that will be unflattened.\n       splitter : {\'tuple\', \'path\', \'underscore\', \'dot\', Callable}\n           The key splitting method. If a Callable is given, the Callable will be\n           used to split `d`.\n           \'tuple\': Use each element in the tuple key as the key of the unflattened dict.\n           \'path\': Use `pathlib.Path.parts` to split keys.\n           \'underscore\': Use underscores to split keys.\n           \'dot\': Use underscores to split keys.\n       inverse : bool\n           Whether you want to invert the key and value before flattening.\n\n       Returns\n       -------\n       unflattened_dict : dict\n       """\n\nExamples\n::::::::\n\n>>> from pprint import pprint\n>>> from flatten_dict import unflatten\n>>> flat_dict = {\n...     (\'a\',): \'0\',\n...     (\'b\', \'a\'): \'1.0\',\n...     (\'b\', \'b\'): \'1.1\',\n...     (\'c\', \'a\'): \'2.0\',\n...     (\'c\', \'b\', \'a\'): \'2.1.0\',\n...     (\'c\', \'b\', \'b\'): \'2.1.1\',\n... }\n>>> pprint(unflatten(flat_dict))\n{\'a\': \'0\',\n \'b\': {\'a\': \'1.0\', \'b\': \'1.1\'},\n \'c\': {\'a\': \'2.0\', \'b\': {\'a\': \'2.1.0\', \'b\': \'2.1.1\'}}}\n>>> flat_dict = {\n...     \'a\': \'0\',\n...     \'b/a\': \'1.0\',\n...     \'b/b\': \'1.1\',\n...     \'c/a\': \'2.0\',\n...     \'c/b/a\': \'2.1.0\',\n...     \'c/b/b\': \'2.1.1\',\n... }\n>>> pprint(unflatten(flat_dict, splitter=\'path\'))\n{\'a\': \'0\',\n \'b\': {\'a\': \'1.0\', \'b\': \'1.1\'},\n \'c\': {\'a\': \'2.0\', \'b\': {\'a\': \'2.1.0\', \'b\': \'2.1.1\'}}}\n>>> flat_dict = {\n...     \'0\': \'a\',\n...     \'1.0\': \'b/a\',\n...     \'1.1\': \'b/b\',\n...     \'2.0\': \'c/a\',\n...     \'2.1.0\': \'c/b/a\',\n...     \'2.1.1\': \'c/b/b\',\n... }\n>>> pprint(unflatten(flat_dict, splitter=\'path\', inverse=True))\n{\'a\': \'0\',\n \'b\': {\'a\': \'1.0\', \'b\': \'1.1\'},\n \'c\': {\'a\': \'2.0\', \'b\': {\'a\': \'2.1.0\', \'b\': \'2.1.1\'}}}\n\nThe `splitter` parameter supports ``\'tuple\'``, ``\'path\'``, ``\'underscore\'``, ``\'dot\'`` and `Callable`. We can customize the reducer using a function:\n\n>>> def underscore_splitter(flat_key):\n...     return flat_key.split("_")\n...\n>>> flat_dict = {\n...     \'a\': \'0\',\n...     \'b_a\': \'1.0\',\n...     \'b_b\': \'1.1\',\n...     \'c_a\': \'2.0\',\n...     \'c_b_a\': \'2.1.0\',\n...     \'c_b_b\': \'2.1.1\',\n... }\n>>> pprint(unflatten(flat_dict, splitter=underscore_splitter))\n{\'a\': \'0\',\n \'b\': {\'a\': \'1.0\', \'b\': \'1.1\'},\n \'c\': {\'a\': \'2.0\', \'b\': {\'a\': \'2.1.0\', \'b\': \'2.1.1\'}}}\n\nThere is also a factory function `make_splitter()` to help you create customized splitter. The function currently only supports customized delimiter:\n\n>>> from flatten_dict.splitters import make_splitter\n>>> pprint(unflatten(flat_dict, splitter=make_splitter(delimiter=\'_\')))\n{\'a\': \'0\',\n \'b\': {\'a\': \'1.0\', \'b\': \'1.1\'},\n \'c\': {\'a\': \'2.0\', \'b\': {\'a\': \'2.1.0\', \'b\': \'2.1.1\'}}}\n',
    'author': 'Ian Lin',
    'author_email': 'you@example.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/ianlini/flatten-dict',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*, !=3.4.*',
}


setup(**setup_kwargs)
