"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.topologicalSort = void 0;
/**
 * Return a topological sort of all elements of xs, according to the given dependency functions
 *
 * Dependencies outside the referenced set are ignored.
 *
 * Not a stable sort, but in order to keep the order as stable as possible, we'll sort by key
 * among elements of equal precedence.
 */
function topologicalSort(xs, keyFn, depFn) {
    const remaining = new Map();
    for (const element of xs) {
        const key = keyFn(element);
        remaining.set(key, { key, element, dependencies: depFn(element) });
    }
    const ret = new Array();
    while (remaining.size > 0) {
        // All elements with no more deps in the set can be ordered
        const selectable = Array.from(remaining.values()).filter((e) => e.dependencies.every((d) => !remaining.has(d)));
        selectable.sort((a, b) => (a.key < b.key ? -1 : b.key < a.key ? 1 : 0));
        for (const selected of selectable) {
            ret.push(selected.element);
            remaining.delete(selected.key);
        }
        // If we didn't make any progress, we got stuck
        if (selectable.length === 0) {
            throw new Error(`Could not determine ordering between: ${Array.from(remaining.keys()).join(", ")}`);
        }
    }
    return ret;
}
exports.topologicalSort = topologicalSort;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidG9wb3NvcnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0b3Bvc29ydC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFHQTs7Ozs7OztHQU9HO0FBQ0gsU0FBZ0IsZUFBZSxDQUM3QixFQUFlLEVBQ2YsS0FBaUIsRUFDakIsS0FBaUI7SUFFakIsTUFBTSxTQUFTLEdBQUcsSUFBSSxHQUFHLEVBQTBCLENBQUM7SUFDcEQsS0FBSyxNQUFNLE9BQU8sSUFBSSxFQUFFLEVBQUU7UUFDeEIsTUFBTSxHQUFHLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzNCLFNBQVMsQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxZQUFZLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQztLQUNwRTtJQUVELE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxFQUFLLENBQUM7SUFDM0IsT0FBTyxTQUFTLENBQUMsSUFBSSxHQUFHLENBQUMsRUFBRTtRQUN6QiwyREFBMkQ7UUFDM0QsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUM3RCxDQUFDLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQy9DLENBQUM7UUFFRixVQUFVLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUV4RSxLQUFLLE1BQU0sUUFBUSxJQUFJLFVBQVUsRUFBRTtZQUNqQyxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUMzQixTQUFTLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUNoQztRQUVELCtDQUErQztRQUMvQyxJQUFJLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzNCLE1BQU0sSUFBSSxLQUFLLENBQ2IseUNBQXlDLEtBQUssQ0FBQyxJQUFJLENBQ2pELFNBQVMsQ0FBQyxJQUFJLEVBQUUsQ0FDakIsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FDZixDQUFDO1NBQ0g7S0FDRjtJQUVELE9BQU8sR0FBRyxDQUFDO0FBQ2IsQ0FBQztBQXBDRCwwQ0FvQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJleHBvcnQgdHlwZSBLZXlGdW5jPFQ+ID0gKHg6IFQpID0+IHN0cmluZztcbmV4cG9ydCB0eXBlIERlcEZ1bmM8VD4gPSAoeDogVCkgPT4gc3RyaW5nW107XG5cbi8qKlxuICogUmV0dXJuIGEgdG9wb2xvZ2ljYWwgc29ydCBvZiBhbGwgZWxlbWVudHMgb2YgeHMsIGFjY29yZGluZyB0byB0aGUgZ2l2ZW4gZGVwZW5kZW5jeSBmdW5jdGlvbnNcbiAqXG4gKiBEZXBlbmRlbmNpZXMgb3V0c2lkZSB0aGUgcmVmZXJlbmNlZCBzZXQgYXJlIGlnbm9yZWQuXG4gKlxuICogTm90IGEgc3RhYmxlIHNvcnQsIGJ1dCBpbiBvcmRlciB0byBrZWVwIHRoZSBvcmRlciBhcyBzdGFibGUgYXMgcG9zc2libGUsIHdlJ2xsIHNvcnQgYnkga2V5XG4gKiBhbW9uZyBlbGVtZW50cyBvZiBlcXVhbCBwcmVjZWRlbmNlLlxuICovXG5leHBvcnQgZnVuY3Rpb24gdG9wb2xvZ2ljYWxTb3J0PFQ+KFxuICB4czogSXRlcmFibGU8VD4sXG4gIGtleUZuOiBLZXlGdW5jPFQ+LFxuICBkZXBGbjogRGVwRnVuYzxUPlxuKTogVFtdIHtcbiAgY29uc3QgcmVtYWluaW5nID0gbmV3IE1hcDxzdHJpbmcsIFRvcG9FbGVtZW50PFQ+PigpO1xuICBmb3IgKGNvbnN0IGVsZW1lbnQgb2YgeHMpIHtcbiAgICBjb25zdCBrZXkgPSBrZXlGbihlbGVtZW50KTtcbiAgICByZW1haW5pbmcuc2V0KGtleSwgeyBrZXksIGVsZW1lbnQsIGRlcGVuZGVuY2llczogZGVwRm4oZWxlbWVudCkgfSk7XG4gIH1cblxuICBjb25zdCByZXQgPSBuZXcgQXJyYXk8VD4oKTtcbiAgd2hpbGUgKHJlbWFpbmluZy5zaXplID4gMCkge1xuICAgIC8vIEFsbCBlbGVtZW50cyB3aXRoIG5vIG1vcmUgZGVwcyBpbiB0aGUgc2V0IGNhbiBiZSBvcmRlcmVkXG4gICAgY29uc3Qgc2VsZWN0YWJsZSA9IEFycmF5LmZyb20ocmVtYWluaW5nLnZhbHVlcygpKS5maWx0ZXIoKGUpID0+XG4gICAgICBlLmRlcGVuZGVuY2llcy5ldmVyeSgoZCkgPT4gIXJlbWFpbmluZy5oYXMoZCkpXG4gICAgKTtcblxuICAgIHNlbGVjdGFibGUuc29ydCgoYSwgYikgPT4gKGEua2V5IDwgYi5rZXkgPyAtMSA6IGIua2V5IDwgYS5rZXkgPyAxIDogMCkpO1xuXG4gICAgZm9yIChjb25zdCBzZWxlY3RlZCBvZiBzZWxlY3RhYmxlKSB7XG4gICAgICByZXQucHVzaChzZWxlY3RlZC5lbGVtZW50KTtcbiAgICAgIHJlbWFpbmluZy5kZWxldGUoc2VsZWN0ZWQua2V5KTtcbiAgICB9XG5cbiAgICAvLyBJZiB3ZSBkaWRuJ3QgbWFrZSBhbnkgcHJvZ3Jlc3MsIHdlIGdvdCBzdHVja1xuICAgIGlmIChzZWxlY3RhYmxlLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgQ291bGQgbm90IGRldGVybWluZSBvcmRlcmluZyBiZXR3ZWVuOiAke0FycmF5LmZyb20oXG4gICAgICAgICAgcmVtYWluaW5nLmtleXMoKVxuICAgICAgICApLmpvaW4oXCIsIFwiKX1gXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIHJldHVybiByZXQ7XG59XG5cbmludGVyZmFjZSBUb3BvRWxlbWVudDxUPiB7XG4gIGtleTogc3RyaW5nO1xuICBkZXBlbmRlbmNpZXM6IHN0cmluZ1tdO1xuICBlbGVtZW50OiBUO1xufVxuIl19