"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KubectlProvider = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const lambda_layer_awscli_1 = require("@aws-cdk/lambda-layer-awscli");
const lambda_layer_kubectl_1 = require("@aws-cdk/lambda-layer-kubectl");
const cluster_1 = require("./cluster");
class KubectlProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        super(scope, id);
        const cluster = props.cluster;
        if (!cluster.kubectlRole) {
            throw new Error('"kubectlRole" is not defined, cannot issue kubectl commands against this cluster');
        }
        if (cluster.kubectlPrivateSubnets && !cluster.kubectlSecurityGroup) {
            throw new Error('"kubectlSecurityGroup" is required if "kubectlSubnets" is specified');
        }
        const memorySize = cluster.kubectlMemory ? cluster.kubectlMemory.toMebibytes() : 1024;
        const handler = new lambda.Function(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'kubectl-handler')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            description: 'onEvent handler for EKS kubectl resource provider',
            memorySize,
            environment: cluster.kubectlEnvironment,
            role: cluster.kubectlLambdaRole ? cluster.kubectlLambdaRole : undefined,
            // defined only when using private access
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            securityGroups: cluster.kubectlSecurityGroup ? [cluster.kubectlSecurityGroup] : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
        });
        // allow user to customize the layer
        if (!props.cluster.kubectlLayer) {
            handler.addLayers(new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer'));
            handler.addLayers(new lambda_layer_kubectl_1.KubectlLayer(this, 'KubectlLayer'));
        }
        else {
            handler.addLayers(props.cluster.kubectlLayer);
        }
        this.handlerRole = handler.role;
        this.handlerRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeCluster'],
            resources: [cluster.clusterArn],
        }));
        // allow this handler to assume the kubectl role
        cluster.kubectlRole.grant(this.handlerRole, 'sts:AssumeRole');
        const provider = new cr.Provider(this, 'Provider', {
            onEventHandler: handler,
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
            securityGroups: cluster.kubectlSecurityGroup ? [cluster.kubectlSecurityGroup] : undefined,
        });
        this.serviceToken = provider.serviceToken;
        this.roleArn = cluster.kubectlRole.roleArn;
    }
    static getOrCreate(scope, cluster) {
        // if this is an "owned" cluster, it has a provider associated with it
        if (cluster instanceof cluster_1.Cluster) {
            return cluster._attachKubectlResourceScope(scope);
        }
        // if this is an imported cluster, we need to provision a custom resource provider in this stack
        // we will define one per stack for each cluster based on the cluster uniqueid
        const uid = `${core_1.Names.nodeUniqueId(cluster.node)}-KubectlProvider`;
        const stack = core_1.Stack.of(scope);
        let provider = stack.node.tryFindChild(uid);
        if (!provider) {
            provider = new KubectlProvider(stack, uid, { cluster });
        }
        return provider;
    }
}
exports.KubectlProvider = KubectlProvider;
//# sourceMappingURL=data:application/json;base64,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