# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dencrypt',
 'dencrypt.base',
 'dencrypt.core',
 'dencrypt.exception',
 'dencrypt.support']

package_data = \
{'': ['*']}

install_requires = \
['cryptography>=35.0.0,<36.0.0', 'rsa>=4.7.2,<5.0.0']

entry_points = \
{'console_scripts': ['dencrypt = cli:main']}

setup_kwargs = {
    'name': 'dencrypt',
    'version': '0.1.0',
    'description': 'Quickly generates the encryption of texts and files. Using RSA and Fernet as a base.',
    'long_description': '# DENcrypt\n#### **v.0.1.0**\n\nMódulo criado com o objetivo de facilitar a criptografia/descriptografia de textos e arquivos.\n\nUtiliza como base as criptografias: \n- RSA (Assimétrica)\n  - Encrypt (Chave Pública)/Decrypt (Chave Privada)\n  - Lento; Uso recomendado em conteúdo com tamanho máximo de 256 bytes\n\n- Fernet (Simétrica)\n  - EncryptFile (Chave Única)/DecryptFile (Chave Única)\n  - Rápido; Uso recomendado para arquivos\n\n\n\n# Requisitos\n\n```python\nrsa = "^4.7.2"\ncryptography = "^35.0.0"\n```\n\n\n\n# Instalação\n\n```\npip install git+https://github.com/edvitor13/dencrypt.git\n```\n\n\n\n# Como Funciona - Via Código\n\n`config. inicial do projeto`\n\n```\nexemplo/\n├─ arquivos/\n   ├─ dados1.txt\n   ├─ dados2.csv\n├─ arquivos-crip/\n├─ keys/\n├─ main.py\n```\n`main.py`\n```python\nfrom dencrypt import (\n    Crypt, # Gera Chaves Pública/Privada RSA e Simétrica Fernet \n    Encrypt, Decrypt, # Realiza En/Decriptação RSA\n    EncryptFile, DecryptFile # Realiza En/Decriptação Fernet\n)\n\n# 1. Gerando as Chaves\ncr = Crypt().generate_all_keys()\ncr.pubkey       # b\'-----BEGIN RSA PUBLIC KEY-----  MIIBCgKCAQEAhBF7c...\'\ncr.privkey      # b\'-----BEGIN RSA PRIVATE KEY----- MIIEqQIBAAKCAQEAd...\'\ncr.symmetrickey # b\'KqqI5Ncke0TSzBJFrE0SM3xuHz11tv689A1PhmkewkE=\'\n\n# 1.1. Salvando arquivos das chaves\ncr.save_all_keys(location="keys/")\n```\n```\nexemplo/\n├─ arquivos/\n   ├─ dados1.txt\n   ├─ dados2.csv\n├─ arquivos-crip/\n├─ keys/\n   ├─ privatekey.key\n   ├─ publickey.key\n   ├─ symmetrickey.key\n├─ main.py\n```\n```python\n# 2. Criptografando Arquivos com a Chave Simétrica\n\n# 2.1. Arquivo dados1.txt\nencr = EncryptFile("arquivos/dados1.txt", key="keys/symmetrickey.key")\nencr.content # Conteúdo Original: b\'Teste 1\\nTeste 2\\nTest...\'\nencr.get()   # Conteúdo Criptog.: b\'\\xc3\\x8a\\xc3\\xa2\\xc3\\xaa\\xc3\\xae...\'\n\n# Salvando arquivo com conteúdo criptografado\nencr.save("arquivos-crip/dados1.txt.dencrypt") \n\n# 2.2. Arquivo dados2.csv\nencr = EncryptFile("arquivos/dados2.csv", key="keys/symmetrickey.key")\nencr.save("arquivos-crip/dados2.csv.dencrypt")\n```\n```\nexemplo/\n├─ arquivos/\n   ├─ dados1.txt\n   ├─ dados2.csv\n├─ arquivos-crip/\n   ├─ dados1.txt.dencrypt\n   ├─ dados2.csv.dencrypt\n├─ keys/\n   ├─ privatekey.key\n   ├─ publickey.key\n   ├─ symmetrickey.key\n├─ main.py\n```\n```python\n# 3. Criptografando Chave Simétrica com Chave Pública RSA\nencr = Encrypt("keys/symmetrickey.key", pubkey="keys/publickey.key")\nencr.save("arquivos-crip/secreta.key")\n```\n```\n├─ arquivos-crip/\n   ├─ secreta.key\n   ├─ dados1.txt.dencrypt\n   ├─ dados2.csv.dencrypt\n```\n```python\n# 4. Descriptografando Arquivos\n\n# 4.1. Descriptografando Chave Simétrica com Chave Privada RSA\nskey = Decrypt("arquivos-crip/secreta.key", privkey="keys/privatekey.key")\nskey = skey.get() # b\'KqqI5Ncke0TSzBJFrE0SM3xuHz11tv689A1PhmkewkE=\'\n\n#  4.2. Descriptografando dados1.txt.dencrypt\ndecr = DecryptFile("arquivos-crip/dados1.txt.dencrypt", key=skey)\ndecr.save("arquivos-crip/dados1.txt")\n\n#  4.3. Descriptografando dados2.csv.dencrypt\ndecr = DecryptFile("arquivos-crip/dados2.csv.dencrypt", key=skey)\ndecr.save("arquivos-crip/dados2.csv")\n```\n```\nexemplo/\n├─ arquivos/\n   ├─ dados1.txt\n   ├─ dados2.csv\n├─ arquivos-crip/\n   ├─ secreta.key\n   ├─ dados1.txt\n   ├─ dados1.txt.dencrypt\n   ├─ dados2.csv\n   ├─ dados2.csv.dencrypt\n├─ keys/\n   ├─ privatekey.key\n   ├─ publickey.key\n   ├─ symmetrickey.key\n├─ main.py\n```\n```python\n# 5. Realizando comparação de conteúdo\nwith (\n    open("arquivos/dados1.txt") as d1_original,\n    open("arquivos/dados2.csv") as d2_original,\n    open("arquivos-crip/dados1.txt.dencrypt") as d1_crip,\n    open("arquivos-crip/dados2.csv.dencrypt") as d2_crip,\n    open("arquivos-crip/dados1.txt") as d1_decrip,\n    open("arquivos-crip/dados2.csv") as d2_decrip\n):\n    # Reads\n    d1_original_read = d1_original.read()\n    d2_original_read = d2_original.read()\n    d1_crip_read = d1_crip.read()\n    d2_crip_read = d2_crip.read()\n    d1_decrip_read = d1_decrip.read()\n    d2_decrip_read = d2_decrip.read()\n\n    # Original com Criptografado\n    d1_original_read == d1_crip_read # False\n    d2_original_read == d2_crip_read # False\n\n    # Original com Descriptografado\n    d1_original_read == d1_decrip_read # True\n    d2_original_read == d2_decrip_read # True\n\n```\n\nDownload: [examples/exemplo.zip](examples/exemplo.zip)\n\n\n\n# Como Funciona - Via CLI\n\n```\nExemplos de uso:\n    + Verificando Versão\n        > dencrypt --version | > dencrypt -v\n\n    + Gerando chaves\n        - Pública RSA (2048), Privada RSA (2048) e Simétrica Fernet\n            > dencrypt --keys | > dencrypt -k\n\n        - Apenas Simétrica Fernet\n            > dencrypt --keys --norsa | > dencrypt -k -nr\n\n        - Apenas Pública/Privada RSA (Tamanho 1024)\n            > dencrypt --keys --nofer -size 1024 | > dencrypt -k -nf -s 1024\n        \n        - Pública/Privada RSA (2048) e Simétrica Fernet escolhendo local para salvar\n            > dencrypt --keys --location "C:/user/User/Desktop" | > dencrypt -l "C:/user/User/Desktop"\n        \n        - Pú./Pr. RSA (2048) e Sim. Fer. adicionando prefixo e sufixo e alterando \n          extensão nos nomes dos arquivos das chaves\n            > dencrypt --keys --prefix "B1_" --sufix "_1B" --ext ".chave" \n            > dencrypt -k -px "B1_" -sx "_1B" -e ".chave"\n            \n            Exemplo de resultado padrão sem aplicar "-px", "-sx" e "-e"\n                - privatekey.key\n                - publickey.key\n                - symmetrickey.key\n            \n            Exemplo de resultado aplicando "-px", "-sx" e "-e"\n                - B1_privatekey_1B.chave\n                - B1_publickey_1B.chave\n                - B1_symmetrickey_1B.chave\n\n    + Encriptando arquivos/conteúdo\n        - Com chave Simétrica Fernet\n            > dencrypt --encrypt --input "arquivo.jpg" --skey "symmetrickey.key"\n                        -enc      -i                    -sk\n            Será gerado um novo arquivo chamado "arquivo.jpg.dencrypt"\n\n        - Com chave Pública RSA\n            > dencrypt -enc -i "arquivo.jpg" --pkey "publickey.key"\n                                              -pk\n            Será gerado um novo arquivo chamado "arquivo.jpg.dencrypt"\n            ! Esta encriptação é limitada ao seu tamanho. Caso a chave seja de \n              2048 bits (Padrão), suportará um conteúdo de no máximo 256 bytes\n\n        - Texto\n            > ... -i "Texto Secreto" -sk ... --forcetext \n                                              -ft\n            Será gerado um novo arquivo chamado "Texto Secreto.dencrypt" com o texto \n              criptografado de conteúdo\n            Porém como podemos observar, o nome do arquivo se mantém com o do conteúdo\n              secreto, abaixo veja exemplos de como renomear o arquivo com `output`      \n        \n        - Utilizando `--ext` e `--output` para salvar arquivo com nome diferente\n            ? DICA: `--output` possui alguns format strings\n                %FILENAME = Retorna o nome completo do arquivo original - Ex: "arquivo.jpg"\n                %FILEEXT  = Retorna apenas a extensão do arquivo original - Ex: ".jpg"\n                %FILE     = Retorna o nome do arquivo original - Ex: "arquivo"\n                %EXT"     = Retorna a extensão enviada através do `--ext` - Por padrão é ".dencrypt"\n                %DATETIME = Retorna a data do momento, seguinto o formado: ddmmaaaahhmmss \n                %RAND"    = Retorna um número aleatório entre 100.000 e 999.999\n            \n            > ... --ext ".louco" --output "a"\n                   -e             -o\n            Será gerado um novo arquivo chamado "a.louco" em vez de "arquivo.jpg.dencrypt"\n            \n            > ... -e ".louco" -o "%FILENAME%EXT"  |  > ... -e ".teste" -o "%FILE_%DATETIME%EXT"\n            Resultará em "arquivo.jpg.louco"      |  Resultará em "arquivo_281020210252.teste"\n            \n            > ... --ext ".abcabc" --output "%RAND%EXT"  |  > ... --ext ".abcabc"\n            Resultará em "734582.abcabc"                |  Resultará em "arquivo.jpg.abcabc" \n        \n        - Múltiplos arquivos\n          > dencrypt --encrypt --allfiles --skey "symmetrickey.key"\n                                -af\n          Todos arquivos da pasta atual serão criptografados e salvos em uma \n          nova pasta (criada no mesmo diretório) chamada "dencrypt_encrypt_202110280323" \n          com a data atual no final\n          O nome dos novos arquivos seguirá o formato "nome.ext.dencrypt", Ex: "imagem.jpg.dencrypt"\n          ! Os arquivos originais ainda serão mantidos\n\n          > ... --allfiles ... --extignore "jpg, png, py"\n                                -eig\n          Realizará o mesmo processo anterior, porém ignorando arquivos com as extensões informadas\n          .jpg, .png e .py\n\n        - Comandos persistentes\n          --replace ou -r = Força a substituição de qualquer arquivo existente com o mesmo nome\n          --force ou -f   = Ignora qualquer aviso, e avança automaticamente \n\n    + Decriptando arquivos/conteúdo\n      - Funcionamento semelhante ao `--encrypt`, porém, agora deve ser passado via `--input`\n        o arquivo já criptografado\n\n      - Com chave Simétrica Fernet\n        > dencrypt --decrypt --input "arquivo.jpg.dencrypt" --skey "symmetrickey.key"\n                    -dec      -i                    -sk\n        Será gerado um novo arquivo chamado "arquivo.jpg"\n\n      - Com chave PRIVADA RSA\n        > dencrypt -dec -i "arquivo.jpg.dencrypt" --pkey "privatekey.key"\n                                                   -pk\n        Será gerado um novo arquivo chamado "arquivo.jpg"\n\n      ...\n```\n',
    'author': 'Vitor Gabriel',
    'author_email': 'edvitor13@hotmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/edvitor13/dencrypt',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
