"""
pyodide-mkdocs-theme
Copyleft GNU GPLv3 🄯 2024 Frédéric Zinelli

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.
If not, see <https://www.gnu.org/licenses/>.
"""



from typing import List
from mkdocs.config import config_options as C

from pyodide_mkdocs_theme.pyodide_macros.plugin.config._string_tools_and_constants import InclusionProfile


from ....tools_and_constants import DecreaseAttemptsMode, DeprecationLevel, HashPathMode, PageInclusion, SequentialFilter, SequentialRun, NamedTestCase
from ..common_tree_src import CommonTreeSrc
from ..config_option_src import ConfigOptionDeprecated, ConfigOptionSrc
from ..sub_config_src import SubConfigSrc
from .docs_dirs_config import DOCS_CONFIG, to_page
from ...tools.options_alterations import sanitize_decrease_attempts_on_user_code_failure



# NOTE: must be a single line, otherwise it well mess `dedent` up.
FORBID_SHORT_CONFIG = (
    "Par défaut, cette situation est considérée comme invalide et `BuildError` sera "
    "levée. Si ce comportement est souhaité, passer cette option à `False`."
)
EN_FORBID_SHORT_CONFIG = (
    "By default, this situation is considered invalid and `BuildError` will be raised. "
    "If this is the desired behavior, set this option to false."
)


CommonTreeSrc.DEFAULT_DOCS_URL_TEMPLATE = to_page(DOCS_CONFIG) / '#{py_macros_path}'














BUILD_CONFIG = SubConfigSrc(
    'build',
    extra_docs = """
    Réglages concernant la construction de la documentation ou qui impactent la façon
    dont le contenu des pages est construit.
    """,
    elements = (

    # KEEP THIS ONE: used for testing purposes.
    ConfigOptionDeprecated(
        'activate_cache', bool, default=None, # default=True
        extra_docs = """
            Active ou non le cache permettant de stocker les données relatives aux codes python et
            REMs des différentes macros (IDE, terminal, ...).

            Si `True` :

            * Rend `mkdocs serve` plus rapide après le premier rendu
            * Il est plus souvent nécessaire d'utiliser les arguments `ID` des différentes macros.

            Si `False`:

            * Les arguments ID ne sont jamais nécessaires pour les macros autres que les IDEs.
            * Les contenus sont recalculés systématiquement à chaque rendu, à partir des données
            des fichiers individuels
        """,
        yaml_desc="""
            Activate or not the macros cache. If activated, the ID argument is more often necessary.
        """,
    ),

    ConfigOptionSrc(
        'deprecation_level', str, conf_type=C.Choice( DeprecationLevel.VALUES, default=DeprecationLevel.error),
        extra_docs = """
            Comportement utilisé lors d'un build/serve lorsqu'une option obsolète est utilisée.

            _ATTENTION:_ concernant les fichiers {{ meta() }}, seule la valeur sélectionnée depuis
            le fichier `mkdocs.yml` est prise en compte. Les valeurs configurées depuis ces fichiers
            n'impactent que les validations des données dans les entêtes des pages markdown.
        """,
        yaml_desc="Behavior when a deprecated feature is used."
    ),

    ConfigOptionSrc(
        'encrypted_js_data', bool, default=True,
        extra_docs="""
            Si `True`, les données de configuration des IDEs, terminaux et py_btns sont encodées.

            Si des problèmes de décompression des données sont rencontrés, cette option peut être
            désactivée, mais cela implique que toutes les données des codes python (notamment les
            contenus des sections `secrets` ou `corr`) seront accessibles à quelqu'un fouillant
            explorant le DOM de la page via l'inspecteur du navigateur.
        """,
        yaml_desc="Compress or not configuration data of IDEs, terminals, ...",
    ),

    ConfigOptionSrc(
        'extra_pyodide_sections', List[str], conf_type=C.ListOfItems(C.Type(str), default=[]),
        extra_docs="""
            Liste de chaînes de caractères additionnelles autorisées en tant que nom de sections
            pour les entêtes `PYODIDE:{section}` des fichiers python du thème.
        """,
        yaml_desc="""
            Extra `PYODIDE:{section}` names authorized in the python files.
        """,
    ),

    ConfigOptionSrc(
        'forbid_macros_override', bool, default=True,
        extra_docs="""
            Si `True`, `PmtMacrosError` est levée lorsque deux macros du même nom sont
            enregistrées par le plugin.
        """,
        yaml_desc="""
            If `True` registering different macros with the same name will raise a
            `PmtMacrosError`.
        """,
    ),

    ConfigOptionSrc(
        'ides_id_hash_mode', str, conf_type=C.Choice( HashPathMode.VALUES, default=HashPathMode.legacy),
        extra_docs = f"""
            Définit le mode de construction des id html des IDEs. Changer de mode rend toutes les
            données utilisateurs stockées en localeStorage invalide !

            Si le mode {HashPathMode.relative!r} est utilisé, un changement de nom de domaine ou de
            la valeur de `site_url` n'affecte pas les valeurs des identifiant html des IDEs.
        """,
        yaml_desc="Behavior used to compute the IDEs' html ids."
    ),

    ConfigOptionSrc(
        'ignore_macros_plugin_diffs', bool, default=False,
        extra_docs = """
            Passer à `#!py True` pour éviter la vérification de compatibilité de la
            configuration du plugin `PyodideMacroPlugin` avec celle du plugin original
            des macros, `MacrosPlugin`.

            ??? note "Raisons de cette vérification"

                Le plugin du thème hérite de celui de la bibliothèque `mkdocs-macros-plugin`,
                `PyodideMacros`.

                Or, la configuration du plugin `MacrosPlugin` est faite "à l'ancienne", avec
                `config_scheme`, alors que celle de `PyodideMacroPlugin` utilise les classes
                `Config` disponibles à partir de mkdocs `1.5+`. Les deux étant incompatibles,
                cela à imposé de reporter en dur la configuration du plugin d'origine dans
                celle du thème. Ceci fait qu'une modification de la configuration du plugin
                d'origine pourrait rendre celle du thème inopérante et ceci sans préavis.

                Cette vérification permet donc d'assurer que le comportement des objets
                `MacrosPlugin` sera celui attendu. Si une différence est constatée entre les
                deux configurations, le build est donc avorté car il n'y a aucune garantie que
                le site construit puisse encore être correct.

                Si les modifications de `MacrosPlugin` sont mineures, il est possible qu'un build
                puisse tout de même fonctionner, et passer cette option à `#!py True` permettra
                donc de faire l'essai. À tenter à vos risques et périls...
        """,
        yaml_desc="""
            Deactivate the compatibility check of PyodideMacrosPlugin configuration against the
            MacrosPlugin one.
        """,
    ),

    ConfigOptionSrc(
        'load_yaml_encoding', str, default='utf-8',
        extra_docs="""
            Encodage à utiliser lors du chargement de données YAML avec les fonctionnalités
            originales de MacrosPlugin :

            La méthode d'origine n'utilise aucun argument d'encodage, ce qui peut entraîner des
            comportements différents entre Windows et Linux (typiquement : lors de l'exécution
            d'un pipeline sur la forge EN par rapport au travail local sous Windows).
        """,
        yaml_desc="""
            Encoding used when loading yaml files within the original macros plugin.
        """,
    ),

    ConfigOptionSrc(
        'macros_with_indents', List[str], conf_type=C.ListOfItems(C.Type(str), default=[]),
        extra_docs="""
            Permet d'enregistrer des macros personnalisées (liste de chaînes de caractères), qui
            insèrent du contenu markdown multilignes, pour pouvoir indenter  correctement le
            contenu dans la page :

            Une fois qu'une macro est enregistrée dans cette liste, elle peut appeler la méthode
            `env.indent_macro(markdown)` durant son exécution pour que le contenu généré soit
            indenté correctement par le plugin.
        """,
        yaml_desc="""
            List of macros inserting multiline contents in the pages (allowing to use
            `plugin.indent_macro(markdown)` from them).
        """,
    ),

    ConfigOptionSrc(
        'meta_yaml_allow_extras', bool, default=False,
        extra_docs="""
            Définit s'il est possible d'ajouter dans les fichiers {{meta()}} des données autres
            que celles relatives au plugin lui-même.

            Lorsque cette valeur est à `#!yaml false`, seules des options du plugin `pyodide_macros`
            sont autorisées, ce qui permet de valider l'intégralité du contenu du fichier, mais
            empêche par exemple de définir des variables pour les macros dans ces fichiers.<br>Si
            la valeur est à `#!yaml true`, il est alors possible d'ajouter d'autres variables,
            mais les fautes de frappes dans les premiers niveaux ne peuvent plus être identifiées
            (exemple : `temrs.cut_feedback` au lieu de `terms.cut_feedback`).

            _ATTENTION:_ concernant les fichiers {{ meta() }}, seule la valeur sélectionnée depuis
            le fichier `mkdocs.yml` est prise en compte. Les valeurs configurées depuis ces fichiers
            n'impactent que les validations des données dans les entêtes des pages markdown.
        """,
        yaml_desc="Allow definition of extra variables/options in the `.meta.pmt.yml` files.",
    ),

    ConfigOptionSrc(
        'meta_yaml_encoding', str, default='utf-8',
        extra_docs="Encodage utilisé pour charger les [fichiers `.meta.pmt.yml`](--custom/metadata/).",
        yaml_desc="Encoding to use when loading `.meta.pmt.yml` files."
    ),

    ConfigOptionSrc(
        'python_libs', List[str], conf_type=C.ListOfItems(C.Type(str), default=['py_libs']),
        extra_docs="""
            Liste de répertoires de [bibliothèques python](--custom-libs) qui doivent être
            importables dans Pyodide.

            Une erreur est levée si :

            * Le nom donné ne correspond pas à un répertoire existant (sauf s'il s'agit de la
            valeur par défaut, `#!py "py_libs"`).
            * Le répertoire n'est pas situé à la racine du projet.
            * Le répertoire n'est pas une bibliothèque Python (c'est-à-dire qu'il ne contient
            pas de fichier `__init__.py`).
        """,
        yaml_desc="""
            List of custom python packages to make available at runtime in Pyodide environment.
        """,
    ),

    ConfigOptionSrc(
        'limit_pypi_install_to', List[str], is_optional=True, conf_type=C.ListOfItems(C.Type(str)),
        extra_docs="""
        Si cette liste est définie, seules les imports dont le nom de bibliothèque
        figure dans cette liste seront autorisés à déclencher une installation
        automatique depuis PyPI. Noter que :

        * C'est le nom de l'import dans le code python qui doit être renseigné
        (ex : `PIL` pour interdire l'installation de `pillow`).

        * Utiliser `[]` interdit toutes les installations automatiques depuis PyPI.

        * Mettre cette option à `null` (valeur par défaut) autorise toutes les
        requêtes vers PyPI.
        """,
        yaml_desc="""
            If defined, only the package names in this list are allowed to be automatically
            installed from PyPI.
        """,
    ),

    ConfigOptionDeprecated(
        'show_cache_refresh', bool, default=None, # default=False
        extra_docs="""
            Si `#!yaml true`, des messages sont ajoutés dans la console permettant de voir quand
            les caches des fichiers du thèmes sont mis à jour (debugging purpose...).
        """,
        yaml_desc="""
            Show additional messages in the console about internal caches updates.
        """,
    ),

    ConfigOptionSrc(
        'skip_py_md_paths_names_validation', bool, default=False,
        extra_docs = """
            Par défaut, les noms de chemin de tous les fichiers `.py` et `.md` présents dans
            le `docs_dir` sont vérifiés pour s'assurer qu'ils ne contiennent aucun caractère
            autre que des lettres, des chiffres, des points ou des tirets. Cela garantit le
            bon fonctionnement des macros liées aux IDEs.

            Si des caractères indésirables sont détectés, une erreur de type `BuildError`
            est levée. Cependant, cette vérification peut être désactivée en assignant `True`
            à ce paramètre.
            ... À Utiliser  à vos risques et périls.
        """,
        yaml_desc="Deactivate the sanity check of the directories and files in the `docs_dir`.",
    ),

    ConfigOptionSrc(
        'tab_to_spaces',int, default=-1,
        extra_docs="""
            Si cette option est définie avec une valeur positive (ou nulle), les tabulations
            trouvées avant un appel à une macro multiligne (voir l'option
            [`macros_with_indenst`](--pyodide_macros_build_macros_with_indents)) seront
            automatiquement converties en utilisant ce nombre d'espaces.

            __Aucune garantie n'est alors donnée quant à la correction du résultat__.
            <br>Si une conversion est effectuée, un avertissement sera affiché dans la console
            pour faciliter la localisation et la modification des appels de macros responsables
            du warning.

            !!! warning "Éviter les caractères de tabulation dans la documentation"

                Régler votre éditeur de code de manière à ce qu'il remplace automatiquement les
                tabulations par des espaces.

                Les caractères de tabulation ne sont pas toujours interprétés de la même façon
                selon le contexte d'utilisation du fichier, tandis que les fichiers markdown
                reposent en bonne partie sur les indentations pour définir la mise en page des
                rendus.
                <br>Les tabulations sont donc à proscrire.
        """,
        yaml_desc="""
            Number os space characters used to replace tabulations on the left of multiline
            macro calls.
        """,
    ),

    ConfigOptionSrc(
        '_pmt_meta_filename', str, default=".meta.pmt.yml",
        inclusion_profile = InclusionProfile.config,
        extra_docs = "Nom des fichiers de configuration des métadonnées pour le thème.",
        yaml_desc="Name used for the Pyodide-MkDoc-Theme meta files."
    ),

    #----------------------------------------------------------------------------

    # ConfigOptionDeprecated(
    #     'encrypt_corrections_and_rems', bool,
    #     moved_to = 'ides.encrypt_corrections_and_rems'
    # ),
    # ConfigOptionDeprecated(
    #     'forbid_secrets_without_corr_or_REMs', bool,
    #     moved_to = 'ides.forbid_secrets_without_corr_or_REMs',
    # ),
    # ConfigOptionDeprecated(
    #     'forbid_hidden_corr_and_REMs_without_secrets', bool,
    #     moved_to = 'ides.forbid_hidden_corr_and_REMs_without_secrets',
    # ),
    # ConfigOptionDeprecated(
    #     'forbid_corr_and_REMs_with_infinite_attempts', bool,
    #     moved_to = 'ides.forbid_corr_and_REMs_with_infinite_attempts',
    # ),
    # ConfigOptionDeprecated(
    #     'bypass_indent_errors', bool, deprecation_status=DeprecationStatus.removed,
    # ),
))













IDES_CONFIG = SubConfigSrc(
    'ides',
    extra_docs = """
    Réglages spécifiques aux IDEs (comportements impactant l'utilisateur et les exécutions).
    """,
    elements = (

    ConfigOptionSrc(
        'deactivate_stdout_for_secrets', bool, default=True,
        extra_docs="""
            Détermine si la sortie standard (stdout) sera visible dans les terminaux lors
            des tests secrets ou non.
        """,
        yaml_desc="""
            Define if the stdout will be shown in terminals to the user or not, during the
            secret tests.
        """,
    ),

    ConfigOptionSrc(
        'decrease_attempts_on_user_code_failure', bool,
        conf_type=C.Choice(DecreaseAttemptsMode.VALUES+(True, False), default=DecreaseAttemptsMode.editor),
        yaml_schema_dct = {"enum": [*DecreaseAttemptsMode.VALUES]},
        value_transfer_processor = sanitize_decrease_attempts_on_user_code_failure,
        extra_docs="""
            Les validations sont grossièrement constituées de 4 étapes, exécutant les éléments
            suivants :

            1. La section `env`, qui ne devrait pas lever d'erreur sauf `AssertionError`.
            1. Le contenu de l'éditeur (y compris l'état actuel des tests publics).
            1. La section `tests` du fichier python, assurant que la version __originale__ des
            tests publics est toujours exécutée.
            1. La section `secrets` du fichier python.

            Les exécutions étant stoppées à la première erreur rencontrée, cette option définit
            à partir de quelle étape une erreur doit consommer un essai :

            1. `#!py "editor"` : Une erreur levée lors de l'exécution de la section `env`
            ou du contenu de l'éditeur sera comptée comme un essai consommé.
            1. `#!py "public"` : seules les erreurs levées depuis les étapes 3 et 4
            décompteront un essai.
            1. `#!py "secrets"` : seules les erreurs levées depuis la section `secrets`
            décompteront un essai.

            --8<-- "docs_tools/inclusions/decrease_attempts_on_user_code_failure.md"

            ??? warning "Options booléennes"

                Les valeurs booléennes sont là uniquement pour la rétrocompatibilité et un
                warning apparaîtra dans la console si elles sont utilisées.

                * `True` correspond à `#!py "editor"`
                * `False` correspond à `#!py "secrets"`
        """,
        yaml_desc="""
            Define from which step an error will consume an attempt, during a validation.
        """,
    ),

    ConfigOptionSrc(
        'encrypt_alpha_mode', str,
        conf_type=C.Choice(('direct', 'shuffle', 'sort'), default='direct'),
        extra_docs="""
            Les contenus (codes, corrections & remarques) sont transmis de mkdocs aux pages html
            en utilisant des données compressées. L'encodage est réalisé avec l'algorithme LZW,
            et cette option contrôle la manière dont l'alphabet/la table initiale est construit à
            partir du contenu à encoder :

            - `#!py "direct"` : l'alphabet utilise les symboles dans l'ordre où ils sont trouvés
            dans le contenu à compresser (utilisé par défaut).
            - `#!py "shuffle"` : l'alphabet est mélangé aléatoirement.
            - `#!py "sort"` : les symboles sont triés dans l'ordre naturel.
        """,
        yaml_desc="""
            Define in what order the characters of the content are pushed in the LZW compression
            table (by default: `direct`, `shuffle`, `sort`).
        """,
    ),

    ConfigOptionSrc(
        'encrypt_corrections_and_rems', bool, default=True,
        extra_docs="""
            Si activé, le contenu de la div HTML de la correction et des remarques, sous
            les IDEs, sera compressé lors de la construction du site.

            Désactiver ceci peut être utile durant le développement, mais {{ red("cette option
            doit toujours être activée pour le site déployé") }}, sans quoi la barre de recherche
            pourraient suggérer le contenu des corrections et des remarques à l'utilisateur.
        """,
        yaml_desc="""
            Compress or not the solutions and remarks below IDEs (deactivate only for
            debugging purpose).
        """,
    ),

    ConfigOptionSrc(
        'export_zip_prefix', str, default="",
        extra_docs="""
            Préfixe ajouté au début du nom des archives zip créées avec les contenus des éditeurs
            des IDEs configurés comme exportable (argument [`EXPORT=True`](--IDE-EXPORT)).
            Si `{{ config_validator('ides.export_zip_prefix',tail=1) }}` n'est pas une chaîne
            vide, un trait d'union sera ajouté automatiquement entre le préfixe et le reste du
            nom de l'archive.
        """,
        yaml_desc="""
            Prefix for the zip archive containing the editor content of all the exportable IDEs
            in the page.
        """,
    ),

    ConfigOptionSrc(
        'export_zip_with_names', bool, default=False,
        extra_docs="""
            Si `#!py True`, au moment où un utilisateur demandera de créer l'archive zip avec
            tous les codes des IDEs de la page [configurés pour être exportés](--IDE-EXPORT),
            une fenêtre s'ouvrira lui demandant d'indiquer son nom. Une fois le nom renseigné,
            il sera ajouté entre l'éventuel préfixe (voir {{config_link('ides.export_zip_prefix',
            tail=1)}}) et le nom normal de l'archive zip, entouré par des traits d'union.
        """,
        yaml_desc="""
            Choose if the user has to give a name when building a zip archive of the IDEs contents.
        """,
    ),

    ConfigOptionSrc(
        'forbid_corr_and_REMs_with_infinite_attempts', bool, default=True,
        extra_docs = f"""
            Lors de la construction des IDEs, si une section `corr`, un fichier `REM` ou `
            VIS_REM` existent et que le nombre de tentatives est illimité, ce contenu ne
            sera jamais accessible à l'utilisateur, sauf s'il réussit les tests.

            { FORBID_SHORT_CONFIG }
        """,
        yaml_desc = EN_FORBID_SHORT_CONFIG,
    ),

    ConfigOptionSrc(
        'forbid_hidden_corr_and_REMs_without_secrets', bool, default=True,
        extra_docs=f"""
            Lors de la construction des IDEs, le bouton de validation n'apparaît que si une
            section `secrets` existe.
            <br>Si des sections `corr` ou des fichiers `REM` existent alors qu'aucune section
            `secrets` n'est présente, leur contenu ne sera jamais disponible pour l'utilisateur
            en raison de l'absence de bouton de validation dans l'interface.

            { FORBID_SHORT_CONFIG }
        """,
        yaml_desc = EN_FORBID_SHORT_CONFIG,
    ),

    ConfigOptionSrc(
        'forbid_secrets_without_corr_or_REMs', bool, default=True,
        extra_docs = FORBID_SHORT_CONFIG,
        yaml_desc = EN_FORBID_SHORT_CONFIG,
    ),

    ConfigOptionSrc(
        'remove_assertions_stacktrace', bool, default=False,
        extra_docs="""
            Si activé (`True`), les stacktrace des messages d'erreur des assertions sont supprimées
            et seul le message d'erreur avec le type d'erreur est conservé (dernière ligne).

            Ceci s'applique à n'importe quel moment des exécutions.
        """,
        yaml_desc="""
            If True, the stacktrace of assertions messages will be removed (except the very last line/
            part of the message).
        """,
    ),

    ConfigOptionSrc(
        'show_only_assertion_errors_for_secrets', bool, default=False,
        extra_docs="""
            Si activé (`True`), la stacktrace des messages d'erreur sera supprimée et
            seuls les messages des assertions resteront inchangées lorsqu'une erreur
            sera levée pendant les tests secrets.

            | `AssertionError` | Pour les autres erreurs |
            |:-:|:-:|
            | {{ pmt_note("Option à `false`",0) }}<br>![AssertionError: message
            normal](!!show_assertions_msg_only__assert_full_png) | {{ pmt_note("Option à `false`",0)
            }}<br>![Autres erreurs: message normal](!!show_assertions_msg_only__error_full_png) |
            | ![AssertionError: sans stacktrace](!!show_assertions_msg_only_assert_no_stack_png){{
            pmt_note("Option à `true`") }} | ![Autres erreurs sans stacktrace ni
            message](!!show_assertions_msg_only_error_no_stack_png){{ pmt_note("Option à `true`")
            }} |
        """,
        yaml_desc="""
            If True, the stacktrace of all error messages will be suppressed and only
            assertion messages will be left unchanged, when an error is raised during the
            secret tests.
        """,
    ),

    ConfigOptionSrc(
        'editor_font_family', str, default="monospace",
        extra_docs = "Police de caractère à utiliser pour les éditeurs des IDEs.",
        yaml_desc = "Font family used in IDEs' editor.",
    ),

    ConfigOptionSrc(
        'editor_font_size', int, default=15,
        extra_docs = "Taille de la police de caractères pour les éditeurs des IDEs.",
        yaml_desc = "Font size used in IDEs' editor.",
    ),

    ConfigOptionSrc(
        'ace_style_dark', str, default="tomorrow_night_bright",
        extra_docs = """
            Thème de couleur utilisé pour les éditeurs des IDEs en mode sombre ([liste des
            thèmes disponibles][ace-themes]: utiliser le noms des fichiers `js` sans l'extension).{{
            pmt_note("Ce réglage est écrasé par l'ancienne façon de modifier le thème, en
            définissant `extra.ace_style.slate` dans le fichier mkdocs.yml.") }}
        """,
        yaml_desc = "Dark theme for IDEs' editor.",
    ),

    ConfigOptionSrc(
        'ace_style_light', str, default="crimson_editor",
        extra_docs = """
            Thème de couleur utilisé pour les éditeurs des IDEs en mode clair ([liste des thèmes
            disponibles][ace-themes]: utiliser le noms des fichiers `js` sans l'extension).{{
            pmt_note("Ce réglage est écrasé par l'ancienne façon de modifier le thème, en
            définissant `extra.ace_style.default` dans le fichier mkdocs.yml.") }}
        """,
        yaml_desc = "Light theme for IDEs' editor.",
    ),


    #--------------------------------------------------------------------------------

    # ConfigOptionDeprecated(
    #     'show_assertion_code_on_failed_test', bool, moved_to='args.IDE.LOGS',
    # ),
    # ConfigOptionDeprecated(
    #     'max_attempts_before_corr_available', int, moved_to='args.IDE.MAX',
    # ),
    # ConfigOptionDeprecated(
    #     'default_ide_height_lines', int, moved_to='args.IDE.MAX_SIZE',
    # ),
))













SEQUENTIAL_CONFIG = SubConfigSrc(
    'sequential',
    long_accessor = True,
    extra_docs = "Réglages Pour lier les exécutions de différents éléments entre elles.",
    elements = (
    ConfigOptionSrc(
        'run', str,
        conf_type = C.Choice(SequentialRun.VALUES, default=SequentialRun.none),
        extra_docs = """
            Cette option permet d'obtenir des executions liées pour différents éléments dans
            une page, un peu à la façon des Notebooks Jupyter, où l'on peut exécuter toutes
            les cellules en une fois.

            `run` peut prendre les valeurs suivantes :

            {{ul_li([
                "`#!py ''` : Pas d'exécutions liées.",
                "`#!py 'dirty'` : Exécute tous les éléments précédents depuis le premier non encore
                exécuté ou modifié, jusqu'à l'élément en cours.",
                "`#!py 'all'` : Exécute tous les éléments précédents jusqu'à l'élément en cours.",
            ])}}

            [Pour plus d'informations...](--redactors/sequential_runs/)
        """,
        yaml_desc = "Ties the executions of some elements in the page together",
    ),

    ConfigOptionSrc(
        'only', List[str],
        conf_type = C.ListOfItems(C.Choice(SequentialFilter.VALUES), default=list(SequentialFilter.VALUES)),
        extra_docs = """
            Cette option permet d'obtenir des executions liées pour différents éléments dans une
            page. Quand un utilisateur lance un élément dans la page (IDE, terminal, py_btn),
            s'il existe des éléments plus haut dans la page qui n'ont pas encore été exécutés,
            ils le seront avant celui en cours.

            Le but est d'obtenir un comportement comparable au "run all cells" des Jupyter
            Notebooks, avec un contrôle plus fin quant à ce qui est exécuté ou non.

            `filter` est une liste d'items, dont les valeurs possibles sont :

            {{ul_li([
                "`#!py 'ide'` : exécute les IDEs (ou IDEvs) précédents.",
                "`#!py 'terminal'` : idem pour les terminaux.",
                "`#!py 'py_btn'` : idem pour les py_btn.",
                "`#!py 'run'` : idem pour les macros run (Note: si ces éléments sont utilisés en
                mode séquentiels, ils devraient également utiliser l'argument `AUTO_RUN=False`
                afin de garantir la reproductibilité des comportements).",
            ])}}

            Par défaut, toutes les options sont actives.

            [Pour plus d'informations...](--redactors/sequential_runs/)
        """,
        yaml_desc = "Select the kind of macros calls that can be involved in sequential runs.",
    ),
))













QCMS_CONFIG = SubConfigSrc(
    'qcms',
    extra_docs = "Réglages spécifiques aux QCMs.",
    elements = (
    ConfigOptionSrc(
        'forbid_no_correct_answers_with_multi', bool, default=True,
        extra_docs="""
            Si désactivé (`False`), une question sans réponse correcte fournie, mais marquée comme
            `multi=True`, est considérée comme valide. Si cette option est réglée à `True`, cette
            situation lèvera une erreur.
        """,
        yaml_desc = "Allow to disambiguate MCQ and SCQ when needed.",
    ),

    ConfigOptionSrc(
        'encrypt_comments', bool, default=True,
        extra_docs="""
            Si activé (`True`), les contenus en commentaires pour chaque question (révélés lors de
            la validation) sont cryptés lors de la création du qcm.
        """,
        yaml_desc = "Allow to disambiguate MCQ and SCQ when needed.",
    ),

    #-----------------------------------------------------------------------------

    # ConfigOptionDeprecated('hide',    bool, moved_to='args.multi_qcm.hide'),
    # ConfigOptionDeprecated('multi',   bool, moved_to='args.multi_qcm.multi'),
    # ConfigOptionDeprecated('shuffle', bool, moved_to='args.multi_qcm.shuffle')
))













TERMS_CONFIG = SubConfigSrc(
    'terms',
    extra_docs = "Réglages spécifiques aux terminaux.",
    elements = (

    ConfigOptionSrc(
        'cut_feedback', bool, default=True,
        extra_docs="""
            Si activé (`True`), les entrées affichées dans les terminaux sont tronquées si elles
            sont trop longues, afin d'éviter des problèmes de performances d'affichage des outils
            `jQuery.terminal`.
        """,
        yaml_desc="""
            If True, the content printed in the terminal will be truncated if it's too long, to
            avoid performances troubles.
        """
    ),

    ConfigOptionSrc(
        'stdout_cut_off', int, default=200,
        extra_docs="""
            Nombre maximal de lignes restant affichées dans un terminal : si de nouvelles
            lignes sont ajoutées, les plus anciennes sont éliminées au fur et à mesure.

            ??? note "Performances d'affichage des terminaux"

                ___Les éléments `jQuery.terminal` deviennent horriblement lents lorsque le
                nombre de caractères affichés est important.___

                Cette option permet de limiter ces problèmes de performance lorsque la sortie
                standard n'est pas tronquée (voir le bouton en haut à droite du terminal).

                Noter par contre que cette option _ne limite pas_ le nombre de caractères dans
                une seule ligne, ce qui veut dire qu'une page figée est toujours possible,
                tandis que l'option de troncature, `cut_feedback` évitera ce problème aussi.
        """,
        yaml_desc = "Maximum number of lines kept in terminals.",
    ),

    #--------------------------------------------------------------------------------

    # ConfigOptionDeprecated('default_height_ide_term',      int, moved_to='args.IDE.TERM_H'),
    # ConfigOptionDeprecated('default_height_isolated_term', int, moved_to='args.terminal.TERM_H')
))













TESTING_CONFIG = SubConfigSrc(
    'testing',
    long_accessor = True,
    extra_docs = """
        Permet de paramétrer la page pour tester automatiquement tous les IDEs de la documentation.
    """,
    elements = (

    ConfigOptionSrc(
        'page', str, default="test_ides",
        extra_docs="""
            Nom de fichier markdown (avec ou sans l'extension `.md`) utilisé pour générer une page
            contenant le nécessaire pour tester de manière semi-automatisée tous les IDEs de
            la documentation.

            * La page n'est créée que si l'option `{{config_validator("testing.include")}}`
            n'est pas à `#!yaml null`.
            * Une erreur est levée si un fichier du même nom existe déjà.
            * Une erreur est levée si le fichier n'est pas à la racine de la documentation.
        """,
        yaml_desc = "Name of the IDEs testing page. Generated only if given."
    ),
    ConfigOptionSrc(
        'include', str,
        conf_type = C.Choice(PageInclusion.VALUES[:3], default=PageInclusion.serve),
        extra_docs = f"""
            Définit si la page de tests des IDEs doit être générée et de quelle façon.
            {'{{'}ul_li([
                "`#!py '{PageInclusion.none}'` (_défaut_) : la page de tests n'est pas générée.",
                "`#!py '{PageInclusion.serve}'` : la page de tests est générée pendant `mkdocs serve`,
                et est ajoutée automatiquement à la navigation.",
                "`#!py '{PageInclusion.site}'` : La page de tests est ajoutée au site construit,
                mais n'y apparaît pas dans la navigation. Elle est aussi présente en `serve`
                (page et navigation).",
            ]){'}}'}
        """,
        yaml_desc="""
            Define when and how to generate the page to tests all the IDEs of the documentation.
        """
    ),
    ConfigOptionSrc(
        'load_buttons', bool, is_optional=True,
        extra_docs="""
            Définit si le bouton pour charger l'ensemble des codes associés à un IDE de la page des tests
            sera présent ou non.

            Le comportement par défaut, avec {{ config_validator("testing.load_buttons", 1, val=None, yaml=1) }}
            dépend de la valeur de l'option {{ config_link('testing.include') }} :

            * Pour {{ config_validator("testing.include", val="serve") }}, le bouton est présent par défaut.
            * Pour {{ config_validator("testing.include", val="site") }}, le bouton est absent par défaut.
        """,
        yaml_desc = "Name of the IDEs testing page. Generated only if given."
    ),
    ConfigOptionSrc(
        'empty_section_fallback', str,
        conf_type = C.Choice(NamedTestCase.VALUES, default='skip'),
        extra_docs="""
            Lorsque la page des tests des IDEs est construite et que la section à tester pour
            un IDE donné ne contient pas de code et que `{{config_validator("testing.empty_section_fallback", 1)}}`
            est définie, c'est cette "stratégie" qui sera utilisée à la place.
        """,
        yaml_desc = "Fallback behavior when the `section` normally tested is empty."
    ),

    ),
)











PLAYGROUND_CONFIG = SubConfigSrc(
    'playground',
    long_accessor = True,
    extra_docs = """
        Permet de paramétrer l'inclusion la page de développement/essais "playground".
    """,
    elements = (

    ConfigOptionSrc(
        'page', str, default="playground",
        extra_docs="""
            Nom de fichier markdown (avec ou sans l'extension `.md`) utilisé pour générer
            la page de développement "playground" permettant de modifier en live toutes
            les sections d'un IDE.

            * La page n'est créée que si l'option `{{config_validator("playground.include")}}`
            n'est pas à `#!yaml null`.
            * Une erreur est levée si un fichier du même nom existe déjà.
            * Une erreur est levée si le fichier n'est pas à la racine de la documentation.
        """,
        yaml_desc = "Name of the IDEs playground page. Generated only if given."
    ),
    ConfigOptionSrc(
        'include', str,
        conf_type = C.Choice(PageInclusion.VALUES, default=PageInclusion.serve),
        extra_docs = f"""
            Définit si la page de "playground" doit être générée et de quelle façon.
            {'{{'}ul_li([
                "`#!py '{PageInclusion.none}'` : la page n'est pas générée.",
                "`#!py '{PageInclusion.serve}'` : la page est générée pendant `mkdocs serve`,
                et est ajoutée automatiquement à la navigation.",
                "`#!py '{PageInclusion.site}'` : La page de tests est ajoutée au site construit,
                mais n'y apparaît pas dans la navigation. Elle est aussi présente en `serve`
                (page et navigation).",
                "`#!py '{PageInclusion.site_with_nav}'` : La page de tests est ajoutée au site
                construit et en `serve`, et est visible dans la navigation dans les deux cas.",
            ]){'}}'}
        """,
        yaml_desc="""
            Define when and how to generate the playground page of the documentation.
        """
    ),
    ),
)







# Kept as archive, but not used anymore:

# OTHERS_CONFIG = SubConfigSrc(
#     '_others',
#     extra_docs = "Réglages provenant de pyodide-mkdocs, mais qui ne sont plus utilisés.",
#     deprecation_status = DeprecationStatus.unsupported,
#     elements = (
#         ConfigOptionDeprecated('scripts_url', str),
#         ConfigOptionDeprecated('site_root',   str),
#     ),
# )
