import os
import pathlib

import click

from ._common import IKeyCreateCommand

_keys_files = {"fullchain": "fullchain.pem", "chain": "chain.pem", "private": "privkey.pem", "cert": "cert.pem"}


class LetsEncryptCertificate(IKeyCreateCommand):

    @property
    def secret_type_name(self) -> str:
        return "certificate"

    @property
    def help(self) -> str:
        return "Creates a certificate key type from a folder containing the typical four files generated by " \
               "Letsencrypt. Hint: run with 'sudo -E' if on a machine using certbot to access the files with " \
               "root permissions while maintaining the current user context."

    @property
    def name(self) -> str:
        return "letsencrypt"

    def configure_command(self, command: click.Command):
        command.params.append(click.Argument(["cert_folder"],
                                             type=click.Path(file_okay=False, dir_okay=True, exists=True,
                                                             resolve_path=True)))

    def on_create(self, **kwargs):
        cert_folder: click.Path = kwargs.get("cert_folder")
        missing_files = []
        results = {}
        for key, file_name in _keys_files.items():
            file_path = pathlib.Path(cert_folder).joinpath(file_name)
            if not file_path.exists():
                missing_files.append(file_name)
                continue
            with open(file_path, "r") as handle:
                results[key] = handle.read()

        if missing_files:
            raise ValueError(f"Missing files in {cert_folder}: {', '.join(missing_files)}")

        return results
