"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Code = exports.InlineCode = exports.AssetCode = exports.Code = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const s3_assets = require("aws-cdk-lib/aws-s3-assets");
const runtime_1 = require("./runtime");
/**
 * The code the canary should execute
 */
class Code {
    /**
     * Specify code inline.
     *
     * @param code The actual handler code (limited to 5MB)
     *
     * @returns `InlineCode` with inline code.
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * Specify code from a local path. Path must include the folder structure `nodejs/node_modules/myCanaryFilename.js`.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     *
     * @param assetPath Either a directory or a .zip file
     *
     * @returns `AssetCode` associated with the specified path.
     */
    static fromAsset(assetPath, options) {
        return new AssetCode(assetPath, options);
    }
    /**
     * Specify code from an s3 bucket. The object in the s3 bucket must be a .zip file that contains
     * the structure `nodejs/node_modules/myCanaryFilename.js`.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     *
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     *
     * @returns `S3Code` associated with the specified S3 object.
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
}
_a = JSII_RTTI_SYMBOL_1;
Code[_a] = { fqn: "@aws-cdk/aws-synthetics-alpha.Code", version: "2.73.0-alpha.0" };
exports.Code = Code;
/**
 * Canary code from an Asset
 */
class AssetCode extends Code {
    /**
     * @param assetPath The path to the asset file or directory.
     */
    constructor(assetPath, options) {
        super();
        this.assetPath = assetPath;
        this.options = options;
        if (!fs.existsSync(this.assetPath)) {
            throw new Error(`${this.assetPath} is not a valid path`);
        }
    }
    bind(scope, handler, family) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_RuntimeFamily(family);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        this.validateCanaryAsset(handler, family);
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.assetPath,
                ...this.options,
            });
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Validates requirements specified by the canary resource. For example, the canary code with handler `index.handler`
     * must be found in the file structure `nodejs/node_modules/index.js`.
     *
     * Requires path to be either zip file or directory.
     * Requires asset directory to have the structure 'nodejs/node_modules'.
     * Requires canary file to be directly inside node_modules folder.
     * Requires canary file name matches the handler name.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html
     *
     * @param handler the canary handler
     */
    validateCanaryAsset(handler, family) {
        if (path.extname(this.assetPath) !== '.zip') {
            if (!fs.lstatSync(this.assetPath).isDirectory()) {
                throw new Error(`Asset must be a .zip file or a directory (${this.assetPath})`);
            }
            const filename = handler.split('.')[0];
            const nodeFilename = `${filename}.js`;
            const pythonFilename = `${filename}.py`;
            if (family === runtime_1.RuntimeFamily.NODEJS && !fs.existsSync(path.join(this.assetPath, 'nodejs', 'node_modules', nodeFilename))) {
                throw new Error(`The canary resource requires that the handler is present at "nodejs/node_modules/${nodeFilename}" but not found at ${this.assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary_Nodejs.html)`);
            }
            if (family === runtime_1.RuntimeFamily.PYTHON && !fs.existsSync(path.join(this.assetPath, 'python', pythonFilename))) {
                throw new Error(`The canary resource requires that the handler is present at "python/${pythonFilename}" but not found at ${this.assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary_Python.html)`);
            }
        }
    }
}
_b = JSII_RTTI_SYMBOL_1;
AssetCode[_b] = { fqn: "@aws-cdk/aws-synthetics-alpha.AssetCode", version: "2.73.0-alpha.0" };
exports.AssetCode = AssetCode;
/**
 * Canary code from an inline string.
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        if (code.length === 0) {
            throw new Error('Canary inline code cannot be empty');
        }
    }
    bind(_scope, handler, _family) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_RuntimeFamily(_family);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        if (handler !== 'index.handler') {
            throw new Error(`The handler for inline code must be "index.handler" (got "${handler}")`);
        }
        return {
            inlineCode: this.code,
        };
    }
}
_c = JSII_RTTI_SYMBOL_1;
InlineCode[_c] = { fqn: "@aws-cdk/aws-synthetics-alpha.InlineCode", version: "2.73.0-alpha.0" };
exports.InlineCode = InlineCode;
/**
 * S3 bucket path to the code zip file
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.bucket = bucket;
        this.key = key;
        this.objectVersion = objectVersion;
    }
    bind(_scope, _handler, _family) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_RuntimeFamily(_family);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        return {
            s3Location: {
                bucketName: this.bucket.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
_d = JSII_RTTI_SYMBOL_1;
S3Code[_d] = { fqn: "@aws-cdk/aws-synthetics-alpha.S3Code", version: "2.73.0-alpha.0" };
exports.S3Code = S3Code;
//# sourceMappingURL=data:application/json;base64,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