"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Test = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk = require("aws-cdk-lib");
const synthetics_canned_metrics_generated_1 = require("aws-cdk-lib/aws-synthetics/lib/synthetics-canned-metrics.generated");
const aws_synthetics_1 = require("aws-cdk-lib/aws-synthetics");
/**
 * Specify a test that the canary should run
 */
class Test {
    /**
     * Specify a custom test with your own code
     *
     * @returns `Test` associated with the specified Code object
     * @param options The configuration options
     */
    static custom(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_CustomTestOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.custom);
            }
            throw error;
        }
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
}
_a = JSII_RTTI_SYMBOL_1;
Test[_a] = { fqn: "@aws-cdk/aws-synthetics-alpha.Test", version: "2.73.0-alpha.0" };
exports.Test = Test;
/**
 * Define a new Canary
 */
class Canary extends cdk.Resource {
    constructor(scope, id, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_CanaryProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Canary);
            }
            throw error;
        }
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.string({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = props.artifactsBucketLocation?.bucket ?? new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
            enforceSSL: true,
        });
        this.role = props.role ?? this.createDefaultRole(props);
        if (props.vpc) {
            // Security Groups are created and/or appended in `createVpcConfig`.
            this._connections = new ec2.Connections({});
        }
        const resource = new aws_synthetics_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject(props.artifactsBucketLocation?.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: props.startAfterCreation ?? true,
            runtimeVersion: props.runtime.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: props.failureRetentionPeriod?.toDays(),
            successRetentionPeriod: props.successRetentionPeriod?.toDays(),
            code: this.createCode(props),
            runConfig: this.createRunConfig(props),
            vpcConfig: this.createVpcConfig(props),
            deleteLambdaResourcesOnCanaryDeletion: props.enableAutoDeleteLambdas,
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Canary
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Canaries have security groups to manage. Supply the "vpc" parameter when creating the Canary.');
        }
        return this._connections;
    }
    /**
     * Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricDuration(options) {
        return new aws_cloudwatch_1.Metric({
            ...synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.durationMaximum({ CanaryName: this.canaryName }),
            ...{ statistic: 'Average' },
            ...options,
        }).attachTo(this);
    }
    /**
     * Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricSuccessPercent(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.successPercentAverage, options);
    }
    /**
     * Measure the number of failed canary runs over a given time period.
     *
     * Default: sum over 5 minutes
     *
     * @param options - configuration options for the metric
     */
    metricFailed(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.failedSum, options);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(props) {
        const prefix = props.artifactsBucketLocation?.prefix;
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.bucketArn],
                    actions: ['s3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: [this.logGroupArn()],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        const managedPolicies = [];
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
            managedPolicies,
        });
    }
    logGroupArn() {
        return cdk.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
            resourceName: '/aws/lambda/cwsyn-*',
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler, props.runtime.family),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: codeConfig.s3Location?.bucketName,
            s3Key: codeConfig.s3Location?.objectKey,
            s3ObjectVersion: codeConfig.s3Location?.objectVersion,
        };
    }
    createRunConfig(props) {
        if (!props.environmentVariables) {
            return undefined;
        }
        return {
            environmentVariables: props.environmentVariables,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        return {
            durationInSeconds: String(`${props.timeToLive?.toSeconds() ?? 0}`),
            expression: props.schedule?.expressionString ?? 'rate(5 minutes)',
        };
    }
    createVpcConfig(props) {
        if (!props.vpc) {
            if (props.vpcSubnets != null || props.securityGroups != null) {
                throw new Error("You must provide the 'vpc' prop when using VPC-related properties.");
            }
            return undefined;
        }
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnetIds.length < 1) {
            throw new Error('No matching subnets found in the VPC.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Canary ' + cdk.Names.uniqueId(this),
            });
            securityGroups = [securityGroup];
        }
        this._connections.addSecurityGroup(...securityGroups);
        return {
            vpcId: props.vpc.vpcId,
            subnetIds,
            securityGroupIds: cdk.Lazy.list({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = cdk.Names.uniqueId(this).toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
    cannedMetric(fn, props) {
        return new aws_cloudwatch_1.Metric({
            ...fn({ CanaryName: this.canaryName }),
            ...props,
        }).attachTo(this);
    }
}
_b = JSII_RTTI_SYMBOL_1;
Canary[_b] = { fqn: "@aws-cdk/aws-synthetics-alpha.Canary", version: "2.73.0-alpha.0" };
exports.Canary = Canary;
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,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