# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['protobuf_to_pydantic',
 'protobuf_to_pydantic.contrib',
 'protobuf_to_pydantic.get_desc',
 'protobuf_to_pydantic.get_desc.from_pb_option']

package_data = \
{'': ['*']}

install_requires = \
['grpcio-tools>=1.47.0,<2.0.0', 'pydantic>=1.9.1,<2.0.0']

extras_require = \
{'lark': ['lark>=1.1.2,<2.0.0']}

setup_kwargs = {
    'name': 'protobuf-to-pydantic',
    'version': '0.1.2',
    'description': 'Convert Protobuf-generated Python objects to Pydantic.BaseModel objects with parameter checksum',
    'long_description': '# protobuf_to_pydantic\nGenerate a `pydantic.BaseModel` with parameter verification function from the `Python` Message object(by the Protobuf file).\n\n> NOTE:\n>  - Only supports proto3\n>\n>  - The project is from [pait](https://github.com/so1n/pait) separated，Therefore, this project also generates the `pydantic.BaseModel` object in memory first, and then generates the corresponding `Python` code according to the generated object.\n\n# 1.Installation\n```bash\npip install protobuf_to_pydantic\n```\n\n# 2.Quick Start\n## 2.1.Generate a `pydantic.BaseModel` object at runtime\n`protobuf_to_pydantic` can generate the corresponding `pydantic.BaseModel` object based on the `Message` object at runtime。\n\nFor example, the `UserMessage` in the Protobuf file below：\n```protobuf\nsyntax = "proto3";\npackage user;\n\nenum SexType {\n  man = 0;\n  women = 1;\n}\n\nmessage UserMessage {\n  string uid=1;\n  int32 age=2;\n  float height=3;\n  SexType sex=4;\n  bool is_adult=5;\n  string user_name=6;\n}\n```\n`protobuf_to_pydantic` can read the Message object generated from the Proto file at runtime and generate the corresponding `pydantic.BaseModel` object:\n```Python\nfrom typing import Type\nfrom protobuf_to_pydantic import msg_to_pydantic_model\nfrom pydantic import BaseModel\n\n# import protobuf gen python obj\nfrom example.python_example_proto_code.example_proto.demo import demo_pb2\n\n\nUserModel: Type[BaseModel] = msg_to_pydantic_model(demo_pb2.UserMessage)\nprint(\n    {\n        k: v.field_info\n        for k, v in UserModel.__fields__.items()\n    }\n)\n\n# output\n# {\n#   \'uid\': FieldInfo(default=\'\', extra={}),\n#   \'age\': FieldInfo(default=0, extra={}),\n#   \'height\': FieldInfo(default=0.0, extra={}),\n#   \'sex\': FieldInfo(default=0, extra={}),\n#   \'is_adult\': FieldInfo(default=False, extra={}),\n#   \'user_name\': FieldInfo(default=\'\', extra={})\n#  }\n```\n## 2.2.Parameter verification\nThe `Message` object generated according to the Protobuf file only carries a small amount of information, and there is not enough information to make the generated `pydantic.BaseModel` have more detailed parameter verification functions, and some additional ways are needed to improve the data of the `Message` object.\nAt present, `protobuf_to_pydantic` supports multiple ways to obtain other information of Message, so that the generated `pydantic.BaseModel` object has the function of parameter verification.\n\n> NOTE: P2P mode is recommended\n### 2.2.1.Text annotation\nDevelopers can write comments that meet the requirements of `protobuf_to_pydantic` for each field in the Protobuf file to provide parameter verification information for `protobuf_to_pydantic`, such as the following example:\n```protobuf\nsyntax = "proto3";\npackage user;\n\nenum SexType {\n  man = 0;\n  women = 1;\n}\n\n// user info\nmessage UserMessage {\n  // p2p: {"miss_default": true, "example": "10086"}\n  // p2p: {"title": "UID"}\n  string uid=1; // p2p: {"description": "user union id"}\n  // p2p: {"example": 18, "title": "use age", "ge": 0}\n  int32 age=2;\n  // p2p: {"ge": 0, "le": 2.5}\n  float height=3;\n  SexType sex=4;\n  bool is_adult=5;\n  // p2p: {"description": "user name"}\n  // p2p: {"default": "", "min_length": 1, "max_length": "10", "example": "so1n"}\n  string user_name=6;\n}\n```\nIn this example, each annotation that can be used by `protobuf_to_pydantic` starts with `p2p:` (supports customization) and is followed by a complete Json string. If you are familiar with the usage of `pydantic`, you can find This Json string contains the verification information corresponding to `pydantic.Field`. For example, the `uid` field in `UserMessage` contains a total of 4 pieces of information as follows：\n\n| Column       | Meaning                                                                               |\n|--------------|---------------------------------------------------------------------------------------|\n| miss_default | Indicates that the generated field does not have a default value                      |\n| example      | An example value representing the generated field is 10086                            |\n| title        | Indicates that the schema name of the field is UID                                    |\n | description  | The schema documentation for the representation field is described as `user_union_id` |\n\n> Note:\n>   - 1.Currently only single-line comments are supported and comments must be a complete Json data (no line breaks).\n>   - 2.multi line comments are not supported。\n\nWhen these annotations are written, `protobuf_to_pydantic` will bring the corresponding information for each field when converting the Message into the corresponding `Pydantic.BaseModel` object, as follows:\n```python\nfrom typing import Type\nfrom protobuf_to_pydantic import msg_to_pydantic_model\nfrom pydantic import BaseModel\n\n# import protobuf gen python obj\nfrom example.python_example_proto_code.example_proto.demo import demo_pb2\n\nUserModel: Type[BaseModel] = msg_to_pydantic_model(demo_pb2.UserMessage, parse_msg_desc_method=demo_pb2)\nprint(\n    {\n        k: v.field_info\n        for k, v in UserModel.__fields__.items()\n    }\n)\n# output\n# {\n#   \'uid\': FieldInfo(default=PydanticUndefined, title=\'UID\', description=\'user union id\', extra={\'example\': \'10086\'}),\n#   \'age\': FieldInfo(default=0, title=\'use age\', ge=0, extra={\'example\': 18}),\n#   \'height\': FieldInfo(default=0.0, ge=0, le=2, extra={}),\n#   \'sex\': FieldInfo(default=0, extra={}),\n#   \'is_adult\': FieldInfo(default=False, extra={}),\n#   \'user_name\': FieldInfo(default=\'\', description=\'user name\', min_length=1, max_length=10, extra={\'example\': \'so1n\'})\n# }\n```\nIt can be seen from the output results that the output fields carry the corresponding information. In addition, the difference between this code and the above is that the `msg_to_pydantic_model` function sets a keyword parameter named `parse_msg_desc_method` and its value is `demo_pb2`, which enables `protobuf_to_pydantic` to obtain additional information for each field in the Message object through comments in the `.pyi` file of the `demo_pb2` module.\n\n> Note：This function requires the use of the [mypy-protobuf](https://github.com/nipunn1313/mypy-protobuf) plugin when generating the corresponding `Python` code from the Protobuf file, and the specified output path of the pyi file is the same as the generated `Python` code path to take effect at the same time.\n\nIn addition to obtaining comments through the `.pyi` file, `protobuf_to_pydantic` also supports setting the value of `parse_msg_desc_method` to the root directory path specified when the Message object is generated, so that `protobuf_to_pydantic` can parse the comments of the Protobuf file corresponding to the Message object. getting information。\n\n\nFor example, the project structure of `protobuf_to_pydantic` is as follows:\n```bash\n./protobuf_to_pydantic/\n├── example/\n│ ├── python_example_proto_code/\n│ └── example_proto/\n├── protobuf_to_pydantic/\n└── /\n```\nThe protobuf file is stored in the `example/example_proto` file, and then the `Python` code file corresponding to protobuf is generated by the following command in the `example` directory:\n```bash\ncd example\n\npython -m grpc_tools.protoc\n  --python_out=./python_example_proto_code \\\n  --grpc_python_out=./python_example_proto_code \\\n  -I. \\\n```\nThen the path to be filled in at this time is `./protobuf_to_pydantic/example`, the code is as follows：\n```python\nfrom typing import Type\nfrom protobuf_to_pydantic import msg_to_pydantic_model\nfrom pydantic import BaseModel\n\n# import protobuf gen python obj\nfrom example.python_example_proto_code.example_proto.demo import demo_pb2\n\nUserModel: Type[BaseModel] = msg_to_pydantic_model(\n    demo_pb2.UserMessage, parse_msg_desc_method="./protobuf_to_pydantic/example"\n)\nprint(\n    {\n        k: v.field_info\n        for k, v in UserModel.__fields__.items()\n    }\n)\n# output\n# {\n#   \'uid\': FieldInfo(default=PydanticUndefined, title=\'UID\', description=\'user union id\', extra={\'example\': \'10086\'}),\n#   \'age\': FieldInfo(default=0, title=\'use age\', ge=0, extra={\'example\': 18}),\n#   \'height\': FieldInfo(default=0.0, ge=0, le=2, extra={}),\n#   \'sex\': FieldInfo(default=0, extra={}),\n#   \'is_adult\': FieldInfo(default=False, extra={}),\n#   \'user_name\': FieldInfo(default=\'\', description=\'user name\', min_length=1, max_length=10, extra={\'example\': \'so1n\'})\n# }\n```\nFrom the result, it can be seen that the information carried by the field is the same as the result obtained by the module\n> NOTE: This method requires [lark](https://github.com/lark-parser/lark) to be installed in advance and the Protobuf file must exist in the running project.\n\n### 2.2.2.PGV\nAt present, the verification method of Protobuf-generated objects that is widely used is to directly use the [protoc-gen-validate](https://github.com/envoyproxy/protoc-gen-validate) project, so that the generated Message object has parameter verification logic.\nHowever, the implementation of `Pgv` in `Python` is quite special. It checks all possibilities every time it checks, so the performance is not very good, and the `pydantic.BaseModel` object generated by `protobuf_to_pydantic` itself It is allowed to carry some verification logic and most of the built-in verification logic is written in C, and the performance is relatively high.\n\n`protobuf_to_pydantic` supports parsing the `Pgv` verification information carried by each field in the Protobuf file so that the generated `pydantic.BaseModel` carries the verification logic corresponding to `Pgv`。\n\nIt is very simple to use `Pgv` check rules in `protobuf_to_pydantic`, as long as you write the corresponding `Pgv` rules in the Protobuf file, and then specify the value of `parse_msg_desc_method` as `PGV`, the code is as follows：\n```Python\nfrom typing import Type\nfrom protobuf_to_pydantic import msg_to_pydantic_model\nfrom pydantic import BaseModel\n\n# import protobuf gen python obj\nfrom example.python_example_proto_code.example_proto.validate import demo_pb2\n\nUserModel: Type[BaseModel] = msg_to_pydantic_model(\n    demo_pb2.FloatTest, parse_msg_desc_method="PGV"\n)\nprint(\n    {\n        k: v.field_info\n        for k, v in UserModel.__fields__.items()\n    }\n)\n# output\n# {\n#   \'const_test\': FieldInfo(default=1.0, const=True, extra={}),\n#   \'range_e_test\': FieldInfo(default=0.0, ge=1, le=10, extra={}),\n#   \'range_test\': FieldInfo(default=0.0, gt=1, lt=10, extra={}),\n#   \'in_test\': FieldInfo(default=0.0, extra={\'in\': [1.0, 2.0, 3.0]}),\n#   \'not_in_test\': FieldInfo(default=0.0, extra={\'not_in\': [1.0, 2.0, 3.0]}),\n#   \'ignore_test\': FieldInfo(default=0.0, extra={})\n# }\n```\n\n> Note:\n>  - 1.For the usage of `Pgv`, see: [protoc-gen-validate doc](https://github.com/envoyproxy/protoc-gen-validate/blob/main/README.md)\n>  - 2.Need to install `Pgv` through `pip install protoc_gen_validate` Or download [validate.proto](https://github.com/so1n/protobuf_to_pydantic/blob/master/example/example_proto/common/validate.proto) to the protobuf directory in the project to write pgv rules in the Protobuf file.\n\n\n### 2.2.3.P2p\nThe verification rules of `Pgv` are written in the Option attribute of each field of `Message`, and there are better code specifications, so the readability of Protobuf files carrying `Pgv` verification rules is higher than that of Protobuf carrying comments At the same time, when writing `Pgv` rules, you can also experience the convenience brought by IDE auto-completion, but it only supports verification-related logic, and the feature richness is not as good as the file comment mode.\n\nThe `P2P` mode combines the advantages of the `PGV` mode and the file comment mode. This mode satisfies most of the attribute customization of each `Field` in the `pydantic.BaseModel`, such as the following Protobuf file:\n```protobuf\nsyntax = "proto3";\npackage p2p_validate_test;\n\nimport "example_proto/common/p2p_validate.proto";\n\n\nmessage FloatTest {\n  float const_test = 1 [(p2p_validate.rules).float.const = 1];\n  float range_e_test = 2 [(p2p_validate.rules).float = {ge: 1, le: 10}];\n  float range_test = 3[(p2p_validate.rules).float = {gt: 1, lt: 10}];\n  float in_test = 4[(p2p_validate.rules).float = {in: [1,2,3]}];\n  float not_in_test = 5[(p2p_validate.rules).float = {not_in: [1,2,3]}];\n  float default_test = 6[(p2p_validate.rules).float.default = 1.0];\n  float not_enable_test = 7[(p2p_validate.rules).float.enable = false];\n  float default_factory_test = 8[(p2p_validate.rules).float.default_factory = "p2p@builtin|float"];\n  float miss_default_test = 9[(p2p_validate.rules).float.miss_default = true];\n  float alias_test = 10 [(p2p_validate.rules).float.alias = "alias"];\n  float desc_test = 11 [(p2p_validate.rules).float.description = "test desc"];\n  float multiple_of_test = 12 [(p2p_validate.rules).float.multiple_of = 3.0];\n  float example_test = 13 [(p2p_validate.rules).float.example = 1.0];\n  float example_factory = 14 [(p2p_validate.rules).float.example_factory = "p2p@builtin|float"];\n  float field_test = 15[(p2p_validate.rules).float.field = "p2p@local|CustomerField"];\n  float type_test = 16[(p2p_validate.rules).float.type = "p2p@local|confloat"];\n  float title_test = 17 [(p2p_validate.rules).float.title = "title_test"];\n}\n```\n`protobuf_to_pydantic` can read the generated Message object at runtime and generate a `pydantic.BaseModel` object with the corresponding information:\n```python\nfrom typing import Type\nfrom protobuf_to_pydantic import msg_to_pydantic_model\nfrom pydantic import BaseModel, confloat\nfrom pydantic.fields import FieldInfo\n\n# import protobuf gen python obj\nfrom example.python_example_proto_code.example_proto.p2p_validate import demo_pb2\n\n\nclass CustomerField(FieldInfo):\n    pass\n\n\nDemoModel: Type[BaseModel] = msg_to_pydantic_model(\n    demo_pb2.FloatTest,\n    local_dict={"CustomerField": CustomerField, "confloat": confloat},\n)\nprint(\n    {\n        k: v.field_info\n        for k, v in DemoModel.__fields__.items()\n    }\n)\n# output:\n# {\n#   \'const_test\': FieldInfo(default=1.0, const=True, extra={}),\n#   \'range_e_test\': FieldInfo(default=0.0, ge=1, le=10, extra={}),\n#   \'range_test\': FieldInfo(default=0.0, gt=1, lt=10, extra={}),\n#   \'in_test\': FieldInfo(default=0.0, extra={\'in\': [1.0, 2.0, 3.0]}),\n#   \'not_in_test\': FieldInfo(default=0.0, extra={\'not_in\': [1.0, 2.0, 3.0]}),\n#   \'default_test\': FieldInfo(default=1.0, extra={}),\n#   \'default_factory_test\': FieldInfo(default=PydanticUndefined, default_factory=<class \'float\'>, extra={}),\n#   \'miss_default_test\': FieldInfo(extra={}),\n#   \'alias_test\': FieldInfo(default=0.0, alias=\'alias\', alias_priority=2, extra={}),\n#   \'desc_test\': FieldInfo(default=0.0, description=\'test desc\', extra={}),\n#   \'multiple_of_test\': FieldInfo(default=0.0, multiple_of=3, extra={}),\n#   \'example_test\': FieldInfo(default=0.0, extra={\'example\': 1.0}),\n#   \'example_factory\': FieldInfo(default=0.0, extra={\'example\': <class \'float\'>}),\n#   \'field_test\': CustomerField(default=0.0, extra={}),\n#   \'type_test\': FieldInfo(default=0.0, extra={}),\n#   \'title_test\': FieldInfo(default=0.0, title=\'title_test\', extra={})\n#   }\n```\nIt is worth noting that this code does not explicitly specify that the value of `parse_msg_desc_method` is `p2p`, because `p2p` is already the default rule of `protobuf_to_pydantic`.\n\n> Note: See the template chapter for the usage of `local_dict`\n\n > Note:\n>  - 1.See the template chapter for the usage of `local_dict`\n>  - 2.If the reference to the Proto file fails, you need to download [p2p_validate.proto](https://github.com/so1n/protobuf_to_pydantic/blob/master/p2p_validate/p2p_validate.proto) in the project and use it in the Protobuf file。\n\n\n\n\n### 2.2.3.Other parameter support\nIn addition to the parameters of `FieldInfo`, the file comment mode and `p2p` mode of `protobuf_to_pydantic` also support the following parameters:\n- miss_default：By default, the default values generated for each field in the `pydantic.BaseModel` object are the same as the default values for each field in the Message. However, you can cancel the default value setting by setting `miss default` to `true`. It should be noted that when `miss default` is set to `true`, the `default` parameter will be invalid.。\n- enable: By default, `pydantic.BaseModel` will convert every field in the Message. If some fields do not want to be converted, you can set `enable` to `false`\n- const: Specifies the value of the field\'s constant. Note: The const of `pydantic.BaseModel` only supports bool variables. When `const` is `True`, the accepted value can only be the value set by `default`, and the default value carried by the Message generated by protobuf corresponds to The null value of type does not match `pydantic.BaseModel`, so `protobuf_to_pydantic` makes some changes to the input of this value, but after `const` sets the value, the `cost` property in the generated field is `True` `, and `default` becomes the corresponding value of the setting.\n- type: To expand the current type, for example, if you want to increase the verification of the bank card number through the `pydantic.types.Payment Card Number` type, you can specify the field type as `Payment Card Number` by the following method:\n  ```protobuf\n  message UserPayMessage {\n    string bank_number=1; // p2p: {"type": "p2p@import|pydantic.types|PaymentCardNumber"}\n  }\n  ```\n\n> Note:\n>   If you don\'t know `pydantic`, you can use the following two URLs to learn what parameters Field supports:\n>\n>   - https://pydantic-docs.helpmanual.io/usage/types/#constrained-types\n>\n>   - https://pydantic-docs.helpmanual.io/usage/schema/#field-customization\n\n### 2.2.4.Template\nIn some cases, the value we fill in is a method or function of a library, such as the value of the `type` parameter and the `default_factory` parameter, but it cannot be implemented through Json syntax. In this case, template parameters can be used. Currently, `protobuf_to_pydantic` supports a variety of template parameters.\n\n#### 2.2.4.1.`p2p@import`\nThis template is used to import variables under other modules. The specific usage is as follows:\n```protobuf\n// comment example\nmessage UserPayMessage {\n  string bank_number=1; // p2p: {"type": "p2p@import|pydantic.types|PaymentCardNumber"}\n}\n// p2p example\nmessage UserPayMessage {\n  string bank_number=1[(p2p_validate.rules).string.pydantic_type = "PaymentCardNumber"];\n}\n// p2p another examlpe\nmessage UserPayMessage {\n  string bank_number=1[(p2p_validate.rules).string.type = "p2p@import|pydantic.types|PaymentCardNumber"];\n}\n```\nThe syntax used here is the format of `{p2p method}|{imported class:A}|{class module:B}`, where the method `p2p@import` at the beginning indicates that it needs to pass `from A import B` introduces the `B` object. Through the definition of the template, `protobuf_to_pydantic` will convert the corresponding Message to the following `pydantic.BaseModel`:\n```python\nfrom pydantic import BaseModel\nfrom pydantic.fields import FieldInfo\n# p2p@import|pydantic.types|PaymentCardNumber\nfrom pydantic.types import PaymentCardNumber\n\nclass UserPayMessage(BaseModel):\n    bank_number: PaymentCardNumber = FieldInfo(default="", extra={})\n```\n\n#### 2.2.4.2.`p2p@import_instance`\n`p2p@import` is just to import the variables of a certain library, and `p2p@import_instance` is to import the class of a certain library first, and then instantiate it with the specified parameters. The usage is as follows:\n```protobuf\nmessage AnyTest {\n  google.protobuf.Any default_test = 23 [\n    (p2p_validate.rules).any.default = \'p2p@import_instance|google.protobuf.any_pb2|Any|{"type_url": "type.googleapis.com/google.protobuf.Duration"}\'\n\n  ];\n}\n```\nHere is the `{p2p method}|{module to be imported}|{corresponding class}|{corresponding parameter}` syntax, through the definition of the template, `protobuf_to_pydantic` will convert the corresponding Message is the following `pydantic.BaseModel` object:\n```python\nfrom google.protobuf.any_pb2 import Any as AnyMessage\nfrom pydantic import BaseModel\nfrom pydantic.fields import FieldInfo\n\n\nclass AnyTest(BaseModel):\n    default_test: AnyMessage = FieldInfo(\n        default=AnyMessage(type_url="type.googleapis.com/google.protobuf.Duration")\n    )\n```\n\n#### 2.2.4.3.`p2p@local`\nThis template is used to introduce user-defined variables. Here, the syntax of the `{p2p method}|{local variable to be used}` format is used, as follows:\n```protobuf\n// comment example\nmessage UserPayMessage {\n  google.protobuf.Timestamp exp=1; // p2p: {"default_factory": "p2p@local|exp_time"}\n}\n```\nThen register the corresponding value through the parameter `local_dict` when calling the `msg_to_pydantic_model` method. The fake code is as follows:\n```Python\ndef exp_time() -> float:\n  return time.time()\n\nmsg_to_pydantic_model(\n    demo_pb2.NestedMessage,\n    local_dict={"exp_time": exp_time},  # <----\n)\n```\nIn this way, `protobuf_to_pydantic` can generate a `pydantic.BaseModel` object that meets the requirements:\n```python\nfrom datetime import datetime\nfrom pydantic import BaseModel\nfrom pydantic.fields import FieldInfo\n\nfrom . import exp_time\n\nclass UserPayMessage(BaseModel):\n    exp: datetime = FieldInfo(default_factory=exp_time, extra={})\n```\n\n> Note: See the sample code for specific calling and generation methods.\n\n#### 2.2.4.4.`p2p@builtin`\nWhen the variable to be used comes from a built-in function, the variable can be used directly (it can be considered as a simplified version of `p2p@local`), the syntax is as follows:\n```protobuf\n// comment example\nmessage UserPayMessage {\n  google.protobuf.Timestamp exp=1; // p2p: {"type": "p2p@builtin|float"}\n}\n```\nThe generated `pydantic.BaseModel` object is as follows:\n```python\nfrom pydantic import BaseModel\nfrom pydantic.fields import FieldInfo\n\n\nclass UserPayMessage(BaseModel):\n    exp: float = FieldInfo()\n```\n#### 2.2.4.5.Custom template\nCurrently, `protobuf_to_pydantic` only supports several templates. If you have more template requirements, you can extend the template by inheriting the `DescTemplate` class.\nFor example, there is a strange requirement that the default value of the field is the timestamp when the Message object is generated as a `pydantic.BaseModel` object, but the timestamp has two versions, one version has a timestamp of length 10 and the other has a length of 13, so write the following Protobuf file:\n```protobuf\nmessage TimestampTest{\n  int32 timestamp_10 = 1[(p2p_validate.rules).int32.default = "p2p@timestamp|10"];\n  int32 timestamp_13 = 2[(p2p_validate.rules).int32.default = "p2p@timestamp|13"];\n}\n```\nThis file uses the syntax of `p2p@timestamp|{x}`, where `x` has only two values 10 and 13, and then you can write code according to this template behavior, the code is as follows:\n```python\nimport time\nfrom typing import Any, List\nfrom protobuf_to_pydantic.gen_model import DescTemplate\n\n\nclass CustomDescTemplate(DescTemplate):\n    def template_timestamp(self, template_var_list: List[str]) -> Any:\n        timestamp: float = time.time()\n        length: str = template_var_list[0]\n        if length == "10":\n            return int(timestamp)\n        elif length == "13":\n            return int(timestamp * 100)\n        else:\n            raise KeyError(f"timestamp template not support value:{length}")\n\n\nfrom .demo_pb2 import TimestampTest # fake code\nfrom protobuf_to_pydantic import msg_to_pydantic_model\n\nmsg_to_pydantic_model(\n    TimestampTest,\n    desc_template=CustomDescTemplate\n)\n```\nThe code creates a `CustomDescTemplate` class that inherits from `DescTemplate`, this class adds a `template_timestamp` method to match the `p2p@timestamp` template, and then passes the `desc_template` key parameter in `msg_to_pydantic_model` to specify the template class to use as `CustomDescTemplate`, so that `msg_to_pydantic_model` will generate the following code (assuming the code generated when the timestamp is 1600000000):\n```python\nfrom pydantic import BaseModel\nfrom pydantic.fields import FieldInfo\n\nclass TimestampTest(BaseModel):\n    timestamp_10: int = FieldInfo(default=1600000000)\n    timestamp_13: int = FieldInfo(default=1600000000000)\n```\n\n## 2.3.Generate the corresponding Python code\nIn addition to generating corresponding `pydantic.BaseModel` objects at runtime, `protobuf_to_pydantic` supports converting `pydantic.BaseModel` objects to `Python` code at runtime (only for `protobuf_to_pydantic` generation `pydantic.BaseModel` object).\n\nAmong them, `protobuf_to_pydantic.pydantic_model_to_py_code` is used to generate the code text, `protobuf_to_pydantic.pydantic_model_to_py_file` is used to generate the code file, and the example code of `protobuf_to_pydantic.pydantic_model_to_py_file` as follows:\n```Python\nfrom protobuf_to_pydantic import msg_to_pydantic_model, pydantic_model_to_py_file\n\n# import protobuf gen python obj\nfrom example.python_example_proto_code.example_proto.demo import demo_pb2\n\npydantic_model_to_py_file(\n    "./demo_gen_code.py",\n    msg_to_pydantic_model(demo_pb2.NestedMessage),\n)\n```\nNote that if `protobuf_to_pydantic` checks that the current environment has `isort` and `autoflake` installed, they will be used to format code by default.\n\n## 3.example\n`protobuf_to_pydantic` provides some simple sample code, the following is the path of the sample code and protobuf file, just for reference:\n\n| Implication                           | Example Protobuf                                                                            | Example code                                                                         |\n|------------------------------|---------------------------------------------------------------------------------------------|--------------------------------------------------------------------------------------|\n| Generate Model code with validation rules based on p2p schema | https://github.com/so1n/protobuf_to_pydantic/tree/master/example/example_proto/p2p_validate | https://github.com/so1n/protobuf_to_pydantic/tree/master/example/p2p_validate_example |\n| Generate the basic Model code               | https://github.com/so1n/protobuf_to_pydantic/tree/master/example/example_proto/demo         | https://github.com/so1n/protobuf_to_pydantic/tree/master/example/simple_example      |\n| Generate Model code with validation rules from .pyi files     | https://github.com/so1n/protobuf_to_pydantic/tree/master/example/example_proto/demo         | https://github.com/so1n/protobuf_to_pydantic/tree/master/example/text_comment_example |\n| Generate Model code with validation rules from protobuf files | https://github.com/so1n/protobuf_to_pydantic/tree/master/example/example_proto/validate     | https://github.com/so1n/protobuf_to_pydantic/tree/master/example/validate_example    |\n',
    'author': 'So1n',
    'author_email': 'so1n897046026@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/so1n/protobuf_to_pydantic',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
