"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.FeatureFlags = void 0;
const ajv_1 = __importDefault(require("ajv"));
const fs = __importStar(require("fs-extra"));
const path = __importStar(require("path"));
const lodash_1 = __importDefault(require("lodash"));
const __1 = require("..");
const featureFlagFileProvider_1 = require("./featureFlagFileProvider");
const featureFlagEnvironmentProvider_1 = require("./featureFlagEnvironmentProvider");
const pathManager_1 = require("../state-manager/pathManager");
const state_manager_1 = require("../state-manager");
const jsonUtilities_1 = require("../jsonUtilities");
class FeatureFlags {
    constructor(environmentProvider, projectPath, useNewDefaults) {
        this.environmentProvider = environmentProvider;
        this.projectPath = projectPath;
        this.useNewDefaults = useNewDefaults;
        this.registrations = new Map();
        this.effectiveFlags = {};
        this.newProjectDefaults = {};
        this.existingProjectDefaults = {};
        this.getValue = (flagName, type) => {
            var _a;
            if (!flagName) {
                throw new Error(`'flagName' argument is required`);
            }
            let value;
            const parts = flagName.toLowerCase().split('.');
            if (parts.length !== 2) {
                throw new Error(`Invalid flagName value: '${flagName}'`);
            }
            const sectionRegistration = this.registrations.get(parts[0]);
            if (!sectionRegistration) {
                throw new Error(`Section '${parts[0]}' is not registered in feature provider`);
            }
            const flagRegistrationEntry = sectionRegistration.find(flag => flag.name === parts[1]);
            if (!flagRegistrationEntry) {
                throw new Error(`Flag '${parts[1]}' within '${parts[0]}' is not registered in feature provider`);
            }
            if (flagRegistrationEntry.type !== type) {
                throw new Error(`'${flagName}' is a ${flagRegistrationEntry.type} type, not ${type}`);
            }
            value = (_a = this.effectiveFlags[parts[0]]) === null || _a === void 0 ? void 0 : _a[parts[1]];
            if (value === undefined) {
                if (this.useNewDefaults) {
                    value = flagRegistrationEntry.defaultValueForNewProjects;
                }
                else {
                    value = flagRegistrationEntry.defaultValueForExistingProjects;
                }
            }
            return value;
        };
        this.buildJSONSchemaFromRegistrations = () => {
            return [...this.registrations.entries()].reduce((s, r) => {
                var _a;
                const currentSection = ((_a = s.properties[r[0].toLowerCase()]) !== null && _a !== void 0 ? _a : {
                    type: 'object',
                    additionalProperties: false,
                });
                currentSection.properties = r[1].reduce((p, fr) => {
                    p[fr.name.toLowerCase()] = {
                        type: fr.type,
                        default: fr.defaultValueForNewProjects,
                    };
                    return p;
                }, {});
                s.properties[r[0].toLowerCase()] = currentSection;
                return s;
            }, {
                $schema: 'http://json-schema.org/draft-07/schema#',
                type: 'object',
                additionalProperties: false,
                properties: {},
            });
        };
        this.buildDefaultValues = () => {
            this.newProjectDefaults = [...this.registrations.entries()].reduce((result, r) => {
                const nest = r[1].reduce((p, fr) => {
                    p[fr.name] = fr.defaultValueForNewProjects;
                    return p;
                }, {});
                result[r[0]] = {
                    ...result[r[0]],
                    ...nest,
                };
                return result;
            }, {});
            this.existingProjectDefaults = [...this.registrations.entries()].reduce((result, r) => {
                const nest = r[1].reduce((p, fr) => {
                    p[fr.name] = fr.defaultValueForExistingProjects;
                    return p;
                }, {});
                result[r[0]] = {
                    ...result[r[0]],
                    ...nest,
                };
                return result;
            }, {});
        };
        this.validateFlags = (allFlags) => {
            const schema = this.buildJSONSchemaFromRegistrations();
            const ajv = new ajv_1.default({
                allErrors: true,
            });
            const schemaValidate = ajv.compile(schema);
            const validator = (target, flags) => {
                var _a;
                const valid = schemaValidate(flags);
                if (!valid && schemaValidate.errors) {
                    const unknownFlags = [];
                    const otherErrors = [];
                    for (const error of schemaValidate.errors) {
                        if (error.keyword === 'additionalProperties') {
                            const additionalProperty = (_a = error.params) === null || _a === void 0 ? void 0 : _a.additionalProperty;
                            let flagName = error.dataPath.length > 0 && error.dataPath[0] === '.' ? `${error.dataPath.slice(1)}.` : '';
                            if (additionalProperty) {
                                flagName += additionalProperty;
                            }
                            if (flagName.length > 0) {
                                unknownFlags.push(flagName);
                            }
                        }
                        else {
                            const errorMessage = error.dataPath.length > 0 && error.dataPath[0] === '.'
                                ? `${error.dataPath.slice(1)}: ${error.message}`
                                : `${error.dataPath}: ${error.message}`;
                            otherErrors.push(errorMessage);
                        }
                    }
                    throw new __1.JSONValidationError('Invalid feature flag configuration', unknownFlags, otherErrors);
                }
            };
            const featureFlagsValidator = (type, features) => {
                validator(`${type} project`, features.project);
                for (let env of Object.keys(features.environments)) {
                    validator(`${type} environment (${env})`, features.environments[env]);
                }
            };
            for (let flagItem of allFlags) {
                featureFlagsValidator(flagItem.name, flagItem.flags);
            }
        };
        this.transformEnvFlags = (features) => {
            const convertValue = (section, flagName, value) => {
                const sectionRegistration = this.registrations.get(section);
                if (!sectionRegistration) {
                    throw new Error(`Section '${section}' is not registered in feature provider`);
                }
                const flagRegistrationEntry = sectionRegistration.find(flag => flag.name === flagName);
                if (!flagRegistrationEntry) {
                    throw new Error(`Flag '${flagName}' within '${section}' is not registered in feature provider`);
                }
                switch (flagRegistrationEntry.type) {
                    case 'boolean':
                        if (value === 'true') {
                            return true;
                        }
                        else if (value === 'false') {
                            return false;
                        }
                        else {
                            throw new Error(`Invalid boolean value: '${value}' for '${flagName}' in section '${section}'`);
                        }
                    case 'string':
                        return value.toString();
                    case 'number': {
                        const n = Number.parseInt(value, 10);
                        if (!Number.isNaN(n)) {
                            return n;
                        }
                        else {
                            throw new Error(`Invalid number value: '${value}' for '${flagName}' in section '${section}'`);
                        }
                    }
                    default:
                        throw new Error(`Invalid number value: ${value} for ${flagName}`);
                }
            };
            const mapFeatureFlagEntry = (input) => Object.keys(input).reduce((result, section) => {
                const sourceObject = input[section];
                result[section] = Object.keys(sourceObject).reduce((resultFlag, flagName) => {
                    const sourceValue = sourceObject[flagName];
                    resultFlag[flagName] = convertValue(section, flagName, sourceValue);
                    return resultFlag;
                }, {});
                return result;
            }, {});
            features.project = mapFeatureFlagEntry(features.project);
            for (let env of Object.keys(features.environments)) {
                features.environments[env] = mapFeatureFlagEntry(features.environments[env]);
            }
            return features;
        };
        this.loadValues = async () => {
            var _a, _b;
            const fileFlags = await this.fileValueProvider.load();
            const envFlags = this.transformEnvFlags(await this.envValueProvider.load());
            this.validateFlags([
                {
                    name: 'File',
                    flags: fileFlags,
                },
                {
                    name: 'Environment',
                    flags: envFlags,
                },
            ]);
            this.buildDefaultValues();
            this.effectiveFlags = lodash_1.default.merge(this.useNewDefaults ? this.newProjectDefaults : this.existingProjectDefaults, fileFlags.project, (_a = fileFlags.environments[this.environmentProvider.getCurrentEnvName()]) !== null && _a !== void 0 ? _a : {}, envFlags.project, (_b = envFlags.environments[this.environmentProvider.getCurrentEnvName()]) !== null && _b !== void 0 ? _b : {});
        };
        this.registerFlag = (section, flags) => {
            var _a;
            if (!section) {
                throw new Error(`'section' argument is required`);
            }
            if (!flags) {
                throw new Error(`'flags' argument is required`);
            }
            const newFlags = (_a = this.registrations.get(section.toLowerCase())) !== null && _a !== void 0 ? _a : new Array();
            for (let flag of flags) {
                if (!flag.name || flag.name.trim().length === 0) {
                    throw new Error(`Flag does not have a name specified`);
                }
                if (newFlags.find(f => f.name === flag.name.toLowerCase())) {
                    throw new Error(`Flag with name: '${flag.name}' is already registered in section: '${section}'`);
                }
                flag.name = flag.name.toLowerCase();
                newFlags.push(flag);
            }
            this.registrations.set(section.toLowerCase(), newFlags);
        };
        this.registerFlags = () => {
            this.registerFlag('graphQLTransformer', [
                {
                    name: 'addMissingOwnerFields',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'improvePluralization',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'validateTypeNameReservedWords',
                    type: 'boolean',
                    defaultValueForExistingProjects: true,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'useExperimentalPipelinedTransformer',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: false,
                },
                {
                    name: 'enableIterativeGSIUpdates',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'secondaryKeyAsGSI',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'skipOverrideMutationInputTypes',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
            ]);
            this.registerFlag('frontend-ios', [
                {
                    name: 'enableXcodeIntegration',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
            ]);
            this.registerFlag('auth', [
                {
                    name: 'enableCaseInsensitivity',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'useInclusiveTerminology',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'breakCircularDependency',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
            ]);
            this.registerFlag('codegen', [
                {
                    name: 'useAppSyncModelgenPlugin',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'useDocsGeneratorPlugin',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'useTypesGeneratorPlugin',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'cleanGeneratedModelsDirectory',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'retainCaseStyle',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'addTimestampFields',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'handleListNullabilityTransparently',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'emitAuthProvider',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'generateIndexRules',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
                {
                    name: 'enableDartNullSafety',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true
                }
            ]);
            this.registerFlag('appSync', [
                {
                    name: 'generateGraphQLPermissions',
                    type: 'boolean',
                    defaultValueForExistingProjects: false,
                    defaultValueForNewProjects: true,
                },
            ]);
        };
    }
}
exports.FeatureFlags = FeatureFlags;
FeatureFlags.initialize = async (environmentProvider, useNewDefaults = false, additionalFlags) => {
    var _a;
    if (typeof jest === 'undefined' && FeatureFlags.instance) {
        throw new Error('FeatureFlags can only be initialzied once');
    }
    if (!environmentProvider) {
        throw new Error(`'environmentProvider' argument is required`);
    }
    const projectPath = (_a = pathManager_1.pathManager.findProjectRoot()) !== null && _a !== void 0 ? _a : process.cwd();
    await FeatureFlags.removeOriginalConfigFile(projectPath);
    const instance = new FeatureFlags(environmentProvider, projectPath, useNewDefaults);
    instance.registerFlags();
    if (additionalFlags) {
        for (const sectionName of Object.keys(additionalFlags)) {
            const flags = additionalFlags[sectionName];
            instance.registerFlag(sectionName, flags);
        }
    }
    instance.fileValueProvider = new featureFlagFileProvider_1.FeatureFlagFileProvider(environmentProvider, {
        projectPath,
    });
    instance.envValueProvider = new featureFlagEnvironmentProvider_1.FeatureFlagEnvironmentProvider({
        projectPath,
    });
    await instance.loadValues();
    FeatureFlags.instance = instance;
};
FeatureFlags.ensureDefaultFeatureFlags = async (newProject) => {
    FeatureFlags.ensureInitialized();
    let config = state_manager_1.stateManager.getCLIJSON(FeatureFlags.instance.projectPath, undefined, {
        throwIfNotExist: false,
        preserveComments: true,
    });
    if (!config || !config.features) {
        config = {
            ...(config !== null && config !== void 0 ? config : {}),
            features: newProject ? FeatureFlags.getNewProjectDefaults() : FeatureFlags.getExistingProjectDefaults(),
        };
        state_manager_1.stateManager.setCLIJSON(FeatureFlags.instance.projectPath, config);
    }
};
FeatureFlags.getBoolean = (flagName) => {
    FeatureFlags.ensureInitialized();
    return FeatureFlags.instance.getValue(flagName, 'boolean');
};
FeatureFlags.getString = (flagName) => {
    FeatureFlags.ensureInitialized();
    return FeatureFlags.instance.getValue(flagName, 'string');
};
FeatureFlags.getNumber = (flagName) => {
    FeatureFlags.ensureInitialized();
    return FeatureFlags.instance.getValue(flagName, 'number');
};
FeatureFlags.getEffectiveFlags = () => {
    FeatureFlags.ensureInitialized();
    return FeatureFlags.instance.effectiveFlags;
};
FeatureFlags.getNewProjectDefaults = () => {
    FeatureFlags.ensureInitialized();
    return FeatureFlags.instance.newProjectDefaults;
};
FeatureFlags.getExistingProjectDefaults = () => {
    FeatureFlags.ensureInitialized();
    return FeatureFlags.instance.existingProjectDefaults;
};
FeatureFlags.removeFeatureFlagConfiguration = async (removeProjectConfiguration, envNames) => {
    FeatureFlags.ensureInitialized();
    if (!envNames) {
        throw new Error(`'envNames' argument is required`);
    }
    if (removeProjectConfiguration) {
        const configFileName = pathManager_1.pathManager.getCLIJSONFilePath(FeatureFlags.instance.projectPath);
        await fs.remove(configFileName);
    }
    for (let envName of envNames) {
        const configFileName = pathManager_1.pathManager.getCLIJSONFilePath(FeatureFlags.instance.projectPath, envName);
        await fs.remove(configFileName);
    }
};
FeatureFlags.isInitialized = () => {
    return FeatureFlags.instance !== undefined;
};
FeatureFlags.reloadValues = async () => {
    FeatureFlags.ensureInitialized();
    await FeatureFlags.instance.loadValues();
};
FeatureFlags.removeOriginalConfigFile = async (projectPath) => {
    const originalConfigFileName = 'amplify.json';
    try {
        if (!projectPath) {
            return;
        }
        const originalConfigFilePath = path.join(projectPath, originalConfigFileName);
        const configFileData = jsonUtilities_1.JSONUtilities.readJson(originalConfigFilePath, {
            throwIfNotExist: false,
        });
        if ((configFileData === null || configFileData === void 0 ? void 0 : configFileData.features) !== undefined) {
            fs.removeSync(originalConfigFilePath);
        }
    }
    catch (_a) {
    }
};
FeatureFlags.ensureInitialized = () => {
    if (!FeatureFlags.instance) {
        throw new Error('FeatureFlags is not initialized');
    }
};
//# sourceMappingURL=featureFlags.js.map