"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TransformerFeatureFlagProvider = exports.SchemaTransformer = void 0;
const fs = require("fs");
const path_1 = require("path");
const graphql_auth_transformer_1 = require("graphql-auth-transformer");
const graphql_connection_transformer_1 = require("graphql-connection-transformer");
const graphql_dynamodb_transformer_1 = require("graphql-dynamodb-transformer");
const graphql_http_transformer_1 = require("graphql-http-transformer");
const graphql_key_transformer_1 = require("graphql-key-transformer");
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphql_ttl_transformer_1 = require("graphql-ttl-transformer");
const graphql_versioned_transformer_1 = require("graphql-versioned-transformer");
const cdk_transformer_1 = require("./cdk-transformer");
const custom_vtl_transformer_1 = require("./custom-vtl-transformer");
// Import this way because FunctionTransformer.d.ts types were throwing an eror. And we didn't write this package so hope for the best :P
// eslint-disable-next-line
const { FunctionTransformer } = require('graphql-function-transformer');
class SchemaTransformer {
    constructor(props) {
        var _a, _b, _c, _d;
        this.schemaPath = (_a = props.schemaPath) !== null && _a !== void 0 ? _a : './schema.graphql';
        this.outputPath = (_b = props.outputPath) !== null && _b !== void 0 ? _b : './appsync';
        this.isSyncEnabled = (_c = props.syncEnabled) !== null && _c !== void 0 ? _c : false;
        this.customVtlTransformerRootDirectory = (_d = props.customVtlTransformerRootDirectory) !== null && _d !== void 0 ? _d : process.cwd();
        this.outputs = {};
        this.resolvers = {};
        // TODO: Make this better?
        this.authTransformerConfig = {
            authConfig: {
                defaultAuthentication: {
                    authenticationType: 'AMAZON_COGNITO_USER_POOLS',
                    userPoolConfig: {
                        userPoolId: '12345xyz',
                    },
                },
                additionalAuthenticationProviders: [
                    {
                        authenticationType: 'API_KEY',
                        apiKeyConfig: {
                            description: 'Testing',
                            apiKeyExpirationDays: 100,
                        },
                    },
                    {
                        authenticationType: 'AWS_IAM',
                    },
                    {
                        authenticationType: 'OPENID_CONNECT',
                        openIDConnectConfig: {
                            name: 'OIDC',
                            issuerUrl: 'https://cognito-idp.us-east-1.amazonaws.com/us-east-1_XXX',
                        },
                    },
                ],
            },
        };
    }
    transform(preCdkTransformers = [], postCdkTransformers = []) {
        var _a, _b, _c;
        const transformConfig = this.isSyncEnabled ? this.loadConfigSync() : {};
        const provider = new TransformerFeatureFlagProvider();
        // Note: This is not exact as we are omitting the @searchable transformer as well as some others.
        const transformer = new graphql_transformer_core_1.GraphQLTransform({
            transformConfig: transformConfig,
            featureFlags: provider,
            transformers: [
                new graphql_dynamodb_transformer_1.DynamoDBModelTransformer(),
                new graphql_ttl_transformer_1.default(),
                new graphql_versioned_transformer_1.VersionedModelTransformer(),
                new FunctionTransformer(),
                new graphql_key_transformer_1.KeyTransformer(),
                new graphql_connection_transformer_1.ModelConnectionTransformer(),
                new graphql_auth_transformer_1.ModelAuthTransformer(this.authTransformerConfig),
                new graphql_http_transformer_1.HttpTransformer(),
                new custom_vtl_transformer_1.CustomVTLTransformer(this.customVtlTransformerRootDirectory),
                ...preCdkTransformers,
                new cdk_transformer_1.CdkTransformer(),
                ...postCdkTransformers,
            ],
        });
        const schema = fs.readFileSync(this.schemaPath);
        const cfdoc = transformer.transform(schema.toString());
        // TODO: Get Unauth Role and Auth Role policies for authorization stuff
        this.unauthRolePolicy = ((_a = cfdoc.rootStack.Resources) === null || _a === void 0 ? void 0 : _a.UnauthRolePolicy01) || undefined;
        this.authRolePolicy = ((_b = cfdoc.rootStack.Resources) === null || _b === void 0 ? void 0 : _b.AuthRolePolicy01) || undefined;
        this.writeSchema(cfdoc.schema);
        this.writeResolversToFile(cfdoc.resolvers);
        // Outputs shouldn't be null but default to empty map
        this.outputs = (_c = cfdoc.rootStack.Outputs) !== null && _c !== void 0 ? _c : {};
        return this.outputs;
    }
    /**
     * Gets the resolvers from the `./appsync/resolvers` folder
     * @returns all resolvers
     */
    getResolvers() {
        const statements = ['Query', 'Mutation'];
        const resolversDirPath = path_1.normalize('./appsync/resolvers');
        if (fs.existsSync(resolversDirPath)) {
            const files = fs.readdirSync(resolversDirPath);
            files.forEach(file => {
                // Example: Mutation.createChannel.response
                let args = file.split('.');
                let typeName = args[0];
                let fieldName = args[1];
                let templateType = args[2]; // request or response
                // default to composite key of typeName and fieldName, however if it
                // is Query, Mutation or Subscription (top level) the compositeKey is the
                // same as fieldName only
                let compositeKey = `${typeName}${fieldName}`;
                if (statements.indexOf(typeName) >= 0) {
                    if (!this.outputs.noneResolvers || !this.outputs.noneResolvers[compositeKey])
                        compositeKey = fieldName;
                }
                let filepath = path_1.normalize(`${resolversDirPath}/${file}`);
                if (statements.indexOf(typeName) >= 0 || (this.outputs.noneResolvers && this.outputs.noneResolvers[compositeKey])) {
                    if (!this.resolvers[compositeKey]) {
                        this.resolvers[compositeKey] = {
                            typeName: typeName,
                            fieldName: fieldName,
                        };
                    }
                    if (templateType === 'req') {
                        this.resolvers[compositeKey].requestMappingTemplate = filepath;
                    }
                    else if (templateType === 'res') {
                        this.resolvers[compositeKey].responseMappingTemplate = filepath;
                    }
                }
                else if (this.isHttpResolver(typeName, fieldName)) {
                    if (!this.resolvers[compositeKey]) {
                        this.resolvers[compositeKey] = {
                            typeName: typeName,
                            fieldName: fieldName,
                        };
                    }
                    if (templateType === 'req') {
                        this.resolvers[compositeKey].requestMappingTemplate = filepath;
                    }
                    else if (templateType === 'res') {
                        this.resolvers[compositeKey].responseMappingTemplate = filepath;
                    }
                }
                else { // This is a GSI
                    if (!this.resolvers.gsi) {
                        this.resolvers.gsi = {};
                    }
                    if (!this.resolvers.gsi[compositeKey]) {
                        this.resolvers.gsi[compositeKey] = {
                            typeName: typeName,
                            fieldName: fieldName,
                            tableName: fieldName.charAt(0).toUpperCase() + fieldName.slice(1),
                        };
                    }
                    if (templateType === 'req') {
                        this.resolvers.gsi[compositeKey].requestMappingTemplate = filepath;
                    }
                    else if (templateType === 'res') {
                        this.resolvers.gsi[compositeKey].responseMappingTemplate = filepath;
                    }
                }
            });
        }
        return this.resolvers;
    }
    /**
     * decides if this is a resolver for an HTTP datasource
     * @param typeName
     * @param fieldName
     */
    isHttpResolver(typeName, fieldName) {
        if (!this.outputs.httpResolvers)
            return false;
        for (const endpoint in this.outputs.httpResolvers) {
            for (const resolver of this.outputs.httpResolvers[endpoint]) {
                if (resolver.typeName === typeName && resolver.fieldName === fieldName)
                    return true;
            }
        }
        return false;
    }
    /**
       * Writes the schema to the output directory for use with @aws-cdk/aws-appsync
       * @param schema
       */
    writeSchema(schema) {
        if (!fs.existsSync(this.outputPath)) {
            fs.mkdirSync(this.outputPath);
        }
        fs.writeFileSync(`${this.outputPath}/schema.graphql`, schema);
    }
    /**
       * Writes all the resolvers to the output directory for loading into the datasources later
       * @param resolvers
       */
    writeResolversToFile(resolvers) {
        if (!fs.existsSync(this.outputPath)) {
            fs.mkdirSync(this.outputPath);
        }
        const resolverFolderPath = path_1.normalize(this.outputPath + '/resolvers');
        if (fs.existsSync(resolverFolderPath)) {
            const files = fs.readdirSync(resolverFolderPath);
            files.forEach(file => fs.unlinkSync(resolverFolderPath + '/' + file));
            fs.rmdirSync(resolverFolderPath);
        }
        if (!fs.existsSync(resolverFolderPath)) {
            fs.mkdirSync(resolverFolderPath);
        }
        Object.keys(resolvers).forEach((key) => {
            const resolver = resolvers[key];
            const fileName = key.replace('.vtl', '');
            const resolverFilePath = path_1.normalize(`${resolverFolderPath}/${fileName}`);
            fs.writeFileSync(resolverFilePath, resolver);
        });
    }
    /**
       * @returns {@link TransformConfig}
      */
    loadConfigSync(projectDir = 'resources') {
        // Initialize the config always with the latest version, other members are optional for now.
        let config = {
            Version: graphql_transformer_core_1.TRANSFORM_CURRENT_VERSION,
            ResolverConfig: {
                project: {
                    ConflictHandler: "OPTIMISTIC_CONCURRENCY" /* OPTIMISTIC */,
                    ConflictDetection: 'VERSION',
                },
            },
        };
        const configDir = path_1.join(__dirname, '..', '..', projectDir);
        try {
            const configPath = path_1.join(configDir, graphql_transformer_core_1.TRANSFORM_CONFIG_FILE_NAME);
            const configExists = fs.existsSync(configPath);
            if (configExists) {
                const configStr = fs.readFileSync(configPath);
                config = JSON.parse(configStr.toString());
            }
            return config;
        }
        catch (err) {
            return config;
        }
    }
}
exports.SchemaTransformer = SchemaTransformer;
/**
 * Grabbed from Amplify
 * https://github.com/aws-amplify/amplify-cli/blob/eb9257eaee117d0ed53ebc23aa28ecd7b7510fa1/packages/graphql-transformer-core/src/FeatureFlags.ts
 */
class TransformerFeatureFlagProvider {
    getBoolean(featureName, options) {
        switch (featureName) {
            case 'improvePluralization':
                return true;
            case 'validateTypeNameReservedWords':
                return false;
            default:
                return this.getValue(featureName, options);
        }
    }
    getString(featureName, options) {
        return this.getValue(featureName, options);
    }
    getNumber(featureName, options) {
        return this.getValue(featureName, options);
    }
    getObject() {
        // Todo: for future extensibility
        throw new Error('Not implemented');
    }
    getValue(featureName, defaultValue) {
        if (defaultValue !== undefined) {
            return defaultValue;
        }
        throw new Error(`No value found for feature ${featureName}`);
    }
}
exports.TransformerFeatureFlagProvider = TransformerFeatureFlagProvider;
//# sourceMappingURL=data:application/json;base64,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