"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppSyncTransformer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const aws_appsync_1 = require("@aws-cdk/aws-appsync");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const schema_transformer_1 = require("./transformer/schema-transformer");
const defaultAuthorizationConfig = {
    defaultAuthorization: {
        authorizationType: aws_appsync_1.AuthorizationType.API_KEY,
        apiKeyConfig: {
            description: 'Auto generated API Key from construct',
            name: 'dev',
        },
    },
};
/**
 * (experimental) AppSyncTransformer Construct.
 *
 * @experimental
 */
class AppSyncTransformer extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(scope, id);
        this.props = props;
        this.tableMap = {};
        this.isSyncEnabled = props.syncEnabled ? props.syncEnabled : false;
        this.pointInTimeRecovery = (_b = props.enableDynamoPointInTimeRecovery) !== null && _b !== void 0 ? _b : false;
        const transformerConfiguration = {
            schemaPath: props.schemaPath,
            syncEnabled: (_c = props.syncEnabled) !== null && _c !== void 0 ? _c : false,
            customVtlTransformerRootDirectory: props.customVtlTransformerRootDirectory,
        };
        // Combine the arrays so we only loop once
        // Test each transformer to see if it implements ITransformer
        const allCustomTransformers = [...(_d = props.preCdkTransformers) !== null && _d !== void 0 ? _d : [], ...(_e = props.postCdkTransformers) !== null && _e !== void 0 ? _e : []];
        if (allCustomTransformers && allCustomTransformers.length > 0) {
            allCustomTransformers.forEach(transformer => {
                if (transformer && !this.implementsITransformer(transformer)) {
                    throw new Error(`Transformer does not implement ITransformer from graphql-transformer-core: ${transformer}`);
                }
            });
        }
        const transformer = new schema_transformer_1.SchemaTransformer(transformerConfiguration);
        this.outputs = transformer.transform(props.preCdkTransformers, props.postCdkTransformers);
        const resolvers = transformer.getResolvers();
        this.functionResolvers = (_f = this.outputs.functionResolvers) !== null && _f !== void 0 ? _f : {};
        // Remove any function resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, functionResolvers] of Object.entries(this.functionResolvers)) {
            functionResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.httpResolvers = (_g = this.outputs.httpResolvers) !== null && _g !== void 0 ? _g : {};
        // Remove any http resolvers from the total list of resolvers
        // Otherwise it will add them twice
        for (const [_, httpResolvers] of Object.entries(this.httpResolvers)) {
            httpResolvers.forEach((resolver) => {
                switch (resolver.typeName) {
                    case 'Query':
                    case 'Mutation':
                    case 'Subscription':
                        delete resolvers[resolver.fieldName];
                        break;
                }
            });
        }
        this.resolvers = resolvers;
        this.nestedAppsyncStack = new core_1.NestedStack(this, (_h = props.nestedStackName) !== null && _h !== void 0 ? _h : 'appsync-nested-stack');
        // AppSync
        this.appsyncAPI = new aws_appsync_1.GraphqlApi(this.nestedAppsyncStack, `${id}-api`, {
            name: props.apiName ? props.apiName : `${id}-api`,
            authorizationConfig: props.authorizationConfig
                ? props.authorizationConfig
                : defaultAuthorizationConfig,
            logConfig: {
                fieldLogLevel: props.fieldLogLevel
                    ? props.fieldLogLevel
                    : aws_appsync_1.FieldLogLevel.NONE,
            },
            schema: aws_appsync_1.Schema.fromAsset('./appsync/schema.graphql'),
            xrayEnabled: (_j = props.xrayEnabled) !== null && _j !== void 0 ? _j : false,
        });
        let tableData = (_k = this.outputs.cdkTables) !== null && _k !== void 0 ? _k : {};
        // Check to see if sync is enabled
        if (tableData.DataStore) {
            this.isSyncEnabled = true;
            this.syncTable = this.createSyncTable(tableData.DataStore);
            delete tableData.DataStore; // We don't want to create this again below so remove it from the tableData map
        }
        this.tableNameMap = this.createTablesAndResolvers(tableData, resolvers, props.tableNames);
        if (this.outputs.noneResolvers) {
            this.createNoneDataSourceAndResolvers(this.outputs.noneResolvers, resolvers);
        }
        this.createHttpResolvers();
        this.publicResourceArns = this.getResourcesFromGeneratedRolePolicy(transformer.unauthRolePolicy);
        this.privateResourceArns = this.getResourcesFromGeneratedRolePolicy(transformer.authRolePolicy);
        // Outputs so we can generate exports
        new core_1.CfnOutput(scope, 'appsyncGraphQLEndpointOutput', {
            value: this.appsyncAPI.graphqlUrl,
            description: 'Output for aws_appsync_graphqlEndpoint',
        });
    }
    /**
     * graphql-transformer-core needs to be jsii enabled to pull the ITransformer interface correctly.
     * Since it's not in peer dependencies it doesn't show up in the jsii deps list.
     * Since it's not jsii enabled it has to be bundled.
     * The package can't be in BOTH peer and bundled dependencies
     * So we do a fake test to make sure it implements these and hope for the best
     * @param transformer
     */
    implementsITransformer(transformer) {
        return 'name' in transformer
            && 'directive' in transformer
            && 'typeDefinitions' in transformer;
    }
    /**
     * Creates NONE data source and associated resolvers
     * @param noneResolvers The resolvers that belong to the none data source
     * @param resolvers The resolver map minus function resolvers
     */
    createNoneDataSourceAndResolvers(noneResolvers, resolvers) {
        const noneDataSource = this.appsyncAPI.addNoneDataSource('NONE');
        Object.keys(noneResolvers).forEach((resolverKey) => {
            const resolver = resolvers[resolverKey];
            new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: noneDataSource,
                requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
            });
        });
    }
    /**
     * Creates each dynamodb table, gsis, dynamodb datasource, and associated resolvers
     * If sync is enabled then TTL configuration is added
     * Returns tableName: table map in case it is needed for lambda functions, etc
     * @param tableData The CdkTransformer table information
     * @param resolvers The resolver map minus function resolvers
     */
    createTablesAndResolvers(tableData, resolvers, tableNames = {}) {
        const tableNameMap = {};
        Object.keys(tableData).forEach((tableKey) => {
            var _b;
            const tableName = (_b = tableNames[tableKey]) !== null && _b !== void 0 ? _b : undefined;
            const table = this.createTable(tableData[tableKey], tableName);
            this.tableMap[tableKey] = table;
            const dataSource = this.appsyncAPI.addDynamoDbDataSource(tableKey, table);
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appsync-datasource-deltasyncconfig.html
            if (this.isSyncEnabled && this.syncTable) {
                //@ts-ignore - ds is the base CfnDataSource and the db config needs to be versioned - see CfnDataSource
                dataSource.ds.dynamoDbConfig.versioned = true;
                //@ts-ignore - ds is the base CfnDataSource - see CfnDataSource
                dataSource.ds.dynamoDbConfig.deltaSyncConfig = {
                    baseTableTtl: '43200',
                    deltaSyncTableName: this.syncTable.tableName,
                    deltaSyncTableTtl: '30',
                };
                // Need to add permission for our datasource service role to access the sync table
                dataSource.grantPrincipal.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: [
                        'dynamodb:*',
                    ],
                    resources: [this.syncTable.tableArn],
                }));
            }
            const dynamoDbConfig = dataSource.ds
                .dynamoDbConfig;
            tableNameMap[tableKey] = dynamoDbConfig.tableName;
            // Loop the basic resolvers
            tableData[tableKey].resolvers.forEach((resolverKey) => {
                const resolver = resolvers[resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
            // Loop the gsi resolvers
            tableData[tableKey].gsiResolvers.forEach((resolverKey) => {
                const resolver = resolvers.gsi[resolverKey];
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: dataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.requestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromFile(resolver.responseMappingTemplate),
                });
            });
        });
        return tableNameMap;
    }
    createTable(tableData, tableName) {
        var _b;
        // I do not want to force people to pass `TypeTable` - this way they are only passing the @model Type name
        const modelTypeName = tableData.tableName.replace('Table', '');
        const streamSpecification = this.props.dynamoDbStreamConfig && this.props.dynamoDbStreamConfig[modelTypeName];
        const tableProps = {
            tableName,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
            pointInTimeRecovery: this.pointInTimeRecovery,
            sortKey: tableData.sortKey && tableData.sortKey.name
                ? {
                    name: tableData.sortKey.name,
                    type: this.convertAttributeType(tableData.sortKey.type),
                } : undefined,
            timeToLiveAttribute: ((_b = tableData === null || tableData === void 0 ? void 0 : tableData.ttl) === null || _b === void 0 ? void 0 : _b.enabled) ? tableData.ttl.attributeName : undefined,
            stream: streamSpecification,
        };
        const table = new aws_dynamodb_1.Table(this.nestedAppsyncStack, tableData.tableName, tableProps);
        tableData.localSecondaryIndexes.forEach((lsi) => {
            table.addLocalSecondaryIndex({
                indexName: lsi.indexName,
                sortKey: {
                    name: lsi.sortKey.name,
                    type: this.convertAttributeType(lsi.sortKey.type),
                },
                projectionType: this.convertProjectionType(lsi.projection.ProjectionType),
            });
        });
        tableData.globalSecondaryIndexes.forEach((gsi) => {
            table.addGlobalSecondaryIndex({
                indexName: gsi.indexName,
                partitionKey: {
                    name: gsi.partitionKey.name,
                    type: this.convertAttributeType(gsi.partitionKey.type),
                },
                sortKey: gsi.sortKey && gsi.sortKey.name
                    ? {
                        name: gsi.sortKey.name,
                        type: this.convertAttributeType(gsi.sortKey.type),
                    } : undefined,
                projectionType: this.convertProjectionType(gsi.projection.ProjectionType),
            });
        });
        return table;
    }
    /**
     * Creates the sync table for Amplify DataStore
     * https://docs.aws.amazon.com/appsync/latest/devguide/conflict-detection-and-sync.html
     * @param tableData The CdkTransformer table information
     */
    createSyncTable(tableData) {
        var _b;
        return new aws_dynamodb_1.Table(this, 'appsync-api-sync-table', {
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: tableData.partitionKey.name,
                type: this.convertAttributeType(tableData.partitionKey.type),
            },
            sortKey: {
                name: tableData.sortKey.name,
                type: this.convertAttributeType(tableData.sortKey.type),
            },
            timeToLiveAttribute: ((_b = tableData.ttl) === null || _b === void 0 ? void 0 : _b.attributeName) || '_ttl',
        });
    }
    convertAttributeType(type) {
        switch (type) {
            case 'N':
                return aws_dynamodb_1.AttributeType.NUMBER;
            case 'B':
                return aws_dynamodb_1.AttributeType.BINARY;
            case 'S': // Same as default
            default:
                return aws_dynamodb_1.AttributeType.STRING;
        }
    }
    convertProjectionType(type) {
        switch (type) {
            case 'INCLUDE':
                return aws_dynamodb_1.ProjectionType.INCLUDE;
            case 'KEYS_ONLY':
                return aws_dynamodb_1.ProjectionType.KEYS_ONLY;
            case 'ALL': // Same as default
            default:
                return aws_dynamodb_1.ProjectionType.ALL;
        }
    }
    createHttpResolvers() {
        for (const [endpoint, httpResolvers] of Object.entries(this.httpResolvers)) {
            const strippedEndpoint = endpoint.replace(/[^_0-9A-Za-z]/g, '');
            const httpDataSource = this.appsyncAPI.addHttpDataSource(`${strippedEndpoint}`, endpoint);
            httpResolvers.forEach((resolver) => {
                new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                    api: this.appsyncAPI,
                    typeName: resolver.typeName,
                    fieldName: resolver.fieldName,
                    dataSource: httpDataSource,
                    requestMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                    responseMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
                });
            });
        }
    }
    /**
     * This takes one of the autogenerated policies from AWS and builds the list of ARNs for granting GraphQL access later
     * @param policy The auto generated policy from the AppSync Transformers
     * @returns An array of resource arns for use with grants
     */
    getResourcesFromGeneratedRolePolicy(policy) {
        var _b, _c;
        if (!((_c = (_b = policy === null || policy === void 0 ? void 0 : policy.Properties) === null || _b === void 0 ? void 0 : _b.PolicyDocument) === null || _c === void 0 ? void 0 : _c.Statement))
            return [];
        const { region, account } = this.nestedAppsyncStack;
        const resolvedResources = [];
        for (const statement of policy.Properties.PolicyDocument.Statement) {
            const { Resource: resources = [] } = statement !== null && statement !== void 0 ? statement : {};
            for (const resource of resources) {
                const subs = resource['Fn::Sub'][1];
                const { typeName, fieldName } = subs !== null && subs !== void 0 ? subs : {};
                if (fieldName) {
                    resolvedResources.push(`arn:aws:appsync:${region}:${account}:apis/${this.appsyncAPI.apiId}/types/${typeName}/fields/${fieldName}`);
                }
                else {
                    resolvedResources.push(`arn:aws:appsync:${region}:${account}:apis/${this.appsyncAPI.apiId}/types/${typeName}/*`);
                }
            }
        }
        return resolvedResources;
    }
    /**
     * (experimental) Adds the function as a lambdaDataSource to the AppSync api Adds all of the functions resolvers to the AppSync api.
     *
     * @param functionName The function name specified in the.
     * @param id The id to give.
     * @param lambdaFunction The lambda function to attach.
     * @experimental
     * @function directive of the schema
     */
    addLambdaDataSourceAndResolvers(functionName, id, lambdaFunction, options) {
        const functionDataSource = this.appsyncAPI.addLambdaDataSource(id, lambdaFunction, options);
        for (const resolver of this.functionResolvers[functionName]) {
            new aws_appsync_1.Resolver(this.nestedAppsyncStack, `${resolver.typeName}-${resolver.fieldName}-resolver`, {
                api: this.appsyncAPI,
                typeName: resolver.typeName,
                fieldName: resolver.fieldName,
                dataSource: functionDataSource,
                requestMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultRequestMappingTemplate),
                responseMappingTemplate: aws_appsync_1.MappingTemplate.fromString(resolver.defaultResponseMappingTemplate),
            });
        }
        return functionDataSource;
    }
    /**
     * (experimental) Adds a stream to the dynamodb table associated with the type.
     *
     * @returns string - the stream arn token
     * @experimental
     */
    addDynamoDBStream(props) {
        const tableName = `${props.modelTypeName}Table`;
        const table = this.tableMap[tableName];
        if (!table)
            throw new Error(`Table with name '${tableName}' not found.`);
        const cfnTable = table.node.defaultChild;
        cfnTable.streamSpecification = {
            streamViewType: props.streamViewType,
        };
        return cfnTable.attrStreamArn;
    }
    /**
     * (experimental) Allows for overriding the generated request and response mapping templates.
     *
     * @experimental
     */
    overrideResolver(props) {
        const resolver = this.nestedAppsyncStack.node.tryFindChild(`${props.typeName}-${props.fieldName}-resolver`);
        if (!resolver)
            throw new Error(`Resolver with typeName '${props.typeName}' and fieldName '${props.fieldName}' not found`);
        const cfnResolver = resolver.node.defaultChild;
        if (!cfnResolver)
            throw new Error(`Resolver with typeName '${props.typeName}' and fieldName '${props.fieldName}' not found`);
        if (props.requestMappingTemplateFile) {
            cfnResolver.requestMappingTemplate = fs.readFileSync(props.requestMappingTemplateFile).toString('utf-8');
        }
        if (props.responseMappingTemplateFile) {
            cfnResolver.responseMappingTemplate = fs.readFileSync(props.responseMappingTemplateFile).toString('utf-8');
        }
    }
    /**
     * (experimental) Adds an IAM policy statement granting access to the public fields of the AppSync API.
     *
     * Policy is based off of the @auth transformer
     * https://docs.amplify.aws/cli/graphql-transformer/auth
     *
     * @param grantee The principal to grant access to.
     * @experimental
     */
    grantPublic(grantee) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['appsync:GraphQL'],
            resourceArns: this.publicResourceArns,
            scope: this,
        });
    }
    /**
     * (experimental) Adds an IAM policy statement granting access to the private fields of the AppSync API.
     *
     * Policy is based off of the @auth transformer
     * https://docs.amplify.aws/cli/graphql-transformer/auth
     *
     * @experimental
     */
    grantPrivate(grantee) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['appsync:GraphQL'],
            resourceArns: this.privateResourceArns,
        });
    }
}
exports.AppSyncTransformer = AppSyncTransformer;
_a = JSII_RTTI_SYMBOL_1;
AppSyncTransformer[_a] = { fqn: "cdk-appsync-transformer.AppSyncTransformer", version: "1.138.1" };
//# sourceMappingURL=data:application/json;base64,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