from functools import cache

from django.contrib.auth import get_user_model
from django.contrib.auth.backends import ModelBackend
from django.contrib.auth.models import Permission
from django.core.exceptions import PermissionDenied
from django.db.models import ObjectDoesNotExist


class EmailBackend(ModelBackend):
    def _get_user(self, **kwargs):
        try:
            return get_user_model()._default_manager.get(is_active=True, **kwargs)
        except ObjectDoesNotExist:
            return None

    def get_user(self, user_id):
        return self._get_user(pk=user_id)

    def authenticate(self, request, email):
        return self._get_user(email=email)


@cache
def _all_perms():
    queryset = Permission.objects.values_list("content_type__app_label", "codename")
    return [f"{app_label}.{codename}" for app_label, codename in queryset]


class PermissionsBackend(ModelBackend):
    def get_user_permissions(self, user, obj=None):
        attribute = "_user_permissions_cache"
        if not hasattr(user, attribute):
            # ModelBackend can use an optimized variant of this -- we cannot since
            # we don't know what the permission checking callbacks do.
            perms = {perm for perm in _all_perms() if self._has_perm(user, perm, obj)}
            setattr(user, attribute, perms)
        return getattr(user, attribute)

    def _has_perm(self, user, perm, obj):
        try:
            return self.has_perm(user, perm, obj)
        except PermissionDenied:
            return False

    def get_group_permissions(self, user, obj=None):
        return set()

    # def get_all_permissions(self, user, obj=None):

    def has_perm(self, user, perm, obj=None):
        return user._role_has_perm(perm=perm, obj=obj)

    # def has_module_perms(self, user, app_label):
    # def with_perm(self, perm, is_active=True, include_superusers=True, obj=None):

    # The user model and its manager will delegate permission lookup functions (get_user_permissions(), get_group_permissions(), get_all_permissions(), has_perm(), has_module_perms(), and with_perm()) to any authentication backend that implements these functions.
