"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const cdk = require("aws-cdk-lib");
const defaults = require("@aws-solutions-constructs/core");
const ec2 = require("aws-cdk-lib/aws-ec2");
const assertions_1 = require("aws-cdk-lib/assertions");
test('Test new vpc, load balancer, service', () => {
    // An environment with region is required to enable logging on an ALB
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testProps = {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: '10.0.0.0/16',
    });
});
test('Test new load balancer, service, existing vpc', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testProps = {
        existingVpc: defaults.getTestVpc(stack),
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: '172.168.0.0/16'
    });
    template.resourceCountIs('AWS::EC2::VPC', 1);
});
test('Test new service, existing load balancer, vpc', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testName = 'test-value';
    const existingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: existingVpc,
        internetFacing: true,
        loadBalancerName: testName,
    });
    const testProps = {
        existingVpc,
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingLoadBalancerObj: existingAlb,
        listenerProps: {
            protocol: 'HTTP'
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    template.resourceCountIs('AWS::ECS::Service', 1);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.resourceCountIs('AWS::ElasticLoadBalancingV2::Listener', 1);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: testName
    });
    template.resourceCountIs('AWS::ElasticLoadBalancingV2::LoadBalancer', 1);
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: '172.168.0.0/16'
    });
    template.resourceCountIs('AWS::EC2::VPC', 1);
});
test('Test existing load balancer, vpc, service', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testName = 'test-value';
    const existingVpc = defaults.getTestVpc(stack);
    const createFargateServiceResponse = defaults.CreateFargateService(stack, 'test', existingVpc, undefined, defaults.fakeEcrRepoArn);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: existingVpc,
        internetFacing: true,
        loadBalancerName: testName,
    });
    const testProps = {
        existingVpc,
        publicApi: true,
        existingFargateServiceObject: createFargateServiceResponse.service,
        existingContainerDefinitionObject: createFargateServiceResponse.containerDefinition,
        existingLoadBalancerObj: existingAlb,
        listenerProps: {
            protocol: 'HTTP'
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    template.resourceCountIs('AWS::ECS::Service', 1);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.resourceCountIs('AWS::ElasticLoadBalancingV2::Listener', 1);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: testName
    });
    template.resourceCountIs('AWS::ElasticLoadBalancingV2::LoadBalancer', 1);
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: '172.168.0.0/16'
    });
    template.resourceCountIs('AWS::EC2::VPC', 1);
});
test('Test add a second target with rules', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testProps = {
        existingVpc: defaults.getTestVpc(stack),
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
    };
    const firstConstruct = new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    const testPropsTwo = {
        existingVpc: firstConstruct.vpc,
        existingContainerDefinitionObject: firstConstruct.container,
        existingFargateServiceObject: firstConstruct.service,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true,
        ruleProps: {
            conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
            priority: 10
        },
    };
    new lib_1.AlbToFargate(stack, 'test-two-construct', testPropsTwo);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.resourceCountIs('AWS::ElasticLoadBalancingV2::TargetGroup', 2);
    template.resourceCountIs('AWS::ElasticLoadBalancingV2::ListenerRule', 1);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        "*admin*"
                    ]
                }
            }
        ],
    });
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: '172.168.0.0/16'
    });
    template.resourceCountIs('AWS::EC2::VPC', 1);
});
test('Test new vpc, load balancer, service - custom Service Props', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const serviceName = 'test-value';
    const testProps = {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
        fargateServiceProps: {
            serviceName
        }
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
        ServiceName: serviceName,
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true
    });
});
test('Test new vpc, load balancer, service - custom VPC Props', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testCidr = '192.0.0.0/19';
    const testProps = {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        CidrBlock: testCidr,
    });
});
test('Test new vpc, load balancer, service - custom LoadBalancer and targetGroup Props', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testLoadBalancerName = "test-lb";
    const testTargetGroupName = 'test-tg';
    const testProps = {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTP'
        },
        loadBalancerProps: {
            loadBalancerName: testLoadBalancerName,
        },
        targetGroupProps: {
            targetGroupName: testTargetGroupName,
        }
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE',
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: testLoadBalancerName
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        Name: testTargetGroupName
    });
});
test('Test HTTPS API with new vpc, load balancer, service', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const fakeCert = defaults.getFakeCertificate(stack, 'fake-cert');
    const testProps = {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTPS',
            certificates: [fakeCert]
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
        DefaultActions: [
            {
                RedirectConfig: {
                    Port: "443",
                    Protocol: "HTTPS",
                    StatusCode: "HTTP_302"
                },
                Type: "redirect"
            }
        ],
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS',
        Port: 443,
    });
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true
    });
});
test('Test HTTPS API with new vpc, load balancer, service and private API', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const fakeCert = defaults.getFakeCertificate(stack, 'fake-cert');
    const testProps = {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        listenerProps: {
            protocol: 'HTTPS',
            certificates: [fakeCert]
        },
    };
    new lib_1.AlbToFargate(stack, 'test-construct', testProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        LaunchType: 'FARGATE'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
        DefaultActions: [
            {
                RedirectConfig: {
                    Port: "443",
                    Protocol: "HTTPS",
                    StatusCode: "HTTP_302"
                },
                Type: "redirect"
            }
        ],
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS',
        Port: 443,
    });
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
    });
    template.resourceCountIs("AWS::EC2::Subnet", 3);
    template.hasResourceProperties("AWS::EC2::Subnet", {
        Tags: [
            {
                Key: "aws-cdk:subnet-name",
                Value: "isolated"
            },
            {
                Key: "aws-cdk:subnet-type",
                Value: "Isolated"
            },
            {
                Key: "Name",
                Value: "Default/Vpc/isolatedSubnet1"
            }
        ]
    });
    defaults.expectNonexistence(stack, "AWS::EC2::Subnet", {
        Tags: [
            {
                Key: "aws-cdk:subnet-name",
                Value: "Public"
            },
            {
                Key: "aws-cdk:subnet-type",
                Value: "Public"
            },
            {
                Key: "Name",
                Value: "Default/Vpc/PublicSubnet1"
            }
        ]
    });
});
//# sourceMappingURL=data:application/json;base64,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