"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepositoryEventTrigger = exports.Repository = void 0;
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const codecommit_generated_1 = require("./codecommit.generated");
/**
 * Represents a reference to a CodeCommit Repository.
 *
 * If you want to create a new Repository managed alongside your CDK code,
 * use the {@link Repository} class.
 *
 * If you want to reference an already existing Repository,
 * use the {@link Repository.import} method.
 */
class RepositoryBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for repository events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.codecommit'],
            resources: [this.repositoryArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a "CodeCommit Repository State Change" event occurs.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeCommit Repository State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is created (i.e. a new branch/tag is created) to the repository.
     */
    onReferenceCreated(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceCreated'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is updated (i.e. a commit is pushed to an existing or new branch) from the repository.
     */
    onReferenceUpdated(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceCreated', 'referenceUpdated'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a reference is delete (i.e. a branch/tag is deleted) from the repository.
     */
    onReferenceDeleted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({ detail: { event: ['referenceDeleted'] } });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a pull request state is changed.
     */
    onPullRequestStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Pull Request State Change'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a pull request.
     */
    onCommentOnPullRequest(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Comment on Pull Request'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a comment is made on a commit.
     */
    onCommentOnCommit(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({ detailType: ['CodeCommit Comment on Commit'] });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers when a commit is pushed to a branch.
     */
    onCommit(id, options = {}) {
        const rule = this.onReferenceUpdated(id, options);
        if (options.branches) {
            rule.addEventPattern({ detail: { referenceName: options.branches } });
        }
        return rule;
    }
    /**
     * Grant the given principal identity permissions to perform the actions on this repository.
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.repositoryArn],
        });
    }
    /**
     * Grant the given identity permissions to pull this repository.
     */
    grantPull(grantee) {
        return this.grant(grantee, 'codecommit:GitPull');
    }
    /**
     * Grant the given identity permissions to pull and push this repository.
     */
    grantPullPush(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'codecommit:GitPush');
    }
    /**
     * Grant the given identity permissions to read this repository.
     */
    grantRead(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'codecommit:EvaluatePullRequestApprovalRules', 'codecommit:Get*', 'codecommit:Describe*');
    }
}
/**
 * Provides a CodeCommit Repository.
 */
class Repository extends RepositoryBase {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.triggers = new Array();
        this.repository = new codecommit_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: props.repositoryName,
            repositoryDescription: props.description,
            triggers: core_1.Lazy.any({ produce: () => this.triggers }, { omitEmptyArray: true }),
        });
        this.repositoryName = this.getResourceNameAttribute(this.repository.attrName);
        this.repositoryArn = this.getResourceArnAttribute(this.repository.attrArn, {
            service: 'codecommit',
            resource: this.physicalName,
        });
    }
    /**
     * Imports a codecommit repository.
     *
     * @param repositoryArn (e.g. `arn:aws:codecommit:us-east-1:123456789012:MyDemoRepo`).
     */
    static fromRepositoryArn(scope, id, repositoryArn) {
        const stack = core_1.Stack.of(scope);
        const arn = stack.parseArn(repositoryArn);
        const repositoryName = arn.resource;
        const region = arn.region;
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryArn = repositoryArn;
                this.repositoryName = repositoryName;
                this.repositoryCloneUrlHttp = Repository.makeCloneUrl(stack, repositoryName, 'https', region);
                this.repositoryCloneUrlSsh = Repository.makeCloneUrl(stack, repositoryName, 'ssh', region);
            }
        }
        return new Import(scope, id);
    }
    /**
     *
     */
    static fromRepositoryName(scope, id, repositoryName) {
        const stack = core_1.Stack.of(scope);
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = Repository.arnForLocalRepository(repositoryName, scope);
                this.repositoryCloneUrlHttp = Repository.makeCloneUrl(stack, repositoryName, 'https');
                this.repositoryCloneUrlSsh = Repository.makeCloneUrl(stack, repositoryName, 'ssh');
            }
        }
        return new Import(scope, id);
    }
    static makeCloneUrl(stack, repositoryName, protocol, region) {
        return `${protocol}://git-codecommit.${region || stack.region}.${stack.urlSuffix}/v1/repos/${repositoryName}`;
    }
    static arnForLocalRepository(repositoryName, scope) {
        return core_1.Stack.of(scope).formatArn({
            service: 'codecommit',
            resource: repositoryName,
        });
    }
    /**
     * The HTTP clone URL.
     */
    get repositoryCloneUrlHttp() {
        return this.repository.attrCloneUrlHttp;
    }
    /**
     * The SSH clone URL.
     */
    get repositoryCloneUrlSsh() {
        return this.repository.attrCloneUrlSsh;
    }
    /**
     * Create a trigger to notify another service to run actions on repository events.
     *
     * @param arn Arn of the resource that repository events will notify.
     * @param options Trigger options to run actions.
     */
    notify(arn, options) {
        let evt = options && options.events;
        if (evt && evt.length > 1 && evt.indexOf(RepositoryEventTrigger.ALL) > -1) {
            evt = [RepositoryEventTrigger.ALL];
        }
        const customData = options && options.customData;
        const branches = options && options.branches;
        let name = options && options.name;
        if (!name) {
            name = this.node.path + '/' + arn;
        }
        if (this.triggers.find(prop => prop.name === name)) {
            throw new Error(`Unable to set repository trigger named ${name} because trigger names must be unique`);
        }
        this.triggers.push({
            destinationArn: arn,
            name,
            customData,
            branches,
            events: evt || [RepositoryEventTrigger.ALL],
        });
        return this;
    }
}
exports.Repository = Repository;
/**
 * Repository events that will cause the trigger to run actions in another service.
 */
var RepositoryEventTrigger;
(function (RepositoryEventTrigger) {
    RepositoryEventTrigger["ALL"] = "all";
    RepositoryEventTrigger["UPDATE_REF"] = "updateReference";
    RepositoryEventTrigger["CREATE_REF"] = "createReference";
    RepositoryEventTrigger["DELETE_REF"] = "deleteReference";
})(RepositoryEventTrigger = exports.RepositoryEventTrigger || (exports.RepositoryEventTrigger = {}));
//# sourceMappingURL=data:application/json;base64,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