#!/usr/bin/env node
"use strict";
/**
 * automatically creates a module for any CloudFormation namespaces that do not
 * have an AWS construct library.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const cfnspec = require("../lib");
// don't be a prude:
/* eslint-disable no-console */
/* eslint-disable quote-props */
async function main() {
    const root = path.join(__dirname, '..', '..');
    if (path.basename(root) !== '@aws-cdk') {
        throw new Error(`Something went wrong. We expected ${root} to be the "packages/@aws-cdk" directory. Did you move me?`);
    }
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const version = require('../package.json').version;
    // iterate over all cloudformation namespaces
    for (const namespace of cfnspec.namespaces()) {
        const module = cfnspec.createModuleDefinitionFromCfnNamespace(namespace);
        const lowcaseModuleName = module.moduleName.toLocaleLowerCase();
        const packagePath = path.join(root, module.moduleName);
        // we already have a module for this namesapce, move on.
        if (await fs.pathExists(packagePath)) {
            const packageJsonPath = path.join(packagePath, 'package.json');
            // eslint-disable-next-line @typescript-eslint/no-require-imports
            const packageJson = require(packageJsonPath);
            let scopes = packageJson['cdk-build'].cloudformation;
            if (typeof scopes === 'string') {
                scopes = [scopes];
            }
            if (scopes.indexOf(namespace) !== -1) {
                // V2-style module is already modeled in the root package, nothing to be done!
                continue;
            }
            else if (await fs.pathExists(path.join(root, `${module.moduleFamily}-${module.moduleBaseName}`.toLocaleLowerCase()))) {
                // V2-style package already has it's own package (legacy behavior), nothing to be done!
                continue;
            }
            else {
                // V2-style package needs to be added to it's "V1" package... Get down to business!
                console.error(`Adding ${namespace} to ${module.packageName}`);
                scopes.push(namespace);
                packageJson['cdk-build'].cloudformation = scopes;
                await fs.writeJson(packageJsonPath, packageJson, { encoding: 'utf-8', spaces: 2 });
                const indexTsPath = path.join(packagePath, 'lib', 'index.ts');
                const indexTs = [
                    (await fs.readFile(indexTsPath, { encoding: 'utf8' })).trimRight(),
                    `// ${namespace} CloudFormation Resources:`,
                    `export * from './${lowcaseModuleName}.generated';`,
                ].join('\n');
                await fs.writeFile(indexTsPath, indexTs, { encoding: 'utf8' });
                continue;
            }
        }
        async function write(relativePath, contents) {
            const fullPath = path.join(packagePath, relativePath);
            const dir = path.dirname(fullPath);
            await fs.mkdirp(dir);
            let data;
            if (typeof contents === 'string') {
                data = contents.trimLeft(); // trim first newline
            }
            else if (Array.isArray(contents)) {
                data = contents.join('\n');
            }
            else if (typeof contents === 'object') {
                data = JSON.stringify(contents, undefined, 2);
            }
            else {
                throw new Error('Invalid type of contents: ' + contents);
            }
            await fs.writeFile(fullPath, data + '\n');
        }
        console.log(`generating module for ${module.packageName}...`);
        await write('package.json', {
            name: module.packageName,
            version,
            description: `The CDK Construct Library for ${namespace}`,
            main: 'lib/index.js',
            types: 'lib/index.d.ts',
            jsii: {
                outdir: 'dist',
                projectReferences: true,
                targets: {
                    dotnet: {
                        namespace: module.dotnetPackage,
                        packageId: module.dotnetPackage,
                        signAssembly: true,
                        assemblyOriginatorKeyFile: '../../key.snk',
                        iconUrl: 'https://raw.githubusercontent.com/aws/aws-cdk/master/logo/default-256-dark.png',
                    },
                    java: {
                        package: `${module.javaGroupId}.${module.javaPackage}`,
                        maven: {
                            groupId: module.javaGroupId,
                            artifactId: module.javaArtifactId,
                        },
                    },
                    python: {
                        classifiers: [
                            'Framework :: AWS CDK',
                            'Framework :: AWS CDK :: 1',
                        ],
                        distName: module.pythonDistName,
                        module: module.pythonModuleName,
                    },
                },
                metadata: {
                    jsii: {
                        rosetta: {
                            strict: true,
                        },
                    },
                },
            },
            repository: {
                type: 'git',
                url: 'https://github.com/aws/aws-cdk.git',
                directory: `packages/${module.packageName}`,
            },
            homepage: 'https://github.com/aws/aws-cdk',
            scripts: {
                build: 'cdk-build',
                watch: 'cdk-watch',
                lint: 'cdk-lint',
                test: 'cdk-test',
                integ: 'cdk-integ',
                pkglint: 'pkglint -f',
                package: 'cdk-package',
                awslint: 'cdk-awslint',
                cfn2ts: 'cfn2ts',
                'build+test': 'yarn build && yarn test',
                'build+test+package': 'yarn build+test && yarn package',
                compat: 'cdk-compat',
                gen: 'cfn2ts',
                'rosetta:extract': 'yarn --silent jsii-rosetta extract',
                'build+extract': 'yarn build && yarn rosetta:extract',
                'build+test+extract': 'yarn build+test && yarn rosetta:extract',
            },
            'cdk-build': {
                cloudformation: namespace,
                jest: true,
                env: {
                    AWSLINT_BASE_CONSTRUCT: 'true',
                },
            },
            keywords: [
                'aws',
                'cdk',
                'constructs',
                namespace,
                module.moduleName,
            ],
            author: {
                name: 'Amazon Web Services',
                url: 'https://aws.amazon.com',
                organization: true,
            },
            license: 'Apache-2.0',
            devDependencies: {
                '@aws-cdk/assertions': version,
                '@aws-cdk/cdk-build-tools': version,
                '@aws-cdk/cfn2ts': version,
                '@aws-cdk/pkglint': version,
                '@types/jest': '^26.0.22',
            },
            dependencies: {
                '@aws-cdk/core': version,
            },
            peerDependencies: {
                '@aws-cdk/core': version,
            },
            engines: {
                node: '>= 10.13.0 <13 || >=13.7.0',
            },
            stability: 'experimental',
            maturity: 'cfn-only',
            awscdkio: {
                announce: false,
            },
            publishConfig: {
                tag: 'latest',
            },
        });
        await write('.gitignore', [
            '*.js',
            '*.js.map',
            '*.d.ts',
            'tsconfig.json',
            'node_modules',
            '*.generated.ts',
            'dist',
            '.jsii',
            '',
            '.LAST_BUILD',
            '.nyc_output',
            'coverage',
            '.nycrc',
            '.LAST_PACKAGE',
            '*.snk',
            'nyc.config.js',
            '!.eslintrc.js',
            '!jest.config.js',
            'junit.xml',
        ]);
        await write('.npmignore', [
            '# Don\'t include original .ts files when doing `npm pack`',
            '*.ts',
            '!*.d.ts',
            'coverage',
            '.nyc_output',
            '*.tgz',
            '',
            'dist',
            '.LAST_PACKAGE',
            '.LAST_BUILD',
            '!*.js',
            '',
            '# Include .jsii',
            '!.jsii',
            '',
            '*.snk',
            '',
            '*.tsbuildinfo',
            '',
            'tsconfig.json',
            '',
            '.eslintrc.js',
            'jest.config.js',
            '',
            '# exclude cdk artifacts',
            '**/cdk.out',
            'junit.xml',
            'test/',
            '!*.lit.ts',
        ]);
        await write('lib/index.ts', [
            `// ${namespace} CloudFormation Resources:`,
            `export * from './${lowcaseModuleName}.generated';`,
        ]);
        await write(`test/${lowcaseModuleName}.test.ts`, [
            "import '@aws-cdk/assertions';",
            "import {} from '../lib';",
            '',
            "test('No tests are specified for this package', () => {",
            '  expect(true).toBe(true);',
            '});',
        ]);
        await cfnspec.createLibraryReadme(namespace, path.join(packagePath, 'README.md'));
        await write('.eslintrc.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/eslintrc');",
            "baseConfig.parserOptions.project = __dirname + '/tsconfig.json';",
            'module.exports = baseConfig;',
        ]);
        await write('jest.config.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/jest.config');",
            'module.exports = baseConfig;',
        ]);
        await write('rosetta/default.ts-fixture', [
            "import { Construct } from 'constructs';",
            "import { Stack } from '@aws-cdk/core';",
            '',
            'class MyStack extends Stack {',
            '  constructor(scope: Construct, id: string) {',
            '    /// here',
            '  }',
            '}',
        ]);
        const templateDir = path.join(__dirname, 'template');
        for (const file of await fs.readdir(templateDir)) {
            await fs.copy(path.join(templateDir, file), path.join(packagePath, file));
        }
        await addDependencyToMegaPackage(path.join('@aws-cdk', 'cloudformation-include'), module.packageName, version, ['dependencies', 'peerDependencies']);
        await addDependencyToMegaPackage('aws-cdk-lib', module.packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('monocdk', module.packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('decdk', module.packageName, version, ['dependencies']);
    }
}
/**
 * A few of our packages (e.g., decdk, aws-cdk-lib) require a dependency on every service package.
 * This automates adding the dependency (and peer dependency) to the package.json.
 */
async function addDependencyToMegaPackage(megaPackageName, packageName, version, dependencyTypes) {
    const packageJsonPath = path.join(__dirname, '..', '..', '..', megaPackageName, 'package.json');
    const packageJson = JSON.parse(await fs.readFile(packageJsonPath, 'utf8'));
    dependencyTypes.forEach(dependencyType => {
        const unorderedDeps = {
            ...packageJson[dependencyType],
            [packageName]: version,
        };
        packageJson[dependencyType] = {};
        Object.keys(unorderedDeps).sort().forEach(k => {
            packageJson[dependencyType][k] = unorderedDeps[k];
        });
    });
    await fs.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2) + '\n');
}
main().catch(e => {
    console.error(e);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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