import logging
from pathlib import Path

import frontmatter
from markdown2 import markdown
from slugify import slugify
from typing import Any, Optional
from jinja2 import Environment, FileSystemLoader, Template

class Page:
    """Base component used to make web pages.

    All components that represent content in HTML, XML, or user-facing JSON generated by
    Render Engine should be a Page object.

    Pages can be rendered directly from a template or generated from a file.

    Page objects can be used to extend existing page objects.

    .. note::
        Not all attributes are defined by default (those that are marked *optional*) but
        will be checked for in other areas of the code.

    Attributes:
        title (str): **[Optional]** title of the :class:`Page <.Page>` object
        engine (str): **[Optional]**
            The engine the :class:`Site <.site.Site>` should use to generate markup. By
            default this is `Jinja2 <https://palletsprojects.com/p/jinja/>`_.
        template (str): **[Optional]** template filename for the :attr:`engine
            <.site.Site.engine>` to look for. If defined, it must be a valid file.
        content_path (str): **[Optional]** The filepath to load content from.

            The `content_path` will be checked for additional attributes
            and :attr:`base_content <Page.base_content>`.

            Raises:
                FileNotFoundError:
                    If a `content_path` was supplied path that points to a path that does not exist.

            For more information about content paths, markdown and content_path rendering
            see ``:ref:<page_from_content_path.rst>``
        base_content (str):  **[Optional]** Pre-rendered markdown or HTML to be converted to Markup.

            Uses the `Markdown2 <https://pypi.org/project/markdown2/>`_ generator.


            If ``content_path`` is provided the content section of the file will be stored
            as the `base_content`
        list_attrs (typing.Optional[typing.List[str]]): list of attributes whose string
        values should be converted
            to a comma-delimited list

            This is primarily for converting a comma-separated list of values provided in
            your ``content_path``

            Example::

                >>> class myPage(Page):
                    list_attrs = ["foo"]

                # calling myPage will convert the string value of `foo` to a list of values

                >>> myPage(foo='my, values, here')
                >>> myPage.foo
                ["my", "values", "here"]

        slug (str): **[Optional]** The rendered pages filename

            A slug passed in will be `slugified <https://github.com/un33k/python-slugify>`_
            to ensure a valid path is given. ``eg. slug="my cool slug" == "my-cool-slug"``

            If no slug is provided the class name will be used as the slug.::

                class MyPage(Page):
                    pass

                # the path for this page would be https://example.com/mypage

    """

    route: Path = Path("./")
    """the directory in the :attr:`output_path <.site.Site.output_path>` that
    the :class:`Page <.page.Page>` should be created at.

            Example a route of "pages" for a :py:class `render_engine.Page`: object with the
            :py:attr `Page.slug`: "foo" would generate *https://example.com/**pages**/foo*.

            An empty string will apply the route at the root `https://example.com/foo`
    """
    markdown_extras: list[str] = ["fenced-code-blocks", "footnotes"]
    """Plugins to be included when generating HTML from your ``base_content``.

    For more information on available extras or creating your own, see the `Markdown2
    <https://pypi.org/project/markdown2/>`_ documentation
    """

    markdown: Optional[str] = None
    extension: str = ".html"
    engine: Environment = Environment(loader=FileSystemLoader('templates'))

    def __init__(self, **kwargs) -> None:
        for key, val in kwargs.items():
            setattr(self, key, val)
        
        if self.markdown and hasattr(self, "content_path"):
            logging.warning("both `Page.markdown` and `content_path` selected. the content from `content_path` will be used.")

        if hasattr(self, "content_path") and not self.markdown:
            post = frontmatter.load(self.content_path)
            valid_attrs, self.markdown = post.metadata, post.content
            logging.info(f"content_path found! {valid_attrs=}, {self.markdown=}")

            for name, value in valid_attrs.items():
                # comma delimit attributes.
                if name.lower() in getattr(self, "list_attrs", []):
                    value = [attrval.lower() for attrval in value.split(", ")]

                setattr(self, name.lower(), value)        

        if not hasattr(self, "title"):
            self.title = self.__class__.__name__
            logging.info(f"No Title. Assigning {self.title=}")

        if not hasattr(self, "slug"):
            logging.info(f"No slug. Will slugify {self.title=}")
            self.slug = (
                self.title
            )  # Will Slugify in Next Step

        self.slug = slugify(self.slug)
        
    @property
    def url(self) -> str:
        """The first route and the slug of the page."""
        return f"{self.route}/{self.slug}{self._extension}"

    @property
    def _extension(self) -> str:
        """Ensures consistency on extension"""
        if not self.extension.startswith('.'):
            return f".{self.extension}"
        else:
            return self.extension

    @property
    def content(self) -> Optional[str]:
        """html = rendered HTML (not marked up).
        Is `None` if `content == None`
        This is referred to as `content` because it is intended to be applied in the jinja template as {{content}}.
        When referring to the raw content in the Page object use `markdown`.
        """
 
        if not self.markdown:
            return None

        return markdown(
                self.markdown, extras=self.markdown_extras
            )

    @property
    def _template(self) -> Optional[Template]:
        if template:= getattr(self, 'template', None):
            return self.engine.get_template(template)
        
    def __str__(self):
        return self.slug


    def __repr__(self) -> str:
        return f"<Page {self.title}>"

    def _render_content(self, **kwargs) -> str:
        template = self._template

        if template:
            if self.content:
                return template.render(content=self.content, **{**self.__dict__, **kwargs})

            else:
                return template.render(**{**self.__dict__, **kwargs})
        
        elif self.content:
            return self.content
         
        else:
            raise ValueError(f'{self=} must have either content or template')


    def render(self, **kwargs) -> Path:
        """Build the page based on content instructions"""

        markup = self._render_content(**kwargs)
        path = Path(kwargs.get('path', '')).joinpath(f"{self.slug}{self._extension}")
        return path.write_text(markup)