# New matplotlib colormaps by Nathaniel J. Smith, Stefan van der Walt,
# and (in the case of viridis) Eric Firing.
#
# This file and the colormaps in it are released under the CC0 license /
# public domain dedication. We would appreciate credit if you use or
# redistribute these colormaps, but do not impose any legal restrictions.
#
# To the extent possible under law, the persons who associated CC0 with
# mpl-colormaps have waived all copyright and related or neighboring rights
# to mpl-colormaps.
#
# You should have received a copy of the CC0 legalcode along with this
# work.  If not, see <http://creativecommons.org/publicdomain/zero/1.0/>.
from matplotlib.colors import ListedColormap, LinearSegmentedColormap

__all__ = ['fc_cmap', 'germany_cmap', 'germany_r', 'white_red', 'white_red_r']

_germany_data = [[0.00676468, 0.00337139, 0.19021666],
                 [0.00804806, 0.0092624, 0.18577717],
                 [0.01014577, 0.01480928, 0.18230134],
                 [.01294979, 0.02013129, 0.17958165],
                 [0.01640685, 0.02531977, 0.17741095],
                 [0.02046764, 0.03042792, 0.17568978],
                 [0.02507355, 0.03549036, 0.17438615],
                 [0.0301999, 0.04053971, 0.17342227],
                 [0.03586837, 0.04534616, 0.17267593],
                 [0.04197236, 0.04987479, 0.17219228],
                 [0.04806089, 0.05418141, 0.17193634],
                 [0.05409427, 0.05830665, 0.17181829],
                 [0.06004462, 0.06227934, 0.17186355],
                 [0.06590492, 0.0661235, 0.17207171],
                 [0.0717052, 0.0698562, 0.17239],
                 [0.07744942, 0.07349123, 0.17280841],
                 [0.08311926, 0.07704224, 0.17335259],
                 [0.0887344, 0.08051736, 0.17399224],
                 [0.09431722, 0.08392144, 0.17469479],
                 [0.09984189, 0.08726488, 0.17550032],
                 [0.10532152, 0.09055181, 0.17639141],
                 [0.11078888, 0.09378082, 0.17732364],
                 [0.11621418, 0.09696155, 0.1783434],
                 [0.12160756, 0.10009585, 0.17943916],
                 [0.12700649, 0.10317781, 0.18056325],
                 [0.13237997, 0.10621715, 0.1817622],
                 [0.13773826, 0.10921384, 0.1830251],
                 [0.14311885, 0.11215912, 0.18430746],
                 [0.14849148, 0.11506299, 0.1856513],
                 [0.15386921, 0.11792326, 0.18704207],
                 [0.15928483, 0.12073039, 0.18844331],
                 [0.16471006, 0.12349462, 0.18988915],
                 [0.17016335, 0.12621083, 0.1913581],
                 [0.17566691, 0.12887162, 0.1928257],
                 [0.18119549, 0.1314872, 0.19431637],
                 [0.18677584, 0.13404819, 0.19579903],
                 [0.19241187, 0.13655351, 0.1972659],
                 [0.19808313, 0.13901241, 0.19873125],
                 [0.20382852, 0.14140912, 0.20015183],
                 [0.20962423, 0.1437543, 0.20154357],
                 [0.21546219, 0.14605236, 0.20290663],
                 [0.22138644, 0.14828367, 0.20419326],
                 [0.22734736, 0.15047135, 0.20544195],
                 [0.2333683, 0.15260491, 0.20662439],
                 [0.2394521, 0.15468302, 0.20773148],
                 [0.2455655, 0.15672218, 0.2087883],
                 [0.25175846, 0.15869748, 0.20974283],
                 [0.2579813, 0.16063381, 0.21063724],
                 [0.26424905, 0.16252351, 0.21145365],
                 [0.27057172, 0.164361, 0.21217894],
                 [0.27690914, 0.16616749, 0.21285037],
                 [0.28331924, 0.16791127, 0.21340803],
                 [0.28974188, 0.16962483, 0.21391122],
                 [0.29620872, 0.17129031, 0.21432681],
                 [0.30270901, 0.1729132, 0.2146653],
                 [0.30922805, 0.17450168, 0.21494243],
                 [0.31579981, 0.17603525, 0.21512162],
                 [0.32237116, 0.17754512, 0.2152614],
                 [0.32900655, 0.1789917, 0.21529094],
                 [0.33564163, 0.1804139, 0.21528302],
                 [0.34232412, 0.18178146, 0.21518445],
                 [0.34901959, 0.18311557, 0.21503567],
                 [0.35574943, 0.18440194, 0.21481295],
                 [0.36250272, 0.18564703, 0.21453009],
                 [0.36928099, 0.18684927, 0.21418645],
                 [0.37609005, 0.18800399, 0.21377606],
                 [0.38291843, 0.18911849, 0.21331401],
                 [0.38978156, 0.19018118, 0.21278238],
                 [0.39666214, 0.19120371, 0.2122037],
                 [0.40357784, 0.19217248, 0.21155689],
                 [0.41051297, 0.19309828, 0.21086274],
                 [0.4174798, 0.19397107, 0.21010665],
                 [0.42447198, 0.1947948, 0.20929766],
                 [0.43148849, 0.19556945, 0.20843807],
                 [0.43854025, 0.19628531, 0.20751446],
                 [0.44560489, 0.19695961, 0.20655712],
                 [0.45271873, 0.19756128, 0.2055188],
                 [0.45983363, 0.1981297, 0.20446436],
                 [0.46700803, 0.19861371, 0.20331626],
                 [0.47419024, 0.19905685, 0.20214474],
                 [0.48140838, 0.19943323, 0.2009125],
                 [0.48866003, 0.19974342, 0.19962338],
                 [0.49592034, 0.20000926, 0.19831224],
                 [0.50324234, 0.20017926, 0.19690641],
                 [0.51057671, 0.20029855, 0.19547468],
                 [0.51793577, 0.2003541, 0.19400064],
                 [0.52534548, 0.20031788, 0.19244824],
                 [0.53277129, 0.20022286, 0.19086655],
                 [0.54022383, 0.20005663, 0.18924115],
                 [0.54772814, 0.19979008, 0.18753663],
                 [0.55525297, 0.19945428, 0.18579792],
                 [0.56280037, 0.19904521, 0.18402261],
                 [0.5703955, 0.1985318, 0.18217461],
                 [0.57802509, 0.19792583, 0.1802732],
                 [0.58568107, 0.1972339, 0.17833038],
                 [0.59336539, 0.19645092, 0.17634358],
                 [0.60108279, 0.19556779, 0.17430591],
                 [0.6088493, 0.19456004, 0.17219356],
                 [0.61664661, 0.19344654, 0.17003353],
                 [0.6244762, 0.19222121, 0.16782354],
                 [0.63233938, 0.19087764, 0.16556139],
                 [0.6402373, 0.18940915, 0.16324502],
                 [0.64817084, 0.1878088, 0.16087256],
                 [0.65614062, 0.18606947, 0.1584424],
                 [0.66414684, 0.18418396, 0.15595331],
                 [0.67218928, 0.18214515, 0.15340447],
                 [0.68026712, 0.17994615, 0.15079568],
                 [0.68837882, 0.17758062, 0.14812747],
                 [0.69652199, 0.17504318, 0.14540127],
                 [0.70471955, 0.17228188, 0.14257741],
                 [0.71294684, 0.16932732, 0.13969074],
                 [0.72119613, 0.16618189, 0.13674861],
                 [0.72950505, 0.16275685, 0.13368213],
                 [0.73782971, 0.15912149, 0.13055279],
                 [0.746197, 0.15520199, 0.12730143],
                 [0.75459751, 0.15099567, 0.12392171],
                 [0.76302156, 0.14650133, 0.12039671],
                 [0.77146632, 0.14170289, 0.11667645],
                 [0.77992611, 0.13659357, 0.11267075],
                 [0.78835818, 0.1312808, 0.1082536],
                 [0.79668564, 0.12603865, 0.10302419],
                 [0.80427044, 0.12312738, 0.09608916],
                 [0.80831703, 0.1314364, 0.09081277],
                 [0.81052018, 0.1443093, 0.08990281],
                 [0.81236716, 0.15710582, 0.09039944],
                 [0.81407665, 0.16936888, 0.09146726],
                 [0.81571575, 0.18106578, 0.09281939],
                 [0.81730754, 0.19225353, 0.09432212],
                 [0.81886242, 0.20299619, 0.09590444],
                 [0.82039201, 0.21333879, 0.09752765],
                 [0.82190234, 0.22332619, 0.09916741],
                 [0.82339248, 0.23300627, 0.10080614],
                 [0.82486537, 0.24241138, 0.10243394],
                 [0.82632232, 0.25157078, 0.10404413],
                 [0.82776375, 0.26051039, 0.10563221],
                 [0.82919695, 0.26924148, 0.10719879],
                 [0.83061519, 0.27779447, 0.10873931],
                 [0.83201757, 0.28618725, 0.11025279],
                 [0.83340955, 0.29442644, 0.1117422],
                 [0.83478881, 0.30252842, 0.11320733],
                 [0.83615159, 0.31050947, 0.11464765],
                 [0.83750577, 0.31836908, 0.11606911],
                 [0.83884065, 0.32613003, 0.11746827],
                 [0.84016664, 0.33378692, 0.11885329],
                 [0.84147285, 0.34136033, 0.12022111],
                 [0.842767, 0.34884707, 0.12157918],
                 [0.84404233, 0.35626083, 0.12292719],
                 [0.84530073, 0.36360433, 0.12426998],
                 [0.84654189, 0.37088248, 0.12561144],
                 [0.84776121, 0.37810436, 0.1269533],
                 [0.84896333, 0.38526856, 0.12830289],
                 [0.85014255, 0.39238446, 0.12966193],
                 [0.85129764, 0.39945615, 0.13103505],
                 [0.85243143, 0.40648338, 0.13242949],
                 [0.85353837, 0.41347406, 0.13384814],
                 [0.85461739, 0.42043125, 0.13529678],
                 [0.85566999, 0.42735531, 0.13678299],
                 [0.8566916, 0.43425223, 0.13831141],
                 [0.85768074, 0.4411247, 0.13988887],
                 [0.85863701, 0.4479743, 0.1415231],
                 [0.85955758, 0.45480458, 0.14322112],
                 [0.86044026, 0.46161831, 0.14499072],
                 [0.86128331, 0.46841762, 0.14684044],
                 [0.86208451, 0.47520493, 0.14877911],
                 [0.86284145, 0.48198262, 0.150816],
                 [0.86355197, 0.48875265, 0.15296104],
                 [0.8642128, 0.49551774, 0.15522436],
                 [0.86482201, 0.50227931, 0.15761712],
                 [0.86537726, 0.509039, 0.16015088],
                 [0.86587468, 0.51579942, 0.16283747],
                 [0.86631159, 0.52256209, 0.16568966],
                 [0.86668623, 0.52932768, 0.16872089],
                 [0.86699463, 0.53609838, 0.17194486],
                 [0.86723269, 0.5428762, 0.17537607],
                 [0.86739987, 0.54966051, 0.17902963],
                 [0.86749196, 0.55645311, 0.18292107],
                 [0.86750447, 0.56325579, 0.18706674],
                 [0.86743811, 0.57006666, 0.19148267],
                 [0.86728801, 0.57688748, 0.19618606],
                 [0.86705203, 0.583718, 0.20119403],
                 [0.86673171, 0.59055562, 0.20652243],
                 [0.86632159, 0.59740198, 0.21218961],
                 [0.86582546, 0.60425298, 0.21820995],
                 [0.86524379, 0.61110657, 0.22459842],
                 [0.86457718, 0.61796066, 0.23136936],
                 [0.86383228, 0.62480974, 0.23853206],
                 [0.86301241, 0.6316504, 0.24609631],
                 [0.86212627, 0.63847647, 0.25406603],
                 [0.86118307, 0.64528187, 0.26244205],
                 [0.86019438, 0.65205968, 0.27122022],
                 [0.85917013, 0.65880429, 0.28039506],
                 [0.85813239, 0.66550456, 0.28994479],
                 [0.8570877, 0.67215767, 0.29986083],
                 [0.8560562, 0.6787547, 0.31011613],
                 [0.85505771, 0.68528772, 0.32067958],
                 [0.85409788, 0.69175579, 0.33153559],
                 [0.85319692, 0.69815188, 0.3426469],
                 [0.85237556, 0.70446951, 0.35397157],
                 [0.85163266, 0.71071186, 0.36549808],
                 [0.85097842, 0.71687745, 0.37719788],
                 [0.85042167, 0.72296561, 0.38904334],
                 [0.8499815, 0.72897208, 0.40098807],
                 [0.84965318, 0.73490194, 0.41302517],
                 [0.84943771, 0.7407579, 0.42513906],
                 [0.84933782, 0.746542, 0.43731159],
                 [0.84935504, 0.75225669, 0.44952703],
                 [0.84948991, 0.75790468, 0.46177207],
                 [0.84974206, 0.76348889, 0.4740357],
                 [0.85011038, 0.76901238, 0.48630911],
                 [0.85059316, 0.77447825, 0.49858547],
                 [0.85118822, 0.77988959, 0.51085975],
                 [0.85190122, 0.78524717, 0.52310992],
                 [0.85273366, 0.79055303, 0.5353227],
                 [0.85367296, 0.79581285, 0.54751822],
                 [0.85471589, 0.80102944, 0.55969675],
                 [0.85585925, 0.80620546, 0.57185942],
                 [0.85710634, 0.8113419, 0.58399169],
                 [0.85847412, 0.81643678, 0.59604301],
                 [0.85993483, 0.82149825, 0.60807774],
                 [0.86148558, 0.82652831, 0.62009929],
                 [0.86314434, 0.83152478, 0.63205412],
                 [0.8649066, 0.83649028, 0.64394812],
                 [0.86675224, 0.84142994, 0.65583378],
                 [0.86869822, 0.84634196, 0.66765879],
                 [0.87074711, 0.85122741, 0.67941041],
                 [0.87287409, 0.85609135, 0.69116035],
                 [0.87510871, 0.86093057, 0.70281475],
                 [0.87743234, 0.86574889, 0.71442835],
                 [0.87983806, 0.87054801, 0.72602303],
                 [0.88235599, 0.87532557, 0.73749631],
                 [0.88494777, 0.88008645, 0.74897742],
                 [0.88764897, 0.88482816, 0.76034234],
                 [0.89043422, 0.88955353, 0.77168179],
                 [0.8933183, 0.89426207, 0.78294379],
                 [0.89629975, 0.89895444, 0.79413601],
                 [0.89937564, 0.90363096, 0.80527511],
                 [0.90255939, 0.90829166, 0.81631167],
                 [0.90583927, 0.91293636, 0.82730449],
                 [0.90923608, 0.91756533, 0.83817157],
                 [0.9127367, 0.92217725, 0.84898834],
                 [0.91636332, 0.92677297, 0.85966269],
                 [0.92010707, 0.93135011, 0.87026227],
                 [0.92398691, 0.93591003, 0.88070445],
                 [0.92800084, 0.94045057, 0.89102646],
                 [0.93216051, 0.94497351, 0.90117129],
                 [0.93646777, 0.94948003, 0.91112702],
                 [0.94092078, 0.95397327, 0.92087619],
                 [0.94551272, 0.95846338, 0.93035083],
                 [0.95021753, 0.96295699, 0.93959202],
                 [0.95500217, 0.96747815, 0.94849535],
                 [0.95981257, 0.97203699, 0.95717311],
                 [0.9645971, 0.97665896, 0.96557775],
                 [0.96930532, 0.98135103, 0.97383665],
                 [0.97390827, 0.98611952, 0.98200116],
                 [0.97839889, 0.99096109, 0.99013275],
                 [0.98277463, 0.99586566, 0.99833959]]
_germany_data = _germany_data[::-1]

_white_red_data = [[0.88118374, 0.22608963, 0.18798393],
                      [0.8821696, 0.22992773, 0.19218706],
                      [0.88313064, 0.23376667, 0.1963511],
                      [0.88407124, 0.23759875, 0.20047217],
                      [0.88499289, 0.24142153, 0.20455256],
                      [0.88589445, 0.24523692, 0.20859929],
                      [0.88677895, 0.24904025, 0.21261042],
                      [0.88764808, 0.25282915, 0.2165865],
                      [0.88850042, 0.25660628, 0.22053374],
                      [0.88933688, 0.26037056, 0.22445335],
                      [0.89015916, 0.26411988, 0.22834501],
                      [0.89096912, 0.26785197, 0.23220794],
                      [0.89176527, 0.27156962, 0.2360472],
                      [0.8925482, 0.27527245, 0.23986378],
                      [0.89331848, 0.27896019, 0.24365864],
                      [0.89407662, 0.28263264, 0.24743268],
                      [0.89482307, 0.28628971, 0.25118678],
                      [0.89555873, 0.28993071, 0.25492099],
                      [0.89628423, 0.29355535, 0.25863576],
                      [0.89699899, 0.29716499, 0.2623333],
                      [0.89770333, 0.30075971, 0.26601432],
                      [0.89839755, 0.30433965, 0.26967946],
                      [0.89908192, 0.30790497, 0.27332936],
                      [0.89975669, 0.31145585, 0.27696462],
                      [0.90042207, 0.3149925, 0.28058581],
                      [0.90107829, 0.31851514, 0.28419347],
                      [0.90172555, 0.32202399, 0.28778812],
                      [0.90236402, 0.32551929, 0.29137026],
                      [0.90299387, 0.32900129, 0.29494034],
                      [0.90361526, 0.33247024, 0.29849881],
                      [0.90422834, 0.33592638, 0.3020461],
                      [0.90483326, 0.33936996, 0.3055826],
                      [0.90543013, 0.34280125, 0.3091087],
                      [0.9060191, 0.34622049, 0.31262475],
                      [0.90660026, 0.34962793, 0.31613111],
                      [0.90717415, 0.35302337, 0.31962759],
                      [0.90774112, 0.3564068, 0.3231142],
                      [0.90830067, 0.3597791, 0.326592],
                      [0.90885291, 0.36314054, 0.33006128],
                      [0.90939793, 0.36649133, 0.33352229],
                      [0.90993581, 0.36983172, 0.33697532],
                      [0.91046663, 0.37316193, 0.3404206],
                      [0.9109905, 0.37648218, 0.34385836],
                      [0.91150747, 0.37979267, 0.34728882],
                      [0.91201765, 0.38309363, 0.3507122],
                      [0.91252223, 0.38638411, 0.35412737],
                      [0.91302063, 0.38966501, 0.35753535],
                      [0.91351247, 0.39293699, 0.36093682],
                      [0.91399782, 0.39620022, 0.36433197],
                      [0.91447676, 0.3994549, 0.36772094],
                      [0.91494936, 0.40270119, 0.3711039],
                      [0.9154157, 0.40593927, 0.374481],
                      [0.91587731, 0.40916793, 0.37785075],
                      [0.91633311, 0.41238844, 0.3812146],
                      [0.91678285, 0.41560128, 0.38457303],
                      [0.91722659, 0.41880658, 0.38792615],
                      [0.9176644, 0.42200451, 0.39127406],
                      [0.91809637, 0.42519521, 0.39461689],
                      [0.91852437, 0.42837718, 0.39795283],
                      [0.91894669, 0.4315522, 0.40128387],
                      [0.91936333, 0.43472044, 0.40461017],
                      [0.91977439, 0.43788203, 0.4079318],
                      [0.92017992, 0.44103709, 0.41124884],
                      [0.9205813, 0.44418462, 0.41456007],
                      [0.92097785, 0.4473254, 0.41786632],
                      [0.92136906, 0.45046007, 0.42116827],
                      [0.92175499, 0.45358873, 0.42446599],
                      [0.92213572, 0.45671147, 0.42775951],
                      [0.92251279, 0.45982719, 0.43104748],
                      [0.92288516, 0.46293695, 0.43433107],
                      [0.92325249, 0.46604115, 0.43761071],
                      [0.92361488, 0.4691399, 0.44088644],
                      [0.92397269, 0.47223303, 0.44415801],
                      [0.92432719, 0.4753197, 0.44742438],
                      [0.92467691, 0.47840125, 0.45068703],
                      [0.92502192, 0.48147774, 0.45394601],
                      [0.92536232, 0.48454925, 0.45720133],
                      [0.92569945, 0.48761488, 0.46045186],
                      [0.9260325, 0.4906754, 0.46369848],
                      [0.92636111, 0.49373124, 0.46694161],
                      [0.92668535, 0.49678245, 0.47018127],
                      [0.9270061, 0.49982852, 0.47341677],
                      [0.92732343, 0.50286952, 0.47664816],
                      [0.92763658, 0.50590616, 0.47987622],
                      [0.92794562, 0.5089385, 0.48310096],
                      [0.92825118, 0.51196621, 0.48632193],
                      [0.92855374, 0.51498906, 0.48953881],
                      [0.92885238, 0.51800785, 0.49275248],
                      [0.92914718, 0.52102262, 0.49596295],
                      [0.92943869, 0.52403312, 0.49916986],
                      [0.92972742, 0.52703908, 0.50237283],
                      [0.93001251, 0.53004124, 0.5055727],
                      [0.93029404, 0.53303965, 0.50876946],
                      [0.93057256, 0.53603403, 0.51196275],
                      [0.93084843, 0.53902424, 0.51515235],
                      [0.93112094, 0.5420109, 0.51833892],
                      [0.93139017, 0.54499404, 0.52152246],
                      [0.93165674, 0.54797335, 0.52470256],
                      [0.93192072, 0.55094888, 0.52787923],
                      [0.93218162, 0.55392107, 0.53105295],
                      [0.93243954, 0.55688996, 0.53422369],
                      [0.93269516, 0.55985518, 0.537391],
                      [0.93294824, 0.562817, 0.54055514],
                      [0.93319852, 0.56577568, 0.54371638],
                      [0.93344613, 0.56873125, 0.54687469],
                      [0.93369178, 0.57168332, 0.55002959],
                      [0.93393495, 0.57463234, 0.55318155],
                      [0.93417564, 0.57757839, 0.55633063],
                      [0.934414, 0.58052146, 0.55947679],
                      [0.93465057, 0.58346132, 0.56261967],
                      [0.93488486, 0.58639835, 0.56575974],
                      [0.93511697, 0.58933259, 0.56889696],
                      [0.93534716, 0.59226396, 0.57203121],
                      [0.93557557, 0.59519246, 0.57516244],
                      [0.93580201, 0.59811829, 0.57829088],
                      [0.93602658, 0.60104148, 0.58141648],
                      [0.93624954, 0.60396195, 0.58453914],
                      [0.93647084, 0.60687982, 0.58765896],
                      [0.93669048, 0.60979517, 0.59077598],
                      [0.93690857, 0.61270801, 0.5938902],
                      [0.93712525, 0.61561835, 0.59700156],
                      [0.93734049, 0.61852628, 0.60011016],
                      [0.9375544, 0.62143182, 0.60321598],
                      [0.93776705, 0.62433501, 0.60631901],
                      [0.93797845, 0.62723591, 0.60941932],
                      [0.93818871, 0.63013455, 0.61251688],
                      [0.93839797, 0.63303091, 0.61561167],
                      [0.93860617, 0.63592512, 0.61870376],
                      [0.93881335, 0.63881722, 0.6217932],
                      [0.93901974, 0.64170715, 0.62487988],
                      [0.93922545, 0.64459494, 0.62796378],
                      [0.93943018, 0.64748082, 0.63104516],
                      [0.93963426, 0.65036467, 0.63412386],
                      [0.93983788, 0.65324648, 0.63719979],
                      [0.94004109, 0.65612629, 0.640273],
                      [0.94024345, 0.65900441, 0.64334382],
                      [0.94044556, 0.66188057, 0.6464119],
                      [0.94064754, 0.66475479, 0.64947721],
                      [0.94084923, 0.66762723, 0.65253994],
                      [0.94105041, 0.67049807, 0.65560026],
                      [0.94125169, 0.67336706, 0.65865783],
                      [0.94145319, 0.6762342, 0.66171264],
                      [0.94165444, 0.67909981, 0.66476503],
                      [0.94185562, 0.68196387, 0.66781494],
                      [0.94205723, 0.68482617, 0.67086211],
                      [0.94225939, 0.68768672, 0.6739065],
                      [0.94246136, 0.69054599, 0.67694865],
                      [0.9426637, 0.69340374, 0.67998824],
                      [0.94286682, 0.69625982, 0.68302508],
                      [0.94307084, 0.69911425, 0.68605914],
                      [0.94327472, 0.70196762, 0.68909111],
                      [0.94347942, 0.7048195, 0.69212046],
                      [0.94368526, 0.70766981, 0.69514703],
                      [0.94389233, 0.71051856, 0.69817083],
                      [0.9440994, 0.71336644, 0.70119265],
                      [0.94430769, 0.71621289, 0.70421179],
                      [0.94451746, 0.71905785, 0.70722817],
                      [0.94472881, 0.72190135, 0.71024175],
                      [0.94494044, 0.72474408, 0.71325338],
                      [0.94515356, 0.7275855, 0.71626236],
                      [0.94536851, 0.73042553, 0.71926857],
                      [0.9455854, 0.73326417, 0.72227198],
                      [0.94580304, 0.73610206, 0.72527333],
                      [0.94602227, 0.73893885, 0.72827217],
                      [0.94624369, 0.74177432, 0.73126823],
                      [0.9464674, 0.74460849, 0.73426148],
                      [0.94669263, 0.74744179, 0.73725241],
                      [0.9469193, 0.75027431, 0.74024111],
                      [0.94714851, 0.7531056, 0.743227],
                      [0.94738037, 0.75593566, 0.74621008],
                      [0.9476149, 0.75876456, 0.74919037],
                      [0.94785037, 0.76159315, 0.75216889],
                      [0.94808875, 0.76442059, 0.75514458],
                      [0.94833014, 0.76724688, 0.75811744],
                      [0.94857466, 0.77007204, 0.76108745],
                      [0.94882155, 0.77289648, 0.76405506],
                      [0.94907053, 0.77572038, 0.76702048],
                      [0.9493229, 0.77854322, 0.76998303],
                      [0.94957877, 0.781365, 0.77294272],
                      [0.94983825, 0.78418573, 0.77589952],
                      [0.95010039, 0.78700592, 0.77885398],
                      [0.95036528, 0.78982556, 0.78180609],
                      [0.95063405, 0.79264423, 0.7847553],
                      [0.95090683, 0.79546192, 0.78770159],
                      [0.95118373, 0.79827865, 0.79064495],
                      [0.95146444, 0.80109461, 0.79358557],
                      [0.95174779, 0.80391039, 0.79652409],
                      [0.95203555, 0.80672525, 0.79945963],
                      [0.95232785, 0.80953922, 0.8023922],
                      [0.9526248, 0.8123523, 0.80532175],
                      [0.95292652, 0.8151645, 0.80824829],
                      [0.95323259, 0.81797607, 0.81117204],
                      [0.95354218, 0.82078741, 0.81409346],
                      [0.95385688, 0.82359792, 0.81701179],
                      [0.95417682, 0.8264076, 0.81992702],
                      [0.95450211, 0.82921645, 0.82283912],
                      [0.95483289, 0.8320245, 0.82574806],
                      [0.95516928, 0.83483174, 0.82865382],
                      [0.95551112, 0.83763832, 0.83155651],
                      [0.9558574, 0.8404447, 0.83445663],
                      [0.95620972, 0.84325029, 0.83735346],
                      [0.95656821, 0.84605509, 0.84024696],
                      [0.95693303, 0.8488591, 0.8431371],
                      [0.95730432, 0.85166232, 0.84602383],
                      [0.95768224, 0.85446477, 0.84890711],
                      [0.95806695, 0.85726642, 0.8517869],
                      [0.95845861, 0.86006729, 0.85466315],
                      [0.95885738, 0.86286737, 0.85753579],
                      [0.95926345, 0.86566665, 0.86040479],
                      [0.95967676, 0.86846522, 0.86327017],
                      [0.96009718, 0.8712632, 0.86613201],
                      [0.9605256, 0.87406029, 0.86898995],
                      [0.96096222, 0.87685648, 0.87184389],
                      [0.96140728, 0.87965174, 0.87469376],
                      [0.96186104, 0.88244603, 0.87753945],
                      [0.96232376, 0.88523934, 0.88038086],
                      [0.96279572, 0.8880316, 0.88321788],
                      [0.96327722, 0.89082278, 0.88605038],
                      [0.9637686, 0.89361281, 0.88887824],
                      [0.96427019, 0.89640163, 0.8917013],
                      [0.96478238, 0.89918917, 0.8945194],
                      [0.96530554, 0.90197535, 0.89733239],
                      [0.96583889, 0.90476055, 0.90014056],
                      [0.96638393, 0.90754426, 0.9029433],
                      [0.96694119, 0.91032637, 0.90574038],
                      [0.96751124, 0.91310672, 0.90853154],
                      [0.96809472, 0.91588516, 0.9113165],
                      [0.96869235, 0.91866149, 0.91409497],
                      [0.96930492, 0.92143549, 0.91686659],
                      [0.96993317, 0.92420697, 0.91963106],
                      [0.97057458, 0.92697697, 0.92238927],
                      [0.9712336, 0.92974391, 0.92513948],
                      [0.97191149, 0.93250738, 0.92788117],
                      [0.97260975, 0.93526694, 0.93061374],
                      [0.9733251, 0.93802391, 0.93333827],
                      [0.97406336, 0.94077622, 0.93605257],
                      [0.97482753, 0.94352286, 0.93875547],
                      [0.97561389, 0.94626533, 0.94144807],
                      [0.97643072, 0.9490007, 0.94412722],
                      [0.97727809, 0.95172909, 0.94679245],
                      [0.97815973, 0.95444933, 0.94944174],
                      [0.97907633, 0.95716136, 0.95207418],
                      [0.98002346, 0.95986626, 0.95469757],
                      [0.98099291, 0.96256054, 0.95737842],
                      [0.98216249, 0.9651729, 0.96011629],
                      [0.98352471, 0.96773091, 0.96267983],
                      [0.98488115, 0.97030653, 0.96514497],
                      [0.98619811, 0.97290617, 0.96757952],
                      [0.98747894, 0.97552703, 0.96999703],
                      [0.98872999, 0.97816674, 0.97239524],
                      [0.989953, 0.98082545, 0.97476565],
                      [0.99115338, 0.98350268, 0.97709183],
                      [0.99233646, 0.98619941, 0.9793433],
                      [0.99352216, 0.98891487, 0.98145147],
                      [0.99481944, 0.99162674, 0.98321654],
                      [1.0, 1.0, 1.0]]

cmaps = {}
for (name, data) in (('germany', _germany_data),
                     ('germany_r', _germany_data[::-1]),
                     ('white_red', _white_red_data[::-1]),
                     ('white_red_r', _white_red_data)):

    cmaps[name] = ListedColormap(data, name=name)

fc_cdict = {
    'blue': [(0.0, 0.12156862765550613, 0.12156862765550613),
             (0.08, 0.16862745583057404, 0.16862745583057404),
             (0.16, 0.3019607961177826, 0.3019607961177826),
             (0.24, 0.5098039507865906, 0.5098039507865906),
             (0.32, 0.7803921699523926, 0.7803921699523926),
             (0.40, 0.9686274528503418, 0.9686274528503418),
             (0.48, 1.0, 1.0),
             (0.52, 1.0, 1.0),
             (0.68, 0.9411764740943909, 0.9411764740943909),
             (0.76, 0.8705882430076599, 0.8705882430076599),
             (0.84, 0.7647058963775635, 0.7647058963775635),
             (0.92, 0.6745098233222961, 0.6745098233222961),
             (1.0, 0.3803921639919281, 0.3803921639919281)],
    'green': [(0.0, 0.0, 0.0),
              (0.08, 0.0941176488995552, 0.0941176488995552),
              (0.16, 0.3764705955982208, 0.3764705955982208),
              (0.24, 0.6470588445663452, 0.6470588445663452),
              (0.32, 0.8588235378265381, 0.8588235378265381),
              (0.4, 0.9686274528503418, 0.9686274528503418),
              (0.48, 1.0, 1.0),
              (0.52, 1.0, 1.0),
              (0.68, 0.8980392217636108, 0.8980392217636108),
              (0.76, 0.772549033164978, 0.772549033164978),
              (0.84, 0.5764706134796143, 0.5764706134796143),
              (0.92, 0.4000000059604645, 0.4000000059604645),
              (1.0, 0.1882352977991104, 0.1882352977991104)],
    'red': [(0.0, 0.40392157435417175, 0.40392157435417175),
            (0.08, 0.6980392336845398, 0.6980392336845398),
            (0.16, 0.8392156958580017, 0.8392156958580017),
            (0.24, 0.95686274766922, 0.95686274766922),
            (0.32, 0.9921568632125854, 0.9921568632125854),
            (0.40, 0.9686274528503418, 0.9686274528503418),
            (0.48, 1.0, 1.0),
            (0.52, 1.0, 1.0),
            (0.68, 0.8196078538894653, 0.8196078538894653),
            (0.76, 0.572549045085907, 0.572549045085907),
            (0.84, 0.26274511218070984, 0.26274511218070984),
            (0.92, 0.12941177189350128, 0.12941177189350128),
            (1.0, 0.019607843831181526, 0.019607843831181526)]
}
fc_cmap_r = LinearSegmentedColormap('RdBuWhitespace_r', fc_cdict)


def reverse_colormap(cmap, name='my_cmap_r'):
    """
    In:
    cmap, name
    Out:
    my_cmap_r

    Explanation:
    t[0] goes from 0 to 1
    row i:   x  y0  y1 -> t[0] t[1] t[2]
                   /
                  /
    row i+1: x  y0  y1 -> t[n] t[1] t[2]

    so the inverse should do the same:
    row i+1: x  y1  y0 -> 1-t[0] t[2] t[1]
                   /
                  /
    row i:   x  y1  y0 -> 1-t[n] t[2] t[1]
    """
    reverse = []
    k = []

    for key in cmap._segmentdata:
        k.append(key)
        channel = cmap._segmentdata[key]
        data = []

        for t in channel:
            data.append((1 - t[0], t[2], t[1]))
        reverse.append(sorted(data))

    lin = dict(zip(k, reverse))
    my_cmap_r = LinearSegmentedColormap(name, lin)
    return my_cmap_r

fc_cmap = reverse_colormap(fc_cmap_r, 'RdBuWhitespace')

germany_cmap = cmaps['germany']
germany_r = cmaps['germany_r']
white_red = cmaps['white_red']
white_red_r = cmaps['white_red_r']
