# Copyright (C) 2019-2020, TomTom (http://tomtom.com).
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Tests for python template helpers."""

import pytest

from asciidoxy.generator.filters import InsertionFilter
from asciidoxy.model import Compound, Member, Parameter, ReturnValue, TypeRef, InnerTypeReference
from asciidoxy.templates.python.helpers import params, PythonTemplateHelper


@pytest.fixture
def python_class():
    compound = Compound("python")
    compound.name = "MyClass"

    def generate_member_function(name: str,
                                 has_return_value: bool = True,
                                 is_static: bool = False) -> Member:
        member = Member("python")
        member.kind = "function"
        member.name = name
        member.prot = "public"
        if has_return_value:
            member.returns = ReturnValue()
        if is_static:
            member.static = True
        return member

    def generate_member_variable(name: str) -> Member:
        member_variable = Member("python")
        member_variable.kind = "variable"
        member_variable.name = name
        member_variable.prot = "public"
        member_variable.returns = ReturnValue()
        member_variable.returns.type = TypeRef("python")
        return member_variable

    def generate_inner_class(name: str) -> InnerTypeReference:
        nested_class = Compound("python")
        nested_class.name = name
        inner_class_reference = InnerTypeReference(language="python")
        inner_class_reference.name = nested_class.name
        inner_class_reference.referred_object = nested_class
        inner_class_reference.prot = "public"
        return inner_class_reference

    compound.members = [
        generate_member_function("__init__", has_return_value=False, is_static=False),
        generate_member_function("public_static_method", is_static=True),
        generate_member_function("_private_static_method", is_static=True),
        generate_member_function("__mangled_private_static_method", is_static=True),
        generate_member_function("public_method"),
        generate_member_function("_private_method"),
        generate_member_function("__mangled_private_method"),
        generate_member_function("__add__"),
        generate_member_variable("public_variable"),
        generate_member_variable("_private_variable"),
        generate_member_variable("__mangled_private_variable"),
    ]

    compound.inner_classes = [
        generate_inner_class("NestedClass"),
        generate_inner_class("_PrivateNestedClass"),
        generate_inner_class("__MangledPrivateNestedClass"),
    ]

    return compound


def test_params__empty():
    member = Member("lang")
    assert list(params(member)) == []


def test_params__normal():
    type1 = TypeRef("lang")
    type1.name = "int"

    type2 = TypeRef("lang")
    type2.name = "float"

    param1 = Parameter()
    param1.type = type1
    param1.name = "arg1"

    param2 = Parameter()
    param2.type = type2
    param2.name = "arg2"

    member = Member("lang")
    member.params = [param1, param2]

    assert list(params(member)) == [param1, param2]


def test_params__self():
    type1 = TypeRef("lang")
    type1.name = "self"

    type2 = TypeRef("lang")
    type2.name = "float"

    param1 = Parameter()
    param1.type = type1

    param2 = Parameter()
    param2.type = type2
    param2.name = "arg2"

    member = Member("lang")
    member.params = [param1, param2]

    assert list(params(member)) == [param2]


def test_params__cls():
    type1 = TypeRef("lang")
    type1.name = "cls"

    type2 = TypeRef("lang")
    type2.name = "float"

    param1 = Parameter()
    param1.type = type1

    param2 = Parameter()
    param2.type = type2
    param2.name = "arg2"

    member = Member("lang")
    member.params = [param1, param2]

    assert list(params(member)) == [param2]


def test_params__no_type():
    param1 = Parameter()
    param1.type = None
    param1.name = "arg1"

    param2 = Parameter()
    param2.type = None
    param2.name = "arg2"

    member = Member("lang")
    member.params = [param1, param2]

    assert list(params(member)) == [param1, param2]


@pytest.fixture
def helper(empty_context, python_class):
    return PythonTemplateHelper(empty_context, python_class, InsertionFilter())


def test_public_static_methods__no_filter(helper):
    result = list(m.name for m in helper.public_static_methods())
    assert sorted(result) == sorted(["public_static_method"])


def test_public_static_methods__filter_match(helper):
    helper.insert_filter = InsertionFilter(members="ALL")
    result = list(m.name for m in helper.public_static_methods())
    assert sorted(result) == sorted(["public_static_method"])


def test_public_static_methods__filter_no_match(helper):
    helper.insert_filter = InsertionFilter(members="NONE")
    result = list(m.name for m in helper.public_static_methods())
    assert not result


def test_public_methods__no_filter(helper):
    result = list(m.name for m in helper.public_methods())
    assert sorted(result) == sorted(["public_method"])


def test_public_methods__filter_match(helper):
    helper.insert_filter = InsertionFilter(members="ALL")
    result = list(m.name for m in helper.public_methods())
    assert sorted(result) == sorted(["public_method"])


def test_public_methods__filter_no_match(helper):
    helper.insert_filter = InsertionFilter(members="NONE")
    result = list(m.name for m in helper.public_methods())
    assert not result


def test_public_constructors__no_filter(helper):
    result = list(m.name for m in helper.public_constructors())
    assert sorted(result) == sorted(["__init__"])


def test_public_constructors__filter_match(helper):
    helper.insert_filter = InsertionFilter(members="ALL")
    result = list(m.name for m in helper.public_constructors())
    assert sorted(result) == sorted(["__init__"])


def test_public_constructors__filter_no_match(helper):
    helper.insert_filter = InsertionFilter(members="NONE")
    result = list(m.name for m in helper.public_constructors())
    assert not result


def test_public_enclosed_types__no_filter(helper):
    result = list(m.name for m in helper.public_complex_enclosed_types())
    assert sorted(result) == sorted(["NestedClass"])


def test_public_enclosed_types__filter_match(helper):
    helper.insert_filter = InsertionFilter(inner_classes="ALL")
    result = list(m.name for m in helper.public_complex_enclosed_types())
    assert sorted(result) == sorted(["NestedClass"])


def test_public_enclosed_types__filter_no_match(helper):
    helper.insert_filter = InsertionFilter(inner_classes="NONE")
    result = list(m.name for m in helper.public_complex_enclosed_types())
    assert not result


def test_public_variables__no_filter(helper):
    result = list(m.name for m in helper.public_variables())
    assert sorted(result) == sorted(["public_variable"])


def test_public_variables__filter_match(helper):
    helper.insert_filter = InsertionFilter(members="ALL")
    result = list(m.name for m in helper.public_variables())
    assert sorted(result) == sorted(["public_variable"])


def test_public_variables__filter_no_match(helper):
    helper.insert_filter = InsertionFilter(members="NONE")
    result = list(m.name for m in helper.public_variables())
    assert not result


def test_method_signature__no_params(empty_context):
    method = Member("python")
    method.name = "ShortMethod"

    method.returns = ReturnValue()
    method.returns.type = TypeRef("python", "None")

    helper = PythonTemplateHelper(empty_context)
    assert helper.method_signature(method) == "def ShortMethod() -> None"


def test_method_signature__single_param(empty_context):
    method = Member("python")
    method.name = "ShortMethod"

    method.returns = ReturnValue()
    method.returns.type = TypeRef("python", "int")

    method.params = [Parameter()]
    method.params[0].name = "value"
    method.params[0].type = TypeRef("python", "int")

    helper = PythonTemplateHelper(empty_context)
    assert helper.method_signature(method) == "def ShortMethod(value: int) -> int"


def test_method_signature__single_param__too_wide(empty_context):
    method = Member("python")
    method.name = "ShortMethod"

    method.returns = ReturnValue()
    method.returns.type = TypeRef("python", "str")

    method.params = [Parameter()]
    method.params[0].name = "value"
    method.params[0].type = TypeRef("python", "int")

    helper = PythonTemplateHelper(empty_context)
    assert (helper.method_signature(method, max_width=20) == """\
def ShortMethod(
    value: int) -> str""")


def test_method_signature__multiple_params(empty_context):
    method = Member("python")
    method.name = "ShortMethod"

    method.returns = ReturnValue()
    method.returns.type = TypeRef("python", "None")

    method.params = [Parameter(), Parameter(), Parameter()]
    method.params[0].name = "value"
    method.params[0].type = TypeRef("python", "int")
    method.params[1].name = "other_value"
    method.params[1].type = TypeRef("python", "float")
    method.params[2].name = "text"
    method.params[2].type = TypeRef("python", "str")

    helper = PythonTemplateHelper(empty_context)
    assert (helper.method_signature(method) == """\
def ShortMethod(value: int,
                other_value: float,
                text: str) -> None""")


def test_method_signature__multiple_params__first_param_too_wide(empty_context):
    method = Member("python")
    method.name = "ShortMethod"

    method.returns = ReturnValue()
    method.returns.type = TypeRef("python", "None")

    method.params = [Parameter(), Parameter(), Parameter()]
    method.params[0].name = "value"
    method.params[0].type = TypeRef("python", "int")
    method.params[1].name = "other_value"
    method.params[1].type = TypeRef("python", "float")
    method.params[2].name = "text"
    method.params[2].type = TypeRef("python", "str")

    helper = PythonTemplateHelper(empty_context)
    assert (helper.method_signature(method, max_width=20) == """\
def ShortMethod(
    value: int,
    other_value: float,
    text: str) -> None""")


def test_method_signature__multiple_params__last_param_too_wide(empty_context):
    method = Member("python")
    method.name = "ShortMethod"

    method.returns = ReturnValue()
    method.returns.type = TypeRef("python", "Type")

    method.params = [Parameter(), Parameter(), Parameter()]
    method.params[0].name = "value"
    method.params[0].type = TypeRef("python", "int")
    method.params[1].name = "other_value"
    method.params[1].type = TypeRef("python", "float")
    method.params[2].name = "text" * 10
    method.params[2].type = TypeRef("python", "str")

    helper = PythonTemplateHelper(empty_context)
    assert (helper.method_signature(method, max_width=40) == f"""\
def ShortMethod(
    value: int,
    other_value: float,
    {"text" * 10}: str) -> Type""")


def test_method_signature__ignore_return_type_xref_length(empty_context):
    method = Member("python")
    method.name = "ShortMethod"

    method.returns = ReturnValue()
    method.returns.type = TypeRef("python", "Type")
    method.returns.type.id = "ab" * 80

    method.params = [Parameter()]
    method.params[0].name = "value"
    method.params[0].type = TypeRef("python", "int")

    helper = PythonTemplateHelper(empty_context)
    assert (
        helper.method_signature(method) == f"def ShortMethod(value: int) -> xref:{'ab' * 80}[Type]")


def test_method_signature__ignore_param_type_xref_length(empty_context):
    method = Member("python")
    method.name = "ShortMethod"

    method.returns = ReturnValue()
    method.returns.type = TypeRef("python", "None")

    method.params = [Parameter()]
    method.params[0].name = "value"
    method.params[0].type = TypeRef("python", "int")
    method.params[0].type.id = "ab" * 80

    helper = PythonTemplateHelper(empty_context)
    assert (
        helper.method_signature(method) == f"def ShortMethod(value: xref:{'ab' * 80}[int]) -> None")


def test_parameter(empty_context):
    ref = TypeRef("lang")
    ref.name = "MyType"
    ref.id = "lang-tomtom_1_MyType"

    param = Parameter()
    param.type = ref
    param.name = "arg"

    helper = PythonTemplateHelper(empty_context)
    assert helper.parameter(param) == "arg: xref:lang-tomtom_1_MyType[MyType]"


def test_parameter__no_link(empty_context):
    ref = TypeRef("lang")
    ref.name = "MyType"
    ref.id = "lang-tomtom_1_MyType"

    param = Parameter()
    param.type = ref
    param.name = "arg"

    helper = PythonTemplateHelper(empty_context)
    assert helper.parameter(param, link=False) == "arg: MyType"


def test_parameter__default_value(empty_context):
    ref = TypeRef("lang")
    ref.name = "MyType"
    ref.id = "lang-tomtom_1_MyType"

    param = Parameter()
    param.type = ref
    param.name = "arg"
    param.default_value = "12"

    helper = PythonTemplateHelper(empty_context)
    assert helper.parameter(param,
                            default_value=True) == "arg: xref:lang-tomtom_1_MyType[MyType] = 12"


def test_parameter__ignore_default_value(empty_context):
    ref = TypeRef("lang")
    ref.name = "MyType"
    ref.id = "lang-tomtom_1_MyType"

    param = Parameter()
    param.type = ref
    param.name = "arg"
    param.default_value = "12"

    helper = PythonTemplateHelper(empty_context)
    assert helper.parameter(param, default_value=False) == "arg: xref:lang-tomtom_1_MyType[MyType]"
