# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['cloudsigma']

package_data = \
{'': ['*']}

install_requires = \
['backports.ssl-match-hostname>=3.4.0.2',
 'configobj>=5.0.6',
 'future>=0.18.2',
 'iso8601>=0.1.12',
 'nose>=1.3.7',
 'pyserial>=3.4',
 'requests>=2.23',
 'simplejson>=3.17.0',
 'six>=1.14.0',
 'websocket-client>=0.57.0']

setup_kwargs = {
    'name': 'cloudsigma',
    'version': '1.0',
    'description': "CloudSigma's official python library.",
    'long_description': '# CloudSigma\'s Python Library\n\n## Config file\n\nIn order for the CloudSigma library to interact with the API, you need to provide your credentials. These are set in the file `~/.cloudsigma.conf`. Here\'s a sample version of the file that talks to the Zürich datacenter. If you instead want to use the Frankfurt datacenter, simply replace \'zrh\' with \'fra\' in the api_endpoint and ws_endpoint. Please note that this is not required in order to read back meta data on the server.\n\n\n```python\napi_endpoint = https://zrh.cloudsigma.com/api/2.0/\nws_endpoint = wss://direct.zrh.cloudsigma.com/websocket\nusername = user@domain.com\npassword = secret\n\n# Only needed for the integration/unit tests.\npersistent_drive_name=foobar\npersistent_drive_ssh_password=sshsecret\n```\n\nSince this file includes credentials, it is highly recommended that you set the permission of the file to 600 (`chmod 600 ~/.cloudsigma.conf`)\n\n\n## Installation\n\n\n### Mac OS X\n\n```bash\nsudo pip install cloudsigma\n```\n\n### Ubuntu / Debian\n\n```bash\nsudo apt -y install python-pip\npip install cloudsigma\n```\n\n### CentOS / RHEL\n\n```bash\nyum install -y python3-pip\npip install cloudsigma\n```\n\n## Using pycloudsigma\n\n### Imports and definitions\n\n```python\nimport cloudsigma\nfrom pprint import pprint\n\ndrive = cloudsigma.resource.Drive()\nserver = cloudsigma.resource.Server()\n```\n\n#### Create a drive\n\n```python\ntest_disk = {\n    \'name\': \'test_drive\',\n    \'size\': 1073741824 * 1,\n    \'media\': \'disk\'\n}\nmy_test_disk = drive.create(test_disk)\n```\n\nPrint back the result:\n\n```python\npprint(my_test_disk)\n{u\'affinities\': [],\n u\'allow_multimount\': False,\n u\'jobs\': [],\n u\'licenses\': [],\n u\'media\': u\'disk\',\n u\'meta\': {},\n u\'mounted_on\': [],\n u\'name\': u\'test_drive\',\n u\'owner\': {u\'resource_uri\': u\'/api/2.0/user/b4b9XXX-ba52-4ad0-9837-a2672652XXX/\',\n            u\'uuid\': u\'b4b9XXX-ba52-4ad0-9837-a2672652XXX\'},\n u\'resource_uri\': u\'/api/2.0/drives/5c33e407-23b9-XXX-b007-3a302eXXXX/\',\n u\'size\': 1073741824,\n u\'status\': u\'creating\',\n u\'storage_type\': None,\n u\'tags\': [],\n u\'uuid\': u\'5c33e407-23b9-XXX-b007-3a302eXXXX\'}\n```\n\n### Create a server without a drive\n\n```python\ntest_server = {\n    \'name\': \'My Test Server\',\n    \'cpu\': 1000,\n    \'mem\': 512 * 1024 ** 2,\n    \'vnc_password\': \'test_server\'\n}\nmy_test_server = server.create(test_server)\n```\n\nPrint back the result:\n\n```python\npprint(my_test_server)\n{u\'context\': True,\n u\'cpu\': 1000,\n u\'cpus_instead_of_cores\': False,\n u\'drives\': [],\n u\'enable_numa\': False,\n u\'hv_relaxed\': False,\n u\'hv_tsc\': False,\n u\'mem\': 536870912,\n u\'meta\': {},\n u\'name\': u\'My Test Server\',\n u\'nics\': [],\n u\'owner\': {u\'resource_uri\': u\'/api/2.0/user/b4b9XXX-ba52-4ad0-9837-a2672652XXX/\',\n            u\'uuid\': u\'b4b9XXX-ba52-4ad0-9837-a2672652XXX\'},\n u\'requirements\': [],\n u\'resource_uri\': u\'/api/2.0/servers/4d5bXXX-4da0-43cd-83aY-18b047014XXXX/\',\n u\'runtime\': None,\n u\'smp\': 1,\n u\'status\': u\'stopped\',\n u\'tags\': [],\n u\'uuid\': u\'4d5bXXX-4da0-43cd-83aY-18b047014XXXX\',\n u\'vnc_password\': u\'test_server\'}\n```\n\n### Attach a drive the drive and a NIC to the server\n\nWe could of course have attached this above, but in order to keep things simple, let\'s do this in two phases.\n\nAttach the drive:\n\n```python\nmy_test_server[\'drives\'] = [{\n    \'boot_order\': 1,\n    \'dev_channel\': \'0:0\',\n    \'device\': \'virtio\',\n    \'drive\': my_test_disk[\'uuid\']\n}]\n```\n\nAttach a public network interface:\n\n```python\nmy_test_server[\'nics\']  = [{\n    \'ip_v4_conf\': {\n        \'conf\': \'dhcp\',\n        \'ip\': None\n    },\n    \'model\': \'virtio\',\n    \'vlan\': None\n}]\n```\n\n**Optional**: Add a user-defined SSH key:\n\n```python\nmy_test_server[\'meta\'] = {\'ssh_public_key\': \'ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABAQDoHuFV7Skbu9G1iVokXBdB+zN4wEbqGKijlExUPmxuB6MXDBWCmXUEmMRLerTm3a8QMA+8Vyech0/TWQscYvs8xzM/HkRAqKwhhjPMRlfHgy+QKjRD8P989AYMnNcSYe8DayElFXoLYKwsDmoUcsnbf5H+f6agiBkWqz5odb8fvc2rka0X7+p3tDyKFJRt2OugPqLR9fhWddie65DBxAcycnScoqLW0+YAxakfWlKDUqwerIjuRN2VJ7T7iHywcXhvAU060CEtpWW7bE9T/PIoj/N753QDLYrmqtvqAQqU0Ss5rIqS8bYJXyM0zTKwIuncek+k+b9ButBf/Nx5ehjN vagrant@precise64\'}\n```\n\nPush the settings:\n\n```python\nserver.update(my_test_server[\'uuid\'], my_test_server)\n```\n\n### Start the server\n\n```python\nserver.start(my_test_server[\'uuid\'])\n```\n\n### Stop the server\n\n```python\nserver.stop(my_test_server[\'uuid\'])\n```\n\n### Reading Meta Data\n\nCloudSigma supports the notion of exposing meta data to guests. Using the Python library, this can be done very easily. **Please note** that you do not need `~/.cloudsigma.conf` in order to use this feature, as the data is read directly from `/dev/ttyS1`. More information on how to this works can be found [here](https://cloudsigma-docs.readthedocs.io/en/2.14.1/server_context.html#setting-up-the-virtual-serial-port).\n\nBy default, various system information is exposed, but it is also possible to push user-defined data, such as an SSH-key to the guest.\n\nHere\'s snippet that demonstrates how to read the meta meta data from a given server using the python library:\n\n```python\nimport cloudsigma\nmetadata = cloudsigma.metadata.GetServerMetadata().get()\n\nfrom pprint import pprint\npprint(metadata)\n{u\'cpu\': 1000,\n u\'cpus_instead_of_cores\': False,\n u\'drives\': [{u\'boot_order\': 1,\n              u\'dev_channel\': u\'0:0\',\n              u\'device\': u\'virtio\',\n              u\'drive\': {u\'affinities\': [],\n                         u\'allow_multimount\': False,\n                         u\'licenses\': [],\n                         u\'media\': u\'disk\',\n                         u\'meta\': {u\'description\': u\'This is my test disk.\'},\n                         u\'name\': u\'SomeName\',\n                         u\'size\': 21474836480,\n                         u\'tags\': [],\n                         u\'uuid\': u\'19757XXX-8173-46ba-8822-YYYYc6bZZZZ\'}}],\n u\'enable_numa\': False,\n u\'hv_relaxed\': False,\n u\'hv_tsc\': False,\n u\'mem\': 536870912,\n u\'meta\': {u\'description\': u\'This is my test server.\'},\n u\'name\': u\'MyTestServer\',\n u\'nics\': [{u\'boot_order\': None,\n            u\'ip_v4_conf\': {u\'conf\': u\'dhcp\',\n                            u\'ip\': {u\'gateway\': u\'123.123.123.123\',\n                                    u\'meta\': {},\n                                    u\'nameservers\': [u\'123.123.123.123\',\n                                                     u\'123.123.123.123\',\n                                                     u\'8.8.8.8\'],\n                                    u\'netmask\': 24,\n                                    u\'tags\': [],\n                                    u\'uuid\': u\'123.123.123.123\'}},\n            u\'ip_v6_conf\': None,\n            u\'mac\': u\'22:bd:c4:XX:XX:XX\',\n            u\'model\': u\'virtio\',\n            u\'vlan\': None}],\n u\'requirements\': [],\n u\'smp\': 1,\n u\'tags\': [],\n u\'uuid\': u\'6cc0XXX-d024-4ecf-b0de-83dbc29ZZZ\',\n u\'vnc_password\': u\'NotMyPassword\'}\n```\n\nIf you get a permission error while running the above command, make sure you have access to read from `/dev/ttyS1`. For instance, on Ubuntu, the default owner is `root` and the group is set to `dialout`. Hence you need to either change the permission, or add your user to the group `sudo usermod -a -G dialout $(whoami)`. Please note that you will need to logout and log in again for the permission change to take effect.\n\nFor more examples on how to read and write meta data, please visit our [API documentation](https://autodetect.cloudsigma.com/docs/meta.html#examples).\n\n## Sample: Install SSH key from meta data\n\nIn the example above, we pushed an SSH key as meta data to a server. That\'s great, but what if we want to put this to use? Don\'t worry, we got you covered.\n\nThe code snippet below assumes that you have installed your SSH key into the server\'s meta data with the key \'ssh_public_key\'.\n\n```python\nimport cloudsigma\nimport os\nimport stat\n\nmetadata = cloudsigma.metadata.GetServerMetadata().get()\nssh_key = metadata[\'meta\'][\'ssh_public_key\']\n\n# Define paths\nhome = os.path.expanduser("~")\nssh_path = os.path.join(home, \'.ssh\')\nauthorized_keys = os.path.join(ssh_path, \'authorized_keys\')\n\n\ndef get_permission(path):\n    return oct(os.stat(ssh_path)[stat.ST_MODE])[-4:]\n\nif not os.path.isdir(ssh_path):\n    print(\'Creating folder %s\' % ssh_path)\n    os.makedirs(ssh_path)\n\nif get_permission(ssh_path) != 0700:\n    print(\'Setting permission for %s\' % ssh_path)\n    os.chmod(ssh_path, 0700)\n\n# We\'ll have to assume that there might be other keys installed.\n# We could do something fancy, like checking if the key is installed already,\n# but in order to keep things simple, we\'ll simply append the key.\nwith open(authorized_keys, \'a\') as auth_file:\n    auth_file.write(ssh_key + \'\\n\')\n\nif get_permission(authorized_keys) != 0600:\n    print(\'Setting permission for %s\' % authorized_keys)\n    os.chmod(authorized_keys, 0600)\n```\n\n[Download](https://raw.github.com/cloudsigma/pycloudsigma/master/samples/set_ssh_key.py)\n\n## Sample: Monitor websocket activity\n\nHere\'s a sample application that listens to activity on the websocket. You can run this application to see activity from the web interface.\n\n```python\nfrom cloudsigma.generic import GenericClient\nfrom cloudsigma.resource import Websocket\nfrom cloudsigma.errors import ClientError, PermissionError\n\nws = Websocket(timeout=None)\nclient = GenericClient()\n\nprint("Display Websocket activity.\\nExit with ^C.")\n\nwhile True:\n    try:\n        get_action = ws.ws.recv()\n        action_uri = get_action[\'resource_uri\']\n        print(\'Received Action: %s\' % get_action)\n        print(\'Result:\\n%s\' % client.get(action_uri))\n    except ClientError as e:\n        if e.args[0] == 404:\n            print("Resource %s was deleted" % action_uri)\n        else:\n            print(\'Error retrieving: %s\' % e)\n    except PermissionError as e:\n        print("No permissions for resource %s" % action_uri)\n```\n\n[Download](https://raw.github.com/cloudsigma/pycloudsigma/master/samples/monitor_websocket_activity.py)\n\n## Sample: A CLI for taking snapshots\n\nOne handy functionality in our API is the ability to take snapshots of a drive. Since the drive can be running while you take the snapshot, combined with the fact that you only pay for the delta between the original disk and the snapshot, this becomes a very powerful backup tool.\n\nUsing the snippet below, you can easily create automated backups. All you need to do is to run the snapshot blow in an automated fashion, such as using the crontab.\n\n\n```python\nimport cloudsigma\nimport sys\nfrom time import sleep\n\nsnapshot = cloudsigma.resource.Snapshot()\nsnapshot_done = False\n\nif len(sys.argv) < 3:\n    print(\'\\nUsage: ./snapshot.py drive-uuid snapshot-name\\n\')\n    sys.exit(1)\n\nsnapshot_data = {\n    \'drive\': sys.argv[1],\n    \'name\': sys.argv[2],\n}\n\ncreate_snapshot = snapshot.create(snapshot_data)\n\nwhile not snapshot_done:\n    snapshot_status = snapshot.get(create_snapshot[\'uuid\'])\n\n    if snapshot_status[\'status\'] == \'available\':\n        snapshot_done = True\n        print(\'\\nSnapshot successfully created\\n\')\n    else:\n        sleep(1)\n```\n\n[Download](https://raw.github.com/cloudsigma/pycloudsigma/master/samples/snapshot.py)\n\nThere\'s also another script named [snapshot_purge.py](https://raw.githubusercontent.com/cloudsigma/pycloudsigma/master/samples/snapshot_purge.py) that can used for automatically purging old snapshots.\n\n## Running the tests\n\nThere must be a VM available by the name that matches `persistent_drive_name`. This VM should be a server with SSH installed, where one can be log in as `root` with the password set in `persistent_drive_ssh_password`.\n\n',
    'author': 'CloudSigma AG',
    'author_email': 'dev-support@cloudsigma.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/cloudsigma/pycloudsigma',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.5,<4.0',
}


setup(**setup_kwargs)
