"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const cognito_generated_1 = require("./cognito.generated");
/**
 * Define a user pool domain
 */
class UserPoolDomain extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        if (!!props.customDomain === !!props.cognitoDomain) {
            throw new Error('One of, and only one of, cognitoDomain or customDomain must be specified');
        }
        if (((_a = props.cognitoDomain) === null || _a === void 0 ? void 0 : _a.domainPrefix) && !/^[a-z0-9-]+$/.test(props.cognitoDomain.domainPrefix)) {
            throw new Error('domainPrefix for cognitoDomain can contain only lowercase alphabets, numbers and hyphens');
        }
        const domainName = ((_b = props.cognitoDomain) === null || _b === void 0 ? void 0 : _b.domainPrefix) || ((_c = props.customDomain) === null || _c === void 0 ? void 0 : _c.domainName);
        const resource = new cognito_generated_1.CfnUserPoolDomain(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            domain: domainName,
            customDomainConfig: props.customDomain ? { certificateArn: props.customDomain.certificate.certificateArn } : undefined,
        });
        this.domainName = resource.ref;
    }
    /**
     * The domain name of the CloudFront distribution associated with the user pool domain.
     */
    get cloudFrontDomainName() {
        const sdkCall = {
            service: 'CognitoIdentityServiceProvider',
            action: 'describeUserPoolDomain',
            parameters: {
                Domain: this.domainName,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.domainName),
        };
        const customResource = new custom_resources_1.AwsCustomResource(this, 'CloudFrontDomainName', {
            resourceType: 'Custom::UserPoolCloudFrontDomainName',
            onCreate: sdkCall,
            onUpdate: sdkCall,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                // DescribeUserPoolDomain only supports access level '*'
                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html#amazoncognitouserpools-actions-as-permissions
                resources: ['*'],
            }),
        });
        return customResource.getResponseField('DomainDescription.CloudFrontDistribution');
    }
}
exports.UserPoolDomain = UserPoolDomain;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNlci1wb29sLWRvbWFpbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInVzZXItcG9vbC1kb21haW4udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFDQSx3Q0FBK0Q7QUFDL0QsZ0VBQXVIO0FBQ3ZILDJEQUF3RDtBQTBFeEQ7O0dBRUc7QUFDSCxNQUFhLGNBQWUsU0FBUSxlQUFRO0lBRzFDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMEI7O1FBQ2xFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLENBQUMsS0FBSyxDQUFDLFlBQVksS0FBSyxDQUFDLENBQUMsS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLDBFQUEwRSxDQUFDLENBQUM7U0FDN0Y7UUFFRCxJQUFJLE9BQUEsS0FBSyxDQUFDLGFBQWEsMENBQUUsWUFBWSxLQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQyxFQUFFO1lBQy9GLE1BQU0sSUFBSSxLQUFLLENBQUMsMEZBQTBGLENBQUMsQ0FBQztTQUM3RztRQUVELE1BQU0sVUFBVSxHQUFHLE9BQUEsS0FBSyxDQUFDLGFBQWEsMENBQUUsWUFBWSxNQUFJLE1BQUEsS0FBSyxDQUFDLFlBQVksMENBQUUsVUFBVyxDQUFBLENBQUM7UUFDeEYsTUFBTSxRQUFRLEdBQUcsSUFBSSxxQ0FBaUIsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3ZELFVBQVUsRUFBRSxLQUFLLENBQUMsUUFBUSxDQUFDLFVBQVU7WUFDckMsTUFBTSxFQUFFLFVBQVU7WUFDbEIsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsRUFBRSxjQUFjLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDdkgsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFVBQVUsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsb0JBQW9CO1FBQzdCLE1BQU0sT0FBTyxHQUFlO1lBQzFCLE9BQU8sRUFBRSxnQ0FBZ0M7WUFDekMsTUFBTSxFQUFFLHdCQUF3QjtZQUNoQyxVQUFVLEVBQUU7Z0JBQ1YsTUFBTSxFQUFFLElBQUksQ0FBQyxVQUFVO2FBQ3hCO1lBQ0Qsa0JBQWtCLEVBQUUscUNBQWtCLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUM7U0FDM0QsQ0FBQztRQUNGLE1BQU0sY0FBYyxHQUFHLElBQUksb0NBQWlCLENBQUMsSUFBSSxFQUFFLHNCQUFzQixFQUFFO1lBQ3pFLFlBQVksRUFBRSxzQ0FBc0M7WUFDcEQsUUFBUSxFQUFFLE9BQU87WUFDakIsUUFBUSxFQUFFLE9BQU87WUFDakIsTUFBTSxFQUFFLDBDQUF1QixDQUFDLFlBQVksQ0FBQztnQkFDM0Msd0RBQXdEO2dCQUN4RCxrSUFBa0k7Z0JBQ2xJLFNBQVMsRUFBRSxDQUFFLEdBQUcsQ0FBRTthQUNuQixDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxjQUFjLENBQUMsZ0JBQWdCLENBQUMsMENBQTBDLENBQUMsQ0FBQztJQUNyRixDQUFDO0NBQ0Y7QUFoREQsd0NBZ0RDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgSUNlcnRpZmljYXRlIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWNlcnRpZmljYXRlbWFuYWdlcic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIElSZXNvdXJjZSwgUmVzb3VyY2UgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IEF3c0N1c3RvbVJlc291cmNlLCBBd3NDdXN0b21SZXNvdXJjZVBvbGljeSwgQXdzU2RrQ2FsbCwgUGh5c2ljYWxSZXNvdXJjZUlkIH0gZnJvbSAnQGF3cy1jZGsvY3VzdG9tLXJlc291cmNlcyc7XG5pbXBvcnQgeyBDZm5Vc2VyUG9vbERvbWFpbiB9IGZyb20gJy4vY29nbml0by5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSVVzZXJQb29sIH0gZnJvbSAnLi91c2VyLXBvb2wnO1xuXG4vKipcbiAqIFJlcHJlc2VudHMgYSB1c2VyIHBvb2wgZG9tYWluLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElVc2VyUG9vbERvbWFpbiBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgZG9tYWluIHRoYXQgd2FzIHNwZWNpZmllZCB0byBiZSBjcmVhdGVkLlxuICAgKiBJZiBgY3VzdG9tRG9tYWluYCB3YXMgc2VsZWN0ZWQsIHRoaXMgaG9sZHMgdGhlIGZ1bGwgZG9tYWluIG5hbWUgdGhhdCB3YXMgc3BlY2lmaWVkLlxuICAgKiBJZiB0aGUgYGNvZ25pdG9Eb21haW5gIHdhcyB1c2VkLCBpdCBjb250YWlucyB0aGUgcHJlZml4IHRvIHRoZSBDb2duaXRvIGhvc3RlZCBkb21haW4uXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBPcHRpb25zIHdoaWxlIHNwZWNpZnlpbmcgY3VzdG9tIGRvbWFpblxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvY29nbml0by11c2VyLXBvb2xzLWFkZC1jdXN0b20tZG9tYWluLmh0bWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDdXN0b21Eb21haW5PcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBjdXN0b20gZG9tYWluIG5hbWUgdGhhdCB5b3Ugd291bGQgbGlrZSB0byBhc3NvY2lhdGUgd2l0aCB0aGlzIFVzZXIgUG9vbC5cbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGNlcnRpZmljYXRlIHRvIGFzc29jaWF0ZSB3aXRoIHRoaXMgZG9tYWluLlxuICAgKi9cbiAgcmVhZG9ubHkgY2VydGlmaWNhdGU6IElDZXJ0aWZpY2F0ZTtcbn1cblxuLyoqXG4gKiBPcHRpb25zIHdoaWxlIHNwZWNpZnlpbmcgYSBjb2duaXRvIHByZWZpeCBkb21haW4uXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9jb2duaXRvLXVzZXItcG9vbHMtYXNzaWduLWRvbWFpbi1wcmVmaXguaHRtbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvZ25pdG9Eb21haW5PcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBwcmVmaXggdG8gdGhlIENvZ25pdG8gaG9zdGVkIGRvbWFpbiBuYW1lIHRoYXQgd2lsbCBiZSBhc3NvY2lhdGVkIHdpdGggdGhlIHVzZXIgcG9vbC5cbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpblByZWZpeDogc3RyaW5nO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gY3JlYXRlIGEgVXNlclBvb2xEb21haW5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBVc2VyUG9vbERvbWFpbk9wdGlvbnMge1xuICAvKipcbiAgICogQXNzb2NpYXRlIGEgY3VzdG9tIGRvbWFpbiB3aXRoIHlvdXIgdXNlciBwb29sXG4gICAqIEVpdGhlciBgY3VzdG9tRG9tYWluYCBvciBgY29nbml0b0RvbWFpbmAgbXVzdCBiZSBzcGVjaWZpZWQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1wb29scy1hZGQtY3VzdG9tLWRvbWFpbi5odG1sXG4gICAqIEBkZWZhdWx0IC0gbm90IHNldCBpZiBgY29nbml0b0RvbWFpbmAgaXMgc3BlY2lmaWVkLCBvdGhlcndpc2UsIHRocm93cyBhbiBlcnJvci5cbiAgICovXG4gIHJlYWRvbmx5IGN1c3RvbURvbWFpbj86IEN1c3RvbURvbWFpbk9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIEFzc29jaWF0ZSBhIGNvZ25pdG8gcHJlZml4IGRvbWFpbiB3aXRoIHlvdXIgdXNlciBwb29sXG4gICAqIEVpdGhlciBgY3VzdG9tRG9tYWluYCBvciBgY29nbml0b0RvbWFpbmAgbXVzdCBiZSBzcGVjaWZpZWQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1wb29scy1hc3NpZ24tZG9tYWluLXByZWZpeC5odG1sXG4gICAqIEBkZWZhdWx0IC0gbm90IHNldCBpZiBgY3VzdG9tRG9tYWluYCBpcyBzcGVjaWZpZWQsIG90aGVyd2lzZSwgdGhyb3dzIGFuIGVycm9yLlxuICAgKi9cbiAgcmVhZG9ubHkgY29nbml0b0RvbWFpbj86IENvZ25pdG9Eb21haW5PcHRpb25zO1xufVxuXG4vKipcbiAqIFByb3BzIGZvciBVc2VyUG9vbERvbWFpbiBjb25zdHJ1Y3RcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBVc2VyUG9vbERvbWFpblByb3BzIGV4dGVuZHMgVXNlclBvb2xEb21haW5PcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSB1c2VyIHBvb2wgdG8gd2hpY2ggdGhpcyBkb21haW4gc2hvdWxkIGJlIGFzc29jaWF0ZWQuXG4gICAqL1xuICByZWFkb25seSB1c2VyUG9vbDogSVVzZXJQb29sO1xufVxuXG4vKipcbiAqIERlZmluZSBhIHVzZXIgcG9vbCBkb21haW5cbiAqL1xuZXhwb3J0IGNsYXNzIFVzZXJQb29sRG9tYWluIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJVXNlclBvb2xEb21haW4ge1xuICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBVc2VyUG9vbERvbWFpblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmICghIXByb3BzLmN1c3RvbURvbWFpbiA9PT0gISFwcm9wcy5jb2duaXRvRG9tYWluKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09uZSBvZiwgYW5kIG9ubHkgb25lIG9mLCBjb2duaXRvRG9tYWluIG9yIGN1c3RvbURvbWFpbiBtdXN0IGJlIHNwZWNpZmllZCcpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5jb2duaXRvRG9tYWluPy5kb21haW5QcmVmaXggJiYgIS9eW2EtejAtOS1dKyQvLnRlc3QocHJvcHMuY29nbml0b0RvbWFpbi5kb21haW5QcmVmaXgpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2RvbWFpblByZWZpeCBmb3IgY29nbml0b0RvbWFpbiBjYW4gY29udGFpbiBvbmx5IGxvd2VyY2FzZSBhbHBoYWJldHMsIG51bWJlcnMgYW5kIGh5cGhlbnMnKTtcbiAgICB9XG5cbiAgICBjb25zdCBkb21haW5OYW1lID0gcHJvcHMuY29nbml0b0RvbWFpbj8uZG9tYWluUHJlZml4IHx8IHByb3BzLmN1c3RvbURvbWFpbj8uZG9tYWluTmFtZSE7XG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuVXNlclBvb2xEb21haW4odGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgdXNlclBvb2xJZDogcHJvcHMudXNlclBvb2wudXNlclBvb2xJZCxcbiAgICAgIGRvbWFpbjogZG9tYWluTmFtZSxcbiAgICAgIGN1c3RvbURvbWFpbkNvbmZpZzogcHJvcHMuY3VzdG9tRG9tYWluID8geyBjZXJ0aWZpY2F0ZUFybjogcHJvcHMuY3VzdG9tRG9tYWluLmNlcnRpZmljYXRlLmNlcnRpZmljYXRlQXJuIH0gOiB1bmRlZmluZWQsXG4gICAgfSk7XG5cbiAgICB0aGlzLmRvbWFpbk5hbWUgPSByZXNvdXJjZS5yZWY7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGRvbWFpbiBuYW1lIG9mIHRoZSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbiBhc3NvY2lhdGVkIHdpdGggdGhlIHVzZXIgcG9vbCBkb21haW4uXG4gICAqL1xuICBwdWJsaWMgZ2V0IGNsb3VkRnJvbnREb21haW5OYW1lKCk6IHN0cmluZyB7XG4gICAgY29uc3Qgc2RrQ2FsbDogQXdzU2RrQ2FsbCA9IHtcbiAgICAgIHNlcnZpY2U6ICdDb2duaXRvSWRlbnRpdHlTZXJ2aWNlUHJvdmlkZXInLFxuICAgICAgYWN0aW9uOiAnZGVzY3JpYmVVc2VyUG9vbERvbWFpbicsXG4gICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgIERvbWFpbjogdGhpcy5kb21haW5OYW1lLFxuICAgICAgfSxcbiAgICAgIHBoeXNpY2FsUmVzb3VyY2VJZDogUGh5c2ljYWxSZXNvdXJjZUlkLm9mKHRoaXMuZG9tYWluTmFtZSksXG4gICAgfTtcbiAgICBjb25zdCBjdXN0b21SZXNvdXJjZSA9IG5ldyBBd3NDdXN0b21SZXNvdXJjZSh0aGlzLCAnQ2xvdWRGcm9udERvbWFpbk5hbWUnLCB7XG4gICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OlVzZXJQb29sQ2xvdWRGcm9udERvbWFpbk5hbWUnLFxuICAgICAgb25DcmVhdGU6IHNka0NhbGwsXG4gICAgICBvblVwZGF0ZTogc2RrQ2FsbCxcbiAgICAgIHBvbGljeTogQXdzQ3VzdG9tUmVzb3VyY2VQb2xpY3kuZnJvbVNka0NhbGxzKHtcbiAgICAgICAgLy8gRGVzY3JpYmVVc2VyUG9vbERvbWFpbiBvbmx5IHN1cHBvcnRzIGFjY2VzcyBsZXZlbCAnKidcbiAgICAgICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9uY29nbml0b3VzZXJwb29scy5odG1sI2FtYXpvbmNvZ25pdG91c2VycG9vbHMtYWN0aW9ucy1hcy1wZXJtaXNzaW9uc1xuICAgICAgICByZXNvdXJjZXM6IFsgJyonIF0sXG4gICAgICB9KSxcbiAgICB9KTtcbiAgICByZXR1cm4gY3VzdG9tUmVzb3VyY2UuZ2V0UmVzcG9uc2VGaWVsZCgnRG9tYWluRGVzY3JpcHRpb24uQ2xvdWRGcm9udERpc3RyaWJ1dGlvbicpO1xuICB9XG59XG4iXX0=