"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CelltestsWidget = void 0;
/******************************************************************************
 *
 * Copyright (c) 2019, the nbcelltest authors.
 *
 * This file is part of the nbcelltest library, distributed under the terms of
 * the Apache License 2.0.  The full license can be found in the LICENSE file.
 *
 */
/* eslint-disable max-classes-per-file */
/* eslint-disable id-blacklist */
const widgets_1 = require("@lumino/widgets");
const cells_1 = require("@jupyterlab/cells");
const codeeditor_1 = require("@jupyterlab/codeeditor");
const codemirror_1 = require("@jupyterlab/codemirror");
const utils_1 = require("./utils");
const circleSvg = require("../style/circle.svg").default;
/**
 * Widget responsible for holding test controls
 *
 * @class      ControlsWidget (name)
 */
class ControlsWidget extends widgets_1.BoxPanel {
    constructor() {
        super({ direction: "top-to-bottom" });
        // eslint-disable-next-line @typescript-eslint/no-empty-function
        this.add = () => { };
        // eslint-disable-next-line @typescript-eslint/no-empty-function
        this.save = () => { };
        // eslint-disable-next-line @typescript-eslint/no-empty-function
        this.clear = () => { };
        /* Section Header */
        this.label = document.createElement("label");
        this.label.textContent = "Tests";
        this.svglabel = document.createElement("label");
        this.svg = document.createElement("svg");
        this.svg.innerHTML = circleSvg;
        this.svg = this.svg.firstChild;
        const div1 = document.createElement("div");
        div1.appendChild(this.label);
        const div2 = document.createElement("div");
        div1.appendChild(div2);
        div2.appendChild(this.svglabel);
        div2.appendChild(this.svg);
        this.node.appendChild(div1);
        this.node.classList.add(utils_1.CELLTEST_TOOL_CONTROLS_CLASS);
        /* Add button */
        const div3 = document.createElement("div");
        const add = document.createElement("button");
        add.textContent = "Add";
        add.onclick = () => {
            this.add();
        };
        /* Save button */
        const save = document.createElement("button");
        save.textContent = "Save";
        save.onclick = () => {
            this.save();
        };
        /* Clear button */
        const clear = document.createElement("button");
        clear.textContent = "Clear";
        clear.onclick = () => {
            this.clear();
        };
        /* add to container */
        div3.appendChild(add);
        div3.appendChild(save);
        div3.appendChild(clear);
        this.node.appendChild(div3);
    }
}
/**
 * Widget responsible for holding test controls
 *
 * @class      ControlsWidget (name)
 */
class RulesWidget extends widgets_1.BoxPanel {
    constructor() {
        super({ direction: "top-to-bottom" });
        // eslint-disable-next-line @typescript-eslint/no-empty-function
        this.save = () => { };
        /* Section Header */
        this.label = document.createElement("label");
        this.label.textContent = "Lint Rules";
        this.node.appendChild(this.label);
        this.node.classList.add(utils_1.CELLTEST_TOOL_RULES_CLASS);
        /* Add button */
        const div = document.createElement("div");
        for (const val of [].slice.call(utils_1.CELLTEST_RULES)) {
            const row = document.createElement("div");
            const span = document.createElement("span");
            span.textContent = val.label;
            const chkbx = document.createElement("input");
            chkbx.type = "checkbox";
            chkbx.name = val.key;
            const number = document.createElement("input");
            number.type = "number";
            number.name = val.key;
            chkbx.onchange = () => {
                number.disabled = !chkbx.checked;
                number.value = number.disabled ? "" : val.value;
                this.save();
            };
            number.onchange = () => {
                this.save();
            };
            if (val.min !== undefined) {
                number.min = val.min;
            }
            if (val.max !== undefined) {
                number.max = val.max;
            }
            if (val.step !== undefined) {
                number.step = val.step;
            }
            row.appendChild(span);
            row.appendChild(chkbx);
            row.appendChild(number);
            this.setByKey(val.key, row);
            div.appendChild(row);
        }
        this.node.appendChild(div);
    }
    getByKey(key) {
        switch (key) {
            case "lines_per_cell": {
                return this.lines_per_cell;
            }
            case "cells_per_notebook": {
                return this.cells_per_notebook;
            }
            case "function_definitions": {
                return this.function_definitions;
            }
            case "class_definitions": {
                return this.class_definitions;
            }
            case "cell_coverage": {
                return this.cell_coverage;
            }
        }
    }
    setByKey(key, elem) {
        switch (key) {
            case "lines_per_cell": {
                this.lines_per_cell = elem;
                break;
            }
            case "cells_per_notebook": {
                this.cells_per_notebook = elem;
                break;
            }
            case "function_definitions": {
                this.function_definitions = elem;
                break;
            }
            case "class_definitions": {
                this.class_definitions = elem;
                break;
            }
            case "cell_coverage": {
                this.cell_coverage = elem;
                break;
            }
        }
    }
    getValuesByKey(key) {
        let elem;
        switch (key) {
            case "lines_per_cell": {
                elem = this.lines_per_cell;
                break;
            }
            case "cells_per_notebook": {
                elem = this.cells_per_notebook;
                break;
            }
            case "function_definitions": {
                elem = this.function_definitions;
                break;
            }
            case "class_definitions": {
                elem = this.class_definitions;
                break;
            }
            case "cell_coverage": {
                elem = this.cell_coverage;
                break;
            }
        }
        const chkbx = elem.querySelector('input[type="checkbox"]');
        const input = elem.querySelector('input[type="number"]');
        return { key, enabled: chkbx.checked, value: Number(input.value) };
    }
    setValuesByKey(key, checked = true, value = null) {
        let elem;
        switch (key) {
            case "lines_per_cell": {
                elem = this.lines_per_cell;
                break;
            }
            case "cells_per_notebook": {
                elem = this.cells_per_notebook;
                break;
            }
            case "function_definitions": {
                elem = this.function_definitions;
                break;
            }
            case "class_definitions": {
                elem = this.class_definitions;
                break;
            }
            case "cell_coverage": {
                elem = this.cell_coverage;
                break;
            }
        }
        const chkbx = elem.querySelector('input[type="checkbox"]');
        const input = elem.querySelector('input[type="number"]');
        if (input) {
            input.value = (value === null ? "" : String(value));
            input.disabled = !checked;
        }
        if (chkbx) {
            chkbx.checked = checked;
        }
    }
}
/**
 * Widget holding the Celltests widget, container for options and editor
 *
 * @class      CelltestsWidget (name)
 */
class CelltestsWidget extends widgets_1.Widget {
    constructor() {
        super();
        this.currentActiveCell = null;
        this.notebookTracker = null;
        this.editor = null;
        /* create layout */
        const layout = (this.layout = new widgets_1.PanelLayout());
        /* create options widget */
        const controls = (this.controls = new ControlsWidget());
        /* create options widget */
        const rules = (this.rules = new RulesWidget());
        /* create codemirror editor */
        // eslint-disable-next-line @typescript-eslint/unbound-method
        const editorOptions = { model: new cells_1.CodeCellModel({}), factory: codemirror_1.editorServices.factoryService.newInlineEditor };
        const editor = (this.editor = new codeeditor_1.CodeEditorWrapper(editorOptions));
        editor.addClass(utils_1.CELLTEST_TOOL_EDITOR_CLASS);
        editor.model.mimeType = "text/x-ipython";
        /* add options and editor to widget */
        layout.addWidget(controls);
        layout.addWidget(editor);
        layout.addWidget(rules);
        /* set add button functionality */
        controls.add = () => {
            this.fetchAndSetTests();
            return true;
        };
        /* set save button functionality */
        controls.save = () => {
            this.saveTestsForActiveCell();
            return true;
        };
        /* set clear button functionality */
        controls.clear = () => {
            this.deleteTestsForActiveCell();
            return true;
        };
        rules.save = () => {
            this.saveRulesForCurrentNotebook();
        };
    }
    fetchAndSetTests() {
        const tests = [];
        const splits = this.editor.model.value.text.split(/\n/);
        // eslint-disable-next-line @typescript-eslint/prefer-for-of
        for (let i = 0; i < splits.length; i++) {
            tests.push(splits[i] + "\n");
        }
        if (this.currentActiveCell !== null && this.currentActiveCell.model.type === "code") {
            this.currentActiveCell.model.metadata.set("celltests", tests);
            this.setIndicatorTests();
        }
    }
    loadTestsForActiveCell() {
        if (this.currentActiveCell !== null && this.currentActiveCell.model.type === "code") {
            let tests = this.currentActiveCell.model.metadata.get("celltests");
            let s = "";
            if (tests === undefined || tests.length === 0) {
                tests = ["# Use %cell to mark where the cell should be inserted, or add a line comment \"# no %cell\" to deliberately skip the cell\n", "%cell\n"];
                this.setIndicatorNoTests();
            }
            else {
                this.setIndicatorTests();
            }
            // eslint-disable-next-line @typescript-eslint/prefer-for-of
            for (let i = 0; i < tests.length; i++) {
                s += tests[i];
            }
            this.editor.model.value.text = s;
            this.editor.editor.setOption("readOnly", false);
        }
        else {
            this.editor.model.value.text = "# Not a code cell";
            this.editor.editor.setOption("readOnly", true);
            this.setIndicatorNonCode();
        }
    }
    saveTestsForActiveCell() {
        /* if currentActiveCell exists */
        if (this.currentActiveCell !== null && this.currentActiveCell.model.type === "code") {
            const tests = [];
            const splits = this.editor.model.value.text.split(/\n/);
            // eslint-disable-next-line @typescript-eslint/prefer-for-of
            for (let i = 0; i < splits.length; i++) {
                tests.push(splits[i] + "\n");
            }
            this.currentActiveCell.model.metadata.set("celltests", tests);
            this.setIndicatorTests();
        }
        else if (this.currentActiveCell !== null) {
            // TODO this?
            this.currentActiveCell.model.metadata.delete("celltests");
            this.setIndicatorNonCode();
        }
    }
    deleteTestsForActiveCell() {
        if (this.currentActiveCell !== null) {
            this.currentActiveCell.model.metadata.delete("celltests");
            this.setIndicatorNoTests();
        }
    }
    loadRulesForCurrentNotebook() {
        if (this.notebookTracker !== null) {
            const metadata = this.notebookTracker.currentWidget
                .model.metadata.get("celltests") || {};
            for (const rule of [].slice.call(utils_1.CELLTEST_RULES)) {
                this.rules.setValuesByKey(rule.key, rule.key in metadata, metadata[rule.key]);
            }
        }
    }
    saveRulesForCurrentNotebook() {
        if (this.notebookTracker !== null) {
            const metadata = {};
            for (const rule of [].slice.call(utils_1.CELLTEST_RULES)) {
                const settings = this.rules.getValuesByKey(rule.key);
                if (settings.enabled) {
                    metadata[settings.key] = settings.value;
                }
            }
            this.notebookTracker.currentWidget.model.metadata.set("celltests", metadata);
        }
    }
    get editorWidget() {
        return this.editor;
    }
    setIndicatorNoTests() {
        this.controls.svg.firstElementChild.firstElementChild.style.fill = "#e75c57";
        this.controls.svglabel.textContent = "(No Tests)";
    }
    setIndicatorTests() {
        this.controls.svg.firstElementChild.firstElementChild.style.fill = "#008000";
        this.controls.svglabel.textContent = "(Tests Exist)";
    }
    setIndicatorNonCode() {
        this.controls.svg.firstElementChild.firstElementChild.style.fill = "var(--jp-inverse-layout-color3)";
        this.controls.svglabel.textContent = "(Non Code Cell)";
    }
}
exports.CelltestsWidget = CelltestsWidget;
