import re
from collections.abc import Callable, Sequence
from typing import Annotated, NamedTuple, Self, TypeVar, cast

from pydantic import BaseModel, ConfigDict

from eval_framework.metrics.llm.graders.language import detect_language_of
from eval_framework.utils.helpers import count_bytes
from template_formatting.formatter import ConcatFormatter, Message, Role


class ConcatCompression(NamedTuple):
    """Helper class for storing compression info for the concat formatter.

    The concat formatter is used to avoid bias towards special tokens.
    """

    num_bytes: int
    num_tokens: int

    @classmethod
    def calculate(
        cls,
        messages: Sequence[Message],
        count_tokens: Callable[[str], int],
        choices: list[str] | None = None,
        completion: str | None = None,
    ) -> Self | None:
        """Calculate the compression info for the given messages and token counting function."""
        if (choices is None) == (completion is None):
            raise ValueError("Either possible_completions or completion must be provided, but not both.")
        concat_str = ConcatFormatter().format(messages, output_mode="string")

        if choices is not None:
            if any(c is None for c in choices):
                return None
            num_bytes = count_bytes(concat_str) + sum(count_bytes(c) for c in choices)
            num_tokens = count_tokens(concat_str) + sum(count_tokens(c) for c in choices)
        else:
            if completion is None:
                return None
            concat_str = f"{concat_str}{completion}"
            num_bytes = count_bytes(concat_str)
            num_tokens = count_tokens(concat_str)

        res = cls(num_bytes=num_bytes, num_tokens=num_tokens)
        if res.num_bytes > 0 and res.num_tokens > 0:
            return res
        else:
            return None


class BaseMetricContext(BaseModel):
    """Base class for metric context"""

    model_config = ConfigDict(extra="allow")


class LanguageMetricContext(BaseMetricContext):
    language: str


class UntemplatedPrompt(BaseMetricContext):
    untemplated_prompt: str


class Error(BaseModel):
    model_config = ConfigDict(extra="forbid")
    error_class: str
    message: str
    traceback: str


class PromptTooLongException(Exception):
    pass


class BaseCompletion(BaseModel):
    model_config = ConfigDict(extra="forbid")
    prompt: Annotated[str, "prompt as passed to the llm"]
    prompt_sequence_positions: Annotated[
        int | None,
        "number of sequence positions that the prompt occupies in the llm architecture (e.g. token count) "
        "or None if the info is not available",
    ]
    completion: Annotated[str, "completion as generated by the llm"]
    concat_compression: Annotated[ConcatCompression | None, "Compression info for the concat formatter."] = None


class RawCompletion(BaseCompletion):
    completion_sequence_positions: Annotated[
        int | None,
        "number of sequence positions that the completion occupies in the llm architecture "
        "(e.g. token count) or None if the info is not available",
    ]
    raw_completion_error: Error | None = None


class Completion(BaseCompletion):
    id: int
    subject: str
    ground_truth: str | None | list[str]
    messages: list[Message] | None  # needed for LLM as a judge
    raw_completion: Annotated[str, "raw completion as generated by the llm"]
    raw_completion_sequence_positions: Annotated[
        int | None,
        "number of sequence positions that the completion occupies in the llm architecture or None "
        "if the info is not available",
    ]
    context: list[BaseMetricContext] | BaseMetricContext | None = None
    error: Error | None = None

    @property
    def ground_truth_list(self) -> list[str] | list[None]:
        if isinstance(self.ground_truth, list):
            return self.ground_truth

        return [self.ground_truth]  # type: ignore[return-value]

    # Use just the raw messages for instructions to LLM judges, not the original prompt with its special formatting.
    # (see https://x.com/karpathy/status/1823418177197646104 for a motivation).
    @property
    def system_user_instruction(self) -> str:
        assert self.messages is not None
        return "\n\n".join([m.content for m in self.messages if m.role in (Role.SYSTEM, Role.USER)])

    @property
    def user_instruction(self) -> str:
        assert self.messages is not None
        return "\n\n".join([m.content for m in self.messages if m.role == Role.USER])

    @property
    def first_user_instruction(self) -> str:
        assert self.messages is not None
        user_messages = [m.content for m in self.messages if m.role == Role.USER]
        return user_messages[0] if user_messages else ""

    @property
    def all_but_first_user_instruction(self) -> str:
        assert self.messages is not None
        user_messages = [m.content for m in self.messages if m.role == Role.USER]
        return "\n\n".join(user_messages[1:]) if len(user_messages) > 1 else ""

    @property
    def last_user_instruction(self) -> str:
        assert self.messages is not None
        user_messages = [m.content for m in self.messages if m.role == Role.USER]
        return user_messages[-1] if user_messages else ""

    @property
    def sanitized_completion(self) -> str:
        # Make sure the completion doesn't contain any obvious special chars either by "breaking" any <|xyz|> pattern.
        return re.sub(r"<\|(\S+)\|>", r"<| \1 |>", self.completion)

    def get_completion_language(self) -> str:
        detected_language = ""
        if self.context and isinstance(self.context, LanguageMetricContext):
            detected_language = self.context.language
        else:
            detected_language_object = detect_language_of(self.completion)
            detected_language = detected_language_object.iso_code_639_1.name.lower() if detected_language_object else ""
        return detected_language

    def get_raw_completion_language(self) -> str:
        detected_language = ""
        if self.context and isinstance(self.context, LanguageMetricContext):
            detected_language = self.context.language
        else:
            detected_language_object = detect_language_of(self.raw_completion)
            detected_language = detected_language_object.iso_code_639_1.name.lower() if detected_language_object else ""
        return detected_language

    def get_instruction_language(self) -> str:
        detected_language = ""
        if self.context and isinstance(self.context, LanguageMetricContext):
            detected_language = self.context.language
        else:
            detected_language_object = detect_language_of(self.user_instruction)
            detected_language = detected_language_object.iso_code_639_1.name.lower() if detected_language_object else ""
        return detected_language


class BaseLoglikelihood(BaseModel):
    model_config = ConfigDict(extra="forbid")
    prompt: str
    prompt_sequence_positions: int | None
    loglikelihoods: dict[str, float]
    loglikelihoods_sequence_positions: dict[str, int]  # Is empty if the model does not provide sequence positions
    concat_compression: Annotated[ConcatCompression | None, "Compression info for the concat formatter"] = None


class RawLoglikelihood(BaseLoglikelihood):
    raw_loglikelihood_error: Error | None = None


class Loglikelihood(BaseLoglikelihood):
    id: int
    subject: str
    ground_truth: str | list[str]
    error: Error | None = None

    @property
    def ground_truth_list(self) -> list[str] | list[None]:
        if isinstance(self.ground_truth, list):
            return self.ground_truth
        return [self.ground_truth]  # type: ignore[return-value]


MetricContext = TypeVar("MetricContext", bound=BaseMetricContext)


def extract_context_metric[MetricContext: BaseMetricContext](
    response: Completion, metric_context_class: type[MetricContext]
) -> MetricContext:
    assert response.context is not None, "Expected context to be provided in the response"
    if not isinstance(response.context, list):
        assert isinstance(response.context, metric_context_class) or isinstance(response.context, BaseMetricContext), (
            f"Expected context to be of type {metric_context_class.__name__}, got {type(response.context).__name__}"
        )
        return cast(MetricContext, response.context)
    else:
        assert len(response.context) > 0, "Expected context to be provided in the response"
        context = [
            metric_context for metric_context in response.context if isinstance(metric_context, metric_context_class)
        ][0]
        assert context is not None, f"Expected {metric_context_class.__name__} to be provided in the response context"
        return cast(MetricContext, context)
