"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SlackNotifier = void 0;
const path = require("path");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
class SlackNotifier extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { slackBotToken, slackSigningSecret, slackChannel, slackChannelId, slackBotName, slackBotIcon, pipeline, stageNames } = props;
        this.environment = {
            SLACK_BOT_TOKEN: slackBotToken,
            SLACK_SIGNING_SECRET: slackSigningSecret,
            SLACK_CHANNEL: slackChannel || '',
            SLACK_CHANNEL_ID: slackChannelId || '',
        };
        if (slackBotName) {
            this.environment.SLACK_BOT_NAME = slackBotName;
        }
        if (slackBotIcon) {
            this.environment.SLACK_BOT_ICON = slackBotIcon;
        }
        const notifier = new aws_lambda_1.Function(scope, 'SlackNotifierFunction', {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code: aws_lambda_1.Code.asset(path.join(__dirname, 'lambdas', 'notifier')),
            environment: this.environment,
        });
        notifier.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [pipeline.pipelineArn],
            actions: ['codepipeline:GetPipelineState', 'codepipeline:GetPipelineExecution'],
        }));
        pipeline.onStateChange('SlackPipelineNotifierRule', {
            target: new aws_events_targets_1.LambdaFunction(notifier),
        });
        const stageRule = new aws_events_1.Rule(this, 'SlackStageNotifierRule');
        stageRule.addTarget(new aws_events_targets_1.LambdaFunction(notifier));
        stageRule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [pipeline.pipelineArn],
            detailType: ['CodePipeline Stage Execution State Change'],
        });
        if (stageNames) {
            stageRule.addEventPattern({
                detail: {
                    stage: stageNames,
                },
            });
        }
    }
    validate() {
        if (this.environment.SLACK_CHANNEL && this.environment.SLACK_CHANNEL_ID) {
            return ['Redundant Configuration: Please configure slackChannel by id (prop slackChannelId) OR name (prop slackChannel)'];
        }
        if (!this.environment.SLACK_CHANNEL && !this.environment.SLACK_CHANNEL_ID) {
            return ['Missing Configuration: Please configure slackChannel by id (prop slackChannelId) or name (prop slackChannel)'];
        }
        return [];
    }
}
exports.SlackNotifier = SlackNotifier;
//# sourceMappingURL=data:application/json;base64,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