"""
ANNA Protocol SDK - Python Client
VersÃ£o: 1.1.0 (com prÃ©-validaÃ§Ã£o)

SDK oficial para interagir com o ANNA Protocol
"""

import json
import time
from typing import Dict, Optional, List, Tuple, Any
from dataclasses import dataclass
from enum import Enum
from web3 import Web3
from eth_account import Account
from eth_account.messages import encode_typed_data


# ============================================================
# CONFIGURAÃ‡Ã•ES E CONSTANTES
# ============================================================

NETWORKS = {
    "polygon-amoy": {
        "rpc": "https://rpc-amoy.polygon.technology/",
        "chain_id": 80002,
        "explorer": "https://www.oklink.com/amoy"
    },
    "polygon-mainnet": {
        "rpc": "https://polygon-rpc.com",
        "chain_id": 137,
        "explorer": "https://polygonscan.com"
    }
}


# ============================================================
# TIPOS E ENUMS
# ============================================================

class VerificationTier(Enum):
    """NÃ­veis de verificaÃ§Ã£o disponÃ­veis"""
    BASIC = "basic"
    STANDARD = "standard"
    PREMIUM = "premium"


class AttestationStatus(Enum):
    """Status possÃ­veis de uma attestation"""
    PENDING = "pending"
    VERIFIED = "verified"
    REJECTED = "rejected"
    CHALLENGED = "challenged"


@dataclass
class ReasoningStep:
    """Estrutura de um passo de raciocÃ­nio"""
    step_number: int
    description: str
    rationale: str


@dataclass
class Reasoning:
    """Estrutura completa de raciocÃ­nio"""
    input: str
    reasoning_steps: List[ReasoningStep]
    conclusion: str
    confidence: float
    
    def to_dict(self) -> Dict:
        """Converte para dicionÃ¡rio"""
        return {
            "input": self.input,
            "reasoning_steps": [
                {
                    "step_number": step.step_number,
                    "description": step.description,
                    "rationale": step.rationale
                }
                for step in self.reasoning_steps
            ],
            "conclusion": self.conclusion,
            "confidence": self.confidence
        }


@dataclass
class AttestationResult:
    """Resultado de uma attestation"""
    attestation_id: str
    tx_hash: str
    status: AttestationStatus
    timestamp: int
    explorer_url: str
    verified: bool = False
    score: Optional[int] = None
    verifier: Optional[str] = None
    verification_time: Optional[int] = None


@dataclass
class Identity:
    """Identidade de um agente"""
    agent_id: int
    did: str
    address: str
    model_type: str
    model_version: str
    specializations: List[str]
    creation_time: int
    token_uri: Optional[str] = None


# ============================================================
# CLIENTE PRINCIPAL
# ============================================================

class ANNAClient:
    """Cliente principal do ANNA Protocol SDK"""
    
    # PadrÃµes proibidos (sincronizado com verifier.py)
    FORBIDDEN_PATTERNS = [
        "ignore previous instructions",
        "ignore all instructions",
        "jailbreak",
        "bypass",
        "hack",
        "disable safety",
        "ignore guidelines",
        "forget everything",
        "new instructions",
        "system prompt",
        "override"
    ]
    
    # Limites de tamanho (anti-spam)
    MIN_REASONING_SIZE = 100      # bytes
    MAX_REASONING_SIZE = 50000    # 50 KB
    
    # Threshold de aprovaÃ§Ã£o Tier 1
    TIER1_PASS_THRESHOLD = 60     # Score mÃ­nimo (0-100)
    
    def __init__(
        self,
        private_key: str,
        network: str = "polygon-amoy",
        identity_contract: Optional[str] = None,
        attestation_contract: Optional[str] = None,
        reputation_contract: Optional[str] = None
    ):
        """Inicializa o cliente ANNA"""
        if network not in NETWORKS:
            raise ValueError(f"Network invÃ¡lida. Use: {list(NETWORKS.keys())}")
        
        self.network = network
        self.network_config = NETWORKS[network]
        self.w3 = Web3(Web3.HTTPProvider(self.network_config["rpc"]))
        
        if not self.w3.is_connected():
            raise ConnectionError(f"NÃ£o foi possÃ­vel conectar ao RPC: {self.network_config['rpc']}")
        
        self.account = Account.from_key(private_key)
        self.address = self.account.address
        
        self.identity_contract = identity_contract
        self.attestation_contract = attestation_contract
        self.reputation_contract = reputation_contract
        
        self._load_abis()
        
        if self.identity_contract:
            self.identity = self.w3.eth.contract(
                address=Web3.to_checksum_address(self.identity_contract),
                abi=self.identity_abi
            )
        
        if self.attestation_contract:
            self.attestation = self.w3.eth.contract(
                address=Web3.to_checksum_address(self.attestation_contract),
                abi=self.attestation_abi
            )
        
        if self.reputation_contract:
            self.reputation = self.w3.eth.contract(
                address=Web3.to_checksum_address(self.reputation_contract),
                abi=self.reputation_abi
            )
    
    def _load_abis(self):
        """Carrega ABIs dos contratos"""
        self.identity_abi = [
            {
                "inputs": [
                    {"name": "agentAddress", "type": "address"},
                    {"name": "did", "type": "string"},
                    {"name": "modelType", "type": "string"},
                    {"name": "modelVersion", "type": "string"},
                    {"name": "specializations", "type": "string[]"}
                ],
                "name": "registerAgent",
                "outputs": [{"name": "", "type": "uint256"}],
                "stateMutability": "nonpayable",
                "type": "function"
            },
            {
                "inputs": [{"name": "agentAddress", "type": "address"}],
                "name": "agentIdByAddress",
                "outputs": [{"name": "", "type": "uint256"}],
                "stateMutability": "view",
                "type": "function"
            }
        ]
        
        self.attestation_abi = [
            {
                "inputs": [
                    {"name": "contentHash", "type": "bytes32"},
                    {"name": "reasoningHash", "type": "bytes32"},
                    {"name": "modelVersion", "type": "string"},
                    {"name": "category", "type": "string"},
                    {"name": "timestamp", "type": "uint256"},
                    {"name": "signature", "type": "bytes"}
                ],
                "name": "submitAttestation",
                "outputs": [{"name": "", "type": "bytes32"}],
                "stateMutability": "nonpayable",
                "type": "function"
            },
            {
                "inputs": [{"name": "", "type": "bytes32"}],
                "name": "attestations",
                "outputs": [
                    {"name": "contentHash", "type": "bytes32"},
                    {"name": "reasoningHash", "type": "bytes32"},
                    {"name": "agent", "type": "address"},
                    {"name": "modelVersion", "type": "string"},
                    {"name": "timestamp", "type": "uint256"},
                    {"name": "status", "type": "uint8"},
                    {"name": "consistencyScore", "type": "uint8"},
                    {"name": "verifier", "type": "address"},
                    {"name": "verificationTime", "type": "uint256"},
                    {"name": "category", "type": "string"}
                ],
                "stateMutability": "view",
                "type": "function"
            }
        ]
        
        self.reputation_abi = [
            {
                "inputs": [{"name": "", "type": "address"}],
                "name": "reputationScore",
                "outputs": [{"name": "", "type": "uint256"}],
                "stateMutability": "view",
                "type": "function"
            }
        ]
    
    def _validate_reasoning_offchain(self, reasoning: dict) -> dict:
        """
        Executa verificaÃ§Ãµes Tier 1 OFF-CHAIN antes de submeter on-chain.
        Economiza gas detectando problemas antes da transaÃ§Ã£o.
        
        Args:
            reasoning: DicionÃ¡rio com estrutura de raciocÃ­nio
            
        Returns:
            dict com keys: passed, score, checks_passed, total_checks, failures
        """
        checks_passed = 0
        total_checks = 7
        failures = []
        
        # Check 1: Hash Integrity
        try:
            reasoning_json = json.dumps(reasoning, sort_keys=True)
            checks_passed += 1
        except Exception as e:
            failures.append(f"Check 1 Failed: Cannot serialize reasoning to JSON - {e}")
        
        # Check 2: Required Fields
        required_fields = ["input", "reasoning_steps", "conclusion", "confidence"]
        missing_fields = [f for f in required_fields if f not in reasoning]
        
        if not missing_fields:
            checks_passed += 1
        else:
            failures.append(f"Check 2 Failed: Missing required fields: {missing_fields}")
        
        # Check 3: Forbidden Patterns
        full_text = json.dumps(reasoning).lower()
        detected_patterns = [p for p in self.FORBIDDEN_PATTERNS if p in full_text]
        
        if not detected_patterns:
            checks_passed += 1
        else:
            failures.append(f"Check 3 Failed: Forbidden patterns detected: {detected_patterns}")
        
        # Check 4: Confidence Range
        confidence = reasoning.get("confidence", -1)
        
        if isinstance(confidence, (int, float)) and 0.0 <= confidence <= 1.0:
            checks_passed += 1
        else:
            failures.append(f"Check 4 Failed: Invalid confidence value: {confidence} (must be 0.0-1.0)")
        
        # Check 5: Reasoning Steps Structure
        steps = reasoning.get("reasoning_steps", [])
        
        if not isinstance(steps, list):
            failures.append(f"Check 5 Failed: reasoning_steps must be a list, got {type(steps)}")
        elif len(steps) < 1:
            failures.append("Check 5 Failed: At least 1 reasoning step required")
        else:
            valid_steps = True
            for i, step in enumerate(steps):
                if not isinstance(step, dict):
                    failures.append(f"Check 5 Failed: Step {i} is not a dict")
                    valid_steps = False
                    break
                
                required_step_fields = ["step_number", "description", "rationale"]
                missing_step_fields = [f for f in required_step_fields if f not in step]
                
                if missing_step_fields:
                    failures.append(f"Check 5 Failed: Step {i} missing fields: {missing_step_fields}")
                    valid_steps = False
                    break
                
                if not step.get("description") or not step.get("rationale"):
                    failures.append(f"Check 5 Failed: Step {i} has empty description or rationale")
                    valid_steps = False
                    break
            
            if valid_steps:
                checks_passed += 1
        
        # Check 6: Size Validation
        try:
            size = len(reasoning_json.encode('utf-8'))
            
            if self.MIN_REASONING_SIZE <= size <= self.MAX_REASONING_SIZE:
                checks_passed += 1
            else:
                failures.append(
                    f"Check 6 Failed: Reasoning size {size} bytes "
                    f"(must be between {self.MIN_REASONING_SIZE}-{self.MAX_REASONING_SIZE})"
                )
        except:
            failures.append("Check 6 Failed: Cannot calculate reasoning size")
        
        # Check 7: Non-Empty Strings
        input_text = reasoning.get("input", "")
        conclusion = reasoning.get("conclusion", "")
        
        if input_text and conclusion and len(input_text) > 0 and len(conclusion) > 0:
            checks_passed += 1
        else:
            failures.append("Check 7 Failed: Input or conclusion is empty")
        
        # Calcular score e resultado
        score = int((checks_passed / total_checks) * 100)
        passed = score >= self.TIER1_PASS_THRESHOLD
        
        return {
            "passed": passed,
            "score": score,
            "checks_passed": checks_passed,
            "total_checks": total_checks,
            "failures": failures
        }
    
    def submit_attestation(
        self,
        content: str,
        reasoning: Reasoning,
        category: str,
        tier: str = "basic",
        pre_validate: bool = True,
        wait_for_confirmation: bool = True,
        timeout: int = 60
    ) -> AttestationResult:
        """
        Submete attestation para a blockchain
        
        Args:
            content: ConteÃºdo gerado pela IA
            reasoning: RaciocÃ­nio estruturado
            category: Categoria da attestation
            tier: NÃ­vel de verificaÃ§Ã£o ("basic", "standard", "premium")
            pre_validate: Se True, valida off-chain antes de submeter (RECOMENDADO)
            wait_for_confirmation: Aguardar confirmaÃ§Ã£o da transaÃ§Ã£o
            timeout: Timeout em segundos
            
        Returns:
            AttestationResult com dados da submissÃ£o
            
        Raises:
            ValueError: Se pre_validate=True e reasoning falhar validaÃ§Ã£o
        """
        # PRÃ‰-VALIDAÃ‡ÃƒO OFF-CHAIN
        if pre_validate:
            reasoning_dict = reasoning.to_dict() if hasattr(reasoning, 'to_dict') else reasoning
            validation_result = self._validate_reasoning_offchain(reasoning_dict)
            
            if not validation_result["passed"]:
                error_msg = (
                    f"\n{'='*60}\n"
                    f"âŒ PRE-VALIDATION FAILED\n"
                    f"{'='*60}\n"
                    f"Score: {validation_result['score']}/100 "
                    f"(threshold: {self.TIER1_PASS_THRESHOLD})\n"
                    f"Checks: {validation_result['checks_passed']}/{validation_result['total_checks']} passed\n"
                    f"\nFailures detected:\n"
                )
                
                for i, failure in enumerate(validation_result["failures"], 1):
                    error_msg += f"  {i}. {failure}\n"
                
                error_msg += (
                    f"\nðŸ’¡ Fix these issues before submitting to avoid wasting gas.\n"
                    f"{'='*60}\n"
                )
                
                raise ValueError(error_msg)
            
            print(f"âœ… Pre-validation passed ({validation_result['score']}/100) - submitting to chain...")
        
        # RESTO DO CÃ“DIGO ORIGINAL
        if not self.attestation_contract:
            raise ValueError("EndereÃ§o do contrato Attestation nÃ£o configurado")
        
        if not 0 <= reasoning.confidence <= 1:
            raise ValueError("Confidence deve estar entre 0 e 1")
        
        if len(reasoning.reasoning_steps) == 0:
            raise ValueError("Reasoning deve ter pelo menos 1 step")
        
        content_hash = Web3.keccak(text=content)
        reasoning_dict = reasoning.to_dict()
        reasoning_json = json.dumps(reasoning_dict, sort_keys=True)
        reasoning_hash = Web3.keccak(text=reasoning_json)
        
        agent_id = self.identity.functions.agentIdByAddress(self.address).call()
        if agent_id == 0:
            raise ValueError("Agente nÃ£o registrado. Execute register_identity() primeiro.")
        
        model_version = "v1.0"
        
        domain_data = {
            "name": "ANNA Protocol",
            "version": "1",
            "chainId": self.network_config["chain_id"],
            "verifyingContract": self.attestation_contract
        }
        
        message_types = {
            "Attestation": [
                {"name": "contentHash", "type": "bytes32"},
                {"name": "reasoningHash", "type": "bytes32"},
                {"name": "agent", "type": "address"},
                {"name": "modelVersion", "type": "string"},
                {"name": "timestamp", "type": "uint256"},
                {"name": "category", "type": "string"}
            ]
        }
        
        timestamp = int(time.time())
        
        message_data = {
            "contentHash": content_hash,
            "reasoningHash": reasoning_hash,
            "agent": self.address,
            "modelVersion": model_version,
            "timestamp": timestamp,
            "category": category
        }
        
        typed_data = {
            "types": {
                "EIP712Domain": [
                    {"name": "name", "type": "string"},
                    {"name": "version", "type": "string"},
                    {"name": "chainId", "type": "uint256"},
                    {"name": "verifyingContract", "type": "address"}
                ],
                **message_types
            },
            "primaryType": "Attestation",
            "domain": domain_data,
            "message": message_data
        }
        
        encoded_data = encode_typed_data(full_message=typed_data)
        signature = self.account.sign_message(encoded_data).signature
        
        tx = self.attestation.functions.submitAttestation(
            content_hash,
            reasoning_hash,
            model_version,
            category,
            timestamp,
            signature
        ).build_transaction({
            'from': self.address,
            'nonce': self.w3.eth.get_transaction_count(self.address),
            'gas': 300000,
            'gasPrice': self.w3.eth.gas_price
        })
        
        signed_tx = self.account.sign_transaction(tx)
        tx_hash = self.w3.eth.send_raw_transaction(signed_tx.raw_transaction)
        tx_hash_hex = tx_hash.hex()
        
        attestation_id = Web3.solidity_keccak(
          ['bytes32', 'bytes32', 'address', 'uint256'],
          [content_hash, reasoning_hash, self.address, timestamp]
        ).hex()
        
        if wait_for_confirmation:
            receipt = self.w3.eth.wait_for_transaction_receipt(tx_hash)
            if receipt['status'] != 1:
                revert_reason = "Unknown"
                try:
                    tx = self.w3.eth.get_transaction(tx_hash)
                    self.w3.eth.call(tx, receipt['blockNumber'])
                except Exception as e: 
                    revert_reason = str(e)
                raise Exception(f"TransaÃ§Ã£o falhou!\nRevert reason: {revert_reason}\nTX hash: {tx_hash.hex()}")
        
        explorer_url = f"{self.network_config['explorer']}/tx/{tx_hash_hex}"
        
        return AttestationResult(
            attestation_id=attestation_id,
            tx_hash=tx_hash_hex,
            status=AttestationStatus.PENDING,
            timestamp=timestamp,
            explorer_url=explorer_url
        )
    
    def register_identity(
        self,
        model_type: str,
        model_version: str,
        specializations: List[str],
        wait_for_confirmation: bool = True
    ) -> Identity:
        """Registra a identidade do agente"""
        if not self.identity_contract:
            raise ValueError("EndereÃ§o do contrato Identity nÃ£o configurado")
        
        agent_id = self.identity.functions.agentIdByAddress(self.address).call()
        if agent_id != 0:
            raise ValueError(f"Agente jÃ¡ registrado com ID: {agent_id}")
        
        # Generate DID
        did = f"did:anna:{self.address.lower()}"
        
        tx = self.identity.functions.registerAgent(
            self.address,      # agentAddress
            did,               # did
            model_type,        # modelType
            model_version,     # modelVersion
            specializations    # specializations
        ).build_transaction({
            'from': self.address,
            'nonce': self.w3.eth.get_transaction_count(self.address),
            'gas': 500000,  # Increased gas limit for safety
            'gasPrice': self.w3.eth.gas_price
        })
        
        signed_tx = self.account.sign_transaction(tx)
        tx_hash = self.w3.eth.send_raw_transaction(signed_tx.raw_transaction)
        
        if wait_for_confirmation:
            receipt = self.w3.eth.wait_for_transaction_receipt(tx_hash)
            if receipt['status'] != 1:
                raise Exception("TransaÃ§Ã£o falhou")
        
        agent_id = self.identity.functions.agentIdByAddress(self.address).call()
        
        return Identity(
            agent_id=agent_id if wait_for_confirmation else 0,
            did=did,
            address=self.address,
            model_type=model_type,
            model_version=model_version,
            specializations=specializations,
            creation_time=int(time.time())
        )
    
    def get_attestation(self, attestation_id: str) -> Dict[str, Any]:
        """Busca informaÃ§Ãµes de uma attestation"""
        if not self.attestation_contract:
            raise ValueError("EndereÃ§o do contrato Attestation nÃ£o configurado")
        
        attestation_data = self.attestation.functions.attestations(attestation_id).call()
        
        return {
            "contract_hash": attestation_data[0].hex(),
            "reasoning_hash": attestation_data[1].hex(),
            "agent": attestation_data[2],
            "model_version": attestation_data[3],
            "timestamp": attestation_data[4],
            "status": AttestationStatus(attestation_data[5]),
            "consistency_score": attestation_data[6],
            "verifier": attestation_data[7],
            "verification_time": attestation_data[8],
            "category": attestation_data[9]
        }
    
    def wait_for_verification(
        self,
        attestation_id: str,
        timeout: int = 60,
        poll_interval: int = 5
    ) -> AttestationResult:
        """Aguarda a verificaÃ§Ã£o de uma attestation"""
        start_time = time.time()
        
        while time.time() - start_time < timeout:
            attestation = self.get_attestation(attestation_id)
            
            if attestation["status"] != AttestationStatus.PENDING:
                return AttestationResult(
                    attestation_id=attestation_id,
                    tx_hash="",
                    status=attestation["status"],
                    timestamp=attestation["timestamp"],
                    explorer_url=f"{self.network_config['explorer']}/address/{self.attestation_contract}",
                    verified=attestation["status"] == AttestationStatus.VERIFIED,
                    score=attestation["consistency_score"],
                    verifier=attestation["verifier"],
                    verification_time=attestation["verification_time"]
                )
            
            time.sleep(poll_interval)
        
        raise TimeoutError(f"VerificaÃ§Ã£o nÃ£o concluÃ­da em {timeout}s")
    
    def get_reputation(self, agent_address: Optional[str] = None) -> int:
        """Busca o score de reputaÃ§Ã£o de um agente"""
        if not self.reputation_contract:
            raise ValueError("EndereÃ§o do contrato Reputation nÃ£o configurado")
        
        address = agent_address or self.address
        score = self.reputation.functions.reputationScore(address).call()
        return score
    
    def get_balance(self) -> float:
        """Retorna o saldo de MATIC da wallet"""
        balance_wei = self.w3.eth.get_balance(self.address)
        return self.w3.from_wei(balance_wei, 'ether')
    
    def get_identity(self, address: Optional[str] = None) -> Optional[Dict]:
        """Busca informaÃ§Ãµes de identidade de um agente"""
        if not self.identity_contract:
            raise ValueError("EndereÃ§o do contrato Identity nÃ£o configurado")
        
        addr = address or self.address
        agent_id = self.identity.functions.agentIdByAddress(addr).call()
        
        if agent_id == 0:
            return None
        
        return {
            "agent_id": agent_id,
            "did": f"did:anna:{addr.lower()}",
            "address": addr
        }


# ============================================================
# HELPER FUNCTIONS
# ============================================================

def create_reasoning(
    input_text: str,
    steps: List[Tuple[str, str]],
    conclusion: str,
    confidence: float
) -> Reasoning:
    """Helper para criar objeto Reasoning facilmente"""
    reasoning_steps = [
        ReasoningStep(i + 1, desc, rat)
        for i, (desc, rat) in enumerate(steps)
    ]
    
    return Reasoning(
        input=input_text,
        reasoning_steps=reasoning_steps,
        conclusion=conclusion,
        confidence=confidence
    )


def calculate_content_hash(content: str) -> str:
    """Calcula hash Keccak256 de um conteÃºdo"""
    return Web3.keccak(text=content).hex()


# ============================================================
# EXCEÃ‡Ã•ES CUSTOMIZADAS
# ============================================================

class ANNAError(Exception):
    """Erro base do SDK"""
    pass


class IdentityNotFoundError(ANNAError):
    """Agente nÃ£o tem identidade registrada"""
    pass


class AttestationNotFoundError(ANNAError):
    """Attestation nÃ£o encontrada"""
    pass


class VerificationTimeoutError(ANNAError):
    """Timeout aguardando verificaÃ§Ã£o"""
    pass


# ============================================================
# VERSÃƒO DO SDK
# ============================================================

__version__ = "1.1.0"
__author__ = "ANNA Protocol Team"
__license__ = "MIT"