"""
ANNA Protocol SDK - Python Client
VersÃƒÂ£o: 1.2.8 (com suporte a metadados estruturados)

SDK oficial para interagir com o ANNA Protocol
"""

import json
import time
from typing import Dict, Optional, List, Tuple, Any, Union
from dataclasses import dataclass, asdict
from enum import Enum
from web3 import Web3
from eth_account import Account
from eth_account.messages import encode_typed_data
import logging

# Configurar logging
logger = logging.getLogger(__name__)



# ============================================================
# CONFIGURAÃƒâ€¡Ãƒâ€¢ES E CONSTANTES
# ============================================================

NETWORKS = {
    "polygon-amoy": {
        "rpc": "https://rpc-amoy.polygon.technology/",
        "chain_id": 80002,
        "explorer": "https://www.oklink.com/amoy"
    },
    "polygon-mainnet": {
        "rpc": "https://polygon-rpc.com",
        "chain_id": 137,
        "explorer": "https://polygonscan.com"
    }
}


# ============================================================
# TIPOS E ENUMS
# ============================================================

class VerificationTier(Enum):
    """NÃƒÂ­veis de verificaÃƒÂ§ÃƒÂ£o disponÃƒÂ­veis"""
    BASIC = "basic"
    STANDARD = "standard"
    PREMIUM = "premium"


class AttestationStatus(Enum):
    """Status possÃƒÂ­veis de uma attestation"""
    PENDING = "pending"
    VERIFIED = "verified"
    REJECTED = "rejected"
    CHALLENGED = "challenged"


@dataclass
class ReasoningStep:
    """Estrutura de um passo de raciocÃƒÂ­nio"""
    step_number: int
    description: str
    rationale: str


@dataclass
class Reasoning:
    """Estrutura completa de raciocÃƒÂ­nio"""
    input: str
    reasoning_steps: List[ReasoningStep]
    conclusion: str
    confidence: float
    
    def to_dict(self) -> Dict:
        """Converte para dicionÃƒÂ¡rio"""
        return {
            "input": self.input,
            "reasoning_steps": [
                {
                    "step_number": step.step_number,
                    "description": step.description,
                    "rationale": step.rationale
                }
                for step in self.reasoning_steps
            ],
            "conclusion": self.conclusion,
            "confidence": self.confidence
        }


@dataclass
class Metadata:
    """
    Metadados estruturados para attestations
    Permite rastreabilidade bidirecional entre sistemas externos e ANNA
    
    Exemplo:
        metadata = Metadata(
            external_id="PROC-2024-12345",
            document_type="contrato_locacao",
            client_name="JoÃƒÂ£o Silva",
            system_origin="LegalTech Pro v2.0",
            custom_fields={"valor": "R$ 2.500,00", "prazo": "12 meses"}
        )
    """
    external_id: Optional[str] = None          # ID do sistema externo (processo, contrato, etc)
    document_type: Optional[str] = None        # Tipo do documento
    client_name: Optional[str] = None          # Nome do cliente
    system_origin: Optional[str] = None        # Sistema de origem
    custom_fields: Optional[Dict[str, Any]] = None  # Campos adicionais customizados
    
    def to_dict(self) -> Dict[str, Any]:
        """Converte para dicionÃƒÂ¡rio, removendo valores None"""
        data = asdict(self)
        return {k: v for k, v in data.items() if v is not None}
    
    def to_json(self) -> str:
        """Converte para string JSON compacta"""
        return json.dumps(self.to_dict(), ensure_ascii=False, separators=(',', ':'))
    
    @classmethod
    def from_json(cls, json_str: str) -> 'Metadata':
        """Cria Metadata a partir de string JSON"""
        data = json.loads(json_str)
        return cls(**data)
    
    def validate(self, max_size: int = 500) -> bool:
        """
        Valida se os metadados estÃƒÂ£o dentro dos limites
        
        Args:
            max_size: Tamanho mÃƒÂ¡ximo em caracteres (default: 500)
            
        Returns:
            True se vÃƒÂ¡lido
            
        Raises:
            ValueError se exceder o tamanho mÃƒÂ¡ximo
        """
        json_str = self.to_json()
        if len(json_str) > max_size:
            raise ValueError(
                f"Metadados muito grandes: {len(json_str)} caracteres "
                f"(mÃƒÂ¡ximo: {max_size}). Considere usar campos mais concisos."
            )
        return True


@dataclass
class AttestationResult:
    """Resultado de uma attestation"""
    attestation_id: str
    tx_hash: str
    status: AttestationStatus
    timestamp: int
    explorer_url: str
    verified: bool = False
    score: Optional[int] = None
    verifier: Optional[str] = None
    verification_time: Optional[int] = None
    metadata: Optional[Metadata] = None  # NOVO: metadados da attestation


@dataclass
class Identity:
    """Identidade de um agente"""
    agent_id: int
    did: str
    address: str
    model_type: str
    model_version: str
    specializations: List[str]
    creation_time: int
    token_uri: Optional[str] = None


# ============================================================
# CLIENTE PRINCIPAL
# ============================================================

class ANNAClient:
    """Cliente principal do ANNA Protocol SDK"""
    
    # PadrÃƒÂµes proibidos (sincronizado com verifier.py)
    FORBIDDEN_PATTERNS = [
        "ignore previous instructions",
        "ignore all instructions",
        "jailbreak",
        "bypass",
        "hack",
        "disable safety",
        "ignore guidelines",
        "forget everything",
        "new instructions",
        "system prompt",
        "override"
    ]
    
    # Limites de tamanho (anti-spam)
    MIN_REASONING_SIZE = 100      # bytes
    MAX_REASONING_SIZE = 50000    # 50 KB
    MAX_METADATA_SIZE = 500       # 500 caracteres para metadados
    
    # Threshold de aprovaÃƒÂ§ÃƒÂ£o Tier 1
    TIER1_PASS_THRESHOLD = 60     # Score mÃƒÂ­nimo (0-100)
    
    def __init__(
        self,
        private_key: str,
        network: str = "polygon-amoy",
        identity_contract: Optional[str] = None,
        attestation_contract: Optional[str] = None,
        reputation_contract: Optional[str] = None
    ):
        """Inicializa o cliente ANNA"""
        if network not in NETWORKS:
            raise ValueError(f"Network invÃƒÂ¡lida. Use: {list(NETWORKS.keys())}")
        
        self.network = network
        self.network_config = NETWORKS[network]
        self.w3 = Web3(Web3.HTTPProvider(self.network_config["rpc"]))
        
        if not self.w3.is_connected():
            raise ConnectionError(f"NÃƒÂ£o foi possÃƒÂ­vel conectar ao RPC: {self.network_config['rpc']}")
        
        self.account = Account.from_key(private_key)
        self.address = self.account.address
        
        self.identity_contract = identity_contract
        self.attestation_contract = attestation_contract
        self.reputation_contract = reputation_contract
        
        self._load_abis()
        
        if self.identity_contract:
            self.identity = self.w3.eth.contract(
                address=Web3.to_checksum_address(self.identity_contract),
                abi=self.identity_abi
            )
        
        if self.attestation_contract:
            self.attestation = self.w3.eth.contract(
                address=Web3.to_checksum_address(self.attestation_contract),
                abi=self.attestation_abi
            )
        
        if self.reputation_contract:
            self.reputation = self.w3.eth.contract(
                address=Web3.to_checksum_address(self.reputation_contract),
                abi=self.reputation_abi
            )
    
    def _load_abis(self):
        """Carrega ABIs dos contratos"""
        self.identity_abi = [
            {
                "inputs": [
                    {"name": "agentAddress", "type": "address"},
                    {"name": "did", "type": "string"},
                    {"name": "modelType", "type": "string"},
                    {"name": "modelVersion", "type": "string"},
                    {"name": "specializations", "type": "string[]"}
                ],
                "name": "registerAgent",
                "outputs": [{"name": "", "type": "uint256"}],
                "stateMutability": "nonpayable",
                "type": "function"
            },
            {
                "inputs": [{"name": "agentAddress", "type": "address"}],
                "name": "agentIdByAddress",
                "outputs": [{"name": "", "type": "uint256"}],
                "stateMutability": "view",
                "type": "function"
            }
        ]
        
        self.attestation_abi = [
            {
                "inputs": [
                    {"name": "contentHash", "type": "bytes32"},
                    {"name": "reasoningHash", "type": "bytes32"},
                    {"name": "modelVersion", "type": "string"},
                    {"name": "category", "type": "string"},
                    {"name": "timestamp", "type": "uint256"},
                    {"name": "signature", "type": "bytes"}
                ],
                "name": "submitAttestation",
                "outputs": [{"name": "", "type": "bytes32"}],
                "stateMutability": "nonpayable",
                "type": "function"
            },
            {
                "inputs": [{"name": "", "type": "bytes32"}],
                "name": "attestations",
                "outputs": [
                    {"name": "contentHash", "type": "bytes32"},
                    {"name": "reasoningHash", "type": "bytes32"},
                    {"name": "agent", "type": "address"},
                    {"name": "modelVersion", "type": "string"},
                    {"name": "timestamp", "type": "uint256"},
                    {"name": "status", "type": "uint8"},
                    {"name": "consistencyScore", "type": "uint256"},
                    {"name": "verifier", "type": "address"},
                    {"name": "verificationTime", "type": "uint256"},
                    {"name": "category", "type": "string"}
                ],
                "stateMutability": "view",
                "type": "function"
            }
        ]
        
        self.reputation_abi = [
            {
                "inputs": [{"name": "", "type": "address"}],
                "name": "reputationScore",
                "outputs": [{"name": "", "type": "uint256"}],
                "stateMutability": "view",
                "type": "function"
            }
        ]
    
    def _validate_reasoning(self, reasoning: Union[str, Reasoning]) -> str:
        """
        Valida o raciocÃƒÂ­nio antes de enviar
        
        Args:
            reasoning: String JSON ou objeto Reasoning
            
        Returns:
            String JSON validada
            
        Raises:
            ValueError: Se a validaÃƒÂ§ÃƒÂ£o falhar
        """
        # Converter para string se for objeto Reasoning
        if isinstance(reasoning, Reasoning):
            reasoning_str = json.dumps(reasoning.to_dict())
        else:
            reasoning_str = reasoning
        
        # Verificar se ÃƒÂ© JSON vÃƒÂ¡lido
        try:
            reasoning_obj = json.loads(reasoning_str)
        except json.JSONDecodeError as e:
            raise ValueError(f"Reasoning invÃƒÂ¡lido (nÃƒÂ£o ÃƒÂ© JSON): {e}")
        
        # Verificar tamanho
        reasoning_bytes = reasoning_str.encode('utf-8')
        if len(reasoning_bytes) < self.MIN_REASONING_SIZE:
            raise ValueError(
                f"Reasoning muito curto: {len(reasoning_bytes)} bytes "
                f"(mÃƒÂ­nimo: {self.MIN_REASONING_SIZE})"
            )
        
        if len(reasoning_bytes) > self.MAX_REASONING_SIZE:
            raise ValueError(
                f"Reasoning muito longo: {len(reasoning_bytes)} bytes "
                f"(mÃƒÂ¡ximo: {self.MAX_REASONING_SIZE})"
            )
        
        # Verificar padrÃƒÂµes proibidos
        reasoning_lower = reasoning_str.lower()
        for pattern in self.FORBIDDEN_PATTERNS:
            if pattern in reasoning_lower:
                raise ValueError(f"PadrÃƒÂ£o proibido detectado: '{pattern}'")
        
        # Tier 1: verificar campos obrigatÃƒÂ³rios bÃƒÂ¡sicos
        required_fields = ["input", "conclusion"]
        for field in required_fields:
            if field not in reasoning_obj:
                raise ValueError(f"Campo obrigatÃƒÂ³rio ausente: '{field}'")
        
        return reasoning_str
    
    def _set_attestation_txhash(
        self,
        attestation_id: bytes,
        tx_hash: str,
        wait_for_confirmation: bool = True
    ) -> Optional[str]:
        """
        Registra transaction hash on-chain (NOVO v1.2.6)
        
        Este método é chamado automaticamente após submitAttestation().
        FAIL-SAFE: Se contrato não suportar (v1.1), apenas retorna None.
        
        Args:
            attestation_id: ID da attestation (bytes32)
            tx_hash: Transaction hash da criação (string hex)
            wait_for_confirmation: Se deve aguardar confirmação
            
        Returns:
            Transaction hash do setAttestationTxHash ou None se falhar
        """
        try:
            # Verificar se método existe (contrato v1.2+)
            if not hasattr(self.attestation.functions, 'setAttestationTxHash'):
                logger.debug("Contract does not support setAttestationTxHash (v1.1)")
                return None
            
            # Converter attestation_id para bytes32 se necessário
            if isinstance(attestation_id, str):
                if attestation_id.startswith('0x'):
                    attestation_id = bytes.fromhex(attestation_id[2:])
                else:
                    attestation_id = bytes.fromhex(attestation_id)
            
            # Converter tx_hash para bytes32
            if tx_hash.startswith('0x'):
                tx_hash_bytes = bytes.fromhex(tx_hash[2:])
            else:
                tx_hash_bytes = bytes.fromhex(tx_hash)
            
            logger.info(f"📝 Registering txHash on-chain: {tx_hash[:16]}...")
            
            # Chamar setAttestationTxHash
            tx = self.attestation.functions.setAttestationTxHash(
                attestation_id,
                tx_hash_bytes
            ).build_transaction({
                'from': self.address,
                'nonce': self.w3.eth.get_transaction_count(self.address),
                'gas': 100000,
                'gasPrice': self.w3.eth.gas_price
            })
            
            signed_tx = self.account.sign_transaction(tx)
            set_tx_hash = self.w3.eth.send_raw_transaction(signed_tx.raw_transaction)
            set_tx_hash_hex = set_tx_hash.hex()
            
            if wait_for_confirmation:
                receipt = self.w3.eth.wait_for_transaction_receipt(set_tx_hash)
                if receipt['status'] != 1:
                    logger.warning(f"⚠️  setAttestationTxHash failed: {set_tx_hash_hex}")
                    return None
                logger.info(f"✅ TxHash registered on-chain: {set_tx_hash_hex[:16]}...")
            
            return set_tx_hash_hex
            
        except Exception as e:
            # FAIL-SAFE: Não quebrar se setAttestationTxHash falhar
            logger.warning(f"⚠️  Could not set txHash on-chain: {e}")
            return None
    
    def submit_attestation(
        self,
        content: str,
        reasoning: Union[str, Reasoning],
        model_version: str = "claude-sonnet-4-20250514",
        category: str = "general",
        wait_for_confirmation: bool = True,
        store_txhash: bool = True  # NOVO v1.2.6
    ) -> AttestationResult:
        """
        Submete uma attestation ao ANNA Protocol
        
        Args:
            content: ConteÃƒÂºdo gerado pelo agente
            reasoning: RaciocÃƒÂ­nio (string JSON ou objeto Reasoning)
            model_version: VersÃƒÂ£o do modelo
            category: Categoria da attestation
            wait_for_confirmation: Se deve esperar confirmaÃƒÂ§ÃƒÂ£o da transaÃƒÂ§ÃƒÂ£o
            
        Returns:
            AttestationResult com informaÃƒÂ§ÃƒÂµes da attestation
            
        Raises:
            ValueError: Se a validaÃƒÂ§ÃƒÂ£o falhar
            Exception: Se a transaÃƒÂ§ÃƒÂ£o falhar
        """
        if not self.attestation_contract:
            raise ValueError("EndereÃƒÂ§o do contrato Attestation nÃƒÂ£o configurado")
        
        # Validar reasoning
        reasoning_str = self._validate_reasoning(reasoning)
        
        # Calcular hashes
        content_hash = Web3.keccak(text=content)
        reasoning_hash = Web3.keccak(text=reasoning_str)
        
        # Preparar timestamp e assinatura
        timestamp = int(time.time())
        
        message_hash = Web3.solidity_keccak(
            ['bytes32', 'bytes32', 'address', 'uint256'],
            [content_hash, reasoning_hash, self.address, timestamp]
        )
        
        encoded_data = encode_typed_data(
            domain_data={
                'name': 'ANNA Protocol',
                'version': '1',
                'chainId': self.network_config['chain_id'],
                'verifyingContract': self.attestation_contract
            },
            message_types={
                'Attestation': [
                    {'name': 'contentHash', 'type': 'bytes32'},
                    {'name': 'reasoningHash', 'type': 'bytes32'},
                    {'name': 'agent', 'type': 'address'},
                    {'name': 'modelVersion', 'type': 'string'},
                    {'name': 'timestamp', 'type': 'uint256'},
                    {'name': 'category', 'type': 'string'}
                ]
            },
            message_data={
                'contentHash': content_hash,
                'reasoningHash': reasoning_hash,
                'agent': self.address,
                'modelVersion': model_version,
                'timestamp': timestamp,
                'category': category
            }
        )
        
        signature = self.account.sign_message(encoded_data).signature
        
        tx = self.attestation.functions.submitAttestation(
            content_hash,
            reasoning_hash,
            model_version,
            category,
            timestamp,
            signature
        ).build_transaction({
            'from': self.address,
            'nonce': self.w3.eth.get_transaction_count(self.address),
            'gas': 500000,
            'gasPrice': self.w3.eth.gas_price
        })
        
        signed_tx = self.account.sign_transaction(tx)
        tx_hash = self.w3.eth.send_raw_transaction(signed_tx.raw_transaction)
        tx_hash_hex = tx_hash.hex()
        
        attestation_id = Web3.solidity_keccak(
          ['bytes32', 'bytes32', 'address', 'uint256'],
          [content_hash, reasoning_hash, self.address, timestamp]
        ).hex()
        
        if wait_for_confirmation:
            receipt = self.w3.eth.wait_for_transaction_receipt(tx_hash)
            if receipt['status'] != 1:
                revert_reason = "Unknown"
                try:
                    from web3.datastructures import AttributeDict
                    failed_tx = self.w3.eth.get_transaction(tx_hash)
                    # Converter para AttributeDict para eth.call funcionar
                    tx_dict = AttributeDict({
                        'from': failed_tx['from'],
                        'to': failed_tx['to'],
                        'data': failed_tx['input'],
                        'value': failed_tx.get('value', 0),
                        'gas': failed_tx.get('gas', 300000)
                    })
                    self.w3.eth.call(tx_dict, receipt['blockNumber'])
                except Exception as e: 
                    revert_reason = str(e)
                raise Exception(f"TransaÃƒÂ§ÃƒÂ£o falhou!\nRevert reason: {revert_reason}\nTX hash: {tx_hash.hex()}")
            
            # NOVO v1.2.7: Salvar txHash on-chain
            if store_txhash:
                self._set_attestation_txhash(
                    attestation_id=attestation_id,
                    tx_hash=tx_hash_hex,
                    wait_for_confirmation=True
                )
        
        explorer_url = f"{self.network_config['explorer']}/tx/{tx_hash_hex}"
        
        return AttestationResult(
            attestation_id=attestation_id,
            tx_hash=tx_hash_hex,
            status=AttestationStatus.PENDING,
            timestamp=timestamp,
            explorer_url=explorer_url
        )
    
    def submit_attestation_with_metadata(
        self,
        content: str,
        reasoning: Union[str, Reasoning],
        metadata: Metadata,
        model_version: str = "claude-sonnet-4-20250514",
        wait_for_confirmation: bool = True,
        store_txhash: bool = True  # NOVO v1.2.6
    ) -> AttestationResult:
        """
        Submete uma attestation com metadados estruturados
        
        MÃƒÂ©todo conveniente que facilita o envio de attestations com rastreabilidade.
        Os metadados sÃƒÂ£o armazenados no campo 'category' como JSON.
        
        Exemplo de uso:
            metadata = Metadata(
                external_id="PROC-2024-12345",
                document_type="contrato_locacao",
                client_name="JoÃƒÂ£o Silva",
                system_origin="LegalTech Pro",
                custom_fields={"valor": "R$ 2.500,00"}
            )
            
            result = client.submit_attestation_with_metadata(
                content=contract_text,
                reasoning=ai_reasoning,
                metadata=metadata
            )
            
            # Guardar no banco local
            db.save({
                "processo_id": "PROC-2024-12345",
                "anna_attestation_id": result.attestation_id
            })
        
        Args:
            content: ConteÃƒÂºdo gerado pelo agente
            reasoning: RaciocÃƒÂ­nio (string JSON ou objeto Reasoning)
            metadata: Objeto Metadata com informaÃƒÂ§ÃƒÂµes estruturadas
            model_version: VersÃƒÂ£o do modelo
            wait_for_confirmation: Se deve esperar confirmaÃƒÂ§ÃƒÂ£o da transaÃƒÂ§ÃƒÂ£o
            
        Returns:
            AttestationResult com informaÃƒÂ§ÃƒÂµes da attestation e metadados
            
        Raises:
            ValueError: Se a validaÃƒÂ§ÃƒÂ£o falhar
            Exception: Se a transaÃƒÂ§ÃƒÂ£o falhar
        """
        # Validar metadados
        metadata.validate(max_size=self.MAX_METADATA_SIZE)
        
        # Converter metadados para JSON
        category = metadata.to_json()
        
        # Submeter attestation normalmente
        result = self.submit_attestation(
            content=content,
            reasoning=reasoning,
            model_version=model_version,
            category=category,
            wait_for_confirmation=wait_for_confirmation,
            store_txhash=store_txhash  # NOVO v1.2.7: passar parâmetro
        )
        
        # Adicionar metadados ao resultado
        result.metadata = metadata
        
        return result
    
    def register_identity(
        self,
        model_type: str,
        model_version: str,
        specializations: List[str],
        wait_for_confirmation: bool = True
    ) -> Identity:
        """Registra a identidade do agente"""
        if not self.identity_contract:
            raise ValueError("EndereÃƒÂ§o do contrato Identity nÃƒÂ£o configurado")
        
        agent_id = self.identity.functions.agentIdByAddress(self.address).call()
        if agent_id != 0:
            raise ValueError(f"Agente jÃƒÂ¡ registrado com ID: {agent_id}")
        
        # Generate DID
        did = f"did:anna:{self.address.lower()}"
        
        tx = self.identity.functions.registerAgent(
            self.address,      # agentAddress
            did,               # did
            model_type,        # modelType
            model_version,     # modelVersion
            specializations    # specializations
        ).build_transaction({
            'from': self.address,
            'nonce': self.w3.eth.get_transaction_count(self.address),
            'gas': 500000,  # Increased gas limit for safety
            'gasPrice': self.w3.eth.gas_price
        })
        
        signed_tx = self.account.sign_transaction(tx)
        tx_hash = self.w3.eth.send_raw_transaction(signed_tx.raw_transaction)
        
        if wait_for_confirmation:
            receipt = self.w3.eth.wait_for_transaction_receipt(tx_hash)
            if receipt['status'] != 1:
                raise Exception("TransaÃƒÂ§ÃƒÂ£o falhou")
        
        agent_id = self.identity.functions.agentIdByAddress(self.address).call()
        
        return Identity(
            agent_id=agent_id if wait_for_confirmation else 0,
            did=did,
            address=self.address,
            model_type=model_type,
            model_version=model_version,
            specializations=specializations,
            creation_time=int(time.time())
        )
    
    def get_attestation(self, attestation_id: str) -> Dict[str, Any]:
        """
        Busca informaÃƒÂ§ÃƒÂµes de uma attestation
        
        Se o campo category contiver JSON com metadados estruturados,
        eles serÃƒÂ£o automaticamente parseados.
        """
        if not self.attestation_contract:
            raise ValueError("EndereÃƒÂ§o do contrato Attestation nÃƒÂ£o configurado")
        
        attestation_data = self.attestation.functions.attestations(attestation_id).call()
        
        # Tentar parsear metadados do category
        category = attestation_data[9]
        metadata = None
        try:
            metadata = Metadata.from_json(category)
        except (json.JSONDecodeError, TypeError):
            # Se nÃƒÂ£o for JSON vÃƒÂ¡lido, category ÃƒÂ© string simples
            pass
        
        return {
            "content_hash": attestation_data[0].hex(),
            "reasoning_hash": attestation_data[1].hex(),
            "agent": attestation_data[2],
            "model_version": attestation_data[3],
            "timestamp": attestation_data[4],
            "status": AttestationStatus(attestation_data[5]),
            "consistency_score": attestation_data[6],
            "verifier": attestation_data[7],
            "verification_time": attestation_data[8],
            "category": category,
            "metadata": metadata  # NOVO: metadados parseados (ou None)
        }
    
    def wait_for_verification(
        self,
        attestation_id: str,
        timeout: int = 60,
        poll_interval: int = 5
    ) -> AttestationResult:
        """Aguarda a verificaÃƒÂ§ÃƒÂ£o de uma attestation"""
        start_time = time.time()
        
        while time.time() - start_time < timeout:
            attestation = self.get_attestation(attestation_id)
            
            if attestation["status"] != AttestationStatus.PENDING:
                return AttestationResult(
                    attestation_id=attestation_id,
                    tx_hash="",
                    status=attestation["status"],
                    timestamp=attestation["timestamp"],
                    explorer_url=f"{self.network_config['explorer']}/address/{self.attestation_contract}",
                    verified=attestation["status"] == AttestationStatus.VERIFIED,
                    score=attestation["consistency_score"],
                    verifier=attestation["verifier"],
                    verification_time=attestation["verification_time"],
                    metadata=attestation.get("metadata")  # NOVO: incluir metadados
                )
            
            time.sleep(poll_interval)
        
        raise TimeoutError(f"VerificaÃƒÂ§ÃƒÂ£o nÃƒÂ£o concluÃƒÂ­da em {timeout}s")
    
    def get_reputation(self, agent_address: Optional[str] = None) -> int:
        """Busca o score de reputaÃƒÂ§ÃƒÂ£o de um agente"""
        if not self.reputation_contract:
            raise ValueError("EndereÃƒÂ§o do contrato Reputation nÃƒÂ£o configurado")
        
        address = agent_address or self.address
        score = self.reputation.functions.reputationScore(address).call()
        return score
    
    def get_balance(self) -> float:
        """Retorna o saldo de MATIC da wallet"""
        balance_wei = self.w3.eth.get_balance(self.address)
        return self.w3.from_wei(balance_wei, 'ether')
    
    def get_identity(self, address: Optional[str] = None) -> Optional[Dict]:
        """Busca informaÃƒÂ§ÃƒÂµes de identidade de um agente"""
        if not self.identity_contract:
            raise ValueError("EndereÃƒÂ§o do contrato Identity nÃƒÂ£o configurado")
        
        addr = address or self.address
        agent_id = self.identity.functions.agentIdByAddress(addr).call()
        
        if agent_id == 0:
            return None
        
        return {
            "agent_id": agent_id,
            "did": f"did:anna:{addr.lower()}",
            "address": addr
        }


# ============================================================
# HELPER FUNCTIONS
# ============================================================

def create_reasoning(
    input_text: str,
    steps: List[Tuple[str, str]],
    conclusion: str,
    confidence: float
) -> Reasoning:
    """Helper para criar objeto Reasoning facilmente"""
    reasoning_steps = [
        ReasoningStep(i + 1, desc, rat)
        for i, (desc, rat) in enumerate(steps)
    ]
    
    return Reasoning(
        input=input_text,
        reasoning_steps=reasoning_steps,
        conclusion=conclusion,
        confidence=confidence
    )


def create_metadata(
    external_id: str,
    document_type: str,
    **kwargs
) -> Metadata:
    """
    Helper para criar objeto Metadata facilmente
    
    Exemplo:
        metadata = create_metadata(
            external_id="PROC-2024-12345",
            document_type="contrato",
            client_name="JoÃƒÂ£o Silva",
            custom_fields={"valor": "R$ 2.500,00"}
        )
    """
    return Metadata(
        external_id=external_id,
        document_type=document_type,
        **kwargs
    )


def calculate_content_hash(content: str) -> str:
    """Calcula hash Keccak256 de um conteÃƒÂºdo"""
    return Web3.keccak(text=content).hex()


# ============================================================
# EXCEÃƒâ€¡Ãƒâ€¢ES CUSTOMIZADAS
# ============================================================

class ANNAError(Exception):
    """Erro base do SDK"""
    pass


class IdentityNotFoundError(ANNAError):
    """Agente nÃƒÂ£o tem identidade registrada"""
    pass


class AttestationNotFoundError(ANNAError):
    """Attestation nÃƒÂ£o encontrada"""
    pass


class VerificationTimeoutError(ANNAError):
    """Timeout aguardando verificaÃƒÂ§ÃƒÂ£o"""
    pass


class MetadataValidationError(ANNAError):
    """Erro na validaÃƒÂ§ÃƒÂ£o de metadados"""
    pass


# ============================================================
# VERSÃƒÆ’O DO SDK
# ============================================================

__version__ = "1.2.8"
__author__ = "ANNA Protocol Team"
__license__ = "MIT"