"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const glue_generated_1 = require("./glue.generated");
/**
 * Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["UNENCRYPTED"] = "Unencrypted";
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    TableEncryption["S3_MANAGED"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    TableEncryption["KMS"] = "SSE-KMS";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the KMS service.
     */
    TableEncryption["KMS_MANAGED"] = "SSE-KMS-MANAGED";
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    TableEncryption["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * A Glue table.
 */
class Table extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = props.s3Prefix || 'data/';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        this.compressed = props.compressed === undefined ? false : props.compressed;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new glue_generated_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: this.physicalName,
                description: props.description || `${props.tableName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    has_encrypted_data: this.encryption !== TableEncryption.UNENCRYPTED
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: this.compressed,
                    storedAsSubDirectories: props.storedAsSubDirectories === undefined ? false : props.storedAsSubDirectories,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className
                    },
                },
                tableType: 'EXTERNAL_TABLE'
            }
        });
        this.tableName = this.getResourceNameAttribute(tableResource.ref);
        this.tableArn = this.stack.formatArn({
            service: 'glue',
            resource: 'table',
            resourceName: `${this.database.databaseName}/${this.tableName}`
        });
    }
    static fromTableArn(scope, id, tableArn) {
        const tableName = core_1.Fn.select(1, core_1.Fn.split('/', core_1.Stack.of(scope).parseArn(tableArn).resourceName));
        return Table.fromTableAttributes(scope, id, {
            tableArn,
            tableName
        });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromTableAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.tableArn = attrs.tableArn;
                this.tableName = attrs.tableName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, readPermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        this.bucket.grantRead(grantee, this.s3Prefix);
        return ret;
    }
    /**
     * Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, writePermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        this.bucket.grantWrite(grantee, this.s3Prefix);
        return ret;
    }
    /**
     * Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, [...readPermissions, ...writePermissions]);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        this.bucket.grantReadWrite(grantee, this.s3Prefix);
        return ret;
    }
    grant(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.tableArn],
            actions,
        });
    }
}
exports.Table = Table;
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error(`column names and partition keys must be unique, but 'p1' is duplicated`);
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3_MANAGED]: s3.BucketEncryption.S3_MANAGED,
    [TableEncryption.KMS_MANAGED]: s3.BucketEncryption.KMS_MANAGED,
    [TableEncryption.KMS]: s3.BucketEncryption.KMS,
    [TableEncryption.CLIENT_SIDE_KMS]: s3.BucketEncryption.UNENCRYPTED,
    [TableEncryption.UNENCRYPTED]: s3.BucketEncryption.UNENCRYPTED,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.UNENCRYPTED;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.UNENCRYPTED && encryption !== TableEncryption.CLIENT_SIDE_KMS)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.CLIENT_SIDE_KMS && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.Key(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.CLIENT_SIDE_KMS) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey
    };
}
const readPermissions = [
    'glue:BatchDeletePartition',
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersions'
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition'
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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