"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CFunction = void 0;
const child_process = require("child_process");
const crypto_1 = require("crypto");
const os_1 = require("os");
const path_1 = require("path");
const process_1 = require("process");
try {
    /* eslint-disable import/no-extraneous-dependencies */
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    require.resolve('esbuild/package.json');
    /* eslint-enable import/no-extraneous-dependencies */
}
catch (e) {
    throw new Error('"esbuild" must be installed on the system');
}
// we are taking a "runtime peer" dependency on esbuild
// eslint-disable-next-line import/no-extraneous-dependencies
const esbuild = require("esbuild");
const fs_extra_1 = require("fs-extra");
/**
 * @stability stable
 */
class CFunction {
    /**
     * @stability stable
     */
    constructor(props) {
        var _a, _b, _c;
        const outdir = fs_extra_1.mkdtempSync(path_1.join(os_1.tmpdir(), '.cf.out.'));
        this.workdir = fs_extra_1.mkdtempSync(path_1.join(os_1.tmpdir(), '.cf.work.'));
        this.env = {};
        const workdir = this.workdir;
        // deploy runtime
        const runtimesrc = path_1.dirname(require.resolve(path_1.join(__dirname, '..', 'resources', 'cfruntime.js')));
        fs_extra_1.copySync(runtimesrc, workdir);
        const outfile = path_1.resolve(outdir, 'cf.js');
        const epcode = new Array();
        this.captureSymbols((_a = props.capture) !== null && _a !== void 0 ? _a : {}, epcode);
        const code = (_b = props.code) !== null && _b !== void 0 ? _b : 'true';
        epcode.push(`const output = ${code};`);
        epcode.push('module.exports = require(\'./cfruntime\').wrap(output);');
        const entrypoint = path_1.join(workdir, 'entrypoint.js');
        fs_extra_1.writeFileSync(entrypoint, epcode.join('\n'));
        const restore = process.cwd();
        process_1.chdir(workdir);
        try {
            const result = esbuild.buildSync({
                bundle: true,
                entryPoints: [path_1.relative(workdir, entrypoint)],
                outfile,
                platform: 'node',
                logLevel: 'silent',
            });
            if ((_c = result.warnings) === null || _c === void 0 ? void 0 : _c.length) {
                throw new Error('esbuild warnings');
            }
        }
        catch (e) {
            throw new Error(`esbuild failed at ${workdir} ${e.message}`);
        }
        finally {
            process_1.chdir(restore);
        }
        this.outfile = outfile;
    }
    /**
     * @stability stable
     */
    static exec(file, options = {}) {
        var _a;
        try {
            const substrate = `require('${file}')().catch(e => { console.error(e.stack); process.exit(1); }).then(o => console.log(JSON.stringify(o)))`;
            const child = child_process.spawnSync(process.execPath, ['-e', substrate], {
                stdio: ['inherit', 'pipe', 'pipe'],
                env: (_a = options.env) !== null && _a !== void 0 ? _a : {},
            });
            const stderr = child.stderr.toString('utf-8');
            if (child.status !== 0) {
                throw new Error(`non-exit from child process: ${stderr}`);
            }
            // allow function to emit logs to STDERR (non-streaming)
            if (stderr.length) {
                console.error(stderr);
            }
            // STDOUT is the result
            return JSON.parse(child.stdout.toString());
        }
        catch (e) {
            throw new Error(`unable to execute ${file}: ${e.message}`);
        }
    }
    /**
     * @stability stable
     */
    toJson() {
        return {
            env: this.env,
            js: fs_extra_1.readFileSync(this.outfile, 'utf-8'),
        };
    }
    captureSymbols(symbols, code) {
        for (const [name, value] of Object.entries(symbols)) {
            // if the value is a cfunction, we copy the cf.js file to our workdir and
            // bind the result of "require()"ing it locally.
            if (value instanceof CFunction) {
                const fileName = hashOf(value.outfile) + '.js';
                fs_extra_1.copySync(value.outfile, path_1.join(this.workdir, fileName));
                code.push(`const ${name} = require('./${fileName}');`);
                continue;
            }
            // bind the value through an environment variable (late binding). technically it should be
            // possible to do early-binding for all values that are not CDK tokens but since we are unable
            // to identify tokens at this layer, we have to deffer to late bindings.
            const cvalue = this.captureValue(value);
            const envVar = symbolToEnv(name);
            code.push(`const ${name} = process.env["${envVar}"];`);
            code.push(`if (${name} === undefined) { throw new Error("missing required environment variable '${envVar}'"); }`);
            this.env[envVar] = cvalue;
        }
    }
    captureValue(x) {
        if (typeof (x) === 'function') {
            throw new Error('cannot capture functions yet');
        }
        ;
        if (typeof (x) === 'symbol') {
            throw new Error('cannot capture symbols yet');
        }
        if (Array.isArray(x)) {
            return this.captureArray(x);
        }
        if (typeof (x) === 'object') {
            return this.captureObject(x);
        }
        return this.capturePrimitive(x);
    }
    capturePrimitive(x) {
        return JSON.stringify(x);
    }
    captureObject(o) {
        if (o.constructor.name != 'Object') {
            throw new Error('classes are not supported yet');
        }
        if ('toJson' in o && typeof (o.toJson === 'function')) {
            o = o.toJson();
        }
        const fields = Array();
        for (const [k, v] of Object.entries(o)) {
            fields.push(`"${k}": ${this.captureValue(v)}`);
        }
        return `{ ${fields.join(',')}}`;
    }
    captureArray(a) {
        return '[' + a.map(x => this.captureValue(x)).join(',') + ']';
    }
}
exports.CFunction = CFunction;
/** generate an environment variable for a named symbol */
function symbolToEnv(symbol) {
    return `__CF__${symbol}__`;
}
function hashOf(file) {
    const sha1 = crypto_1.createHash('sha1');
    sha1.update(fs_extra_1.readFileSync(file));
    return sha1.digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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