#!/usr/bin/env python
# -*- coding: utf-8 -*-
# @Time    : 9/13/16 2:29 PM
# @Author  : Runsheng
# @Email   : Runsheng.lee@gmail.com
# @File    : utils.py

"""
Some general functions used for sequence file parser and coding running
"""

# Third oart import
from os.path import exists

from Bio import SeqIO, Seq
from Bio.Blast.Applications import NcbimakeblastdbCommandline


def reverse_complement(seq):
    """
        Given: A DNA string s of length at most 1000 bp.
        Return: The reverse complement sc of s.
        due to the complement_map,
        the symbol such as \n and something else is illegal
        the input need to be pure sequence
    """
    return str(Seq.reverse_complement(seq))

def fasta2dic(fastafile):
    """
    Give a fasta file name, return a dict contains the name and seq
    Require Biopython SeqIO medule to parse the sequence into dict, a large genome may take a lot of RAM
    """
    handle=open(fastafile, "r")
    record_dict=SeqIO.to_dict(SeqIO.parse(handle,"fasta"))
    handle.close()
    return record_dict


def dic2dic(record_dict):
    """
    :param record_dict: a SeqIO dict generated by Biopython
    :return the dict contain {name:seq}
    """
    seq_dict={}
    for k,v in record_dict.items():
        seq=str(v.seq)
        seq_dict[k]=seq.upper()
    return seq_dict

def dic2fasta(record_dict, out):
    with open(out, "w") as fw:
        for k, v in record_dict.items():
            fw.write(">"+k+"\n")
            fw.write(v)
            fw.write("\n")


def tuple_to_pos_str(pos_t):
    """
    reverse of pos_str_to_tuple
    """
    pos_s=[str(x) for x in pos_t]
    chro, start,end=pos_s
    return chro+":"+start+"-"+end


def pos_str_to_tuple(pos_str):
    """
    pos_str: chro:start-end, same as the input for igv or ucsc web browser
    return (chro, start, end), start and end are int
    """
    chro=pos_str.split(":")[0]
    start, end =pos_str.split(":")[1].split("-") # still str
    return (chro, int(start), int(end))


def chr_select(seq_dict, chro, start,end):
    """
    Note the start and end is 0 based
    give the name of refdic, and the chr, start and end to be used
    return the name and sequence
    for example, chrcut(record_dict, "I", 100,109) returns
     ("I:100_109","AAAAAAAAAA")
    """
    name=tuple_to_pos_str((chro, start, end))
    seq=str(seq_dict[chro])[start:end]
    return name,seq


def makeblastdb(genomefile):
    cline = NcbimakeblastdbCommandline(dbtype="nucl",
                                       input_file=genomefile)
    NcbimakeblastdbCommandline(cmd='makeblastdb', dbtype='prot', input_file='NC_005816.faa')
    print(cline)
    cline()


def checkblastdb(genomefile):
    """
    check if the blastdb exist, if not, create one
    """
    dbfile=genomefile+".nsq"
    if exists(dbfile):
        return 0
    else:
        makeblastdb(genomefile)
        return 1
