# Python implementation for MoDE (Multi-objective 2D Embeddings)
## Important modules
- "MoDE": Contains the main class that implements MoDE.
- "metrics": Contains the functions to compute the three metrics introduced in the paper, i.e, distance, correlation, and order preservation metrics.
- "waterfilling_compression": Contains the implementation of waterfilling algorithm.
## Usage
__MoDE__ embeddings can be trained on exact or inexact distance matrices. In the case of inexact distance information, ranges of lower and upper bounds on the distances in the form of seperate lower and upper bound distance matrices should be given to the `fit_transform` function. The resulting embeddings are in 2D dimensions and the data points are placed in the embedding space such that samples with higher scores are placed in higher angles (in polar coordinates).
The following examples shows how to use this package to compute MoDE embeddings for the S curve dataset with
500 data points.
```
import numpy as np
import matplotlib.pyplot as plt
from mpl_toolkits.mplot3d import Axes3D
%matplotlib inline
from MoDE_embeddings.MoDE import MoDE
from sklearn import datasets
n_points = 500
X, color = datasets.make_s_curve(n_points, random_state=0)
mode = MoDE(n_neighbor=10, max_iter=40000, tol=0.001, verbose=True)
x_2d_mode = mode.fit_transform(X, color)
Axes3D
fig = plt.figure(figsize=(8, 5))
ax = fig.add_subplot(121, projection='3d')
ax.scatter(X[:, 0], X[:, 1], X[:, 2], c=color)
ax.view_init(10, -72)
ax.set_xticks([])
ax.set_yticks([])
ax.set_zticks([])
ax = fig.add_subplot(122)
im = ax.scatter(x_2d_mode[:, 0], x_2d_mode[:,1], c=color)
ax.set_xticks([])
ax.set_yticks([])
fig.subplots_adjust(right=0.98)
cbar_ax = fig.add_axes([0.99, 0.15, 0.01, 0.7])
cbar = fig.colorbar(im, cax=cbar_ax)
cbar.set_ticks([-4, 4])
cbar.set_ticklabels(["low score", "high score"])
```
<img src="https://raw.githubusercontent.com/ahmadajal/MoDE/dfeec9059ec883aeb58635ae19312ef911311ecb/Python_implementation/MoDE_s_curve.jpg" alt="mode_image_scurve" width="500">

Once the MoDE embeddings are trained, you can measure the fidelity of the embedded dataset to the original dataset in terms of preserving distances, correlations and orders. To do so, you can use the metric functions available from metrics module.

```
from MoDE_embeddings.metrics import distance_metric, correlation_metric, order_preservation
print("R_d:", distance_metric(X, x_2d_mode, n_neighbor=10))
print("R_c:", correlation_metric(X, x_2d_mode, n_neighbor=10))
print("R_o:", order_preservation(X, mode.P, n_neighbor=10, score=color.squeeze()))
```
```
R_d: 0.8581267537775847
R_c: 0.9915859179657047
R_o: 0.9506003430531732
```

## Waterfilling algorithm (for data compression)
With waterfilling algorithm you can find tight lower and upper bounds on the pair-wise distances between data points that have been compressed using orthonormal
transforms, e.g, fourier transform. Using the `WaterfillingCompression` class you can compress the data by keeping only a small portion of fourier transform
coefficients. Then by calling the `compute_distance_bounds` method you are able to compute tight lower and upper bounds on pair-wise distances. For more information
on the waterfilling algorithm check out the paper: https://arxiv.org/pdf/1405.5873.pdf
```
from MoDE_embeddings.waterfilling_compression import WaterfillingCompression
comp = WaterfillingCompression(num_coeffs=4, coeffs_to_keep='optimal')
dm_ub, dm_lb = comp.compute_distance_bounds(data)
```
