from ..datatypes import *


class Pool(object):
    def __init__(self):
        self.auth = None
        self.api = None
        self.storage = None
        self.websocket = None

        self.pool_id = None
        self.data_object = None

        self.x_data_handler = None
        self.y_data_handler = None

        self.return_x = False

    def attach(self, auth=None, api=None, storage=None, websocket=None):
        '''attach all required modules to class object'''
        self.auth = auth
        self.api = api
        self.storage = storage
        self.websocket = websocket

    def select_pool(self, pool_name=None, pool_key=None, return_x=False):
        '''select pool to work in, with either pool_name or pool_key'''
        if not self.auth:
            raise UserWarning(
                "No auth object attached.")

        query = """query ListPools(
            $filter: ModelPoolFilterInput
            $limit: Int
            $nextToken: String
        ) {
            listPools(filter: $filter, limit: $limit, nextToken: $nextToken) {
            items {
                id
                title
                privateKey
                catagory {
                    catagory
                    xtype {
                    data
                    }
                    ytype {
                    data
                    }
                }
            }
            }
        }"""

        if pool_key:
            params = {
                "filter": {"privateKey": {"eq": pool_key}, "owner": {"eq": self.auth.username}}
            }
        elif pool_name:
            params = {
                "filter": {"title": {"eq": pool_name}, "owner": {"eq": self.auth.username}}
            }
        response = self.api.execute_gql(query, params)

        try:
            data = response['data']['listPools']['items']
        except:
            raise UserWarning(
                response['errors'][0]['message'])

        if len(data) == 0:
            raise UserWarning('No pools found')
        if len(data) > 1:
            raise UserWarning(
                'multiple pools found, please use pool key instead')

        self.return_x = return_x
        self.data_object = data[0]
        self.pool_id = self.data_object['id']
        self.pool_key = self.data_object['privateKey']

        self.x_data_handler = self.get_data_handler(
            self.data_object['catagory']['xtype']['data'])
        self.y_data_handler = self.get_data_handler(
            self.data_object['catagory']['ytype']['data'])

    def get_backlog(self, limit=100):
        '''get backlog of already completed samples'''
        if not self.auth:
            raise UserWarning(
                "No auth object attached.")

        if not self.pool_id:
            raise UserWarning(
                "No pool selected")

        if self.return_x:
            query = """query ListSamples(
                        $filter: ModelsampleFilterInput
                        $limit: Int
                        $nextToken: String
                    ) {
                        listSamples(filter: $filter, limit: $limit, nextToken: $nextToken) {
                        items {
                            id
                            x
                            y
                        }
                        nextToken
                        }
                    }"""
        else:
            query = """query ListSamples(
                        $filter: ModelsampleFilterInput
                        $limit: Int
                        $nextToken: String
                    ) {
                        listSamples(filter: $filter, limit: $limit, nextToken: $nextToken) {
                        items {
                            id
                            y
                        }
                        nextToken
                        }
                    }"""

        params = {
            "filter": {"poolID": {"eq": self.pool_id}, "labeledStatus": {"eq": "COMPLETED"}},
            "limit": limit
        }

        # print(self.pool_id)

        response = self.api.execute_gql(query, params)

        try:
            data = response['data']['listSamples']['items']
        except:
            raise UserWarning(
                response['errors'][0]['message'])

        if len(data) == 0:
            raise UserWarning('No Samples found')

        if response['data']['listSamples']['nextToken'] and len(data) == limit:
            print('There are more samples available, try increasing the limit')

        if self.return_x:
            return {sample['id']: (self.x_data_handler.get(sample['x']),
                                   self.y_data_handler.get(sample['y'])) for sample in data}
        else:
            return {sample['id']: self.y_data_handler.get(sample['y']) for sample in data}

    def put(self, x, y):
        '''select pool to work in, with either pool_name or pool_key'''
        if not self.auth:
            raise UserWarning(
                "No auth object attached.")

        if not self.pool_id:
            raise UserWarning(
                "No pool selected")

        query = """mutation PutSample($key: String!, $input: sampleXY!) {
            putSample(key: $key, input: $input) {
            id
            }
        }"""

        params = {
            "key": self.pool_key,
            "input": {"x": self.x_data_handler.put(x), "y": self.y_data_handler.put(y)}
        }

        response = self.api.execute_gql(query, params)

        try:
            return response['data']['putSample']['id']
        except:
            raise UserWarning(
                response['errors'][0]['message'])

    def connect(self):

        if not self.pool_id:
            raise UserWarning(
                "No pool selected")

        if not self.auth:
            raise UserWarning(
                "No auth object attached.")

        self.websocket.connect(self.pool_id)

    def disconnect(self):
        self.websocket.disconnect()

    def query(self):
        '''query websocket and convert results to correct format'''
        if not self.auth:
            raise UserWarning(
                "No auth object attached.")

        if not self.websocket.ws:
            raise UserWarning(
                "Not connected to pool")

        data = self.websocket.query()
        if self.return_x:
            return {sample['id']: (self.x_data_handler.get(sample['x']),
                                   self.y_data_handler.get(sample['y'])) for sample in data}
        else:
            return {sample['id']: self.y_data_handler.get(sample['y']) for sample in data}

    def get_data_handler(self, datatype):
        '''will map output to correct format given by the data-types'''
        switcher = {
            'Text': Text,
            'Boolean': Boolean,
            'Image': Image,
            'Gif': Gif
        }

        res = switcher.get(datatype, 'invalid')

        if res == 'invalid':
            raise NotImplementedError(f'{datatype} not implemented yet')

        return res(self.storage)
