# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aioambient']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.7.4,<4.0.0',
 'python-engineio>=3.13.1,<5.0.0',
 'python-socketio>=4.6,<6.0',
 'websockets>=8.1,<9.0']

setup_kwargs = {
    'name': 'aioambient',
    'version': '1.2.4',
    'description': 'A clean, async-friendly library for the Ambient Weather API',
    'long_description': '# 🌤  aioambient: An async library for Ambient Weather Personal Weather Stations\n\n[![CI](https://github.com/bachya/aioambient/workflows/CI/badge.svg)](https://github.com/bachya/aioambient/actions)\n[![PyPi](https://img.shields.io/pypi/v/aioambient.svg)](https://pypi.python.org/pypi/aioambient)\n[![Version](https://img.shields.io/pypi/pyversions/aioambient.svg)](https://pypi.python.org/pypi/aioambient)\n[![License](https://img.shields.io/pypi/l/aioambient.svg)](https://github.com/bachya/aioambient/blob/master/LICENSE)\n[![Code Coverage](https://codecov.io/gh/bachya/aioambient/branch/dev/graph/badge.svg)](https://codecov.io/gh/bachya/aioambient)\n[![Maintainability](https://api.codeclimate.com/v1/badges/81a9f8274abf325b2fa4/maintainability)](https://codeclimate.com/github/bachya/aioambient/maintainability)\n[![Say Thanks](https://img.shields.io/badge/SayThanks-!-1EAEDB.svg)](https://saythanks.io/to/bachya)\n\n`aioambient` is a Python3, asyncio-driven library that interfaces with both the\nREST and Websocket APIs provided by\n[Ambient Weather](https://ambientweather.net).\n\n# Installation\n\n```python\npip install aioambient\n```\n\n# Python Versions\n\n`aioambient` is currently supported on:\n\n* Python 3.7\n* Python 3.8\n* Python 3.9\n\n# API and Application Keys\n\nUtilizing `aioambient` requires both an Application Key and an API Key from\nAmbient Weather. You can generate both from the Profile page in your\n[Ambient Weather Dashboard](https://dashboard.ambientweather.net).\n\n# Usage\n\n## Creating a Client\n\nAn `aioambient` client starts with an\n[aiohttp](https://aiohttp.readthedocs.io/en/stable/) `ClientSession`:\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom aioambient import Client\n\n\nasync def main() -> None:\n    """Create the aiohttp session and run the example."""\n    async with ClientSession() as session:\n        # YOUR CODE HERE\n\n\nasyncio.get_event_loop().run_until_complete(main())\n```\n\nCreate a client, initialize it, then get to it:\n\n```python\nimport asyncio\nfrom datetime import date\n\nfrom aiohttp import ClientSession\n\nfrom aioambient import Client\n\n\nasync def main() -> None:\n    """Create the aiohttp session and run the example."""\n    async with ClientSession() as session:\n        client = Client("<YOUR API KEY>", "<YOUR APPLICATION KEY>", session)\n\n        # Get all devices in an account:\n        await client.api.get_devices()\n\n        # Get all stored readings from a device:\n        await client.api.get_device_details("<DEVICE MAC ADDRESS>")\n\n        # Get all stored readings from a device (starting at a date):\n        await client.api.get_device_details(\n            "<DEVICE MAC ADDRESS>", end_date=date(2019, 1, 16)\n        )\n\n\nasyncio.get_event_loop().run_until_complete(main())\n```\n\n## REST API\n\n```python\nimport asyncio\nfrom datetime import date\n\nfrom aiohttp import ClientSession\n\nfrom aioambient import Client\n\n\nasync def main() -> None:\n    """Create the aiohttp session and run the example."""\n    client = Client("<YOUR API KEY>", "<YOUR APPLICATION KEY>")\n\n    # Get all devices in an account:\n    await client.api.get_devices()\n\n    # Get all stored readings from a device:\n    await client.api.get_device_details("<DEVICE MAC ADDRESS>")\n\n    # Get all stored readings from a device (starting at a datetime):\n    await client.api.get_device_details(\n        "<DEVICE MAC ADDRESS>", end_date=date(2019, 1, 16)\n    )\n\n\nasyncio.run(main())\n```\n\nBy default, the library creates a new connection to Ambient Weather with each coroutine.\nIf you are calling a large number of coroutines (or merely want to squeeze out every\nsecond of runtime savings possible), an\n[`aiohttp`](https://github.com/aio-libs/aiohttp) `ClientSession` can be used for connection\npooling:\n\n```python\nimport asyncio\nfrom datetime import date\n\nfrom aiohttp import ClientSession\n\nfrom aioambient import Client\n\n\nasync def main() -> None:\n    """Create the aiohttp session and run the example."""\n    async with ClientSession() as session:\n        client = Client("<YOUR API KEY>", "<YOUR APPLICATION KEY>", session=session)\n\n        # Get all devices in an account:\n        await client.api.get_devices()\n\n        # Get all stored readings from a device:\n        await client.api.get_device_details("<DEVICE MAC ADDRESS>")\n\n        # Get all stored readings from a device (starting at a datetime):\n        await client.api.get_device_details(\n            "<DEVICE MAC ADDRESS>", end_date=date(2019, 1, 16)\n        )\n\n\nasyncio.run(main())\n```\n\nPlease be aware of Ambient Weather\'s\n[rate limiting policies](https://ambientweather.docs.apiary.io/#introduction/rate-limiting).\n\n## Websocket API\n\n```python\nimport asyncio\n\nfrom aiohttp import ClientSession\n\nfrom aioambient import Client\n\n\nasync def main() -> None:\n    """Create the aiohttp session and run the example."""\n    async with ClientSession() as session:\n        client = Client("<YOUR API KEY>", "<YOUR APPLICATION KEY>", session)\n\n        # Define a method that should be fired when the websocket client\n        # connects:\n        def connect_method():\n            """Print a simple "hello" message."""\n            print("Client has connected to the websocket")\n\n        client.websocket.on_connect(connect_method)\n\n        # Alternatively, define a coroutine handler:\n        async def connect_coroutine():\n            """Waits for 3 seconds, then print a simple "hello" message."""\n            await asyncio.sleep(3)\n            print("Client has connected to the websocket")\n\n        client.websocket.async_on_connect(connect_coroutine)\n\n        # Define a method that should be run upon subscribing to the Ambient\n        # Weather cloud:\n        def subscribed_method(data):\n            """Print the data received upon subscribing."""\n            print(f"Subscription data received: {data}")\n\n        client.websocket.on_subscribed(subscribed_method)\n\n        # Alternatively, define a coroutine handler:\n        async def subscribed_coroutine(data):\n            """Waits for 3 seconds, then print the incoming data."""\n            await asyncio.sleep(3)\n            print(f"Subscription data received: {data}")\n\n        client.websocket.async_on_subscribed(subscribed_coroutine)\n\n        # Define a method that should be run upon receiving data:\n        def data_method(data):\n            """Print the data received."""\n            print(f"Data received: {data}")\n\n        client.websocket.on_data(data_method)\n\n        # Alternatively, define a coroutine handler:\n        async def data_coroutine(data):\n            """Wait for 3 seconds, then print the data received."""\n            await asyncio.sleep(3)\n            print(f"Data received: {data}")\n\n        client.websocket.async_on_data(data_coroutine)\n\n        # Define a method that should be run when the websocket client\n        # disconnects:\n        def disconnect_method(data):\n            """Print a simple "goodbye" message."""\n            print("Client has disconnected from the websocket")\n\n        client.websocket.on_disconnect(disconnect_method)\n\n        # Alternatively, define a coroutine handler:\n        async def disconnect_coroutine(data):\n            """Wait for 3 seconds, then print a simple "goodbye" message."""\n            await asyncio.sleep(3)\n            print("Client has disconnected from the websocket")\n\n        client.websocket.async_on_disconnect(disconnect_coroutine)\n\n        # Connect to the websocket:\n        await client.websocket.connect()\n\n        # At any point, disconnect from the websocket:\n        await client.websocket.disconnect()\n\n\nasyncio.run(main())\n```\n\n# Contributing\n\n1. [Check for open features/bugs](https://github.com/bachya/aioambient/issues)\n  or [initiate a discussion on one](https://github.com/bachya/aioambient/issues/new).\n2. [Fork the repository](https://github.com/bachya/aioambient/fork).\n3. (_optional, but highly recommended_) Create a virtual environment: `python3 -m venv .venv`\n4. (_optional, but highly recommended_) Enter the virtual environment: `source ./venv/bin/activate`\n5. Install the dev environment: `script/setup`\n6. Code your new feature or bug fix.\n7. Write tests that cover your new functionality.\n8. Run tests and ensure 100% code coverage: `script/test`\n9. Update `README.md` with any new documentation.\n10. Add yourself to `AUTHORS.md`.\n11. Submit a pull request!\n',
    'author': 'Aaron Bach',
    'author_email': 'bachya1208@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/bachya/aioambient',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
